<?php

declare(strict_types=1);

/**
 * Class Validators
 *
 * A library of reusable, static validation functions for both PHP and JS.
 * Messages are designed to be friendly and non-technical.
 */
class Validators
{
    /**
     * Checks if a value is not empty.
     * @param mixed $value
     * @return bool
     */
    public static function required($value): bool
    {
        return !empty(trim((string)$value));
    }

    /**
     * Validates an email address.
     * @param string $value
     * @return bool
     */
    public static function email(string $value): bool
    {
        return filter_var($value, FILTER_VALIDATE_EMAIL) !== false;
    }

    /**
     * Checks if a value is numeric.
     * @param mixed $value
     * @return bool
     */
    public static function numeric($value): bool
    {
        return is_numeric($value);
    }

    /**
     * Checks if a string's length is not more than a maximum.
     * @param string $value
     * @param int $max
     * @return bool
     */
    public static function maxlength(string $value, int $max): bool
    {
        return mb_strlen($value) <= $max;
    }

    /**
     * Checks if a string's length is not less than a minimum.
     * @param string $value
     * @param int $min
     * @return bool
     */
    public static function minlength(string $value, int $min): bool
    {
        return mb_strlen($value) >= $min;
    }

    /**
     * Checks if a value is within a list of allowed options (for enums).
     * @param string $value
     * @param array $options
     * @return bool
     */
    public static function allowed(string $value, array $options): bool
    {
        return in_array($value, $options, true);
    }
}