// --- PhoenixForms v1 (forms.js) ---
// Enhanced with cyberpunk styling and better validation

document.addEventListener('DOMContentLoaded', () => {
    const form = document.querySelector('.phoenix-form');
    if (form) {
        applyLiveValidation(form);

        form.addEventListener('submit', (e) => {
            e.preventDefault();
            if (validateForm(form)) {
                showSuccessMessage(form, '¡Formulario validado con éxito! Enviando...');
                // Submit via fetch to mock CRUD API
                submitForm(form);
            } else {
                showErrorMessage(form, 'Por favor corrige los errores antes de continuar.');
                scrollToFirstError(form);
            }
        });
    }
});

function submitForm(form) {
    const table = form.dataset.table;
    const formData = new FormData(form);
    const data = {};
    
    formData.forEach((value, key) => data[key] = value);
    
    // Determine endpoint based on whether we're editing or creating
    const idField = form.querySelector('input[name="id"]');
    const id = idField ? idField.value : null;
    
    const endpoint = id ? 'api/mock_save.php' : 'api/mock_insert.php';
    
    fetch(endpoint, {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            db: localStorage.getItem('currentDb') || 'importer_db',
            table: table,
            data: data,
            id: idField ? idField.value : null
        })
    })
    .then(r => r.json())
    .then(res => {
        if(res.status === 'ok') {
            showSuccessMessage(form, '¡Cambios guardados exitosamente!');
            // Reload the mock CRUD page to refresh the table
            window.location.href = `mockcrud.php?db=${localStorage.getItem('currentDb') || 'importer_db'}&table=${table}`;
        } else {
            showErrorMessage(form, 'Error al guardar: ' + res.message);
        }
    })
    .catch(err => {
        showErrorMessage(form, 'Network error: ' + err.message);
    });
}

/**
 * Attaches validation listeners to all required fields in a form.
 * @param {HTMLFormElement} form
 */
function applyLiveValidation(form) {
    const fields = form.querySelectorAll('[required]');
    fields.forEach(field => {
        field.addEventListener('input', () => validateField(field));
        field.addEventListener('blur', () => validateField(field));
    });
}

/**
 * Validates a single form field and updates its helper text.
 * @param {HTMLElement} el - The input, textarea, or select element.
 * @returns {boolean} - True if valid, false otherwise.
 */
function validateField(el) {
    const helper = el.parentElement.querySelector('.helper');
    let message = '';

    if (el.hasAttribute('required') && el.value.trim() === '') {
        message = 'Este campo es necesario, por favor.';
    } else if (el.type === 'email' && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(el.value)) {
        message = 'Por favor ingresa un correo válido 😊';
    }
    // Add more validation rules here as needed...

    helper.textContent = message;
    return message === '';
}

/**
 * Validates all required fields in the form.
 * @param {HTMLFormElement} form
 * @returns {boolean} - True if the entire form is valid.
 */
function validateForm(form) {
    let isFormValid = true;
    const fields = form.querySelectorAll('[required]');
    fields.forEach(field => {
        if (!validateField(field)) {
            isFormValid = false;
        }
    });
    return isFormValid;
}

/**
 * Scrolls the viewport to the first field with a validation error.
 * @param {HTMLFormElement} form
 */
function scrollToFirstError(form) {
    const firstErrorField = form.querySelector('.helper:not(:empty)');
    if (firstErrorField) {
        firstErrorField.parentElement.scrollIntoView({
            behavior: 'smooth',
            block: 'center'
        });
    }
}

/**
 * Shows a generic success message.
 * @param {HTMLFormElement} form
 * @param {string} message
 */
function showSuccessMessage(form, message) {
    // For now, we'll use a simple alert. A more robust implementation
    // would inject a styled message into the DOM.
    alert(message);
}

/**
 * Shows a generic error message.
 * @param {HTMLFormElement} form
 * @param {string} message
 */
function showErrorMessage(form, message) {
    // For now, we'll use a simple alert.
    alert(message);
}

/**
 * Friendly validation messages.
 */
const validationMessages = {
    required: 'Este campo es necesario, por favor.',
    email: 'Por favor ingresa un correo válido 😊',
    numeric: 'Solo se permiten números aquí.',
};
