<?php
/**
 * tabler_ui.php - The Ledger Whisperer Web Interface
 *
 * Web UI for converting bank statement PDFs to TSV using tabler.php
 *
 * Protocol: tabler.mother.tongue.protocol.v1
 * Doctrine: Miserable-First → Zoom-Out → Reconcile → Export
 *
 * @version 1.0.0
 */

// ============================================================================
// BACKEND PROCESSING (Handle form submission and downloads)
// ============================================================================

session_start();

$conversion_results = [];
$conversion_errors = [];
$zip_file = null;

// Create persistent temp directory for this session
$persistent_temp_dir = sys_get_temp_dir() . '/tabler_ui_persistent_' . session_id();
if (!is_dir($persistent_temp_dir)) {
    mkdir($persistent_temp_dir, 0755, true);
}

// Handle ZIP download request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['download_zip'])) {
    $zip_filename = $_POST['download_zip'];
    $zip_path = sys_get_temp_dir() . '/tabler_ui_' . getmypid() . '/' . $zip_filename;
    
    if (file_exists($zip_path)) {
        // Set headers for file download
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . basename($zip_path) . '"');
        header('Content-Length: ' . filesize($zip_path));
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        // Read and output the file
        readfile($zip_path);
        exit;
    } else {
        die('ZIP file not found.');
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'convert') {
    // Validate uploaded files
    if (!isset($_FILES['pdf_files']) || !is_array($_FILES['pdf_files']['name'])) {
        $conversion_errors[] = "No files uploaded or upload error";
    } else {
        $uploaded_files = $_FILES['pdf_files'];
        
        // Validate PDF files
        $allowed_extensions = ['pdf'];
        $temp_dir = sys_get_temp_dir() . '/tabler_ui_' . getmypid();
        if (!is_dir($temp_dir)) {
            mkdir($temp_dir, 0755, true);
        }
        
        // Process each file sequentially
        for ($i = 0; $i < count($uploaded_files['name']); $i++) {
            // Skip empty files
            if ($uploaded_files['error'][$i] !== UPLOAD_ERR_OK) {
                continue;
            }
            
            $file_name = $uploaded_files['name'][$i];
            $file_tmp_name = $uploaded_files['tmp_name'][$i];
            $file_size = $uploaded_files['size'][$i];
            
            // Validate PDF
            $extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            
            if (!in_array($extension, $allowed_extensions)) {
                $conversion_errors[] = "File '$file_name' is not a PDF";
                continue;
            } elseif ($file_size > 50 * 1024 * 1024) {
                $conversion_errors[] = "File '$file_name' is too large (max 50MB)";
                continue;
            }
            
            // Move uploaded file to temp location
            $temp_pdf = $temp_dir . '/input_' . $i . '.pdf';
            move_uploaded_file($file_tmp_name, $temp_pdf);
            
            // Determine output path
            $output_filename = !empty($_POST['output_filename'])
                ? $_POST['output_filename']
                : pathinfo($file_name, PATHINFO_FILENAME) . '.txt';

            // For multiple files, ensure unique output names
            if (count($uploaded_files['name']) > 1) {
                $output_filename = pathinfo($file_name, PATHINFO_FILENAME) . '.txt';
            }

            $output_path = $temp_dir . '/' . $output_filename;
            // Also save to persistent directory for later copying
            $persistent_output_path = $persistent_temp_dir . '/' . $output_filename;
            $audit_path = !empty($_POST['generate_audit']) ? $temp_dir . '/audit_' . $i . '.json' : null;
            
            // Build command
            $cmd = "php " . escapeshellarg(__DIR__ . '/tabler.php');
            $cmd .= " " . escapeshellarg($temp_pdf);
            $cmd .= " " . escapeshellarg($output_path);
            
            if (!empty($_POST['language'])) {
                $cmd .= " --lang=" . escapeshellarg($_POST['language']);
            }
            
            if (!empty($_POST['generate_debug'])) {
                $cmd .= " --debug";
            }
            
            if ($audit_path) {
                $cmd .= " --audit=" . escapeshellarg($audit_path);
            }
            
            // Execute conversion
            $descriptors = [
                0 => ["pipe", "r"],
                1 => ["pipe", "w"],
                2 => ["pipe", "w"],
            ];
            
            $process = proc_open($cmd, $descriptors, $pipes);
            
            if (is_resource($process)) {
                // Close input pipe
                fclose($pipes[0]);
                
                // Read output
                $stdout = stream_get_contents($pipes[1]);
                fclose($pipes[1]);
                
                // Read stderr
                $stderr = stream_get_contents($pipes[2]);
                fclose($pipes[2]);
                
                // Wait for process
                $return_value = proc_close($process);
                
                if ($return_value === 0 && file_exists($output_path)) {
                    // Read output file
                    $output_content = file_get_contents($output_path);

                    // Copy to persistent directory for later use
                    copy($output_path, $persistent_output_path);

                    $lines = explode("\n", trim($output_content));

                    // Parse TSV
                    $headers = [];
                    $rows = [];

                    foreach ($lines as $index => $line) {
                        $fields = explode("\t", $line);

                        if ($index === 0) {
                            $headers = $fields;
                        } else {
                            $rows[] = $fields;
                        }
                    }

                    $conversion_results[] = [
                        'input_filename' => $file_name,
                        'output_filename' => $output_filename,
                        'output_path' => $output_path,
                        'persistent_path' => $persistent_output_path,
                        'row_count' => count($rows),
                        'headers' => $headers,
                        'rows' => $rows,
                        'audit' => ($audit_path && file_exists($audit_path))
                            ? json_decode(file_get_contents($audit_path), true)
                            : null,
                        'stdout' => $stdout,
                        'stderr' => $stderr,
                    ];
                } else {
                    $conversion_errors[] = "Conversion failed for '$file_name' (exit code: $return_value)\n\nSTDERR:\n$stderr";
                }
            } else {
                $conversion_errors[] = "Could not execute tabler.php for '$file_name'";
            }
            
            // Cleanup temp files
            @unlink($temp_pdf);
            @unlink($output_path);
            if ($audit_path) {
                @unlink($audit_path);
            }
        }
        
        // If we have results and zip option is selected, create zip file
    if (!empty($conversion_results) && !empty($_POST['generate_zip'])) {
        $zip_file = $temp_dir . '/converted_files.zip';
        $zip = new ZipArchive();
        
        if ($zip->open($zip_file, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
            foreach ($conversion_results as $result) {
                // Read the file content and add it to zip
                if (file_exists($result['output_path'])) {
                    $content = file_get_contents($result['output_path']);
                    $zip->addFromString($result['output_filename'], $content);
                }
            }
            $zip->close();
        }
    }
}
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>⚡ tabler_ui - The Ledger Whisperer</title>

<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monospace', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    .panel.cathedral {
        border: 2px solid #00ff88;
        box-shadow: 0 0 40px #00ff88aa;
        animation: cathedral-pulse 2s ease-in-out infinite;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    @keyframes cathedral-pulse {
        0%, 100% { box-shadow: 0 0 40px #00ff88aa; }
        50% { box-shadow: 0 0 60px #00ff88dd, 0 0 80px #00ff8855; }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    /* Input Styling */
    input[type="text"], input[type="number"], input[type="file"], select {
        background: #000;
        color: #00eaff;
        border: 1px solid #00eaff55;
        padding: 8px 12px;
        border-radius: 4px;
        font-family: 'Consolas', monospace;
        transition: all 0.3s ease;
    }

    input[type="file"] {
        padding: 6px;
    }

    input[type="checkbox"] {
        cursor: pointer;
        width: 18px;
        height: 18px;
        accent-color: #00eaff;
    }

    label {
        color: #00ffaa;
        font-weight: bold;
        margin-right: 10px;
    }

    /* Button Styling */
    button {
        background: #00eaff;
        font-size: 16px;
        border-radius: 8px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 14px #00eaffcc;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }

    button:hover {
        transform: scale(1.05);
        box-shadow: 0 0 22px #00eaff;
    }

    button:active {
        transform: scale(0.98);
    }

    button.convert-btn {
        background: linear-gradient(135deg, #00eaff 0%, #00ff88 100%);
        box-shadow: 0 0 20px #00ff88cc;
    }

    button.convert-btn:hover {
        box-shadow: 0 0 30px #00ff88;
    }

    /* Table Styling */
    .table-wrap {
        overflow-x: auto;
        margin-top: 15px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
    }

    th, td {
        border: 1px solid #00eaff33;
        padding: 10px 12px;
        text-align: left;
    }

    th {
        background: rgba(0,234,255,0.15);
        text-shadow: 0 0 6px #00eaff;
        font-weight: bold;
    }

    tr:hover td {
        background: rgba(0,234,255,0.08);
        transition: background 0.2s ease;
    }

    /* Status Messages */
    .status {
        margin-top: 15px;
        padding: 10px;
        border-radius: 6px;
        animation: fadein 0.4s ease;
    }

    .status.success {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
    }

    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }

    .status.info {
        color: #00eaff;
        text-shadow: 0 0 8px #00eaff;
    }

    /* Scrollbar Styling */
    ::-webkit-scrollbar {
        width: 10px;
        height: 10px;
    }

    ::-webkit-scrollbar-track {
        background: #000;
    }

    ::-webkit-scrollbar-thumb {
        background: #00eaff55;
        border-radius: 5px;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }

    /* Form groups */
    .form-group {
        margin-bottom: 20px;
    }

    .form-row {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
    }

    .form-col {
        flex: 1;
    }

    /* Options panel */
    .options-panel {
        background: rgba(0,0,0,0.5);
        border: 1px solid #00eaff22;
        border-radius: 8px;
        padding: 15px;
        margin-top: 20px;
    }

    .options-panel h4 {
        color: #00eaff;
        margin-top: 0;
        margin-bottom: 15px;
        font-size: 14px;
    }

    /* Audit panel */
    .audit-panel {
        background: rgba(0,0,0,0.5);
        border: 1px solid #ffaa0022;
        border-radius: 8px;
        padding: 15px;
        margin-top: 20px;
    }

    .audit-panel h4 {
        color: #ffaa00;
        margin-top: 0;
        margin-bottom: 15px;
        font-size: 14px;
    }

    .audit-json {
        background: #000;
        color: #ffaa00;
        border: 1px solid #ffaa0055;
        padding: 15px;
        border-radius: 8px;
        font-family: 'Consolas', monospace;
        font-size: 12px;
        white-space: pre-wrap;
        overflow-x: auto;
        max-height: 400px;
        overflow-y: auto;
    }

    /* Reconciliation status */
    .recon-badge {
        display: inline-block;
        padding: 4px 10px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: bold;
    }

    .recon-badge.valid {
        background: rgba(0,255,136,0.2);
        color: #00ff88;
        border: 1px solid #00ff88;
    }

    .recon-badge.invalid {
        background: rgba(255,51,102,0.2);
        color: #ff3366;
        border: 1px solid #ff3366;
    }

    /* Loading spinner */
    .spinner {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 2px solid #00eaff;
        border-top-color: transparent;
        border-radius: 50%;
        animation: spin 1s linear infinite;
        vertical-align: middle;
        margin-right: 10px;
    }

    @keyframes spin {
        to { transform: rotate(360deg); }
    }

    /* Download button styling */
    .download-btn {
        display: inline-block;
        background: linear-gradient(135deg, #00ff88 0%, #00eaff 100%);
        color: #000;
        padding: 8px 16px;
        border-radius: 6px;
        text-decoration: none;
        font-weight: bold;
        font-size: 14px;
        margin: 5px 5px 5px 0;
        transition: all 0.3s ease;
        border: 1px solid #00ff88;
        box-shadow: 0 0 12px #00ff8866;
    }

    .download-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 0 20px #00ff88;
        background: linear-gradient(135deg, #00ffaa 0%, #00ecff 100%);
    }

    .download-btn:active {
        transform: translateY(0);
    }

    .download-section {
        background: rgba(0,0,0,0.3);
        border: 1px solid #00eaff33;
        border-radius: 8px;
        padding: 15px;
        margin-top: 20px;
    }

    .download-section h3 {
        color: #00ff88;
        margin-top: 0;
        margin-bottom: 15px;
        font-size: 16px;
    }

    .download-info {
        color: #00eaff;
        font-size: 12px;
        margin-left: 10px;
    }

.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.85);
    z-index: 1000;
    overflow: auto;
    animation: fadein 0.3s ease;
}

.modal-content {
    position: relative;
    background: linear-gradient(135deg, rgba(0,13,24,0.95) 0%, rgba(0,0,0,0.95) 100%);
    border: 2px solid #00eaff;
    border-radius: 12px;
    padding: 30px;
    margin: 50px auto;
    max-width: 800px;
    box-shadow: 0 0 50px #00eaffaa, 0 0 100px #00eaff44;
    animation: slideDown 0.4s ease;
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-30px); }
    to { opacity: 1; transform: translateY(0); }
}

.directory-item {
    padding: 12px 15px;
    margin: 8px 0;
    border-radius: 8px;
    background: rgba(0,234,255,0.08);
    border: 1px solid rgba(0,234,255,0.2);
    cursor: pointer;
    transition: all 0.2s ease;
}

.directory-item:hover {
    background: rgba(0,234,255,0.18);
    border-color: rgba(0,234,255,0.4);
    transform: translateX(5px);
    box-shadow: 0 0 15px rgba(0,234,255,0.3);
}

.directory-path {
    color: #00ffaa;
    font-size: 12px;
}

.close {
    position: absolute;
    top: 15px;
    right: 20px;
    font-size: 32px;
    font-weight: bold;
    cursor: pointer;
    color: #ff3366;
    line-height: 1;
    transition: all 0.3s ease;
}

.close:hover {
    color: #ff5588;
    text-shadow: 0 0 10px #ff3366;
    transform: rotate(90deg);
}
</style>

</head>
<body>

<!-- GATE 1: THE ARRIVAL -->
<div class="panel">
    <a href="index.php" class="back-link">&#8592; Return to Threshold</a>
    <h2>&#128666; Gate 1: The Arrival &mdash; Upload Bank Statement PDF</h2>
    <p style="color: #00ffaa; font-size: 14px; margin-bottom: 15px;">
        Upload a bank statement PDF (text-based or scanned) to convert to TSV.<br>
        <span style="color: #00eaff; font-size: 12px;">
            Protocol: <strong>tabler.mother.tongue.protocol.v1</strong> &bull; 
            Doctrine: <strong>Miserable-First &#8594; Zoom-Out &#8594; Reconcile &#8594; Export</strong>
        </span>
    </p>

    <?php if (!empty($conversion_errors)): ?>
    <div class="status error">
        <?php foreach ($conversion_errors as $error): ?>
        &#9888; <?php echo htmlspecialchars($error); ?><br>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>

    <form method="POST" enctype="multipart/form-data" id="uploadForm">
        <input type="hidden" name="action" value="convert" />

        <div class="form-row">
            <div class="form-col">
                <div class="form-group">
                    <label>Bank Statement (PDF):</label><br>
                    <div id="file-drop-area" class="file-drop-area">
                        <input id="pdf_files" name="pdf_files[]" type="file" accept=".pdf" multiple />
                        <div class="drop-content">
                            <span class="drop-text">Drag & drop PDF files here or click to select</span>
                            <span class="drop-subtext">Supports multiple files</span>
                        </div>
                    </div>
                    <div id="file-list" class="file-list"></div>
                </div>
            </div>
            <div class="form-col">
                <div class="form-group">
                    <label for="output_filename">Output Filename:</label><br>
                    <input id="output_filename" name="output_filename" type="text" placeholder="statement.txt" />
                    <p style="color: #00ffaa; font-size: 12px; margin-top: 5px;">
                        (defaults to input basename + .txt)
                    </p>
                </div>
            </div>
        </div>

        <div class="options-panel">
                <h4>&#9894; Conversion Options</h4>
                
                <div class="form-row">
                    <div class="form-col">
                        <label for="language">Language:</label>
                        <select id="language" name="language">
                            <option value="es">Spanish (es)</option>
                            <option value="en">English (en)</option>
                            <option value="fr">French (fr)</option>
                            <option value="pt">Portuguese (pt)</option>
                        </select>
                    </div>
                    <div class="form-col">
                        <label style="cursor: pointer;">
                            <input type="checkbox" id="generate_debug" name="generate_debug" value="1" />
                            Generate Debug Overlay
                        </label>
                    </div>
                    <div class="form-col">
                        <label style="cursor: pointer;">
                            <input type="checkbox" id="generate_audit" name="generate_audit" value="1" checked />
                            Generate Audit Log
                        </label>
                    </div>
                    <div class="form-col">
                        <label style="cursor: pointer;">
                            <input type="checkbox" id="generate_zip" name="generate_zip" value="1" />
                            Create ZIP Archive
                        </label>
                    </div>
                </div>
            </div>

        <button type="submit" class="convert-btn" onclick="showConverting()">
            &#9886; Convert to TSV
        </button>

        <div id="converting" class="status info" style="display: none;">
            <span class="spinner"></span>
            Converting PDF to TSV... (this may take a moment for scanned PDFs)
        </div>
    </form>
</div>

<!-- PREVIEW PANEL -->
<?php if (!empty($conversion_results)): ?>
<div class="panel cathedral">
    <h2>&#10024; Gate 4: The Cathedral Moment &mdash; Conversion Complete</h2>
    
    <?php if (!empty($conversion_errors)): ?>
    <div class="status error">
        <?php foreach ($conversion_errors as $error): ?>
        &#9888; <?php echo htmlspecialchars($error); ?><br>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>
    
    <div class="status success">
        &#10004; Successfully converted <?php echo count($conversion_results); ?> files
    </div>
    
    <?php foreach ($conversion_results as $index => $result): ?>
    <h3>File: <?php echo htmlspecialchars($result['input_filename']); ?></h3>
    
    <div class="status success">
        &#10004; Converted <?php echo htmlspecialchars($result['input_filename']); ?>
        &#8592; <?php echo htmlspecialchars($result['output_filename']); ?>
        (<?php echo $result['row_count']; ?> rows)
    </div>
    
    <h3>Converted Data (Preview)</h3>
    <div class="table-wrap">
        <table>
            <thead>
                <tr>
                    <?php foreach ($result['headers'] as $header): ?>
                    <th><?php echo htmlspecialchars($header); ?></th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach (array_slice($result['rows'], 0, 20) as $row): ?>
                <tr>
                    <?php foreach ($row as $cell): ?>
                    <td><?php echo htmlspecialchars($cell); ?></td>
                    <?php endforeach; ?>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php if (count($result['rows']) > 20): ?>
    <p style="color: #00eaff; margin-top: 10px;">
        ... and <?php echo count($result['rows']) - 20; ?> more rows
    </p>
    <?php endif; ?>
    
    <h3>Reconciliation Status</h3>
    <?php
    $is_valid = empty($result['audit']['reconciliation_failures']);
    ?>
    <div class="status <?php echo $is_valid ? 'success' : 'error'; ?>">
        <span class="recon-badge <?php echo $is_valid ? 'valid' : 'invalid'; ?>">
            <?php echo $is_valid ? '&#10004; LEDGER VALID' : '&#9888; RECONCILIATION ISSUES'; ?>
        </span>
        <?php if (!$is_valid && !empty($result['audit']['reconciliation_failures'])): ?>
        <br>
        <span style="font-size: 12px; margin-top: 10px; display: block;">
            First failure at row <?php echo $result['audit']['reconciliation_failures'][0]['row'] + 1; ?>
        </span>
        <?php endif; ?>
    </div>
    
    <?php if (!empty($result['audit'])): ?>
    <div class="audit-panel">
        <h4>&#9886; Audit Log (PII Redacted)</h4>
        <div class="audit-json"><?php echo htmlspecialchars(json_encode($result['audit'], JSON_PRETTY_PRINT)); ?></div>
    </div>
    <?php endif; ?>
    
    <?php endforeach; ?>

    <!-- FILE COPY DASHBOARD -->
    <div class="download-section" style="margin-top: 30px;">
        <h3>📁 Copy Files to Folder</h3>
        <p style="color: #00eaff; font-size: 13px; margin-bottom: 15px;">
            Copy your converted files to a specific folder on the server
        </p>

        <div style="display: flex; gap: 15px; align-items: center; margin-bottom: 15px;">
            <button id="copy-to-folder-btn" class="download-btn" style="margin: 0;">
                🗂️ Browse Folders
            </button>
            <div id="selected-destination-display" style="color: #00ff88; font-size: 13px; flex: 1;">
                No folder selected
            </div>
        </div>

        <div style="margin-bottom: 15px;">
            <input type="text" id="selected-directory" readonly placeholder="Selected destination will appear here"
                   style="width: 100%; background: rgba(0,0,0,0.5); cursor: not-allowed;" />
        </div>

        <button id="copy-confirm-btn" class="convert-btn" disabled style="opacity: 0.5; cursor: not-allowed;">
            ⚡ Copy Files Now
        </button>

        <div id="copy-status" style="margin-top: 15px;"></div>
    </div>
</div>

<!-- FOLDER PICKER MODAL -->
<div id="folder-picker-modal" class="modal">
    <div class="modal-content">
        <span class="close">&times;</span>
        <h3 style="margin-top: 0; color: #00eaff;">Select Destination Folder</h3>

        <!-- Breadcrumb navigation -->
        <div id="breadcrumb" style="margin-bottom: 20px; padding: 10px; background: rgba(0,0,0,0.3); border-radius: 6px; color: #00ffaa; font-size: 13px;">
            <span id="breadcrumb-path">/lamp/www</span>
        </div>

        <!-- Directory listing -->
        <div id="directory-list" style="max-height: 400px; overflow-y: auto; margin-bottom: 20px;">
            <div style="text-align: center; padding: 20px; color: #00eaff;">
                <span class="spinner"></span> Loading folders...
            </div>
        </div>

        <div style="display: flex; gap: 10px; justify-content: flex-end;">
            <button id="select-current-folder" class="download-btn">
                ✓ Select This Folder
            </button>
            <button id="cancel-folder-selection" class="download-btn" style="background: rgba(255,51,102,0.2); color: #ff3366; box-shadow: 0 0 12px #ff336666;">
                Cancel
            </button>
        </div>
    </div>
</div>
<?php endif; ?>

<script>
/**
 * Show converting status
 */
function showConverting() {
    var converting = document.getElementById('converting');
    if (converting) {
        converting.style.display = 'block';
    }
}

/**
 * Download output as file
 */
function downloadOutput(filename, content) {
    var blob = new Blob([content], { type: 'text/tab-separated-values' });
    var url = URL.createObjectURL(blob);
    var a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
}

/**
 * Download all files as ZIP
 */
function downloadZip() {
    <?php if ($zip_file && file_exists($zip_file)): ?>
    // Create a form to POST download request
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = 'tabler_ui.php';
    
    const input = document.createElement('input');
    input.type = 'hidden';
    input.name = 'download_zip';
    input.value = '<?php echo basename($zip_file); ?>';
    
    form.appendChild(input);
    document.body.appendChild(form);
    form.submit();
    document.body.removeChild(form);
    <?php else: ?>
    alert('No ZIP file available for download.');
    <?php endif; ?>
}

/**
 * Escape HTML
 */
function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    var div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Auto-fill output filename based on selected PDF file
 */
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('pdf_files').addEventListener('change', function() {
        const outputField = document.getElementById('output_filename');
        const fileList = document.getElementById('file-list');
        fileList.innerHTML = '';
        
        if (this.files.length > 0) {
            // If multiple files, don't auto-fill output filename (user should specify)
            if (this.files.length === 1 && outputField.value === '') {
                const fileName = this.files[0].name;
                const baseName = fileName.replace(/\.pdf$/i, '');
                outputField.value = baseName + '.txt';
            }
            
            // Show list of selected files
            for (let i = 0; i < this.files.length; i++) {
                const file = this.files[i];
                const fileItem = document.createElement('div');
                fileItem.className = 'file-item';
                fileItem.innerHTML = '<span class="file-name">' + file.name + '</span> <span class="file-size">(' + (file.size / 1024).toFixed(1) + ' KB)</span>';
                fileList.appendChild(fileItem);
            }
        }
    });

    // Drag and drop functionality
    const dropArea = document.getElementById('file-drop-area');
    const fileInput = document.getElementById('pdf_files');

    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, preventDefaults, false);
        document.body.addEventListener(eventName, preventDefaults, false);
    });

    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }

    ['dragenter', 'dragover'].forEach(eventName => {
        dropArea.addEventListener(eventName, highlight, false);
    });

    ['dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, unhighlight, false);
    });

    function highlight() {
        dropArea.classList.add('highlight');
    }

    function unhighlight() {
        dropArea.classList.remove('highlight');
    }

    dropArea.addEventListener('drop', handleDrop, false);

    function handleDrop(e) {
        const dt = e.dataTransfer;
        const files = dt.files;
        
        // Clear previous selections
        fileInput.files = files;
        
        // Trigger change event to update file list
        const event = new Event('change');
        fileInput.dispatchEvent(event);
    }

    // Add CSS for drag and drop area
    const style = document.createElement('style');
    style.textContent = `
        .file-drop-area {
            border: 2px dashed #00eaff;
            border-radius: 8px;
            padding: 40px;
            text-align: center;
            cursor: pointer;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            background: rgba(0,234,255,0.05);
        }
        
        .file-drop-area.highlight {
            border-color: #00ff88;
            background: rgba(0,255,136,0.1);
        }
        
        .drop-content {
            margin-bottom: 10px;
        }
        
        .drop-text {
            display: block;
            color: #00eaff;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .drop-subtext {
            color: #00ffaa;
            font-size: 12px;
        }
        
        .file-list {
            max-height: 120px;
            overflow-y: auto;
            margin-bottom: 15px;
            padding: 10px;
            border: 1px solid #00eaff33;
            border-radius: 6px;
            background: rgba(0,0,0,0.2);
        }
        
        .file-item {
            display: flex;
            justify-content: space-between;
            padding: 5px 8px;
            margin: 4px 0;
            border-radius: 4px;
            background: rgba(0,234,255,0.1);
            font-size: 12px;
        }
        
        .file-name {
            color: #00ffaa;
        }
        
        .file-size {
            color: #00eaff;
        }
    `;
    document.head.appendChild(style);

    <?php if (!empty($conversion_results)): ?>
    const resultsContainer = document.querySelector('.cathedral');
    const downloadSection = document.createElement('div');
    downloadSection.className = 'download-section';
    
    const downloadTitle = document.createElement('h3');
    downloadTitle.textContent = 'Download Results';
    downloadSection.appendChild(downloadTitle);
    
    <?php foreach ($conversion_results as $index => $result): ?>
    // Create download item with unique variable name
    const downloadItem<?php echo $index; ?> = document.createElement('div');
    
    const downloadBtn<?php echo $index; ?> = document.createElement('a');
    downloadBtn<?php echo $index; ?>.href = '#';
    downloadBtn<?php echo $index; ?>.className = 'download-btn';
    downloadBtn<?php echo $index; ?>.innerHTML = '&#8595; Download <?php echo htmlspecialchars($result['output_filename']); ?>';
    downloadBtn<?php echo $index; ?>.onclick = function(e) {
        e.preventDefault();
        // Include headers in download
        const headers<?php echo $index; ?> = <?php echo json_encode(implode("\t", $result['headers'])); ?>;
        const rows<?php echo $index; ?> = <?php echo json_encode(implode("\n", array_map(function($r) { return implode("\t", $r); }, $result['rows']))); ?>;
        const content<?php echo $index; ?> = headers<?php echo $index; ?> + "\n" + rows<?php echo $index; ?>;
        downloadOutput('<?php echo htmlspecialchars($result['output_filename'], ENT_QUOTES); ?>', content<?php echo $index; ?>);
    };

    const downloadInfo<?php echo $index; ?> = document.createElement('span');
    downloadInfo<?php echo $index; ?>.className = 'download-info';
    // Calculate size including headers
    const headerSize<?php echo $index; ?> = <?php echo strlen(implode("\t", $result['headers'])) + 1; ?>; // +1 for newline
    const rowsSize<?php echo $index; ?> = <?php echo strlen(implode("\n", array_map(function($r) { return implode("\t", $r); }, $result['rows']))); ?>;
    downloadInfo<?php echo $index; ?>.textContent = '(' + (headerSize<?php echo $index; ?> + rowsSize<?php echo $index; ?>).toLocaleString() + ' bytes)';

    downloadItem<?php echo $index; ?>.appendChild(downloadBtn<?php echo $index; ?>);
    downloadItem<?php echo $index; ?>.appendChild(downloadInfo<?php echo $index; ?>);
    downloadSection.appendChild(downloadItem<?php echo $index; ?>);
    <?php endforeach; ?>
    
    <?php if ($zip_file && file_exists($zip_file)): ?>
        const zipItem = document.createElement('div');
        
        const zipBtn = document.createElement('a');
        zipBtn.href = '#';
        zipBtn.className = 'download-btn';
        zipBtn.innerHTML = '&#8595; Download All Files as ZIP';
        zipBtn.onclick = function(e) {
            e.preventDefault();
            downloadZip();
        };
        
        zipItem.appendChild(zipBtn);
        downloadSection.appendChild(zipBtn);
    <?php endif; ?>
    
    resultsContainer.appendChild(downloadSection);

    // ============================================================================
    // FILE COPY DASHBOARD FUNCTIONALITY
    // ============================================================================

    // Load last used folder from localStorage (default to /lamp/www)
    const lastFolder = localStorage.getItem('tabler_last_folder') || '/lamp/www';
    let currentPath = lastFolder;

    const filesList = [<?php
        $files = [];
        foreach ($conversion_results as $result) {
            $files[] = json_encode($result['output_filename']);
        }
        echo implode(', ', $files);
    ?>];

    // Pre-populate the last used folder on page load
    if (lastFolder !== '/lamp/www') {
        document.getElementById('selected-directory').value = lastFolder;
        document.getElementById('selected-destination-display').innerHTML =
            '✓ Last used: <strong>' + escapeHtml(lastFolder) + '</strong>';

        // Enable copy button since we have a folder selected
        const copyBtn = document.getElementById('copy-confirm-btn');
        copyBtn.disabled = false;
        copyBtn.style.opacity = '1';
        copyBtn.style.cursor = 'pointer';
    }

    // Open folder picker modal
    document.getElementById('copy-to-folder-btn').addEventListener('click', function() {
        document.getElementById('folder-picker-modal').style.display = 'block';
        fetchDirectories(currentPath);
    });

    // Close modal
    document.querySelector('.close').addEventListener('click', function() {
        document.getElementById('folder-picker-modal').style.display = 'none';
    });

    document.getElementById('cancel-folder-selection').addEventListener('click', function() {
        document.getElementById('folder-picker-modal').style.display = 'none';
    });

    // Select current folder
    document.getElementById('select-current-folder').addEventListener('click', function() {
        // Save to localStorage for next time
        localStorage.setItem('tabler_last_folder', currentPath);

        document.getElementById('selected-directory').value = currentPath;
        document.getElementById('selected-destination-display').innerHTML =
            '✓ Selected: <strong>' + escapeHtml(currentPath) + '</strong>';
        document.getElementById('folder-picker-modal').style.display = 'none';

        // Enable copy button
        const copyBtn = document.getElementById('copy-confirm-btn');
        copyBtn.disabled = false;
        copyBtn.style.opacity = '1';
        copyBtn.style.cursor = 'pointer';
    });

    // Fetch and display directories
    function fetchDirectories(path) {
        const directoryList = document.getElementById('directory-list');
        directoryList.innerHTML = '<div style="text-align: center; padding: 20px; color: #00eaff;"><span class="spinner"></span> Loading folders...</div>';

        fetch('api/list_directories.php?path=' + encodeURIComponent(path))
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    directoryList.innerHTML = '<div style="color: #ff3366; padding: 20px; text-align: center;">Error: ' + escapeHtml(data.error) + '</div>';
                    return;
                }

                currentPath = data.current_path;

                // Update breadcrumb
                document.getElementById('breadcrumb-path').textContent = currentPath;

                directoryList.innerHTML = '';

                // Add parent directory option if not at root
                if (data.parent) {
                    const parentElement = document.createElement('div');
                    parentElement.className = 'directory-item';
                    parentElement.style.background = 'rgba(0,234,255,0.15)';
                    parentElement.innerHTML = '<strong>📁 .. (Parent Directory)</strong>';
                    parentElement.addEventListener('click', function() {
                        fetchDirectories(data.parent);
                    });
                    directoryList.appendChild(parentElement);
                }

                // Add subdirectories
                if (data.directories.length === 0 && !data.parent) {
                    directoryList.innerHTML = '<div style="color: #00ffaa; padding: 20px; text-align: center;">No subdirectories found</div>';
                } else {
                    data.directories.forEach(dir => {
                        const dirElement = document.createElement('div');
                        dirElement.className = 'directory-item';
                        dirElement.innerHTML = `
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <span><strong>📁 ${escapeHtml(dir.name)}</strong></span>
                                <span class="directory-path" style="font-size: 11px; opacity: 0.7;">${escapeHtml(dir.path)}</span>
                            </div>
                        `;
                        dirElement.addEventListener('click', function() {
                            fetchDirectories(dir.path);
                        });
                        directoryList.appendChild(dirElement);
                    });
                }
            })
            .catch(error => {
                directoryList.innerHTML = '<div style="color: #ff3366; padding: 20px; text-align: center;">Network error: ' + escapeHtml(error.message) + '</div>';
            });
    }

    // Copy files to selected directory
    document.getElementById('copy-confirm-btn').addEventListener('click', function() {
        const destination = document.getElementById('selected-directory').value;
        if (!destination) {
            alert('Please select a destination folder');
            return;
        }

        const copyStatusDiv = document.getElementById('copy-status');
        copyStatusDiv.innerHTML = '<div class="status info"><span class="spinner"></span> Copying files...</div>';

        // Disable button during copy
        this.disabled = true;
        this.style.opacity = '0.5';
        this.style.cursor = 'not-allowed';

        // Send POST request to copy_files.php
        const formData = new URLSearchParams();
        formData.append('destination', destination);
        formData.append('files', JSON.stringify(filesList));

        fetch('api/copy_files.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: formData.toString()
        })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                copyStatusDiv.innerHTML = '<div class="status error">❌ ' + escapeHtml(data.error);
                if (data.failed) {
                    copyStatusDiv.innerHTML += '<br><br><strong>Failed files:</strong><br>';
                    data.failed.forEach(f => {
                        copyStatusDiv.innerHTML += '• ' + escapeHtml(f.file) + ' (' + escapeHtml(f.reason) + ')<br>';
                    });
                }
                if (data.success && data.success.length > 0) {
                    copyStatusDiv.innerHTML += '<br><strong>Successfully copied:</strong> ' + data.success.length + ' file(s)';
                }
                copyStatusDiv.innerHTML += '</div>';
            } else {
                // Save successful destination to localStorage
                localStorage.setItem('tabler_last_folder', destination);

                copyStatusDiv.innerHTML = '<div class="status success">✅ ' + escapeHtml(data.success) + '<br>' +
                    '<strong>' + data.count + '</strong> file(s) copied to <strong>' + escapeHtml(destination) + '</strong></div>';
            }

            // Re-enable button
            const copyBtn = document.getElementById('copy-confirm-btn');
            copyBtn.disabled = false;
            copyBtn.style.opacity = '1';
            copyBtn.style.cursor = 'pointer';
        })
        .catch(error => {
            copyStatusDiv.innerHTML = '<div class="status error">❌ Network error: ' + escapeHtml(error.message) + '</div>';

            // Re-enable button
            const copyBtn = document.getElementById('copy-confirm-btn');
            copyBtn.disabled = false;
            copyBtn.style.opacity = '1';
            copyBtn.style.cursor = 'pointer';
        });
    });
    <?php endif; ?>
});
</script>

</body>
</html>
