<?php
/**
 * Test Suite: Intelligent Date Format Detection
 *
 * Tests the DateFormatDetector class and the fixed DataCleaner::parseDate()
 * Verifies that European (DD/MM/YYYY), American (MM/DD/YYYY), and ISO formats
 * are correctly detected and parsed.
 */

require_once __DIR__ . '/lib/DateFormatDetector.php';
require_once __DIR__ . '/lib/DataCleaner.php';
require_once __DIR__ . '/lib/DataValidator.php';

echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║           INTELLIGENT DATE FORMAT DETECTION - TEST SUITE                     ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n\n";

$passed = 0;
$failed = 0;

// ========================================
// TEST 1: DateFormatDetector::detectFormat
// ========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "TEST 1: DateFormatDetector::detectFormat()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$testCases = [
    [
        'name' => 'European dates (DD/MM/YYYY)',
        'values' => ['30/11/2025', '15/12/2025', '01/01/2025'],
        'expected' => 'd/m/Y',
    ],
    [
        'name' => 'American dates (MM/DD/YYYY)',
        'values' => ['11/30/2025', '12/15/2025', '01/01/2025'],
        'expected' => 'm/d/Y',
    ],
    [
        'name' => 'ISO dates (YYYY-MM-DD)',
        'values' => ['2025-11-30', '2025-12-15'],
        'expected' => 'Y-m-d',
    ],
    [
        'name' => 'European with dashes',
        'values' => ['30-11-2025', '15-12-2025'],
        'expected' => 'd/m/Y',  // d/m/Y and d-m-Y are functionally equivalent
    ],
    [
        'name' => 'Mixed unambiguous (European wins)',
        'values' => ['30/11/2025', '25/12/2025', '15/01/2025'],
        'expected' => 'd/m/Y',
    ],
    [
        'name' => 'Mixed unambiguous (American wins)',
        'values' => ['11/30/2025', '12/25/2025', '01/15/2025'],
        'expected' => 'm/d/Y',
    ],
];

foreach ($testCases as $tc) {
    $detected = DateFormatDetector::detectFormat($tc['values']);
    $status = $detected === $tc['expected'] ? '✅ PASS' : '❌ FAIL';
    if ($detected === $tc['expected']) {
        $passed++;
    } else {
        $failed++;
    }
    echo sprintf("  %-45s %s (got: %s)\n", $tc['name'], $status, $detected);
}

// ========================================
// TEST 2: DateFormatDetector::parseDate
// ========================================
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "TEST 2: DateFormatDetector::parseDate()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$parseTestCases = [
    [
        'name' => 'Parse European (30/11/2025)',
        'value' => '30/11/2025',
        'format' => 'd/m/Y',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Parse American (11/30/2025)',
        'value' => '11/30/2025',
        'format' => 'm/d/Y',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Parse ISO (2025-11-30)',
        'value' => '2025-11-30',
        'format' => 'Y-m-d',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Parse with datetime format',
        'value' => '30/11/2025 14:30:00',
        'format' => 'd/m/Y H:i:s',
        'target' => 'DATETIME',
        'expected' => '2025-11-30 14:30:00',
    ],
];

foreach ($parseTestCases as $tc) {
    $targetFormat = $tc['target'] ?? 'DATE';
    $result = DateFormatDetector::parseDate($tc['value'], $tc['format'], $targetFormat);
    $status = $result === $tc['expected'] ? '✅ PASS' : '❌ FAIL';
    if ($result === $tc['expected']) {
        $passed++;
    } else {
        $failed++;
    }
    echo sprintf("  %-45s %s (got: %s)\n", $tc['name'], $status, $result ?? 'NULL');
}

// ========================================
// TEST 3: DataCleaner::parseDate with context
// ========================================
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "TEST 3: DataCleaner::parseDate() with column context\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$contextTestCases = [
    [
        'name' => 'European dates with context',
        'value' => '30/11/2025',
        'columnValues' => ['30/11/2025', '15/12/2025', '01/01/2025'],
        'columnName' => 'check_in_date',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'American dates with context',
        'value' => '11/30/2025',
        'columnValues' => ['11/30/2025', '12/15/2025', '01/01/2025'],
        'columnName' => 'check_in_date',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'ISO dates with context',
        'value' => '2025-11-30',
        'columnValues' => ['2025-11-30', '2025-12-15'],
        'columnName' => 'booking_date',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'European with Spanish column name',
        'value' => '30/11/2025',
        'columnValues' => ['30/11/2025', '15/12/2025'],
        'columnName' => 'fecha_entrada',
        'expected' => '2025-11-30',
    ],
];

foreach ($contextTestCases as $tc) {
    $result = DataCleaner::parseDate(
        $tc['value'],
        'DATE',
        $tc['columnValues'],
        $tc['columnName']
    );
    $status = $result === $tc['expected'] ? '✅ PASS' : '❌ FAIL';
    if ($result === $tc['expected']) {
        $passed++;
    } else {
        $failed++;
    }
    echo sprintf("  %-45s %s (got: %s)\n", $tc['name'], $status, $result ?? 'NULL');
}

// ========================================
// TEST 4: DataValidator::validateDateWithContext
// ========================================
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "TEST 4: DataValidator::validateDateWithContext()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$validatorTestCases = [
    [
        'name' => 'Validate European date',
        'value' => '30/11/2025',
        'columnValues' => ['30/11/2025', '15/12/2025'],
        'columnName' => 'check_in',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Validate American date',
        'value' => '11/30/2025',
        'columnValues' => ['11/30/2025', '12/15/2025'],
        'columnName' => 'check_in',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Validate ISO date',
        'value' => '2025-11-30',
        'columnValues' => ['2025-11-30', '2025-12-15'],
        'columnName' => 'created_at',
        'expected' => '2025-11-30',
    ],
    [
        'name' => 'Invalid date returns null',
        'value' => 'not-a-date',
        'columnValues' => ['30/11/2025', '15/12/2025'],
        'columnName' => 'check_in',
        'nullable' => true,
        'expected' => null,
    ],
];

foreach ($validatorTestCases as $tc) {
    $result = DataValidator::validateDateWithContext(
        $tc['value'],
        $tc['nullable'] ?? true,
        $tc['columnValues'],
        $tc['columnName']
    );
    $actualValue = $result['value'];
    $status = $actualValue === $tc['expected'] ? '✅ PASS' : '❌ FAIL';
    if ($actualValue === $tc['expected']) {
        $passed++;
    } else {
        $failed++;
    }
    echo sprintf("  %-45s %s (got: %s)\n", $tc['name'], $status, $actualValue ?? 'NULL');
}

// ========================================
// TEST 5: End-to-end column parsing
// ========================================
echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "TEST 5: DateFormatDetector::parseColumn() - Full column\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$europeanColumn = ['30/11/2025', '15/12/2025', '01/01/2025', '25/02/2025'];
$americanColumn = ['11/30/2025', '12/15/2025', '01/01/2025', '02/25/2025'];
$isoColumn = ['2025-11-30', '2025-12-15', '2025-01-01'];

$columnTests = [
    [
        'name' => 'Parse entire European column',
        'values' => $europeanColumn,
        'columnName' => 'check_in_date',
        'expected' => ['2025-11-30', '2025-12-15', '2025-01-01', '2025-02-25'],
    ],
    [
        'name' => 'Parse entire American column',
        'values' => $americanColumn,
        'columnName' => 'check_in_date',
        'expected' => ['2025-11-30', '2025-12-15', '2025-01-01', '2025-02-25'],
    ],
    [
        'name' => 'Parse entire ISO column',
        'values' => $isoColumn,
        'columnName' => 'booking_date',
        'expected' => ['2025-11-30', '2025-12-15', '2025-01-01'],
    ],
];

foreach ($columnTests as $tc) {
    $result = DateFormatDetector::parseColumn($tc['values'], $tc['columnName']);
    $allMatch = $result['parsed'] === $tc['expected'];
    $status = $allMatch ? '✅ PASS' : '❌ FAIL';
    if ($allMatch) {
        $passed++;
    } else {
        $failed++;
    }
    echo sprintf("  %-45s %s (format: %s)\n", $tc['name'], $status, $result['format']);
}

// ========================================
// SUMMARY
// ========================================
echo "\n╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              SUMMARY                                         ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo sprintf("  Passed: %d  |  Failed: %d  |  Total: %d\n", $passed, $failed, $passed + $failed);

if ($failed === 0) {
    echo "\n  🎉 ALL TESTS PASSED! Date format detection is working correctly.\n\n";
} else {
    echo "\n  ⚠️  Some tests failed. Please review the output above.\n\n";
}
