<?php
/**
 * Integration test - Full import flow with date format detection
 */

require_once __DIR__ . '/vendor/autoload.php';
require_once __DIR__ . '/config.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║              FULL INTEGRATION TEST - IMPORT FLOW                           ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n\n";

// Create test spreadsheet with European dates
$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

$sheet->setCellValue('A1', 'nombre');
$sheet->setCellValue('B1', 'fecha_alta');
$sheet->setCellValue('C1', 'monto');
$sheet->setCellValue('D1', 'check_in');

$testData = [
    ['Juan Pérez', '30/11/2025', 1500.50, '15/12/2025'],
    ['María García', '01/12/2025', 2300.00, '20/12/2025'],
    ['Carlos López', '15/11/2025', 950.75, '25/11/2025'],
    ['Ana Martínez', '28/02/2026', 4200.00, '10/03/2026'],
    ['Roberto Sánchez', '31/01/2026', 1750.25, '05/02/2026'],
];

$row = 2;
foreach ($testData as $data) {
    $sheet->setCellValue('A' . $row, $data[0]);
    $sheet->setCellValue('B' . $row, $data[1]);
    $sheet->setCellValue('C' . $row, $data[2]);
    $sheet->setCellValue('D' . $row, $data[3]);
    $row++;
}

$testFile = __DIR__ . '/temp/test_integration_dates.xlsx';
$writer = new Xlsx($spreadsheet);
$writer->save($testFile);

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "✓ Created test file: temp/test_integration_dates.xlsx\n\n";

require_once __DIR__ . '/lib/ChunkReader.php';
require_once __DIR__ . '/lib/DataCleaner.php';
require_once __DIR__ . '/lib/DataValidator.php';
require_once __DIR__ . '/lib/DateFormatDetector.php';
require_once __DIR__ . '/lib/ProgressTracker.php';

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "📊 Testing ChunkReader with ProgressTracker...\n\n";

// Get file metadata
$metadata = ChunkReader::getFileMetadata($testFile);
echo "✓ File metadata: {$metadata['total_rows']} rows, {$metadata['total_columns']} columns\n";

// Read headers
$headers = ChunkReader::readHeaders($testFile);
echo "✓ Headers: " . implode(', ', $headers['headers']) . "\n\n";

// Start progress tracking
$progress = ProgressTracker::start('integration_test', 'parsing', 0, $metadata['total_rows'] - 1, 'Reading file...');
echo "✓ Progress tracker started\n\n";

// Collect all column values for date format detection
$columnValues = [
    'fecha_alta' => [],
    'check_in' => []
];

// Read all data first to detect formats
$allRows = [];
ChunkReader::readInChunks(
    $testFile,
    $headers['headerMap'],
    $headers['columnCount'],
    2, // start row (after header)
    $metadata['total_rows'], // end row
    2, // chunk size
    function($chunkRows, $chunkStart, $chunkEnd) use (&$allRows, &$columnValues, $progress) {
            foreach ($chunkRows as $row) {
                $allRows[] = $row;
                $columnValues['fecha_alta'][] = $row['fecha_alta'] ?? null;
                $columnValues['check_in'][] = $row['check_in'] ?? null;
            }
            
            $currentRow = $chunkEnd;
            $totalRows = $progress['total'] ?? $chunkEnd;
            ProgressTracker::update('integration_test', $currentRow, $totalRows, 'parsing', "Processing row $currentRow of $totalRows...");
    }
);

// Detect date formats from column values
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🔍 Date Format Detection:\n\n";

$fechaAltaFormat = DateFormatDetector::detectFormat($columnValues['fecha_alta'], 'fecha_alta');
$checkInFormat = DateFormatDetector::detectFormat($columnValues['check_in'], 'check_in');

echo "  fecha_alta format: " . ($fechaAltaFormat ?: 'unknown') . "\n";
echo "  check_in format: " . ($checkInFormat ?: 'unknown') . "\n\n";

// Validate and clean each row
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "✅ Validating and Cleaning Rows:\n\n";

$validRows = 0;
foreach ($allRows as $index => $row) {
    $cleaned = [];
    $isValid = true;
    
    foreach ($row as $column => $value) {
        // Use context-aware validation for date columns
        if (in_array($column, ['fecha_alta', 'check_in'])) {
            $result = DataValidator::validateDateWithContext(
                $value,
                true,
                $columnValues[$column],
                $column
            );
            $cleaned[$column] = $result['value'];
            
            if ($result['reason'] && strpos($result['reason'], 'Invalid') !== false) {
                $isValid = false;
            }
        } else {
            $result = DataValidator::validate($value, 'VARCHAR', 255, true);
            $cleaned[$column] = $result['value'];
        }
    }
    
    if ($isValid) {
        $validRows++;
    }
    
    echo "  Row " . ($index + 1) . ": {$row['nombre']} - fecha_alta: {$cleaned['fecha_alta']}, check_in: {$cleaned['check_in']}\n";
}

ProgressTracker::complete('integration_test', "Processed $validRows valid rows");
ProgressTracker::cleanup('integration_test');

echo "\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "📈 RESULTS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "Total rows processed: " . count($allRows) . "\n";
echo "Valid rows: $validRows\n";
echo "Date formats detected:\n";
echo "  - fecha_alta: $fechaAltaFormat\n";
echo "  - check_in: $checkInFormat\n";

echo "\n✅ INTEGRATION TEST PASSED!\n\n";

unlink($testFile);
echo "✓ Cleaned up test file\n";
