<?php
/**
 * Test Script: Legacy Table Compatibility (Dual-Mode)
 * Tests that existing tables with English fields still work with the new code
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/SchemaConventions.php';

echo "========================================\n";
echo "Legacy Table Compatibility Test\n";
echo "========================================\n\n";

$tableName = 'propietarios';
$dbName = 'mrw';

try {
    // TEST 1: Field Mode Detection
    echo "[1] Testing field mode detection...\n";
    $mode = SchemaConventions::detectFieldMode($tableName, $dbName);
    echo "✓ Detected mode for 'propietarios': {$mode}\n";

    if ($mode !== 'english') {
        throw new Exception("Expected 'english' mode, got '{$mode}'");
    }
    echo "✓ Correctly detected English legacy schema\n\n";

    // TEST 2: Get existing record count
    echo "[2] Checking existing records...\n";
    $rows = DatabaseHelper::getRows($tableName, $dbName, 5, 0);
    $rowCount = count($rows);
    echo "✓ Found {$rowCount} existing records\n";

    if ($rowCount > 0) {
        $firstRow = $rows[0];
        echo "  Sample record ID: " . $firstRow['id'] . "\n";
        echo "  Has 'id' field (legacy): " . (isset($firstRow['id']) ? 'YES' : 'NO') . "\n";
        echo "  Has 'insert_db' field (legacy): " . (isset($firstRow['insert_db']) ? 'YES' : 'NO') . "\n\n";

        // TEST 3: READ existing record
        echo "[3] Testing READ on legacy table...\n";
        $existingId = $firstRow['id'];
        $record = DatabaseHelper::getRow($tableName, $existingId, $dbName);

        if ($record) {
            echo "✓ Successfully retrieved legacy record\n";
            echo "  - id: " . $record['id'] . "\n";
            echo "  - insert_db: " . $record['insert_db'] . "\n";
            echo "  - insert_by: " . $record['insert_by'] . "\n\n";
        } else {
            throw new Exception("Failed to retrieve legacy record");
        }
    } else {
        echo "  No existing records to test READ/UPDATE/DELETE\n\n";
    }

    // TEST 4: INSERT into legacy table
    echo "[4] Testing INSERT into legacy table...\n";
    $testData = [
        'DEPARTAMENTO' => 'TEST-DEPT-001',
        'PROPIETARIO' => 'Test Owner',
        'CORREO' => 'test@example.com'
    ];

    $insertedId = DatabaseHelper::insertRow($tableName, $testData, $dbName);
    echo "✓ Record inserted with ID: {$insertedId}\n";

    // Verify it used English field names
    $insertedRecord = DatabaseHelper::getRow($tableName, $insertedId, $dbName);
    echo "  - Has 'id' field: " . (isset($insertedRecord['id']) ? 'YES' : 'NO') . "\n";
    echo "  - Has 'insert_db' field: " . (isset($insertedRecord['insert_db']) ? 'YES' : 'NO') . "\n";
    echo "  - Has 'insert_by' field: " . (isset($insertedRecord['insert_by']) ? 'YES' : 'NO') . "\n";
    echo "  - insert_by value: " . $insertedRecord['insert_by'] . "\n\n";

    // TEST 5: UPDATE legacy record
    echo "[5] Testing UPDATE on legacy table...\n";
    $updateData = [
        'PROPIETARIO' => 'Test Owner (Updated)',
        'CORREO' => 'updated@example.com'
    ];

    DatabaseHelper::updateRow($tableName, $insertedId, $updateData, $dbName);
    echo "✓ Record updated successfully\n\n";

    // TEST 6: Verify UPDATE and last_change
    echo "[6] Verifying UPDATE (checking last_change)...\n";
    $updatedRecord = DatabaseHelper::getRow($tableName, $insertedId, $dbName);

    if ($updatedRecord) {
        echo "✓ Updated record retrieved\n";
        echo "  - PROPIETARIO: " . $updatedRecord['PROPIETARIO'] . "\n";
        echo "  - last_change: " . ($updatedRecord['last_change'] ?? 'NULL') . "\n";
        echo "  - last_change_by: " . ($updatedRecord['last_change_by'] ?? 'NULL') . "\n";

        if ($updatedRecord['last_change'] && $updatedRecord['last_change_by']) {
            echo "✓ Audit fields (last_change, last_change_by) populated correctly for legacy table\n\n";
        } else {
            echo "✗ WARNING: last_change or last_change_by not populated\n\n";
        }
    } else {
        throw new Exception("Failed to retrieve updated legacy record");
    }

    // TEST 7: DELETE legacy record
    echo "[7] Testing DELETE on legacy table...\n";
    DatabaseHelper::deleteRow($tableName, $insertedId, $dbName);
    echo "✓ Record deleted successfully\n\n";

    // TEST 8: Verify deletion
    echo "[8] Verifying DELETE...\n";
    $deletedRecord = DatabaseHelper::getRow($tableName, $insertedId, $dbName);

    if (!$deletedRecord) {
        echo "✓ Record no longer exists (deletion confirmed)\n\n";
    } else {
        throw new Exception("Record still exists after deletion!");
    }

    echo "========================================\n";
    echo "✓ ALL LEGACY COMPATIBILITY TESTS PASSED!\n";
    echo "========================================\n\n";
    echo "Summary:\n";
    echo "- Field mode detection: english ✓\n";
    echo "- READ with legacy 'id' primary key ✓\n";
    echo "- INSERT with auto-population of insert_db, insert_by ✓\n";
    echo "- UPDATE with auto-population of last_change, last_change_by ✓\n";
    echo "- DELETE using legacy 'id' primary key ✓\n";
    echo "- Dual-mode support working correctly ✓\n";

} catch (Exception $e) {
    echo "\n✗ TEST FAILED: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
