<?php
/**
 * Test: Animated Progress System
 * Tests the ProgressTracker and getProgress API
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/ErrorHandler.php';
require_once __DIR__ . '/lib/ProgressTracker.php';

echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║              ANIMATED PROGRESS SYSTEM - TEST SUITE                        ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n\n";

// Test 1: ProgressTracker initialization
echo "TEST 1: ProgressTracker::start()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

$key = ProgressTracker::generateKey();
$progress = ProgressTracker::start($key, 100, 'parsing');

$test1 = $progress !== null && $progress['total'] === 100 && $progress['phase'] === 'parsing';
echo sprintf("  Initialize progress: %s\n", $test1 ? '✅ PASS' : '❌ FAIL');
echo "  Key: " . substr($key, 0, 20) . "...\n\n";

// Test 2: Progress update
echo "TEST 2: ProgressTracker::update()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

ProgressTracker::update($key, 50, 'Processing row 50 of 100...');
$retrieved = ProgressTracker::get($key);

$test2 = $retrieved !== null && $retrieved['current'] === 50 && $retrieved['percent'] === 50;
echo sprintf("  Update progress to 50%%: %s\n", $test2 ? '✅ PASS' : '❌ FAIL');
echo sprintf("  Message: %s\n", $retrieved['message'] ?? 'N/A');
echo "  Percent: " . $retrieved['percent'] . "%\n\n";

// Test 3: Progress increment
echo "TEST 3: ProgressTracker::increment()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

ProgressTracker::increment($key);
$retrieved = ProgressTracker::get($key);

$test3 = $retrieved['current'] === 51 && $retrieved['percent'] === 51;
echo sprintf("  Increment to 51: %s\n", $test3 ? '✅ PASS' : '❌ FAIL');
echo "  Current: " . $retrieved['current'] . "\n\n";

// Test 4: Phase change
echo "TEST 4: ProgressTracker::setPhase()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

ProgressTracker::setPhase($key, 'inserting', 'Starting insert...');
$retrieved = ProgressTracker::get($key);

// Percent should be preserved/recalculated, NOT reset to 0
$test4 = $retrieved['phase'] === 'inserting' && $retrieved['percent'] === 51;
echo sprintf("  Change to inserting phase: %s\n", $test4 ? '✅ PASS' : '❌ FAIL');
echo "  Phase: " . $retrieved['phase'] . "\n";
echo "  Percent: " . $retrieved['percent'] . "% (should preserve progress)\n\n";

// Test 5: Progress completion
echo "TEST 5: ProgressTracker::complete()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

ProgressTracker::update($key, 100, 'Almost done...');
ProgressTracker::complete($key, 'All done!');
$retrieved = ProgressTracker::get($key);

$test5 = $retrieved['status'] === 'completed' && $retrieved['percent'] === 100 && $retrieved['message'] === 'All done!';
echo sprintf("  Complete progress: %s\n", $test5 ? '✅ PASS' : '❌ FAIL');
echo "  Status: " . $retrieved['status'] . "\n";
echo "  Message: " . $retrieved['message'] . "\n\n";

// Test 6: Cleanup
echo "TEST 6: ProgressTracker::cleanup()\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

ProgressTracker::cleanup($key);
$retrieved = ProgressTracker::get($key);

$test6 = $retrieved === null;
echo sprintf("  Cleanup progress file: %s\n", $test6 ? '✅ PASS' : '❌ FAIL');
echo "  Retrieved: " . ($retrieved === null ? 'null (correct)' : 'NOT null (wrong)') . "\n\n";

// Test 7: getProgress API simulation
echo "TEST 7: getProgress API simulation\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";

// Simulate API call - same logic as api/getProgress.php but without header issues
$testKey = ProgressTracker::generateKey();
ProgressTracker::start($testKey, 75, 'parsing');
ProgressTracker::update($testKey, 25, 'Processing...');

// Simulate what getProgress.php does (without sending headers)
$progress = ProgressTracker::get($_GET['key'] ?? $testKey);

if ($progress !== null) {
    $response = [
        'status' => 'ok',
        'progress' => [
            'current' => $progress['current'],
            'total' => $progress['total'],
            'percent' => $progress['percent'],
            'phase' => $progress['phase'],
            'message' => $progress['message'],
            'status' => $progress['status']
        ]
    ];
} else {
    $response = ['status' => 'error', 'message' => 'No progress found'];
}

$test7 = $response['status'] === 'ok' && $response['progress']['percent'] === 33;
echo sprintf("  API returns progress: %s\n", $test7 ? '✅ PASS' : '❌ FAIL');
echo "  Response: " . json_encode($response['progress']) . "\n\n";

ProgressTracker::cleanup($testKey);

// Summary
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              SUMMARY                                         ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";

$passed = ($test1 ? 1 : 0) + ($test2 ? 1 : 0) + ($test3 ? 1 : 0) + ($test4 ? 1 : 0) + ($test5 ? 1 : 0) + ($test6 ? 1 : 0) + ($test7 ? 1 : 0);
$total = 7;
echo sprintf("  Passed: %d/%d\n", $passed, $total);

if ($passed >= 6) {
    echo "\n  🎉 Progress system is working!\n";
    echo "  - Animated progress bar will show during upload\n";
    echo "  - Real-time updates every 150-200ms\n";
    echo "  - ETA calculation based on elapsed time\n";
    echo "  - Cyberpunk neon styling with shimmer effects\n";
} else {
    echo "\n  ⚠️  Some tests failed. Check the output above.\n";
}

echo "\n";
