<?php
/**
 * DEMO: Hybrid Detection System in Action
 *
 * Shows how the system runs both name-based and value-based detection
 * and intelligently resolves conflicts
 */

require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/SchemaDetector.php';

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                  HYBRID DETECTION SYSTEM - LIVE DEMO                         ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";

$scenarios = [
    [
        'title' => 'SCENARIO 1: ID Field with Numeric Values',
        'description' => 'property_id contains only numbers, but must remain VARCHAR',
        'column' => 'property_id',
        'values' => ['001', '002', '003', '004', '005']
    ],
    [
        'title' => 'SCENARIO 2: Exchange Rate with Integer Values',
        'description' => 'interchange shows whole numbers now, but needs DECIMAL(10,4)',
        'column' => 'interchange',
        'values' => ['17', '18', '19', '20']
    ],
    [
        'title' => 'SCENARIO 3: Date Field - Both Agree',
        'description' => 'bookeddate has proper date format - perfect match!',
        'column' => 'bookeddate',
        'values' => ['2024-01-15', '2024-02-20', '2024-03-10']
    ],
    [
        'title' => 'SCENARIO 4: Phone Field Data Quality Issue',
        'description' => 'guest_phone has abnormally long values - DATA ALERT!',
        'column' => 'guest_phone',
        'values' => [
            'This is way too long for a phone number - probably contains notes or other data mixed in',
            'Another suspiciously long phone entry that exceeds normal length',
            '+52-555-1234567' // Normal one
        ]
    ],
    [
        'title' => 'SCENARIO 5: No Pattern - Value-Based Fallback',
        'description' => 'random_field has no name pattern, uses value analysis',
        'column' => 'random_field_xyz',
        'values' => ['100', '200', '300', '400', '500']
    ]
];

foreach ($scenarios as $idx => $scenario) {
    echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
    echo "║  " . str_pad($scenario['title'], 74) . "║\n";
    echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
    echo "\n";
    echo "📋 Description: {$scenario['description']}\n";
    echo "🔍 Column: {$scenario['column']}\n";
    echo "📊 Sample Values: ";
    $sampleValues = array_slice($scenario['values'], 0, 3);
    echo implode(', ', array_map(function($v) {
        return '"' . (strlen($v) > 30 ? substr($v, 0, 30) . '...' : $v) . '"';
    }, $sampleValues));
    echo "\n\n";

    // Run detection
    $headers = [$scenario['column']];
    $rows = [];
    foreach ($scenario['values'] as $value) {
        $rows[] = [$scenario['column'] => $value];
    }

    $schema = SchemaDetector::detectSchema($rows, $headers);
    $result = $schema[0];

    // Display results
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "                              DETECTION RESULTS                                \n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "\n";

    // Final Decision
    $typeStr = $result['type'] . ($result['length'] ? "({$result['length']})" : "");
    echo "✅ FINAL TYPE: {$typeStr}\n";
    echo "📌 Detection Method: {$result['detection_method']}\n";
    echo "⚠️  Conflict Detected: " . ($result['conflict_detected'] ? 'YES' : 'NO') . "\n";
    echo "\n";

    // Show conflict details if any
    if ($result['conflict_detected']) {
        echo "┌─────────────────────────────────────────────────────────────────────────────┐\n";
        echo "│ 🔍 CONFLICT ANALYSIS                                                        │\n";
        echo "└─────────────────────────────────────────────────────────────────────────────┘\n";
        echo "\n";
        echo "Name-based suggestion:  {$result['name_based_suggestion']}\n";
        echo "Value-based suggestion: {$result['value_based_suggestion']}\n";
        echo "\n";
        echo "💡 Resolution Reason:\n";
        echo "   {$result['conflict_reason']}\n";
        echo "\n";

        // Determine priority level
        $method = $result['detection_method'];
        $priority = '';
        $icon = '';

        if ($method === 'value_based_override') {
            $priority = 'Priority 1: Data Compatibility (CRITICAL)';
            $icon = '🔴';
        } elseif ($method === 'name_based_semantic') {
            $priority = 'Priority 2: Semantic Meaning';
            $icon = '🟡';
        } elseif ($method === 'name_based_precision') {
            $priority = 'Priority 3: Precision Preservation';
            $icon = '🟡';
        } else {
            $priority = 'Priority 4: Name-Based Default';
            $icon = '🟢';
        }

        echo "{$icon} Applied Rule: {$priority}\n";
    } else {
        echo "┌─────────────────────────────────────────────────────────────────────────────┐\n";
        echo "│ ✅ NO CONFLICT                                                              │\n";
        echo "└─────────────────────────────────────────────────────────────────────────────┘\n";
        echo "\n";

        if ($result['detection_method'] === 'both_agree') {
            echo "🎉 HIGH CONFIDENCE: Both name and value analysis agree!\n";
        } else {
            echo "ℹ️  No name pattern detected, using value-based analysis\n";
        }
    }

    echo "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo "\n\n";
}

// Summary
echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                              KEY TAKEAWAYS                                   ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";
echo "1. 🔴 DATA COMPATIBILITY (Priority 1)\n";
echo "   → If values won't fit, VALUES win to prevent truncation\n";
echo "   → Example: guest_phone with 80-char values → VARCHAR(255) not VARCHAR(20)\n";
echo "\n";
echo "2. 🟡 SEMANTIC MEANING (Priority 2)\n";
echo "   → ID/Date fields preserve name-based types\n";
echo "   → Example: property_id with numeric values → VARCHAR not INT\n";
echo "\n";
echo "3. 🟡 PRECISION PRESERVATION (Priority 3)\n";
echo "   → Money fields maintain name-based precision\n";
echo "   → Example: interchange with integers → DECIMAL(10,4) not INT\n";
echo "\n";
echo "4. 🟢 VALUE-BASED FALLBACK (Priority 4)\n";
echo "   → No name pattern → trust the data\n";
echo "   → Example: random_field_xyz → analyze values\n";
echo "\n";
echo "5. 🎉 BOTH AGREE (High Confidence)\n";
echo "   → Name and values suggest same type\n";
echo "   → Example: bookeddate with date-formatted values → DATE\n";
echo "\n";

echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
echo "║                                 SUMMARY                                      ║\n";
echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
echo "\n";
echo "✅ The hybrid system provides:\n";
echo "   • Guaranteed data compatibility (no truncation)\n";
echo "   • Semantic preservation (IDs, dates stay correct)\n";
echo "   • Precision protection (money fields maintain accuracy)\n";
echo "   • Full transparency (metadata explains every decision)\n";
echo "   • High confidence (detects when both methods agree)\n";
echo "\n";
echo "🎯 Result: 'ejejej we'll figure it out' → System TELLS you about conflicts!\n";
echo "\n";
