<?php
/**
 * Test Suite: Conflict Resolution between Name-Based and Value-Based Detection
 *
 * Tests the hybrid detection system that runs BOTH name-based and value-based
 * detection and intelligently resolves conflicts using priority rules
 */

require_once __DIR__ . '/../lib/DataCleaner.php';
require_once __DIR__ . '/../lib/SchemaDetector.php';

class ConflictResolutionTest {
    private $passed = 0;
    private $failed = 0;
    private $tests = [];

    /**
     * Test a conflict scenario
     */
    private function testConflict($columnName, $testValues, $expectedType, $expectedLength, $expectedDetectionMethod, $expectedConflict, $testDescription) {
        $headers = [$columnName];
        $rows = [];
        foreach ($testValues as $value) {
            $rows[] = [$columnName => $value];
        }

        $schema = SchemaDetector::detectSchema($rows, $headers);
        $result = $schema[0];

        $actualType = $result['type'];
        $actualLength = $result['length'];
        $actualMethod = $result['detection_method'] ?? 'unknown';
        $actualConflict = $result['conflict_detected'] ?? false;

        $typeMatch = ($actualType === $expectedType);
        $lengthMatch = ($actualLength == $expectedLength);
        $methodMatch = ($actualMethod === $expectedDetectionMethod);
        $conflictMatch = ($actualConflict === $expectedConflict);

        $passed = $typeMatch && $lengthMatch && $methodMatch && $conflictMatch;

        if ($passed) {
            $this->passed++;
            $status = "✅ PASS";
        } else {
            $this->failed++;
            $status = "❌ FAIL";
        }

        $this->tests[] = [
            'status' => $status,
            'description' => $testDescription,
            'column' => $columnName,
            'expected' => "$expectedType" . ($expectedLength ? "($expectedLength)" : "") . " [$expectedDetectionMethod]" . ($expectedConflict ? " CONFLICT" : ""),
            'actual' => "$actualType" . ($actualLength ? "($actualLength)" : "") . " [$actualMethod]" . ($actualConflict ? " CONFLICT" : ""),
            'passed' => $passed,
            'conflict_reason' => $result['conflict_reason'] ?? null,
            'name_suggestion' => $result['name_based_suggestion'] ?? null,
            'value_suggestion' => $result['value_based_suggestion'] ?? null
        ];

        return $passed;
    }

    /**
     * Run all tests
     */
    public function runTests() {
        echo "\n";
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║              CONFLICT RESOLUTION TEST SUITE                                  ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";

        // SCENARIO 1: ID Field with Numeric Values (Name wins - Semantic)
        echo "🧪 SCENARIO 1: ID Fields with Numeric Values (Name wins - Semantic)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'property_id',
            ['001', '002', '003', '004', '005'], // Numeric-looking values
            'VARCHAR', 100,
            'name_based_semantic', true,
            'ID field with numeric values → Name-based wins to support future alphanumeric IDs'
        );
        echo "\n";

        // SCENARIO 2: Exchange Rate with Integer Values (Name wins - Precision)
        echo "🧪 SCENARIO 2: Exchange Rate with Integer Values (Name wins - Precision)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'exchange_rate',
            ['17', '18', '19', '20'], // Integer values
            'DECIMAL', '10,4',
            'name_based_precision', true,
            'Exchange rate field with integers → Name-based wins to preserve precision for future decimals'
        );
        echo "\n";

        // SCENARIO 3: Date Field with String Values (Both Agree)
        echo "🧪 SCENARIO 3: Date Field with Date-Formatted Values (Both Agree)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'bookeddate',
            ['2024-01-15', '2024-02-20', '2024-03-10'], // Date-formatted strings
            'DATE', null,
            'both_agree', false, // Both name and values detect DATE!
            'Date field with date-formatted values → Both methods agree on DATE'
        );
        echo "\n";

        // SCENARIO 4: Phone Field with Long Values (Values win - Data Compatibility)
        echo "🧪 SCENARIO 4: Phone Field with Oversized Values (Values win - Data Compatibility)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'guest_phone',
            [
                str_repeat('x', 150), // 150 chars (exceeds VARCHAR(20) from name pattern)
                str_repeat('y', 140),
                str_repeat('z', 130)
            ],
            'VARCHAR', 255, // Should upgrade to VARCHAR(255) to fit values
            'value_based_override', true,
            'Phone field with long values → Values win to prevent data truncation'
        );
        echo "\n";

        // SCENARIO 5: Both Agree (No Conflict)
        echo "🧪 SCENARIO 5: Both Detections Agree (No Conflict)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'guest_count',
            ['1', '2', '3', '4', '5'], // Integer values
            'INT', null,
            'both_agree', false, // No conflict!
            'Count field with integer values → Both methods agree on INT'
        );
        echo "\n";

        // SCENARIO 6: No Name Pattern (Value-Based Fallback)
        echo "🧪 SCENARIO 6: No Name Pattern (Value-Based Fallback)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'random_field',
            ['100', '200', '300'], // Numeric values
            'INT', null,
            'value_based', false, // No conflict because no name pattern
            'No name pattern detected → Value-based detection used'
        );
        echo "\n";

        // SCENARIO 7: Money Field - Different DECIMAL Precisions (Name wins - Precision)
        echo "🧪 SCENARIO 7: Money Field with Different Precisions (Name wins - Precision)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'total_amount',
            ['10.50', '20.75', '30.99'], // DECIMAL(10,2) values
            'DECIMAL', '10,2',
            'both_agree', false, // Should agree since values match pattern
            'Money field → Both methods should agree on DECIMAL(10,2)'
        );
        echo "\n";

        // SCENARIO 8: Description Field with Short Values (Name wins)
        echo "🧪 SCENARIO 8: Description Field with Short Values (Name wins)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'guest_notes',
            ['Short note', 'Another note', 'Third note'], // Short values
            'TEXT', null,
            'name_based', true,
            'Description field with short values → Name-based wins (TEXT for future long notes)'
        );
        echo "\n";

        // SCENARIO 9: No Name Pattern, Long Values → TEXT (Value-Based Fallback)
        echo "🧪 SCENARIO 9: No Name Pattern with Long Values (Value-Based Fallback)\n";
        echo "────────────────────────────────────────────────────────────────────────────────\n";
        $this->testConflict(
            'customer_name',
            [
                str_repeat('Long name ', 60), // ~600 chars (exceeds VARCHAR(255))
                str_repeat('Another long name ', 50),
                'Normal name'
            ],
            'TEXT', null,
            'value_based', false, // No name pattern, so no conflict
            'No name pattern detected, long values → Value-based detection chooses TEXT'
        );
        echo "\n";

        // Print detailed results
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                              DETAILED RESULTS                                ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";

        foreach ($this->tests as $test) {
            printf("%s %s\n", $test['status'], $test['description']);
            printf("   Column: %s\n", $test['column']);
            printf("   Expected: %s\n", $test['expected']);
            printf("   Actual: %s\n", $test['actual']);

            if (!$test['passed']) {
                echo "   ❌ MISMATCH DETECTED\n";
            }

            if ($test['conflict_reason']) {
                printf("   💡 Reason: %s\n", $test['conflict_reason']);
            }

            if ($test['name_suggestion'] && $test['value_suggestion']) {
                printf("   📊 Name suggested: %s | Values suggested: %s\n",
                    $test['name_suggestion'],
                    $test['value_suggestion']
                );
            }

            echo "\n";
        }

        // Print summary
        echo "╔══════════════════════════════════════════════════════════════════════════════╗\n";
        echo "║                              TEST SUMMARY                                    ║\n";
        echo "╚══════════════════════════════════════════════════════════════════════════════╝\n";
        echo "\n";
        printf("Total Tests: %d\n", $this->passed + $this->failed);
        printf("✅ Passed: %d\n", $this->passed);
        printf("❌ Failed: %d\n", $this->failed);
        echo "\n";

        if ($this->failed === 0) {
            echo "🎉 ALL CONFLICT RESOLUTION TESTS PASSED! 🎉\n";
            echo "\n";
            echo "✅ The hybrid detection system correctly:\n";
            echo "   1. Prevents data truncation (Data Compatibility)\n";
            echo "   2. Preserves semantic meaning (ID/Date fields)\n";
            echo "   3. Maintains precision (Money fields)\n";
            echo "   4. Falls back to value-based when no pattern matches\n";
            echo "   5. Detects and reports conflicts with clear reasoning\n";
            echo "\n";
            return true;
        } else {
            echo "⚠️  SOME TESTS FAILED - Please review the detailed results above.\n";
            echo "\n";
            return false;
        }
    }
}

// Run the tests
$tester = new ConflictResolutionTest();
$allPassed = $tester->runTests();

exit($allPassed ? 0 : 1);
