<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>Gate 5: Table Viewer</title>
<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
        color: #e0f7ff;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    /* Button Styling */
    button {
        background: #00eaff;
        color: #000;
        padding: 10px 20px;
        font-size: 14px;
        border-radius: 6px;
        border: none;
        cursor: pointer;
        font-weight: bold;
        box-shadow: 0 0 10px #00eaffcc;
        transition: all 0.2s ease;
        font-family: 'Consolas', monospace;
    }

    button:hover:not(:disabled) {
        transform: scale(1.05);
        box-shadow: 0 0 18px #00eaff;
    }

    button:active:not(:disabled) {
        transform: scale(0.98);
    }
    
    button:disabled {
        background: #333;
        color: #666;
        box-shadow: none;
        cursor: not-allowed;
    }

    /* Table Styling */
    .table-wrap {
        overflow-x: auto;
        margin-top: 15px;
        min-height: 200px;
    }

    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
    }

    th, td {
        border: 1px solid #00eaff33;
        padding: 10px 12px;
        text-align: left;
        white-space: nowrap;
    }

    th {
        background: rgba(0,234,255,0.15);
        text-shadow: 0 0 6px #00eaff;
        font-weight: bold;
    }

    tr:hover td {
        background: rgba(0,234,255,0.08);
        transition: background 0.2s ease;
    }

    /* Pagination */
    .pagination {
        display: flex;
        gap: 20px;
        margin-top: 25px;
        justify-content: center;
        align-items: center;
    }
    .pagination span {
        color: #00eaff;
        font-weight: bold;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }
    
    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }
</style>
</head>
<body>

<div class="panel">
    <a href="dashboard.php" class="back-link">← Return to Observatory</a>
    <h2 id="tableTitle">Viewer</h2>
    <div id="status" class="status"></div>
</div>

<div id="dataPanel" class="panel">
    <div class="table-wrap">
        <table id="dataTable">
            <thead><tr id="tableHead"></tr></thead>
            <tbody id="tableBody"></tbody>
        </table>
    </div>
    
    <div class="pagination">
        <button onclick="changePage(-1)" id="btnPrev">Previous</button>
        <span id="pageInfo">Page 1</span>
        <button onclick="changePage(1)" id="btnNext">Next</button>
    </div>
</div>

<script>
const urlParams = new URLSearchParams(window.location.search);
const dbName = urlParams.get('db');
const tableName = urlParams.get('table');
let currentPage = 1;
let totalPages = 1;

document.addEventListener('DOMContentLoaded', () => {
    if (!dbName || !tableName) {
        showError("Database and table parameters are required.");
        return;
    }
    
    document.getElementById('tableTitle').textContent = `Viewer: [${tableName}] @ ${dbName}`;
    loadData(1);
});

function loadData(page) {
    const status = document.getElementById('status');
    const tbody = document.getElementById('tableBody');
    const thead = document.getElementById('tableHead');
    
    status.innerHTML = '🌊 Fetching data stream...';
    status.className = 'status';
    
    // Dim table while loading
    tbody.style.opacity = '0.5';
    
    fetch(`api/getTableRows.php?db=${encodeURIComponent(dbName)}&table=${encodeURIComponent(tableName)}&page=${page}`)
        .then(r => r.json())
        .then(data => {
            tbody.style.opacity = '1';
            status.innerHTML = '';
            
            if (data.status === 'ok') {
                currentPage = data.page;
                totalPages = data.pages;
                
                renderTable(data.rows);
                updatePagination();
            } else {
                showError(data.message);
            }
        })
        .catch(err => {
            tbody.style.opacity = '1';
            showError('Network error: ' + err.message);
        });
}

function renderTable(rows) {
    const tbody = document.getElementById('tableBody');
    const thead = document.getElementById('tableHead');
    
    tbody.innerHTML = '';
    thead.innerHTML = '';
    
    if (!rows || rows.length === 0) {
        tbody.innerHTML = '<tr><td colspan="100" style="text-align:center;">No data found in this sector.</td></tr>';
        return;
    }
    
    // Headers
    const cols = Object.keys(rows[0]);
    cols.forEach(c => {
        const th = document.createElement('th');
        th.textContent = c;
        thead.appendChild(th);
    });
    
    // Rows
    rows.forEach(r => {
        const tr = document.createElement('tr');
        cols.forEach(c => {
            const td = document.createElement('td');
            const val = r[c];
            td.textContent = (val === null) ? 'NULL' : val;
            if (val === null) td.style.color = '#555';
            tr.appendChild(td);
        });
        tbody.appendChild(tr);
    });
}

function updatePagination() {
    document.getElementById('pageInfo').textContent = `Page ${currentPage} of ${totalPages || 1}`;
    document.getElementById('btnPrev').disabled = (currentPage <= 1);
    document.getElementById('btnNext').disabled = (currentPage >= totalPages);
}

function changePage(delta) {
    const newPage = currentPage + delta;
    if (newPage >= 1 && newPage <= totalPages) {
        loadData(newPage);
    }
}

function showError(msg) {
    const status = document.getElementById('status');
    status.className = 'status error';
    status.textContent = '⚠ ' + msg;
}
</script>

</body>
</html>
