<?php
/**
 * Exodus Backup API
 *
 * Handles backup creation and deletion requests
 */

header('Content-Type: application/json');

require_once __DIR__ . '/../config.php';
require_once 'lib/ExodusEngine.php';
require_once 'lib/ErrorHandler.php';

try {
    // Get JSON input
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data || !isset($data['action'])) {
        throw new Exception('Invalid request - missing action');
    }

    $action = $data['action'];

    switch ($action) {
        case 'create':
            // Create backup
            $options = $data['options'] ?? [];

            $result = ExodusEngine::createFullBackup($options);

            echo json_encode([
                'success' => true,
                'filename' => $result['filename'],
                'file_path' => $result['file_path'],
                'size_mb' => $result['size_mb'],
                'file_count' => $result['file_count'],
                'format' => $result['format'],
                'manifest' => $result['manifest'],
                'database_backup' => $result['database_backup'] ?? null
            ]);
            break;

        case 'delete':
            // Delete backup
            $filename = $data['filename'] ?? '';

            if (empty($filename)) {
                throw new Exception('Filename required for delete action');
            }

            $success = ExodusEngine::deleteBackup($filename);

            echo json_encode([
                'success' => $success,
                'message' => $success ? 'Backup deleted successfully' : 'Failed to delete backup'
            ]);
            break;

        case 'list':
            // List recent backups
            $limit = $data['limit'] ?? 10;

            $backups = ExodusEngine::listRecentBackups($limit);

            echo json_encode([
                'success' => true,
                'backups' => $backups,
                'count' => count($backups)
            ]);
            break;

        default:
            throw new Exception('Unknown action: ' . $action);
    }

} catch (Exception $e) {
    ErrorHandler::logError('Exodus backup API error: ' . $e->getMessage());

    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
