<?php
/**
 * api/tableOperations.php - API endpoint for table operations
 * Handles copy, move, rename, delete, and export operations
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/TableOperations.php';
require_once __DIR__ . '/../lib/ErrorHandler.php';

header('Content-Type: application/json');

try {
    // Handle GET requests (getTables, getTableInfo)
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $action = $_GET['action'] ?? '';

        switch ($action) {
            case 'getTables':
                $database = $_GET['database'] ?? '';
                if (empty($database)) {
                    ErrorHandler::jsonError('Database name required', 'MISSING_PARAM', 400);
                }

                $tables = TableOperations::getTablesInDatabase($database);
                echo json_encode([
                    'success' => true,
                    'tables' => $tables
                ]);
                break;

            case 'getTableInfo':
                $database = $_GET['database'] ?? '';
                $table = $_GET['table'] ?? '';

                if (empty($database) || empty($table)) {
                    ErrorHandler::jsonError('Database and table names required', 'MISSING_PARAM', 400);
                }

                $info = TableOperations::getTableInfo($database, $table);
                echo json_encode([
                    'success' => true,
                    'info' => $info
                ]);
                break;

            default:
                ErrorHandler::jsonError('Invalid action', 'INVALID_ACTION', 400);
        }
        exit;
    }

    // Handle POST requests (copy, move, rename, delete, export)
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);

        if (!$data) {
            ErrorHandler::jsonError('Invalid JSON input', 'INVALID_JSON', 400);
        }

        $action = $data['action'] ?? '';

        switch ($action) {
            case 'copy':
                $sourceDb = $data['sourceDb'] ?? '';
                $sourceTable = $data['sourceTable'] ?? '';
                $targetDb = $data['targetDb'] ?? '';
                $targetTable = $data['targetTable'] ?? '';
                $options = $data['options'] ?? [];

                if (empty($sourceDb) || empty($sourceTable) || empty($targetDb) || empty($targetTable)) {
                    ErrorHandler::jsonError('Missing required parameters', 'MISSING_PARAM', 400);
                }

                $result = TableOperations::copyTable($sourceDb, $sourceTable, $targetDb, $targetTable, $options);
                echo json_encode($result);
                break;

            case 'move':
                $sourceDb = $data['sourceDb'] ?? '';
                $sourceTable = $data['sourceTable'] ?? '';
                $targetDb = $data['targetDb'] ?? '';
                $targetTable = $data['targetTable'] ?? '';
                $options = $data['options'] ?? [];

                if (empty($sourceDb) || empty($sourceTable) || empty($targetDb) || empty($targetTable)) {
                    ErrorHandler::jsonError('Missing required parameters', 'MISSING_PARAM', 400);
                }

                $result = TableOperations::moveTable($sourceDb, $sourceTable, $targetDb, $targetTable, $options);
                echo json_encode($result);
                break;

            case 'rename':
                $database = $data['database'] ?? '';
                $oldName = $data['oldName'] ?? '';
                $newName = $data['newName'] ?? '';

                if (empty($database) || empty($oldName) || empty($newName)) {
                    ErrorHandler::jsonError('Missing required parameters', 'MISSING_PARAM', 400);
                }

                $result = TableOperations::renameTable($database, $oldName, $newName);
                echo json_encode($result);
                break;

            case 'delete':
                $database = $data['database'] ?? '';
                $tableName = $data['tableName'] ?? '';

                if (empty($database) || empty($tableName)) {
                    ErrorHandler::jsonError('Missing required parameters', 'MISSING_PARAM', 400);
                }

                $result = TableOperations::deleteTable($database, $tableName);
                echo json_encode($result);
                break;

            case 'export':
                $database = $data['database'] ?? '';
                $tableName = $data['tableName'] ?? '';
                $options = $data['options'] ?? [];

                if (empty($database) || empty($tableName)) {
                    ErrorHandler::jsonError('Missing required parameters', 'MISSING_PARAM', 400);
                }

                $result = TableOperations::exportStructure($database, $tableName, $options);
                echo json_encode($result);
                break;

            default:
                ErrorHandler::jsonError('Invalid action', 'INVALID_ACTION', 400);
        }
        exit;
    }

    // Invalid method
    ErrorHandler::jsonError('Method not allowed', 'INVALID_METHOD', 405);

} catch (Exception $e) {
    ErrorHandler::handleException($e);
}
