<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>Gate 5: The Observatory</title>
<style>
    * { box-sizing: border-box; }

    body {
        background: radial-gradient(circle at top, #020d18 0%, #000000 70%);
        font-family: 'Consolas', 'Monaco', monospace;
        color: #e0f7ff;
        margin: 0;
        padding: 0;
        min-height: 100vh;
    }

    .panel {
        background: rgba(0,0,0,0.35);
        border: 1px solid #00eaff55;
        box-shadow: 0 0 20px #00eaff44;
        margin: 20px auto;
        padding: 25px;
        border-radius: 12px;
        width: 90%;
        max-width: 1200px;
        animation: fadein 0.6s ease;
    }

    @keyframes fadein {
        from { opacity:0; transform: translateY(20px); }
        to { opacity:1; transform: translateY(0); }
    }

    h2 {
        text-shadow: 0 0 12px #00eaff;
        font-weight: bold;
        margin-bottom: 12px;
        margin-top: 0;
        color: #e0f7ff;
    }

    h3 {
        color: #00ff88;
        text-shadow: 0 0 8px #00ff88;
        margin-top: 20px;
    }

    select {
        background: #000;
        color: #00eaff;
        border: 1px solid #00eaff55;
        padding: 8px 12px;
        border-radius: 4px;
        font-family: 'Consolas', monospace;
        transition: all 0.3s ease;
        min-width: 200px;
    }

    select:focus {
        outline: none;
        border-color: #00eaff;
        box-shadow: 0 0 12px #00eaff88;
    }

    label {
        color: #00ffaa;
        font-weight: bold;
        margin-right: 10px;
    }

    .status {
        margin-top: 15px;
        padding: 10px;
        border-radius: 6px;
    }
    .status.error {
        color: #ff3366;
        text-shadow: 0 0 12px #ff3366;
        font-weight: bold;
    }

    /* Grid Layout */
    .grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
        gap: 20px;
        margin-top: 20px;
    }

    .card {
        background: rgba(0, 10, 20, 0.6);
        border: 1px solid #00eaff33;
        padding: 20px;
        border-radius: 10px;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        text-decoration: none;
        display: block;
        position: relative;
        overflow: hidden;
    }

    .card:hover {
        border-color: #00eaff;
        box-shadow: 0 0 30px #00eaff33;
        transform: translateY(-5px);
        background: rgba(0, 20, 30, 0.8);
    }

    .card h3 {
        margin: 0 0 15px 0;
        color: #00eaff;
        font-size: 1.4em;
        border-bottom: 1px solid #00eaff33;
        padding-bottom: 10px;
    }

    .card-meta {
        font-size: 0.9em;
        color: #8899aa;
        margin-bottom: 8px;
        display: flex;
        justify-content: space-between;
    }

    .card-meta span {
        color: #00ff88;
        font-weight: bold;
    }

    .card-actions {
        margin-top: 15px;
        display: flex;
        gap: 10px;
    }

    .btn-small {
        background: rgba(0, 234, 255, 0.1);
        border: 1px solid #00eaff;
        color: #00eaff;
        padding: 5px 10px;
        border-radius: 4px;
        font-size: 0.8em;
        text-decoration: none;
        transition: all 0.2s;
        text-transform: uppercase;
    }

    .btn-small:hover {
        background: #00eaff;
        color: #000;
        box-shadow: 0 0 10px #00eaff;
    }

    .back-link {
        display: inline-block;
        margin-bottom: 20px;
        color: #00eaff;
        text-decoration: none;
        font-weight: bold;
        transition: all 0.3s;
    }
    .back-link:hover {
        text-shadow: 0 0 8px #00eaff;
        transform: translateX(-5px);
    }
</style>
</head>
<body>

<div class="panel">
    <a href="index.php" class="back-link">← Return to Threshold</a>
    <h2>✨ Gate 5: The Observatory</h2>
    <p style="color: #8899aa; margin-bottom: 25px;">Select a database to materialize its structure.</p>
    
    <div style="margin-bottom: 10px;">
        <label>Target Database:</label>
        <select id="dbSelect" onchange="loadTables()">
            <option value="">-- Loading --</option>
        </select>
    </div>
    <div id="status" class="status"></div>
</div>

<div id="tablesPanel" class="panel" style="display:none;">
    <h3 id="dbTitle">Available Tables</h3>
    <div id="grid" class="grid"></div>
</div>

    <script>
document.addEventListener('DOMContentLoaded', loadDatabases);
    
// Modal functionality
function openCRUDModal(url) {
    fetch(url)
        .then(r => r.text())
        .then(html => {
            document.querySelector('#crudModal').innerHTML = html;
            document.querySelector('#crudModal').classList.add('open');
        })
        .catch(err => {
            console.error('Modal loading error:', err);
        });
}

// Added missing viewer modal function
function openViewerModal(db, table) {
    const url = `viewer.php?db=${encodeURIComponent(db)}&table=${encodeURIComponent(table)}`;
    fetch(url)
        .then(r => r.text())
        .then(html => {
            // Reuse the same modal container
            const modal = document.querySelector('#crudModal');
            if (modal) {
                modal.innerHTML = html;
                modal.classList.add('open');
            }
        })
        .catch(err => {
            console.error('Viewer modal loading error:', err);
        });
}

function closeModal() {
    document.getElementById('crudModal').classList.remove('open');
}

// Close modal when clicking outside content
document.addEventListener('click', function(event) {
    const modal = document.getElementById('crudModal');
    if (modal && modal.classList.contains('open')) {
        if (event.target === modal) {
            closeModal();
        }
    }
});

// Close modal with Escape key
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        closeModal();
    }
});

// Track currently selected database 
let currentDatabase = '';

// Function to update CRUD button href 
function updateCRUDButtonHref() {
    const crudButton = document.querySelector('.full-crud-button');
    if (crudButton && currentDatabase) {
        crudButton.href = `/importer/crud/?db=${encodeURIComponent(currentDatabase)}&t=eleyeme_cfdi_emitidos`;
    }
}

// Function to handle table click 
function handleTableClick(db, table) {
    currentDatabase = db;
    updateCRUDButtonHref();
}


function loadDatabases() {
    fetch('api/getDatabases.php')
        .then(r => r.json())
        .then(data => {
            const select = document.getElementById('dbSelect');
            select.innerHTML = '<option value="">-- Select a Database --</option>';
            
            if(data.status === 'ok') {
                data.databases.forEach(db => {
                    const opt = document.createElement('option');
                    opt.value = db;
                    opt.textContent = db;
                    select.appendChild(opt);
                });
                
                // Auto-select if in memory
                const lastDb = localStorage.getItem('lastObservedDb');
                if (lastDb && data.databases.includes(lastDb)) {
                    select.value = lastDb;
                    loadTables();
                }
            } else {
                showError(data.message);
            }
        })
        .catch(err => showError('Failed to load databases: ' + err.message));
}

function loadTables() {
    const db = document.getElementById('dbSelect').value;
    const panel = document.getElementById('tablesPanel');
    const grid = document.getElementById('grid');
    const status = document.getElementById('status');
    
    if (!db) {
        panel.style.display = 'none';
        return;
    }
    
    // Save to daemon memory
    localStorage.setItem('lastObservedDb', db);
    
    status.innerHTML = '🌊 Scanning sector...';
    status.className = 'status';
    
    fetch(`api/getTables.php?db=${encodeURIComponent(db)}`)
        .then(r => r.json())
        .then(data => {
            if(data.status === 'ok') {
                status.innerHTML = '';
                renderGrid(data.tables, db);
                panel.style.display = 'block';
                document.getElementById('dbTitle').textContent = `Tables in [${db}]`;
                currentDatabase = db;
                updateCRUDButtonHref();
            } else {
                showError(data.message);
                panel.style.display = 'none';
            }
        })
        .catch(err => showError('Failed to load tables: ' + err.message));
}

function renderGrid(tables, db) {
    const grid = document.getElementById('grid');
    grid.innerHTML = '';
    
    if (tables.length === 0) {
        grid.innerHTML = '<div style="color: #888; grid-column: 1/-1; text-align: center;">No tables found in this sector.</div>';
        return;
    }
    
    tables.forEach(t => {
        const card = document.createElement('div');
        card.className = 'card';
        card.onclick = (e) => {
            if (!e.target.classList.contains('btn-small')) {
                window.location.href = `viewer.php?db=${encodeURIComponent(db)}&table=${encodeURIComponent(t.name)}`;
            }
        };
        
        card.innerHTML = `
            <h3>${escapeHtml(t.name)}</h3>
            <div class="card-meta">Rows: <span>${formatNumber(t.rows)}</span></div>
            <div class="card-meta">Size: <span>${formatBytes(t.data_length)}</span></div>
            <div class="card-meta">Engine: <span>${t.engine}</span></div>
            <div class="card-meta">Created: <span>${t.created?.split(' ')[0]}</span></div>
            
            <div class="card-actions">
                <a href="viewer.php?db=${escapeHtml(db)}&table=${escapeHtml(t.name)}" class="btn-small">View Data</a>
                <a href="#" class="btn-small full-crud-button" onclick="handleTableClick('${escapeHtml(db)}', '${escapeHtml(t.name)}')" style="border-color: #00ff88; color: #00ff88;">✨ Full CRUD</a>
                <button class="btn-small" onclick="openCRUDModal('/importer/crud/?db=${encodeURIComponent(db)}&t=${encodeURIComponent(t.name)}')" style="border-color: #ff3366; color: #ff3366;">🌀 CRUD Modal</button>
                <button class="btn-small" style="border-color: #00ff88; color: #00ff88;" onclick="generateAllForms('${escapeHtml(db)}', '${escapeHtml(t.name)}', event)">📜 Build All Forms</button>
            </div>
        `;
        
        grid.appendChild(card);
    });
}

function generateAllForms(db, table, event) {
    event.preventDefault();
    event.stopPropagation();

    const status = document.getElementById('status');
    status.className = 'status';
    status.textContent = `⚡ Generating all forms for [${table}]...`;

    const devices = ['mobile', 'tablet', 'desktop'];
    const promises = devices.map(device => {
        return fetch(`api/generate_form.php?db=${encodeURIComponent(db)}&table=${encodeURIComponent(table)}&device=${encodeURIComponent(device)}`)
            .then(r => r.json())
            .then(data => {
                if (data.status !== 'ok') {
                    throw new Error(`Failed to generate ${device} form: ${data.message}`);
                }
                return { device, path: data.path };
            });
    });

    Promise.all(promises)
        .then(results => {
            status.className = 'status success';
            let linksHtml = results.map(res => 
                `<a href="${res.path}" target="_blank" style="color: #00ff88; margin-right: 15px;">Open ${res.device} form</a>`
            ).join('');
            status.innerHTML = `✓ All forms generated! ${linksHtml}`;
        })
        .catch(err => {
            showError('Form generation failed: ' + err.message);
        });
}

function generateForm(db, table, device, event) {
    event.preventDefault();
    event.stopPropagation();

    const status = document.getElementById('status');
    status.className = 'status';
    status.textContent = `⚡ Generating ${device} form for [${table}]...`;

    fetch(`api/generate_form.php?db=${encodeURIComponent(db)}&table=${encodeURIComponent(table)}&device=${encodeURIComponent(device)}`)
        .then(r => r.json())
        .then(data => {
            if (data.status === 'ok') {
                status.className = 'status success';
                status.innerHTML = `✓ Form generated! <a href="${data.path}" target="_blank" style="color: #00ff88;">Open ${device} form</a>`;
            } else {
                showError(data.message);
            }
        })
        .catch(err => {
            showError('Form generation failed: ' + err.message);
        });
}

function showError(msg) {
    const status = document.getElementById('status');
    status.className = 'status error';
    status.textContent = '⚠ ' + msg;
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatNumber(num) {
    return new Intl.NumberFormat().format(num);
}

function formatBytes(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}
</script>

<!-- Modal container for viewer and CRUD -->
<div id="crudModal" class="modal"></div>

</body>
