<?php
/**
 * 🌌 ExodusEngine - Cross-Platform Migration System
 *
 * Handles backup creation, archive management, and platform-aware migrations
 * for the Cyberpunk Data Importer.
 *
 * Features:
 * - Smart backup creation (code, vendor, uploads, database)
 * - Multi-format archives (tar.gz, zip)
 * - Platform detection and adaptation (Linux ↔ Windows)
 * - Manifest generation with checksums
 * - Differential backup support
 *
 * "The grid remembers. The exodus preserves."
 */

class ExodusEngine {

    const VERSION = '1.0.0';
    const BACKUP_DIR = __DIR__ . '/../backups/';
    const MIGRATION_DIR = __DIR__ . '/../migrations/';

    /**
     * Create a full backup with all options
     *
     * @param array $options Backup configuration
     * @return array Result with file path and metadata
     */
    public static function createFullBackup($options = []) {
        $defaults = [
            'include_code' => true,
            'include_vendor' => true,
            'include_uploads' => true,
            'include_logs' => false,  // Exclude by default (can be large)
            'include_database' => true,
            'include_tmp' => false,  // Exclude tmp by default
            'format' => 'tar.gz',  // tar.gz or zip
            'platform' => 'auto'  // auto, linux, windows
        ];

        $options = array_merge($defaults, $options);

        // Ensure backup directory exists
        self::ensureBackupDirectory();

        // Detect platform if auto
        if ($options['platform'] === 'auto') {
            $options['platform'] = self::detectPlatform();
        }

        // Generate backup filename
        $timestamp = date('Y-m-d_H-i-s');
        $platform = $options['platform'];
        $ext = $options['format'] === 'zip' ? 'zip' : 'tar.gz';
        $filename = "exodus_{$timestamp}_{$platform}_full.{$ext}";
        $filepath = self::BACKUP_DIR . $filename;

        // Collect files to backup
        $filesToBackup = self::collectFiles($options);

        // Create manifest
        $manifest = self::createManifest([
            'filename' => $filename,
            'created_at' => date('Y-m-d H:i:s'),
            'platform_source' => self::detectPlatform(),
            'platform_target' => $platform,
            'options' => $options,
            'files' => $filesToBackup,
            'version' => self::VERSION
        ]);

        // Create archive
        if ($options['format'] === 'zip') {
            $result = self::createZip($filesToBackup, $filepath, $manifest);
        } else {
            $result = self::createTarGz($filesToBackup, $filepath, $manifest);
        }

        // Generate database backup if requested
        if ($options['include_database']) {
            $dbBackup = self::createDatabaseBackup();
            $result['database_backup'] = $dbBackup;
        }

        // Add manifest to result
        $result['manifest'] = $manifest;

        return $result;
    }

    /**
     * Create code-only backup (minimal deployment)
     *
     * @return array Result with file path
     */
    public static function createCodeBackup() {
        return self::createFullBackup([
            'include_code' => true,
            'include_vendor' => false,
            'include_uploads' => false,
            'include_logs' => false,
            'include_database' => false,
            'format' => 'tar.gz'
        ]);
    }

    /**
     * Create database schema backup with migrations
     *
     * @return array Database backup info
     */
    public static function createDatabaseBackup() {
        require_once 'DatabaseMigrator.php';

        $timestamp = date('Y-m-d_H-i-s');
        $sqlFile = self::MIGRATION_DIR . "schema_{$timestamp}.sql";

        // Get all databases to backup
        $databases = ['importer_db'];  // Primary database

        $allTables = [];
        $totalSize = 0;

        foreach ($databases as $dbName) {
            $tables = DatabaseMigrator::exportDatabaseSchema($dbName, $sqlFile);
            $allTables = array_merge($allTables, $tables);
            $totalSize += filesize($sqlFile);
        }

        return [
            'sql_file' => $sqlFile,
            'databases' => $databases,
            'tables' => $allTables,
            'size_bytes' => $totalSize,
            'created_at' => date('Y-m-d H:i:s')
        ];
    }

    /**
     * Create tar.gz archive
     *
     * @param array $files Files to include
     * @param string $destination Output file path
     * @param array $manifest Manifest data
     * @return array Result
     */
    public static function createTarGz($files, $destination, $manifest) {
        $baseDir = __DIR__ . '/../';

        // Create temporary manifest file
        $manifestPath = sys_get_temp_dir() . '/exodus_manifest_' . uniqid() . '.json';
        file_put_contents($manifestPath, json_encode($manifest, JSON_PRETTY_PRINT));

        // Build tar command
        $tarFiles = [];
        foreach ($files as $file) {
            // Make path relative to base directory
            $relativePath = str_replace($baseDir, '', $file);
            $tarFiles[] = escapeshellarg($relativePath);
        }

        // Add manifest
        $tarFiles[] = escapeshellarg(str_replace($baseDir, '', $manifestPath));

        // Create tar.gz (using system tar for better compression)
        $tarCommand = sprintf(
            'cd %s && tar -czf %s %s 2>&1',
            escapeshellarg($baseDir),
            escapeshellarg($destination),
            implode(' ', $tarFiles)
        );

        exec($tarCommand, $output, $returnCode);

        // Clean up temp manifest
        if (file_exists($manifestPath)) {
            unlink($manifestPath);
        }

        if ($returnCode !== 0) {
            throw new Exception('Failed to create tar.gz: ' . implode("\n", $output));
        }

        $fileSize = filesize($destination);

        return [
            'success' => true,
            'file_path' => $destination,
            'filename' => basename($destination),
            'size_bytes' => $fileSize,
            'size_mb' => round($fileSize / 1024 / 1024, 2),
            'file_count' => count($files),
            'format' => 'tar.gz'
        ];
    }

    /**
     * Create ZIP archive (Windows-friendly)
     *
     * @param array $files Files to include
     * @param string $destination Output file path
     * @param array $manifest Manifest data
     * @return array Result
     */
    public static function createZip($files, $destination, $manifest) {
        if (!class_exists('ZipArchive')) {
            throw new Exception('ZipArchive extension not available');
        }

        $zip = new ZipArchive();
        if ($zip->open($destination, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new Exception('Failed to create ZIP archive');
        }

        $baseDir = __DIR__ . '/../';

        // Add files to ZIP
        foreach ($files as $file) {
            if (is_file($file)) {
                $relativePath = str_replace($baseDir, '', $file);
                $zip->addFile($file, $relativePath);
            }
        }

        // Add manifest
        $zip->addFromString('exodus_manifest.json', json_encode($manifest, JSON_PRETTY_PRINT));

        $zip->close();

        $fileSize = filesize($destination);

        return [
            'success' => true,
            'file_path' => $destination,
            'filename' => basename($destination),
            'size_bytes' => $fileSize,
            'size_mb' => round($fileSize / 1024 / 1024, 2),
            'file_count' => count($files),
            'format' => 'zip'
        ];
    }

    /**
     * Collect files to backup based on options
     *
     * @param array $options Backup options
     * @return array List of file paths
     */
    private static function collectFiles($options) {
        $baseDir = __DIR__ . '/../';
        $files = [];

        // Always include essential files
        $essentialFiles = [
            'config.php',
            'index.php',
            'readme.md',
            '.htaccess'
        ];

        foreach ($essentialFiles as $file) {
            $path = $baseDir . $file;
            if (file_exists($path)) {
                $files[] = $path;
            }
        }

        // Include code files
        if ($options['include_code']) {
            $codeDirs = ['lib', 'api', 'crud', 'forms'];
            foreach ($codeDirs as $dir) {
                $dirPath = $baseDir . $dir;
                if (is_dir($dirPath)) {
                    $files = array_merge($files, self::scanDirectory($dirPath));
                }
            }

            // Include root PHP files
            $rootPhpFiles = glob($baseDir . '*.php');
            $files = array_merge($files, $rootPhpFiles);

            // Include markdown docs
            $docFiles = glob($baseDir . '*.md');
            $files = array_merge($files, $docFiles);
        }

        // Include vendor directory
        if ($options['include_vendor']) {
            $vendorPath = $baseDir . 'vendor';
            if (is_dir($vendorPath)) {
                $files = array_merge($files, self::scanDirectory($vendorPath));
            }
        }

        // Include uploads
        if ($options['include_uploads']) {
            $uploadsPath = $baseDir . 'uploads';
            if (is_dir($uploadsPath)) {
                $files = array_merge($files, self::scanDirectory($uploadsPath));
            }
        }

        // Include logs
        if ($options['include_logs']) {
            $logsPath = $baseDir . 'logs';
            if (is_dir($logsPath)) {
                $files = array_merge($files, self::scanDirectory($logsPath));
            }
        }

        // Include tmp directory (usually excluded)
        if ($options['include_tmp']) {
            $tmpPath = $baseDir . 'tmp';
            if (is_dir($tmpPath)) {
                $files = array_merge($files, self::scanDirectory($tmpPath));
            }
        }

        // Remove duplicates
        $files = array_unique($files);

        return $files;
    }

    /**
     * Recursively scan directory for files
     *
     * @param string $dir Directory path
     * @return array List of file paths
     */
    private static function scanDirectory($dir) {
        $files = [];

        if (!is_dir($dir)) {
            return $files;
        }

        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $files[] = $file->getPathname();
            }
        }

        return $files;
    }

    /**
     * Create manifest file with metadata and checksums
     *
     * @param array $data Manifest data
     * @return array Manifest
     */
    public static function createManifest($data) {
        $manifest = [
            'exodus_version' => self::VERSION,
            'created_at' => $data['created_at'] ?? date('Y-m-d H:i:s'),
            'platform_source' => $data['platform_source'] ?? self::detectPlatform(),
            'platform_target' => $data['platform_target'] ?? 'auto',
            'php_version' => PHP_VERSION,
            'importer_version' => '1.0.0',  // TODO: Get from version file
            'filename' => $data['filename'] ?? 'unknown',
            'options' => $data['options'] ?? [],
            'statistics' => [
                'total_files' => count($data['files'] ?? []),
                'total_size_bytes' => self::calculateTotalSize($data['files'] ?? []),
                'directories' => self::getDirectoryList($data['files'] ?? [])
            ],
            'checksums' => self::generateChecksums($data['files'] ?? [])
        ];

        return $manifest;
    }

    /**
     * Calculate total size of files
     *
     * @param array $files File paths
     * @return int Total size in bytes
     */
    private static function calculateTotalSize($files) {
        $total = 0;
        foreach ($files as $file) {
            if (is_file($file)) {
                $total += filesize($file);
            }
        }
        return $total;
    }

    /**
     * Get list of unique directories
     *
     * @param array $files File paths
     * @return array Directory names
     */
    private static function getDirectoryList($files) {
        $baseDir = __DIR__ . '/../';
        $dirs = [];

        foreach ($files as $file) {
            $relativePath = str_replace($baseDir, '', $file);
            $dirParts = explode('/', dirname($relativePath));
            if (!empty($dirParts[0])) {
                $dirs[] = $dirParts[0];
            }
        }

        return array_values(array_unique($dirs));
    }

    /**
     * Generate MD5 checksums for critical files
     *
     * @param array $files File paths
     * @return array Checksums (filename => md5)
     */
    private static function generateChecksums($files) {
        $checksums = [];
        $baseDir = __DIR__ . '/../';

        // Only checksum PHP files and config files (skip uploads, vendor, etc.)
        foreach ($files as $file) {
            if (!is_file($file)) continue;

            $ext = pathinfo($file, PATHINFO_EXTENSION);
            $relativePath = str_replace($baseDir, '', $file);

            // Checksum: PHP files, config files, SQL files
            if (in_array($ext, ['php', 'sql', 'htaccess']) || basename($file) === 'config.php') {
                $checksums[$relativePath] = md5_file($file);
            }
        }

        return $checksums;
    }

    /**
     * Detect current platform (Linux or Windows)
     *
     * @return string 'linux' or 'windows'
     */
    public static function detectPlatform() {
        return strtoupper(substr(PHP_OS, 0, 3)) === 'WIN' ? 'windows' : 'linux';
    }

    /**
     * Get optimal archive format for platform
     *
     * @param string $platform Target platform
     * @return string 'tar.gz' or 'zip'
     */
    public static function getOptimalArchiveFormat($platform = null) {
        if ($platform === null) {
            $platform = self::detectPlatform();
        }

        return $platform === 'windows' ? 'zip' : 'tar.gz';
    }

    /**
     * Ensure backup directory exists
     */
    private static function ensureBackupDirectory() {
        if (!is_dir(self::BACKUP_DIR)) {
            mkdir(self::BACKUP_DIR, 0755, true);
        }

        // Create .gitignore to exclude backups from git
        $gitignorePath = self::BACKUP_DIR . '.gitignore';
        if (!file_exists($gitignorePath)) {
            file_put_contents($gitignorePath, "# Exclude backup files\n*.tar.gz\n*.zip\n*.sql\n");
        }
    }

    /**
     * Ensure migration directory exists
     */
    public static function ensureMigrationDirectory() {
        if (!is_dir(self::MIGRATION_DIR)) {
            mkdir(self::MIGRATION_DIR, 0755, true);
        }
    }

    /**
     * List recent backups
     *
     * @param int $limit Number of backups to return
     * @return array List of backups with metadata
     */
    public static function listRecentBackups($limit = 10) {
        self::ensureBackupDirectory();

        $backups = [];
        $files = glob(self::BACKUP_DIR . 'exodus_*');

        // Sort by modification time (newest first)
        usort($files, function($a, $b) {
            return filemtime($b) - filemtime($a);
        });

        $files = array_slice($files, 0, $limit);

        foreach ($files as $file) {
            $backups[] = [
                'filename' => basename($file),
                'path' => $file,
                'size_bytes' => filesize($file),
                'size_mb' => round(filesize($file) / 1024 / 1024, 2),
                'created_at' => date('Y-m-d H:i:s', filemtime($file)),
                'age_hours' => round((time() - filemtime($file)) / 3600, 1)
            ];
        }

        return $backups;
    }

    /**
     * Delete a backup file
     *
     * @param string $filename Backup filename
     * @return bool Success
     */
    public static function deleteBackup($filename) {
        $filepath = self::BACKUP_DIR . basename($filename);  // Sanitize

        if (file_exists($filepath) && strpos($filepath, 'exodus_') !== false) {
            return unlink($filepath);
        }

        return false;
    }
}
