<?php
/**
 * SchemaConventions - Central Authority for Database Field Naming
 *
 * This class provides the single source of truth for all standard field naming
 * conventions in the importer system. It handles:
 * - Table name singularization (plural → singular)
 * - Dynamic primary key naming ({singular_table}_id)
 * - Spanish canonical audit field definitions
 * - Field categorization (auto-managed vs user-defined)
 * - Dual-mode support (English legacy vs Spanish canonical)
 *
 * @package Importer
 * @version 2.0.0
 */

class SchemaConventions
{
    /**
     * Explicit pluralization map for known tables
     * Format: 'plural_table_name' => 'singular_form'
     *
     * This whitelist ensures deterministic behavior for critical tables.
     * When adding new tables, add them here to avoid fallback rule ambiguity.
     */
    private const IRREGULAR_PLURALS = [
        'propietarios' => 'propietario',
        'departamentos' => 'departamento',
        'alumnos_becados' => 'alumno_becado',
        'eleyeme_cfdi_emitidos' => 'eleyeme_cfdi_emitido',
        'productos' => 'producto',
        'servicios' => 'servicio',
        'empleados' => 'empleado',
        'clientes' => 'cliente',
        'proveedores' => 'proveedor',
        'contratos' => 'contrato',
        'facturas' => 'factura',
        'pagos' => 'pago',
        'usuarios' => 'usuario',
        'roles' => 'rol',
        'permisos' => 'permiso',
        'catalogos' => 'catalogo',
        'categorias' => 'categoria',
        'sucursales' => 'sucursal',
        'almacenes' => 'almacen',
        'inventarios' => 'inventario',
        'movimientos' => 'movimiento',
        'documentos' => 'documento',
        'archivos' => 'archivo',
        'reportes' => 'reporte',
        'configuraciones' => 'configuracion',
        'bitacoras' => 'bitacora',
        'auditorias' => 'auditoria',
        'notificaciones' => 'notificacion',
        'mensajes' => 'mensaje',
        'tareas' => 'tarea',
        'proyectos' => 'proyecto',
        'eventos' => 'evento',
        'registros' => 'registro',
    ];

    /**
     * Spanish canonical field definitions (NEW STANDARD)
     * These are the standard audit fields for all new tables.
     */
    private const SPANISH_FIELDS = [
        'alta_db' => [
            'type' => 'TIMESTAMP',
            'constraint' => 'DEFAULT CURRENT_TIMESTAMP',
            'comment' => 'Timestamp del momento de inserción del registro'
        ],
        'alta_por' => [
            'type' => 'VARCHAR(32)',
            'constraint' => "DEFAULT 'system'",
            'comment' => 'Usuario que insertó el registro'
        ],
        'ultimo_cambio' => [
            'type' => 'TIMESTAMP',
            'constraint' => 'NULL DEFAULT NULL',
            'comment' => 'Timestamp del último cambio al registro'
        ],
        'ultimo_cambio_por' => [
            'type' => 'VARCHAR(32)',
            'constraint' => 'NULL DEFAULT NULL',
            'comment' => 'Usuario que hizo el último cambio'
        ]
    ];

    /**
     * Legacy English field definitions (BACKWARD COMPATIBILITY)
     * Used for existing tables that haven't been migrated yet.
     */
    private const LEGACY_ENGLISH_FIELDS = [
        'insert_db' => [
            'type' => 'TIMESTAMP',
            'constraint' => 'DEFAULT CURRENT_TIMESTAMP',
            'comment' => 'The timestamp of the moment the row was inserted'
        ],
        'insert_by' => [
            'type' => 'VARCHAR(32)',
            'constraint' => "DEFAULT 'system'",
            'comment' => 'The nick of the user who inserted the row'
        ],
        'last_change' => [
            'type' => 'TIMESTAMP',
            'constraint' => 'NULL DEFAULT NULL',
            'comment' => 'The timestamp of the moment the row was updated'
        ],
        'last_change_by' => [
            'type' => 'VARCHAR(32)',
            'constraint' => 'NULL DEFAULT NULL',
            'comment' => 'The nick of the user who updated the row'
        ]
    ];

    /**
     * Converts plural table name to singular form
     *
     * Strategy:
     * 1. Check explicit IRREGULAR_PLURALS whitelist first (deterministic)
     * 2. If ends with 's', strip trailing 's' (fallback rule)
     * 3. Otherwise, return unchanged
     *
     * @param string $tableName Plural table name (e.g., 'propietarios')
     * @return string Singular form (e.g., 'propietario')
     */
    public static function singularize(string $tableName): string
    {
        // Priority 1: Check explicit whitelist
        if (isset(self::IRREGULAR_PLURALS[$tableName])) {
            return self::IRREGULAR_PLURALS[$tableName];
        }

        // Priority 2: Fallback rule - strip trailing 's'
        if (substr($tableName, -1) === 's') {
            error_log("SchemaConventions: Using fallback singularization for '{$tableName}' (stripped trailing 's')");
            return substr($tableName, 0, -1);
        }

        // Priority 3: Already singular or unknown pattern
        error_log("SchemaConventions: Table '{$tableName}' appears already singular or has unknown plural pattern");
        return $tableName;
    }

    /**
     * Generates dynamic primary key name for a table
     *
     * @param string $tableName Table name (plural, e.g., 'propietarios')
     * @return string Primary key field name (e.g., 'propietario_id')
     */
    public static function getPrimaryKeyName(string $tableName): string
    {
        $singular = self::singularize($tableName);
        return "{$singular}_id";
    }

    /**
     * Gets standard audit fields for a table
     *
     * Supports dual-mode: auto-detects whether table uses English or Spanish fields.
     * If $dbName is provided, checks actual table schema to determine mode.
     *
     * @param string $tableName Table name
     * @param string|null $dbName Database name (optional, for dual-mode detection)
     * @return array Standard field definitions
     */
    public static function getStandardFields(string $tableName, ?string $dbName = null): array
    {
        // If database name provided, detect field mode from actual schema
        if ($dbName !== null) {
            $mode = self::detectFieldMode($tableName, $dbName);
            return ($mode === 'english') ? self::LEGACY_ENGLISH_FIELDS : self::SPANISH_FIELDS;
        }

        // Default: Spanish canonical for new tables
        return self::SPANISH_FIELDS;
    }

    /**
     * Gets list of auto-managed field names for a table
     *
     * These fields are automatically populated by the system and should be
     * excluded from user input during INSERT/UPDATE operations.
     *
     * @param string $tableName Table name
     * @param string|null $dbName Database name (optional, for dual-mode detection)
     * @return array List of auto-managed field names
     */
    public static function getAutoManagedFields(string $tableName, ?string $dbName = null): array
    {
        $primaryKey = self::getPrimaryKeyName($tableName);
        $standardFields = array_keys(self::getStandardFields($tableName, $dbName));

        // Combine primary key + all audit fields
        return array_merge([$primaryKey], $standardFields);
    }

    /**
     * Checks if a field name is a standard audit field
     *
     * Checks both Spanish canonical and English legacy field names.
     *
     * @param string $fieldName Field name to check
     * @return bool True if field is a standard audit field
     */
    public static function isStandardField(string $fieldName): bool
    {
        // Check if it's a primary key pattern (ends with _id)
        if (preg_match('/_id$/', $fieldName)) {
            return true;
        }

        // Check Spanish canonical fields
        if (array_key_exists($fieldName, self::SPANISH_FIELDS)) {
            return true;
        }

        // Check English legacy fields (for backward compatibility)
        if (array_key_exists($fieldName, self::LEGACY_ENGLISH_FIELDS)) {
            return true;
        }

        // Also check for old 'id' field (legacy)
        if ($fieldName === 'id') {
            return true;
        }

        return false;
    }

    /**
     * Gets the definition for a specific standard field
     *
     * @param string $fieldName Field name
     * @return array|null Field definition array, or null if not a standard field
     */
    public static function getStandardFieldDefinition(string $fieldName): ?array
    {
        // Check Spanish fields first
        if (array_key_exists($fieldName, self::SPANISH_FIELDS)) {
            return self::SPANISH_FIELDS[$fieldName];
        }

        // Check English legacy fields
        if (array_key_exists($fieldName, self::LEGACY_ENGLISH_FIELDS)) {
            return self::LEGACY_ENGLISH_FIELDS[$fieldName];
        }

        return null;
    }

    /**
     * Detects whether a table uses English (legacy) or Spanish (canonical) field naming
     *
     * Checks the actual table schema in the database:
     * - If 'insert_db' exists → English mode
     * - If 'alta_db' exists → Spanish mode
     * - Default: Spanish (for new tables)
     *
     * @param string $tableName Table name
     * @param string $dbName Database name
     * @return string 'english' or 'spanish'
     */
    public static function detectFieldMode(string $tableName, string $dbName): string
    {
        try {
            // Get actual column names from table schema
            require_once __DIR__ . '/DatabaseHelper.php';
            $columns = DatabaseHelper::getColumns($tableName, $dbName);
            $columnNames = array_column($columns, 'name');

            // Check for English legacy fields
            if (in_array('insert_db', $columnNames)) {
                return 'english';
            }

            // Check for Spanish canonical fields
            if (in_array('alta_db', $columnNames)) {
                return 'spanish';
            }

            // Default for new tables or tables without audit fields
            return 'spanish';
        } catch (Exception $e) {
            // If table doesn't exist or error occurs, default to Spanish
            error_log("SchemaConventions: Error detecting field mode for {$tableName}: " . $e->getMessage());
            return 'spanish';
        }
    }

    /**
     * Gets the legacy primary key name ('id') for backward compatibility
     *
     * @return string Legacy primary key name
     */
    public static function getLegacyPrimaryKeyName(): string
    {
        return 'id';
    }

    /**
     * Checks if a table exists in the irregular plurals whitelist
     *
     * @param string $tableName Table name to check
     * @return bool True if table is in whitelist
     */
    public static function isKnownTable(string $tableName): bool
    {
        return isset(self::IRREGULAR_PLURALS[$tableName]);
    }

    /**
     * Gets all known table mappings (for debugging/admin purposes)
     *
     * @return array Array of plural => singular mappings
     */
    public static function getKnownTables(): array
    {
        return self::IRREGULAR_PLURALS;
    }

    /**
     * Maps English field names to Spanish equivalents
     *
     * @param string $englishFieldName English field name
     * @return string Spanish field name
     */
    public static function mapEnglishToSpanish(string $englishFieldName): string
    {
        $mapping = [
            'id' => 'dynamic', // Special case - depends on table
            'insert_db' => 'alta_db',
            'insert_by' => 'alta_por',
            'last_change' => 'ultimo_cambio',
            'last_change_by' => 'ultimo_cambio_por',
        ];

        return $mapping[$englishFieldName] ?? $englishFieldName;
    }

    /**
     * Maps Spanish field names to English equivalents (for migration/compatibility)
     *
     * @param string $spanishFieldName Spanish field name
     * @return string English field name
     */
    public static function mapSpanishToEnglish(string $spanishFieldName): string
    {
        $mapping = [
            'alta_db' => 'insert_db',
            'alta_por' => 'insert_by',
            'ultimo_cambio' => 'last_change',
            'ultimo_cambio_por' => 'last_change_by',
        ];

        return $mapping[$spanishFieldName] ?? $spanishFieldName;
    }
}
