<?php
/**
 * Test the complete import flow to verify logging works correctly
 * This simulates what happens when a user imports a file
 */

// Start session like upload.php does
session_start();

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/ImportLogger.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/DataCleaner.php';

echo "<h1>Import Flow Test - Verifying Logging Fix</h1>\n";
echo "<hr>\n";

try {
    // ===== STEP 1: SIMULATE UPLOAD.PHP =====
    echo "<h2>Step 1: Simulating upload.php (file parsing)</h2>\n";
    
    // Simulate file metadata that upload.php would store in session
    $fileMetadata = [
        'file_name' => 'test_import_flow.csv',
        'file_type' => 'CSV',
        'file_size' => 1024,
        'table_name' => 'test_import_flow',
        'csv_delimiter' => 'comma',
        'csv_encoding' => 'UTF-8'
    ];
    
    // Store in session (this is what upload.php now does)
    $_SESSION['file_metadata'] = $fileMetadata;
    
    echo "<pre>✓ File metadata stored in session:\n";
    echo json_encode($fileMetadata, JSON_PRETTY_PRINT) . "\n</pre>\n";
    
    // ===== STEP 2: SIMULATE INSERT.PHP =====
    echo "<h2>Step 2: Simulating insert.php (database import)</h2>\n";
    
    // This is what insert.php now does
    $databaseName = 'importer_db';
    $tableName = 'test_import_flow';
    
    // Retrieve file metadata from session (set by upload.php)
    $sessionFileMetadata = $_SESSION['file_metadata'] ?? [];
    
    if (empty($sessionFileMetadata)) {
        throw new Exception('No file metadata found in session!');
    }
    
    echo "<pre>✓ Retrieved file metadata from session\n</pre>\n";
    
    // Merge with database information
    $logMetadata = array_merge($sessionFileMetadata, [
        'database_name' => $databaseName,
        'table_name' => $tableName,
        'alta_por' => 'system'
    ]);
    
    echo "<pre>✓ Merged with database information:\n";
    echo json_encode($logMetadata, JSON_PRETTY_PRINT) . "\n</pre>\n";
    
    // Start import log (this should now work!)
    $importLogId = ImportLogger::startImport($logMetadata);
    
    echo "<pre><strong>✓ Import log started successfully!</strong>\n";
    echo "Import Log ID: {$importLogId}\n</pre>\n";
    
    // Clear file metadata from session
    unset($_SESSION['file_metadata']);
    
    echo "<pre>✓ Cleared file metadata from session\n</pre>\n";
    
    // ===== STEP 3: VERIFY THE LOG ENTRY =====
    echo "<h2>Step 3: Verifying the log entry</h2>\n";
    
    $db = DatabaseHelper::getInstance();
    $conn = $db->getConnection();
    
    $stmt = $conn->prepare("SELECT * FROM import_logs WHERE import_log_id = ?");
    $stmt->bind_param("s", $importLogId);
    $stmt->execute();
    $result = $stmt->get_result();
    $logEntry = $result->fetch_assoc();
    
    if ($logEntry) {
        echo "<pre><strong>✓ Log entry found in database:</strong>\n";
        echo "Import ID: {$logEntry['import_log_id']}\n";
        echo "Database Name: {$logEntry['database_name']}\n";
        echo "Table Name: {$logEntry['table_name']}\n";
        echo "File Name: {$logEntry['file_name']}\n";
        echo "File Type: {$logEntry['file_type']}\n";
        echo "File Size: {$logEntry['file_size']}\n";
        echo "Status: {$logEntry['operation_status']}\n";
        echo "Date: {$logEntry['alta_db']}\n</pre>\n";
        
        // ===== VERIFY CRITICAL FIELDS =====
        echo "<h2>Step 4: Validation</h2>\n";
        
        $allGood = true;
        
        if ($logEntry['database_name'] === 'importer_db') {
            echo "<pre>✓ Database name is correctly set to 'importer_db'</pre>\n";
        } else {
            echo "<pre style='color: red;'>✗ Database name is NOT set correctly (expected: 'importer_db', got: '{$logEntry['database_name']}')</pre>\n";
            $allGood = false;
        }
        
        if ($logEntry['table_name'] === 'test_import_flow') {
            echo "<pre>✓ Table name is correctly set</pre>\n";
        } else {
            echo "<pre style='color: red;'>✗ Table name is NOT set correctly</pre>\n";
            $allGood = false;
        }
        
        if ($logEntry['file_name'] === 'test_import_flow.csv') {
            echo "<pre>✓ File name is correctly set</pre>\n";
        } else {
            echo "<pre style='color: red;'>✗ File name is NOT set correctly</pre>\n";
            $allGood = false;
        }
        
        if ($logEntry['operation_status'] === 'pending') {
            echo "<pre>✓ Status is correctly set to 'pending'</pre>\n";
        } else {
            echo "<pre style='color: red;'>✗ Status is NOT correct (expected: 'pending', got: '{$logEntry['operation_status']}')</pre>\n";
            $allGood = false;
        }
        
        // ===== FINAL RESULT =====
        echo "<hr>\n";
        if ($allGood) {
            echo "<h2 style='color: green;'>✓ SUCCESS! The logging fix is working correctly!</h2>\n";
            echo "<p>The import logger now properly captures the database name because logging is initialized in insert.php where the database name is available.</p>\n";
        } else {
            echo "<h2 style='color: red;'>✗ FAILED! There are still issues with the logging.</h2>\n";
        }
        
        // Clean up test entry
        echo "<hr>\n";
        echo "<h2>Cleanup</h2>\n";
        $conn->query("DELETE FROM import_logs WHERE import_log_id = '{$importLogId}'");
        echo "<pre>✓ Test log entry deleted</pre>\n";
        
    } else {
        throw new Exception("Log entry not found in database!");
    }
    
} catch (Exception $e) {
    echo "<pre style='color: red;'><strong>ERROR:</strong> {$e->getMessage()}\n";
    echo $e->getTraceAsString() . "</pre>\n";
}

echo "<hr>\n";
echo "<h3>Next Steps:</h3>\n";
echo "<ol>\n";
echo "<li>The fix has been tested and validated</li>\n";
echo "<li>You can now import files normally via <a href='index.php'>index.php</a></li>\n";
echo "<li>All imports will be properly logged with the correct database name</li>\n";
echo "<li>Check logs at any time using <a href='test_logging_fix.php'>test_logging_fix.php</a></li>\n";
echo "</ol>\n";
?>
