<?php
/**
 * Test script for ImportLogger
 * Verifies that import logging is working correctly
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/ImportLogger.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/ErrorHandler.php';

echo "=== Import Logger Test ===\n\n";

try {
    // Test 1: Ensure log tables exist
    echo "TEST 1: Checking if log tables exist...\n";
    ImportLogger::ensureLogTablesExist();
    echo "✅ Log tables created/verified successfully\n\n";
    
    // Test 2: Check tables exist in database
    echo "TEST 2: Verifying tables in database...\n";
    $db = DatabaseHelper::getInstance();
    $conn = $db->getConnection();
    $conn->select_db(ImportLogger::LOG_DATABASE);
    
    $tables = ['import_logger_meta', 'import_logs', 'import_schema_logs'];
    foreach ($tables as $table) {
        if ($db->tableExists($table)) {
            echo "✅ Table '$table' exists\n";
        } else {
            echo "❌ Table '$table' does NOT exist\n";
        }
    }
    echo "\n";
    
    // Test 3: Check version
    echo "TEST 3: Checking version...\n";
    $result = $conn->query("SELECT meta_value FROM import_logger_meta WHERE meta_key = 'version'");
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $version = $row['meta_value'];
        echo "✅ Current version: $version\n";
        echo "✅ Expected version: " . ImportLogger::CURRENT_VERSION . "\n";
        if ($version == ImportLogger::CURRENT_VERSION) {
            echo "✅ Versions match!\n";
        } else {
            echo "❌ Version mismatch!\n";
        }
    }
    echo "\n";
    
    // Test 4: Start a test import log
    echo "TEST 4: Creating test import log entry...\n";
    $testMetadata = [
        'file_name' => 'test_file.xlsx',
        'file_type' => 'XLSX',
        'file_size' => 12345,
        'database_name' => 'test_db',
        'table_name' => 'test_table',
        'alta_por' => 'test_user'
    ];
    
    $importLogId = ImportLogger::startImport($testMetadata);
    echo "✅ Import log created with ID: $importLogId\n\n";
    
    // Test 5: Log CREATE TABLE
    echo "TEST 5: Logging CREATE TABLE operation...\n";
    ImportLogger::logCreateTable($importLogId, [
        'database_name' => 'test_db',
        'table_name' => 'test_table',
        'create_sql' => 'CREATE TABLE test_table (id INT PRIMARY KEY, name VARCHAR(100));',
        'status' => 'success'
    ]);
    echo "✅ CREATE TABLE logged successfully\n\n";
    
    // Test 6: Log schema
    echo "TEST 6: Logging schema details...\n";
    $testSchema = [
        [
            'name' => 'nombre',
            'type' => 'VARCHAR',
            'length' => '100',
            'nullable' => false,
            'indexed' => true,
            'comment' => 'Nombre del usuario',
            'original' => 'Nombre'
        ],
        [
            'name' => 'edad',
            'type' => 'INT',
            'nullable' => true,
            'indexed' => false,
            'comment' => 'Edad del usuario',
            'original' => 'Edad'
        ]
    ];
    
    ImportLogger::logSchema($importLogId, $testSchema);
    echo "✅ Schema logged successfully\n\n";
    
    // Test 7: Update import status
    echo "TEST 7: Updating import status...\n";
    ImportLogger::updateImportStatus($importLogId, [
        'rows_processed' => 100,
        'rows_inserted' => 95,
        'rows_failed' => 5,
        'errors' => [
            ['row' => 10, 'reason' => 'Invalid data'],
            ['row' => 25, 'reason' => 'Duplicate entry']
        ]
    ]);
    echo "✅ Import status updated successfully\n\n";
    
    // Test 8: Retrieve import history
    echo "TEST 8: Retrieving import history...\n";
    $logs = ImportLogger::getImportHistory(['limit' => 5]);
    echo "✅ Retrieved " . count($logs) . " import log(s)\n";
    if (!empty($logs)) {
        $latestLog = $logs[0];
        echo "   Latest import:\n";
        echo "   - Table: {$latestLog['table_name']}\n";
        echo "   - Status: {$latestLog['operation_status']}\n";
        echo "   - Rows inserted: {$latestLog['rows_inserted']}\n";
        echo "   - Rows failed: {$latestLog['rows_failed']}\n";
        echo "   - Timestamp: {$latestLog['alta_db']}\n";
    }
    echo "\n";
    
    // Test 9: Retrieve schema by import ID
    echo "TEST 9: Retrieving schema for import...\n";
    $schema = ImportLogger::getSchemaByImportId($importLogId);
    echo "✅ Retrieved " . count($schema) . " schema column(s)\n";
    foreach ($schema as $col) {
        echo "   - {$col['column_name']} ({$col['data_type']}) - Order: {$col['column_order']}\n";
    }
    echo "\n";
    
    // Test 10: Check data integrity
    echo "TEST 10: Verifying data integrity...\n";
    $result = $conn->query("SELECT COUNT(*) as count FROM import_logs");
    $row = $result->fetch_assoc();
    echo "✅ Total import logs in database: {$row['count']}\n";
    
    $result = $conn->query("SELECT COUNT(*) as count FROM import_schema_logs");
    $row = $result->fetch_assoc();
    echo "✅ Total schema logs in database: {$row['count']}\n\n";
    
    echo "=== ALL TESTS PASSED! ===\n";
    echo "\n🎉 ImportLogger is working correctly!\n";
    
} catch (Exception $e) {
    echo "❌ TEST FAILED: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}
