<?php
/**
 * Test Subset Schema Matching
 * Tests that a file with fewer columns can use a saved schema with more columns
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/DatabaseHelper.php';
require_once __DIR__ . '/lib/ImportLogger.php';
require_once __DIR__ . '/lib/ErrorHandler.php';

echo "=== Testing Subset Schema Matching ===\n\n";

try {
    // Get database instance
    $db = DatabaseHelper::getInstance();
    $conn = $db->getConnection();

    // Select database
    if (!$conn->select_db('importer_db')) {
        die("Failed to select database: " . $conn->error . "\n");
    }

    echo "✓ Connected to database: importer_db\n\n";

    // Check if table exists first
    $tableCheck = $conn->query("SHOW TABLES LIKE 'import_logs'");
    echo "import_logs table exists: " . ($tableCheck && $tableCheck->num_rows > 0 ? "YES" : "NO") . "\n";

    // Get last schema for airbnb_transactions
    echo "Retrieving saved schema for 'airbnb_transactions'...\n";
    $savedSchema = ImportLogger::getLastSuccessfulSchema('airbnb_transactions', null); // Don't filter by database

    if ($savedSchema === null) {
        die("❌ No saved schema found for airbnb_transactions\n");
    }

    echo "✓ Found saved schema from import: " . $savedSchema['import_date'] . "\n";
    echo "  Total columns in saved schema: " . count($savedSchema['schema']) . "\n\n";

    // Print all column names
    echo "Saved schema columns:\n";
    foreach ($savedSchema['schema'] as $idx => $col) {
        echo "  " . ($idx + 1) . ". " . $col['name'] . " (" . $col['type'] . ")\n";
    }
    echo "\n";

    // Simulate a file with only SUBSET of columns (10 out of 16)
    $fileHeaders = [
        'fecha',
        'tipo',
        'codigo_de_confirmacion',
        'huesped',
        'espacio',
        'moneda',
        'monto',
        'ingresos_recibidos',
        'tarifa_de_servicio',
        'ingresos_brutos'
    ];

    echo "Simulating file upload with " . count($fileHeaders) . " columns:\n";
    foreach ($fileHeaders as $idx => $header) {
        echo "  " . ($idx + 1) . ". " . $header . "\n";
    }
    echo "\n";

    // Test subset matching logic
    echo "Testing canUseSchemaMemory() function...\n";

    // Extract saved column names
    $savedColumns = array_map(function($col) {
        return $col['name'];
    }, $savedSchema['schema']);

    // Check if ALL file headers exist in saved schema
    $canUseMemory = true;
    $missingInSaved = [];

    foreach ($fileHeaders as $header) {
        if (!in_array($header, $savedColumns)) {
            $canUseMemory = false;
            $missingInSaved[] = $header;
        }
    }

    if ($canUseMemory) {
        echo "✓ All file columns exist in saved schema - CAN USE MEMORY!\n\n";

        // Find which columns are in saved schema but NOT in file
        $missingInFile = [];
        foreach ($savedColumns as $savedCol) {
            if (!in_array($savedCol, $fileHeaders)) {
                $missingInFile[] = $savedCol;
            }
        }

        echo "Columns in saved schema but MISSING from file (" . count($missingInFile) . "):\n";
        foreach ($missingInFile as $idx => $col) {
            echo "  " . ($idx + 1) . ". " . $col . " (will use default value)\n";
        }
        echo "\n";

        echo "✅ SUBSET MATCHING WORKS!\n";
        echo "   File has: " . count($fileHeaders) . " columns\n";
        echo "   Saved schema has: " . count($savedSchema['schema']) . " columns\n";
        echo "   Missing in file: " . count($missingInFile) . " columns (will use defaults)\n";

    } else {
        echo "❌ File has columns not in saved schema - CANNOT USE MEMORY\n";
        echo "   Missing columns: " . implode(', ', $missingInSaved) . "\n";
    }

} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

echo "\n=== Test Complete ===\n";
