# AGENTS.md - Quantix Development Guidelines

## Build/Lint/Test Commands

### PHP Linting (Syntax Check)
```bash
# Lint all PHP files recursively (Windows batch)
lint_php.bat

# Manual lint for single file
/lamp/php/bin/php -l /path/to/file.php

# Lint specific directory
cd /lamp/www/quantix
lint_php.bat  # Creates lint_php_result.txt
```

### PHPUnit Testing
```bash
# Run all tests (from parent vitex directory, NOT quantix)
cd /wamp/www/vitex
C:\wamp\bin\php\php5.6.16\phpdbg -qrr -c tests/phpIniForPhoUnit.ini -d memory_limit=-1 tools/phpunit-9.5.21.phar

# Run specific test suite
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite History
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite Cobranza
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite NotaBodega

# Run single test file
phpdbg -qrr tools/phpunit-9.5.21.phar tests/inc/NotaBodegaTest.php

# Generate coverage report
phpdbg -qrr tools/phpunit-9.5.21.phar --coverage-html tests/reports/html-coverage
```

### JavaScript Linting
```bash
# Install global dependencies first
npm install -g eslint globals eslint-plugin-jquery eslint-plugin-metrics

# Lint JavaScript files
eslint [file.js]

# Using project config
eslint --config eslint.config.mjs [file.js]
```

### Database Operations
```bash
# MySQL connection
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix

# Execute SQL script
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix < script.sql

# Check MySQL service status
systemctl status lamp-mysql
```

## Code Style Guidelines

### PHP Standards

#### File Encoding
- **MUST**: UTF-8 without BOM
- **Spanish characters**: Use proper encoding (ñ, á, é, í, ó, ú)
- **Editor settings**: Configure for UTF-8, Unix line endings (LF)

#### Class Naming
```php
// App classes (models)
class app_propietario extends iacase_base { }
class app_propiedad extends iacase_base { }

// Framework classes (do not edit unless fixing core)
class iacase_base { }
class iaJQGrid { }
```

#### Method Naming
```php
// iaCase lifecycle methods (order matters)
function __construct() { }           // Constructor
function campos_default() { }         // Phase 1: Auto-generate fields
function campos_final() { }           // Phase 2: Customize fields
function listme_pre($grid) { }       // Grid customization
function validate() { }               // Custom validation

// Hook methods (override framework behavior)
function insert_before($values) { }   // Before insert
function insert_after($id, $values) { } // After insert
function update_before($values) { }   // Before update
function delete_before($id) { }        // Before delete
```

#### Database Query Patterns
```php
// ✅ CORRECT: Get rows for iteration (most common)
$rows = ia_sqlArrayIndx("SELECT * FROM propiedad");
foreach ($rows as $row) {
    echo $row['name'];
}

// ✅ CORRECT: Get rows keyed by ID (requires second parameter!)
$users = ia_sqlArray("SELECT * FROM usuarios", 'usuario_id');
echo $users['uuid-here']['name'];

// ❌ WRONG: Missing key parameter (will fail!)
$data = ia_sqlArray("SELECT * FROM usuarios");

// Single value retrieval
$count = ia_singleton("SELECT COUNT(*) FROM usuarios");
$name = ia_singleread("SELECT name FROM usuarios WHERE id=1", "Default");

// Framework operations (returns SQL string, must execute!)
$sql = ia_update('propiedad', $values, "propiedad_id='123'");
ia_query($sql); // Required: actually execute the query

// Direct SQL execution
ia_query("UPDATE propiedad SET nombre='Test' WHERE propiedad_id='123'");
```

#### Field Addition Pattern (iaCase)
When adding new database columns to existing iaCase classes:

1. **Add column to database**
2. **Update `campos_default()` method** (inside `/* TALBE_DEFAULT_INFO START */` markers)
3. **Update comment headers** (`CAMPOS_EN_DB_START` and `CAMPOS_START`)
4. **Customize in `campos_final()`** (add to `$orden` array)
5. **Update `appRelate.php`** if foreign key relationship

```php
// Step 2: Add to campos_default()
/* TALBE_DEFAULT_INFO START */
public function campos_default() {
    global $gAppRelate;
    $campos = array(
        // existing fields...
        'new_field' => array(
            'title' => 'Field description',
            'display_group' => '',
            'Null' => true,
            'required' => false,
            'Type' => 'varchar',
            'maxlength' => '50',
            'modo' => 'R/W',
        ),
        // more fields...
    );
    $this->campos = $gAppRelate->campos_incorpora($this->table, $campos);
}
/* TALBE_DEFAULT_INFO END */

// Step 4: Customize in campos_final()
public function campos_final() {
    $orden = array(
        'existing_field',
        'new_field',        // Add in desired position
        'another_field',
    );
    $this->campos_reorder($orden);
    
    // Optional customizations
    $this->campo['new_field']['label'] = 'Custom Label';
    $this->campo['new_field']['required'] = true;
}
```

### JavaScript Standards

#### jQuery Integration
```javascript
// Use provided jQuery global
$(document).ready(function() {
    $('#myElement').on('click', function() {
        // Handle click
    });
});

// jqGrid integration
$("#grid_id").jqGrid('getGridParam', 'colModel');
$("#grid_id").jqGrid('setGridParam', {postData: filters});
```

#### Module Pattern
```javascript
// Use modern ES6+ patterns
export function initializeGrid() {
    // Grid initialization
}

export function validateForm(formData) {
    // Form validation
    return isValid;
}
```

### Error Handling

#### PHP Error Handling
```php
// Framework provides automatic error logging
// Use try-catch for critical operations
try {
    $result = ia_query($sql);
} catch (Exception $e) {
    error_log("Database error: " . $e->getMessage());
    return false;
}

// Validation in iaCase classes
function validate() {
    $valid = parent::validate();
    
    if (empty($this->values['required_field'])) {
        $this->msg_err .= "<li>Required field cannot be empty</li>";
        $valid = false;
    }
    
    return $valid;
}
```

#### Session Management
```php
// Session is auto-configured in config.php
// User type checks
if (usuarioTipoRony($user_id)) {
    // Super admin access
}

// Session timeout handled automatically
// Use sessionTimeOut() for custom timeout checks
```

### Security Requirements

#### SQL Injection Prevention
```php
// ✅ SAFE: Use framework helpers
$values = array('name' => $_POST['name'], 'email' => $_POST['email']);
$sql = ia_insert('users', $values);
ia_query($sql);

// ❌ UNSAFE: Direct concatenation
$sql = "INSERT INTO users SET name = '" . $_POST['name'] . "'";
```

#### Permission Checks
```php
// In iaCase constructors
$this->permiso_insert = true;     // Allow insert
$this->permiso_update = false;    // Disallow update
$this->permiso_delete = usuarioTipoRony(); // Only super admins can delete

// Runtime permission checks
if (!$this->permiso_update) {
    die('Access denied');
}
```

### File Organization

#### New Entity Creation
1. **Database**: `CREATE TABLE new_entity (...)`
2. **Model**: `/app/app_new_entity.php`
3. **UI**: `/backoffice/new_entity.php`
4. **Schema**: Regenerate `appRelateBase.php` if needed
5. **Relationships**: Add to `appRelate.php`

#### Helper Scripts
- **Location**: `/backoffice/helper/script_name.php`
- **Requirements**: Must inherit session from parent
- **Pattern**: Preview mode → Apply → Verification
- **Database**: Use `ia_query()` for execution

### Import/Export Patterns

#### CSV Import
```php
// Use framework-provided CSV parsing
// Validate each row before insertion
// Log import results with statistics
```

#### jqGrid Export
```php
// Framework handles Excel/PDF export
// Configure via $this->export_buttons array
// Custom export logic in export_custom() method
```

## Development Environment Setup

### Requirements
- **PHP**: 5.6.16 (as specified in PHPUnit config)
- **MySQL**: With socket support at `/lamp/mysql/mysql.sock`
- **PHPUnit**: 9.5.21 (uses custom phpdbg wrapper)
- **Node.js**: For JavaScript linting
- **Composer**: Dependencies managed via `/lamp/www/quantix/composer.json`

### File Permissions
- **Development**: Requires `/wamp/www/showErrors.vin` file
- **Production**: Errors suppressed, logged to files
- **Session**: SameSite=Strict, automatic timeout

### Testing Requirements
- **Test Bootstrap**: Sets `$_SESSION['usuario_id']='1'` (super admin)
- **Database**: Uses same quantix database as production
- **Coverage**: Reports generated in `tests/reports/`

## Critical Gotchas

### Database Function Usage
- **`ia_sqlArrayIndx($sql)`**: Returns indexed array `[0=>[...], 1=>[...]]`
- **`ia_sqlArray($sql, $key)`**: Returns associative array `[$key=>[...]]` (MUST provide key!)
- **`ia_update()`/`ia_insert()`**: Return SQL string only, must call `ia_query()` to execute

### Session-Required Scripts
- **Helper scripts**: Need browser session, cannot run via CLI
- **Web execution**: Use `curl` with session cookies or browser
- **Database scripts**: Can run via CLI if they don't use framework

### Encoding Issues
- **Spanish characters**: Ensure files are UTF-8 without BOM
- **Double-encoding**: Check for "Ã±" instead of "ñ" in source files
- **Database**: Use proper UTF-8 collations (utf8mb4_unicode_ci, utf8mb4_0900_ai_ci)

## Framework Patterns

### Request Flow
```
User Request → /backoffice/page.php
    ↓
require config.php (session, auth, DB, autoloader)
    ↓
$f = new app_model()
    ↓
$f->process_action()  // Handle ?iah=r|e|a|s|d|l
    ↓
$f->show()  // Render HTML/JSON
```

### Grid Customization
```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'field_name' => array(
            'width' => '200',
            'classes' => 'bold',
            'searchoptions' => "{ sopt:['cn'] }",
            'comboBoxAutoComplete' => [/* autocomplete config */],
        ),
    );
    return true;
}
```

This system prioritizes rapid development through convention over configuration, with strong emphasis on data integrity, security, and maintaining backward compatibility.