# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## ⚡ Quick Start

### Accessing the System

**Web Interface**: [https://dev-app.filemonprime.net/quantix/](https://dev-app.filemonprime.net/quantix/)

**PHP Binary Location**: `/lamp/php/bin/php`

**MySQL CLI Connection**:
```bash
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock
```

### Running Scripts

**CLI Execution (Local)**:
```bash
/lamp/php/bin/php /lamp/www/quantix/script_name.php
```

**Web Execution (Remote)**:
```bash
curl -s https://dev-app.filemonprime.net/quantix/script_name.php
```

Both methods produce identical results. Use CLI for development/debugging, web execution for integration testing.

### Database Access

**MySQL Shell**:
```bash
# Interactive shell
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock

# Execute query directly
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock -e "SHOW DATABASES;"

# Connect to specific database
/lamp/mysql/bin/mysql -u root -pM@chiavell1 --socket=/lamp/mysql/mysql.sock quantix
```

**MySQL Stability Notes**:
```bash
# Verify MySQL is running stable (was restarting every 90 seconds - now fixed)
systemctl status lamp-mysql

# Test database connectivity
/lamp/mysql/bin/mysql -u root -pM@chiavell1 \
  --socket=/lamp/mysql/mysql.sock \
  -e "SELECT 'MySQL is alive!' as status;"
```

**System Status**:
- ✅ **MySQL stable** - Service runs indefinitely (no more 90-second restarts)
- ✅ **Full connectivity** - All database operations working
- ✅ **Proper monitoring** - Systemd correctly tracks MySQL PID
- ✅ **Production ready** - Survived 2+ hour stability test

**Key Configuration**:
- Service file: `/etc/systemd/system/lamp-mysql.service`
- Socket: `/lamp/mysql/mysql.sock`
- Binary: `/lamp/mysql/bin/mysqld`
- Data directory: `/lamp/mysql/data`

**Useful Commands**:
```bash
# Restart MySQL service
systemctl restart lamp-mysql

# View MySQL logs
journalctl -u lamp-mysql -f

# Reload systemd after config changes
systemctl daemon-reload
```

## Commands

### Testing
```bash
# Run all PHPUnit tests (from /wamp/www/vitex directory, not quantix)
C:\wamp\bin\php\php5.6.16\phpdbg -qrr -c tests/phpIniForPhoUnit.ini -d memory_limit=-1 tools/phpunit-9.5.21.phar

# Run specific test suite
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite History
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite Cobranza
phpdbg -qrr tools/phpunit-9.5.21.phar --testsuite NotaBodega

# Note: Tests require /wamp/www/showErrors.vin file to exist
# Tests bootstrap sets $_SESSION['usuario_id']='1' as super user
```

### Linting
```bash
# PHP syntax check (all *.php files recursively)
lint_php.bat
# Creates lint_php_result.txt with error details

# JavaScript linting (requires global npm packages)
npm install -g eslint globals eslint-plugin-jquery eslint-plugin-metrics
eslint [file.js]
```

### Database
```bash
# Migration scripts are organized by month in /db/ directory
# Example: db/enero_2025/, db/febrero_2025/
# No automated migration runner - execute SQL files manually
```

### WebSocket Server
```bash
# Start Workerman WebSocket server (async processing)
php websocket_server.php start

# Server listens on ws://0.0.0.0:2000
# 10 worker processes for handling async function calls
```

### Watchdog Service
```bash
# Start file monitoring service
watchdog\start_watchdog.bat

# Stop monitoring
watchdog\stop_watchdog.bat

# Monitors file changes and triggers rebuilds automatically
```

## Architecture Overview

**Quantix** is a custom-built PHP business application framework (formerly "iaKeyValue") following a modified MVC pattern centered around the **iaCase framework** - a declarative CRUD engine that auto-generates UIs from database schema.

### Core Philosophy (from manifesto_ia_case_v1.0.md)
- Reality-driven execution: built for production under pressure, not theoretical elegance
- Minimal abstraction: "The fewer the files, the clearer the path"
- Database-first: Schema drives UI generation automatically
- Security by design: Session management, SQL safety, CSRF protection built-in

### Directory Structure

#### `/inc/` - Core Framework
- `config.php` - **Entry point**: Session, auth, DB connection, global setup
- `iacase.php` - **Core CRUD engine** (~8000 lines): Auto-generates forms, grids, validation
- `iaJQGrid.php` - jqGrid integration for data tables
- `ia_utilerias.php` - Database abstraction (mysqli wrapper)
  - **Query Functions:**
    - `ia_query($sql)` - Execute query, returns mysqli_result
    - `ia_singleton($sql, $default=null)` - Returns single value (first column of first row)
    - `ia_singleread($sql, $default=null)` - Alias for ia_singleton()
  - **Array Fetching Functions (IMPORTANT - Choose the right one!):**
    - `ia_sqlArrayIndx($sql)` - Returns **indexed array** `[0 => ['col1'=>'val1', ...], 1 => [...]]`
      - Use when you need a simple numeric-indexed list of all rows
      - **Most common use case** for loops and iterations
      - Example: `foreach(ia_sqlArrayIndx($sql) as $row) { ... }`
    - `ia_sqlArray($sql, $indexField)` - Returns **associative array** `['key1' => ['col1'=>'val1', ...], 'key2' => [...]]`
      - Use when you need rows keyed by a specific column (like ID)
      - **Requires second parameter** - the field name to use as array key
      - Example: `ia_sqlArray($sql, 'id')` returns `['123' => [...], '456' => [...]]`
      - **WARNING:** If you omit the second parameter, it will fail or behave unexpectedly!
  - **Insert/Update Functions:**
    - `ia_insert($table, $values)` - Insert row, auto-escapes values
    - `ia_update($table, $values)` - Update row, auto-escapes values
- `helpers.php` - Business logic utilities
- `appRelateBase.php` - Auto-generated schema metadata (regenerated from DB)
- `Permisador.php` - Permission system (user types: Rony/Power/Normal)

#### `/app/` - Model Layer
- Pattern: `app_[tablename].php` extends `iacase_base`
- Each class represents a database table with auto-CRUD
- `appRelate.php` - Custom schema definitions (extends appRelateBase)
- `iacase_base.php` - Application-specific base customizations

#### `/backoffice/` - Main UI
- Pattern: `[tablename].php` instantiates `app_[tablename]` class
- Standard page structure:
  ```php
  require_once("../inc/config.php");
  $f = new app_iac_usr();
  $f->process_action();  // Handles ?iah=r|e|a|s|d|l
  $f->show();
  ```
- `/ajax/` - AJAX endpoints (jqGrid data, autocomplete, validation)
- `/helper/` - UI helper files

#### `/cobranza/` - Collections/Billing Module
- Separate business module with own `/inc/` directory
- `vitex.php` - Module loader
- `vitex_*.php` - Business logic files
- `vitex_async.php`, `vitex_async_ws.php` - Async processing layers

#### `/tests/` - PHPUnit Tests
- `bootstrap.php` - Sets up session as user_id=1 (super admin)
- Organized by domain: History, Cobranza, NotaBodega, SQL, etc.
- Coverage reports in `tests/reports/`

### Database Layer

**Connection** (config.php):
```php
$gIAsql = [
    'host' => 'localhost',
    'port' => '3306',
    'user' => 'root',
    'pwd' => 'M@chiavell1',
    'dbname' => 'quantix'
];
```

**Database Query Functions Quick Reference**:

| Function | Returns | Use Case | Example |
|----------|---------|----------|---------|
| `ia_query($sql)` | mysqli_result | Raw query execution | `ia_query("UPDATE users SET active=1")` |
| `ia_singleton($sql)` | Single value | Get one value (count, sum, etc.) | `$total = ia_singleton("SELECT COUNT(*) FROM users")` |
| `ia_singleread($sql, $default)` | Single value | Get one value with fallback | `$name = ia_singleread("SELECT name FROM users WHERE id=1", "Unknown")` |
| **`ia_sqlArrayIndx($sql)`** | **Indexed array** | **Loop through all rows** | `foreach(ia_sqlArrayIndx($sql) as $row) { ... }` |
| `ia_sqlArray($sql, $key)` | Associative array | Fast ID lookups | `$users = ia_sqlArray($sql, 'id'); echo $users['123']['name'];` |
| `ia_insert($table, $values)` | Insert result | Add new record | `ia_insert('users', ['name'=>'John', 'email'=>'john@example.com'])` |
| `ia_update($table, $values)` | Update result | Modify existing record | `ia_update('users', ['id'=>1, 'name'=>'Jane'])` |

**⚠️ CRITICAL: `ia_sqlArray()` vs `ia_sqlArrayIndx()`**

- **Use `ia_sqlArrayIndx($sql)` for iteration** - 99% of use cases
- **Use `ia_sqlArray($sql, 'fieldname')` only when you need keyed access**
- **Never use `ia_sqlArray()` without the second parameter!**

```php
// ✅ CORRECT - Simple iteration
$rows = ia_sqlArrayIndx("SELECT * FROM users");
foreach ($rows as $row) { echo $row['name']; }

// ✅ CORRECT - Keyed by ID for lookups
$users = ia_sqlArray("SELECT * FROM users", 'id');
$user = $users['123'];

// ❌ WRONG - Missing key parameter
$rows = ia_sqlArray("SELECT * FROM users");  // Will fail!
```

**Schema Metadata System**:
- `appRelateBase` - Auto-generated from DB schema (DO NOT edit manually)
- `appRelate` - Extends base with custom relationships, validations, enums
- Defines: table relationships, foreign keys, validation rules, dropdown options

**Migration Pattern**:
- SQL files in `/db/[mes]_[año]/`
- No ORM migrations - raw SQL applied manually

### Request Flow

```
User Request → /backoffice/[tablename].php
  ↓
require ../inc/config.php (session, auth, DB, autoloader)
  ↓
$f = new app_[tablename]()
  ↓
$f->process_action() handles ?iah parameter:
  'r' = Read/View
  'e' = Edit form
  'a' = Add form
  's' = Save (insert/update)
  'd' = Delete
  'l' or '' = List (jqGrid)
  ↓
$f->show() renders HTML (or JSON for AJAX)
```

### iaCase Framework Pattern

**Declarative CRUD** - Define schema, get full UI automatically:

```php
class app_iac_usr extends iacase_base {
    function __construct() {
        parent::__construct('r', 'iac_usr');  // 'r'=read permission, table name

        $this->campos_default();  // Auto-generate from DB schema

        // Customize specific fields
        $this->campo['password']['type'] = 'password';
        $this->campo['rol']['options'] = ['admin', 'user', 'guest'];

        $this->campos_final();  // Finalize configuration

        // UI behavior
        $this->modo = 'cardex_window';  // Display mode
        $this->permiso_delete = true;
        $this->jqGridFilters = true;
    }
}
```

**Auto-generated features**:
- HTML forms with validation
- jqGrid data tables with sorting/filtering
- CRUD operations (insert/update/delete)
- Permission checks (user type + table + field level)
- Audit logging

### Async Processing Architecture

**Three-tier system** for background tasks:

1. **HTTP Async** (`vitex_async.php`):
   - `get_async($url, $postData)` - Fire-and-forget HTTP
   - Non-blocking socket connections
   - Simplest layer, no response handling

2. **WebSocket Async** (`vitex_async_ws.php`):
   - `resilient_async_call($functionName, $params)` - Primary method
   - Connects to Workerman server (ws://localhost:2000)
   - Falls back to HTTP async if WebSocket unavailable
   - Uses React PHP event loop

3. **Amphp Parallel**:
   - CPU-intensive operations
   - True parallel processing (multi-process)
   - Autoload namespace: `Vitex\Parallel\`

**Workerman WebSocket Server** (`websocket_server.php`):
- Receives async function calls via WebSocket
- Executes PHP functions in background workers
- Returns results to connected clients

### Frontend Stack

**JavaScript Libraries** (loaded via iaHeader):
- jQuery 3.7.1 - Core DOM manipulation
- jQuery UI (Redmond theme) - Dialogs, datepickers, autocomplete
- jqGrid 5.0.1 - Primary data grid component
- TinyMCE 5 - Rich text editor
- autoNumeric - Number formatting
- BlockUI - Loading states
- Font Awesome Pro 6.4.0

**CSS**:
- `iastyles.min.css` - Custom framework styles
- jQuery UI themes (configurable via `$jQueryUIStyle` in config.php)

**Page Head Generation**:
```php
global $gIaHeader;
$gIaHeader = new iaHeader();
$gIaHeader->add_css('custom.css');
$gIaHeader->add_js('custom.js');
$gIaHeader->html_head_echo();  // Outputs all <head> content
```

### Permission System

**Three levels**:
1. **User Type** (`iac_usr.tipo` field):
   - `usuarioTipoRony($uid)` - Super admin (full access)
   - Power user - Most features
   - Normal - Restricted access

2. **Table-level** (iaCase properties):
   - `$this->permiso_insert`, `$this->permiso_update`, `$this->permiso_delete`

3. **Field-level** (`iac_field_permission` table):
   - Per-user field visibility/editability

**Session Security**:
- Timeout: 70 min (before 6pm), 37 min (after 6pm)
- Auto-regenerate session ID on login
- Cookie SameSite=Strict
- `sessionTimeOut($minutes)` in config.php

### Key Integrations

**Composer Dependencies**:
- Websockets: `cboden/ratchet`, `workerman/workerman`
- Async: `amphp/amp`, `amphp/parallel`, `react/react`
- PDF: `mpdf/mpdf`, `setasign/fpdf`, `setasign/fpdi`
- HTTP: `guzzlehttp/guzzle`
- Redis: `predis/predis`
- WhatsApp: `green-api/whatsapp-api-client-php`
- AI: `ardagnsrn/ollama-php`

### Naming Conventions

**Files**:
- Application models: `app_[tablename].php`
- Backoffice pages: `[tablename].php`
- Utilities: `ia_*.php` (framework), `vitex_*.php` (Cobranza module)
- AJAX endpoints: Descriptive names in `/ajax/` subdirectories

**Classes**:
- Models extend: `iacase_base` → `iacase` → Framework base
- Prefix `ia` for framework classes: `iacase`, `iaJQGrid`, `iaHeader`

**Database**:
- System tables: `iac_*` prefix (iac_usr, iac_parametros, iac_log)
- Application tables: Descriptive names matching business domain

**URL Parameters**:
- `?iah=` - Action: r(ead), e(dit), a(dd), s(ave), d(elete), l(ist)
- `?id=` - Record primary key
- `?q=` - Search query (jqGrid)

### Development Notes

**Environment Detection**:
- Development requires `/wamp/www/showErrors.vin` file to exist
- Production mode: Errors suppressed, logging to files
- Check via: `file_exists(__DIR__ . '/../../showErrors.vin')`

**Session/Program Configuration** (config.php):
```php
$gWebDir = 'quantix';        // URL path component
$gDBName = 'quantix';        // Database name
$gProgramName = 'Quantix';   // Display name
$jQueryUIStyle = 'redmond';  // UI theme
```

**Caching**:
- OpcacheGUI: `amnuts/opcache-gui` for PHP opcode cache monitoring
- Warmup: `/inc/warmUp.php` pre-loads files into opcache
- Application cache: `/libs/cacher/` custom layer
- Browser: LocalStorage for client-side caching

**Autoloading** (config.php):
```php
// Composer autoload
require_once(__DIR__ . '/vendor/autoload.php');

// Custom autoloader for /app/ and /inc/ classes
spl_autoload_register(function($class) { /* ... */ });
```

### Common Patterns

**Creating a new entity**:
1. Create database table
2. Regenerate `appRelateBase.php` from schema
3. Add custom metadata to `appRelate.php` if needed:
   ```php
   $this->tables['mytable']['parent'] = 'parent_table';
   $this->links['mytable']['parent_id'] = 'parent_table';
   ```
4. Create `/app/app_mytable.php`:
   ```php
   class app_mytable extends iacase_base {
       function __construct() {
           parent::__construct('r', 'mytable');
           $this->campos_default();
           // Customizations here
           $this->campos_final();
       }
   }
   ```
5. Create `/backoffice/mytable.php`:
   ```php
   require_once("../inc/config.php");
   $f = new app_mytable();
   $f->process_action();
   $f->show();
   ```

**Adding custom AJAX endpoint**:
```php
// /backoffice/ajax/mytable_custom.php
require_once("../../inc/config.php");
header('Content-Type: application/json');
// Business logic
echo json_encode($result);
```

**Fetching data from database**:

```php
// Get a single value (first column of first row)
$count = ia_singleton("SELECT COUNT(*) FROM users WHERE active=1");
$name = ia_singleread("SELECT name FROM users WHERE id=123", "Unknown");

// Get all rows as indexed array (MOST COMMON)
$users = ia_sqlArrayIndx("SELECT * FROM users ORDER BY name");
foreach ($users as $user) {
    echo $user['name'];  // Access by column name
}

// Get rows keyed by ID field (for lookups)
$users_by_id = ia_sqlArray("SELECT * FROM users", 'id');
echo $users_by_id['123']['name'];  // Access by ID

// WRONG - Missing second parameter!
$data = ia_sqlArray("SELECT * FROM users");  // ❌ Will fail!

// RIGHT - Use ia_sqlArrayIndx instead!
$data = ia_sqlArrayIndx("SELECT * FROM users");  // ✅ Works!
```

**Common pattern - Loading data for dropdowns/matching**:
```php
// Load all records as indexed array for iteration
$propiedades = ia_sqlArrayIndx("SELECT * FROM propiedades ORDER BY nombre");
$propietarios = ia_sqlArrayIndx("SELECT * FROM propietarios ORDER BY propietario");

// Loop through and match
foreach ($propiedades as $prop) {
    foreach ($propietarios as $owner) {
        if ($prop['nombre'] == $owner['departamento']) {
            // Match found!
        }
    }
}

// Load records keyed by ID for fast lookups
$users_lookup = ia_sqlArray("SELECT id, name, email FROM users", 'id');
$user_name = $users_lookup[$some_id]['name'] ?? 'Unknown';
```

**Calling async function**:
```php
require_once('cobranza/inc/vitex_async_ws.php');
resilient_async_call('myFunctionName', ['param1' => 'value']);
// Function executes in background, no wait for response
```

**Reading system parameters**:
```php
global $gIAParametros;
leeParametrosVitex();  // Called in config.php
$value = $gIAParametros['parameter_key'];
```

### Adding New Fields to iaCase Classes

**CRITICAL:** When adding a new database column to an existing iaCase class, you MUST follow this exact pattern. Skipping steps will cause the field to not appear or break the form.

#### The Two-Phase Initialization Pattern

iaCase classes initialize fields in two phases:

1. **Phase 1 - `campos_default()`**: Defines what fields exist (auto-generated from DB schema)
2. **Phase 2 - `campos_final()`**: Customizes existing fields (manual configuration)

**Golden Rule:** You cannot customize a field in Phase 2 that doesn't exist after Phase 1.

#### Step-by-Step Guide

##### 1. Add Column to Database

```sql
ALTER TABLE propiedades
ADD COLUMN propietario_id CHAR(36) NULL
COMMENT 'Reference to propietarios table'
AFTER nombre_propiedad,
ADD KEY idx_propietario_id (propietario_id);
```

##### 2. Regenerate `campos_default()` Method

**CRITICAL STEP** - Do NOT skip this!

The `campos_default()` method must be updated to include the new field. This section is marked with:
```php
/* TALBE_DEFAULT_INFO START */
// ... field definitions ...
/* TALBE_DEFAULT_INFO END */
```

Add the new field definition inside the `$campos` array:

```php
'propietario_id' =>
array (
  'title' => 'Reference to propietarios table',  // From COMMENT in DB
  'display_group' => '',
  'Null' => true,              // Match DB column definition
  'required' => false,         // false if NULL allowed, true if NOT NULL
  'Type' => 'char',            // DB column type
  'maxlength' => '36',         // For string types only
  'modo' => 'R/W',             // 'R/W' = editable, 'R/O' = read-only
),
```

**Field Definition Attributes:**

| Attribute | Description | Example Values |
|-----------|-------------|----------------|
| `title` | Label/description (from DB comment) | `'Owner name'` |
| `display_group` | Group fields together in UI | `'Contact Info'`, `''` |
| `Null` | Whether DB column allows NULL | `true`, `false` |
| `required` | Whether required for insert | `true`, `false` |
| `Type` | Database column type | `'char'`, `'varchar'`, `'int'`, `'timestamp'`, `'enum'`, `'text'` |
| `maxlength` | Max length for string types | `'36'`, `'100'`, `'255'` |
| `modo` | Edit mode | `'R/W'` (editable), `'R/O'` (read-only) |
| `PK` | Is primary key (optional) | `true` |
| `PK_type` | Primary key type (optional) | `'value'`, `'auto_increment'` |
| `Default` | Default value (optional) | `'CURRENT_TIMESTAMP'`, `'system'` |

##### 3. Update Comment Headers

Update both `CAMPOS_EN_DB_START` and `CAMPOS_START` comments to include the new field:

```php
/*
   CAMPOS_EN_DB_START
   primary key: 'propiedade_id' al 2025-12-30 16:28
   propiedade_id,num_deptos,nombre_propiedad,propietario_id,direccion,alta_db,alta_por,ultimo_cambio,ultimo_cambio_por
   CAMPOS_EN_DB_END
*/

/*
   CAMPOS_START
   primary key: 'propiedade_id' al 2025-12-30 16:28
   propiedade_id,num_deptos,nombre_propiedad,propietario_id,direccion,alta_db,alta_por,ultimo_cambio,ultimo_cambio_por
   CAMPOS_END
*/
```

##### 4. Update `campos_final()` Method

Add the new field to the `$orden` array to control display order:

```php
public function campos_final() {
    // Reorder fields (controls display order in forms)
    $orden = array(
        'num_deptos',
        'nombre_propiedad',
        'propietario_id',      // NEW FIELD - add in desired position
        'direccion',
        'alta_db',
        'alta_por',
        'ultimo_cambio',
        'ultimo_cambio_por'
    );
    $this->campos_reorder($orden);

    // Optional: Customize field attributes
    // $this->campo['propietario_id']['label'] = 'Owner';
    // $this->campo['propietario_id']['required'] = true;
}
```

##### 5. (Optional) Configure Relationship in `appRelate.php`

If the field is a foreign key reference:

```php
// In app/appRelate.php, inside links_final() method:
public function links_final() {
    // Define parent-child relationship
    $this->tables['propiedades']['parent'] = 'propietarios';
    $this->links['propiedades']['propietario_id'] = 'propietarios';
}
```

This enables:
- Dropdown/autocomplete for selecting related records
- Parent-child UI navigation
- Cascade behaviors

#### Common Customizations in `campos_final()`

**Change field label:**
```php
$this->campo['propietario_id']['label'] = 'Property Owner';
```

**Make field required:**
```php
$this->campo['propietario_id']['required'] = true;
```

**Make field read-only:**
```php
$this->campo['propietario_id']['modo'] = 'R/O';
```

**Hide field in forms:**
```php
$this->campo['propietario_id']['hidden'] = true;
```

**Add line break after field:**
```php
$this->campo['propietario_id']['br'] = true;
```

**Set field to specific display group:**
```php
$this->campo['propietario_id']['display_group'] = 'Owner Information';
$this->campo['propietario_id']['display_group_label'] = 'Owner Details';
```

**Configure as lookup/foreign key dropdown:**
```php
$this->campo['propietario_id']['type'] = 'lookup';
```

**Set input size:**
```php
$this->campo['propietario_id']['input_size'] = 50;
```

**Add custom validation:**
```php
// Override validate() method in the class
function validate() {
    $valid = parent::validate();

    if (empty($this->values['propietario_id'])) {
        $this->msg_err .= "<li>Property must have an owner</li>";
        $valid = false;
    }

    return $valid;
}
```

#### Common Mistakes (Anti-Patterns)

**❌ DON'T: Try to configure a field that doesn't exist in `campos_default()`**

```php
// WRONG - This will fail if propietario_id isn't in campos_default()
public function campos_final() {
    $this->campo['propietario_id']['label'] = 'Owner';  // ERROR: field doesn't exist!
}
```

**❌ DON'T: Manually edit inside the `/* TALBE_DEFAULT_INFO START/END */` markers**

These sections are auto-generated. Your changes will be overwritten.

**❌ DON'T: Forget to add the field to the `$orden` array**

If the field isn't in `$orden`, it may not display in the UI or appear in unexpected positions.

**❌ DON'T: Skip updating the comment headers**

The comments serve as documentation and must match reality.

**❌ DON'T: Add a field to `campos_final()` without adding it to `campos_default()` first**

Always define the field in `campos_default()` before customizing it in `campos_final()`.

#### Field Type Examples

**String fields (char/varchar):**
```php
'nombre' => array(
    'Type' => 'varchar',
    'maxlength' => '100',
    'Null' => false,
    'required' => true,
    'modo' => 'R/W',
),
```

**Integer fields:**
```php
'cantidad' => array(
    'Type' => 'int',
    'Null' => false,
    'required' => true,
    'modo' => 'R/W',
),
```

**Timestamp fields (auto-managed):**
```php
'alta_db' => array(
    'Type' => 'timestamp',
    'Null' => true,
    'modo' => 'R/O',                    // Read-only
    'Default' => 'CURRENT_TIMESTAMP',
),
```

**Enum fields:**
```php
'es_dueno' => array(
    'Type' => 'enum',
    'Null' => false,
    'required' => true,
    'modo' => 'R/W',
),
// Then in appRelate.php enums_final():
$this->enums['propietarios.es_dueno'] = ['Si' => 'Si', 'No' => 'No'];
```

**Text fields (large text):**
```php
'descripcion' => array(
    'Type' => 'text',
    'Null' => true,
    'modo' => 'R/W',
),
```

**Foreign key fields:**
```php
'propietario_id' => array(
    'Type' => 'char',
    'maxlength' => '36',
    'Null' => true,
    'modo' => 'R/W',
),
// Then in appRelate.php links_final():
$this->links['propiedades']['propietario_id'] = 'propietarios';
```

#### Troubleshooting

**Field doesn't appear in form:**
- ✓ Check that field exists in `campos_default()` `$campos` array
- ✓ Verify field is in `$orden` array in `campos_final()`
- ✓ Check that `modo` is not `'R/O'` when editing
- ✓ Verify field is not marked as `hidden`

**Field appears but can't edit:**
- ✓ Check `modo` is `'R/W'` not `'R/O'`
- ✓ Verify permissions: `$this->permiso_update`
- ✓ Check field-level permissions in `iac_field_permission` table

**Field validation fails:**
- ✓ Check `required` matches your intent
- ✓ Verify `Null` attribute matches DB column definition
- ✓ Check `maxlength` is sufficient for your data

**Dropdown doesn't show options:**
- ✓ Verify relationship configured in `appRelate.php` `links_final()`
- ✓ Check that referenced table has data
- ✓ Verify foreign key field name matches pattern

**Form breaks/PHP errors:**
- ✓ Ensure field exists in `campos_default()` before customizing in `campos_final()`
- ✓ Check for syntax errors in field definition array
- ✓ Verify all array elements are properly quoted

#### Complete Example: Adding a Status Field

**1. Database:**
```sql
ALTER TABLE propiedades
ADD COLUMN status ENUM('active','inactive','pending') NOT NULL DEFAULT 'active'
COMMENT 'Property status'
AFTER direccion;
```

**2. Update `app/app_propiedades.php` - `campos_default()`:**
```php
/* TALBE_DEFAULT_INFO START */
public function campos_default() {
    global $gAppRelate;
    $campos = array(
        // ... existing fields ...
        'status' =>
        array (
            'title' => 'Property status',
            'display_group' => '',
            'Null' => false,
            'required' => true,
            'Type' => 'enum',
            'modo' => 'R/W',
            'Default' => 'active',
        ),
        // ... more fields ...
    );

    $this->campos = $gAppRelate->campos_incorpora($this->table, $campos);
}
/* TALBE_DEFAULT_INFO END */
```

**3. Update `app/app_propiedades.php` - `campos_final()`:**
```php
public function campos_final() {
    // Customize the status field
    $this->campo['status']['label'] = 'Property Status';
    $this->campo['status']['display_group'] = 'General Info';

    // Set field order
    $orden = array(
        'num_deptos',
        'nombre_propiedad',
        'propietario_id',
        'direccion',
        'status',              // New field in desired position
        'alta_db',
        'alta_por',
        'ultimo_cambio',
        'ultimo_cambio_por'
    );
    $this->campos_reorder($orden);
}
```

**4. Update `app/appRelate.php` - `enums_final()`:**
```php
public function enums_final() {
    $this->enums['propiedades.status'] = [
        'active' => 'Active',
        'inactive' => 'Inactive',
        'pending' => 'Pending Approval',
    ];
}
```

**5. Update comment headers in `app/app_propiedades.php`:**
```php
/*
   CAMPOS_EN_DB_START
   propiedade_id,num_deptos,nombre_propiedad,propietario_id,direccion,status,alta_db,alta_por,ultimo_cambio,ultimo_cambio_por
   CAMPOS_EN_DB_END
*/
```

Done! The field will now appear in forms with a dropdown showing the three status options.

## jqGrid Customization & Autocomplete System

### Overview

The Quantix framework uses a three-layer architecture for data grids:

```
Database Schema → iaCase (PHP) → iaJQGrid (PHP) → jqGrid (JavaScript)
```

1. **Auto-generation**: Column definitions are automatically generated from the database schema in `campos_default()`
2. **Customization layer**: Override specific column behaviors using `colModel_overRide` in the `listme_pre()` method
3. **Frontend rendering**: JavaScript autocomplete widgets are configured via `comboBoxAutoComplete` settings

This system allows you to start with automatic CRUD grids and then progressively enhance specific columns with custom widgets, validation, formatting, and search capabilities.

### Architecture Components

#### 1. Backend: `colModel_overRide` (PHP)
Defined in **app_[tablename].php** within the `listme_pre($grid)` method. This is where you customize how columns appear and behave in the jqGrid.

#### 2. Backend: `obtenCatalogo()` (PHP)
Located in **/cobranza/inc/vitex_catalogos.php**. Generates JSON catalog files for autocomplete dropdowns. Catalogs are cached as HTML files in `/backoffice/json/`.

#### 3. Frontend: `comboBoxAutoComplete` (JavaScript)
Configuration object that tells the JavaScript autocomplete widget how to fetch and display data. Processed by `initAutoCompleteAjax()` in **/js2/app_iacase.js**.

### Data Flow Diagram

```
┌─────────────────────────────────────────────────────────────┐
│  1. User types in jqGrid search field                        │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  2. JavaScript detects 'comboBoxAutoComplete' config         │
│     in colModel (from colModel_overRide)                     │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  3. AJAX request to:                                         │
│     /backoffice/obtenCatalogo.php?catalogo=propietario       │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  4. obtenCatalogo() function executes:                       │
│     - Checks cache: /backoffice/json/propietario.html        │
│     - If missing/stale: queries DB, generates JSON           │
│     - Returns: [{value:"id1",label:"Name1"}, ...]            │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  5. JavaScript renders autocomplete dropdown                 │
│     - Filters results based on user input                    │
│     - Shows matching items                                   │
└────────────────────┬────────────────────────────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────┐
│  6. User selects item → value stored → grid search runs      │
└─────────────────────────────────────────────────────────────┘
```

---

## Part 1: `colModel_overRide` - Customizing Grid Columns

### When to Use colModel_overRide

Use `colModel_overRide` when you need to:
- Add autocomplete/search to a column
- Change column width, alignment, or CSS classes
- Customize search operators
- Add formatters or custom rendering
- Override any auto-generated jqGrid colModel property

### Where to Define It

Override columns in the **`listme_pre($grid)`** method of your app class:

**File**: `/app/app_[tablename].php`

```php
class app_propietario extends iacase_base {

    function listme_pre($grid) {
        // Define your column overrides here
        $this->colModel_overRide = array(
            'field_name' => array(
                'property1' => 'value1',
                'property2' => 'value2',
                // ...
            ),
        );

        return true; // Must return true to continue
    }
}
```

### How It Works

1. **Auto-generation phase**: iaJQGrid builds colModel from `$this->campos` (defined in `campos_default()`)
2. **Override phase**: Properties from `colModel_overRide` are **merged** into the auto-generated colModel
3. **Override rules**:
   - Simple values: Overwrite the auto-generated value
   - Arrays: Merge recursively (sub-properties are added/overwritten)
   - Non-existent properties: Added to colModel

**Implementation** (in `/inc/iaJQGrid.php` lines 997-1016):

```php
if($colModel_override && array_key_exists($fieldName, $colModel_override)) {
    foreach($colModel_override[$fieldName] as $opt => $value) {
        if(is_array($value)) {
            if(!array_key_exists($opt, $col))
                $col[$opt] = array();
            if(!empty($value)) {
                foreach($value as $subopt => $subvalue) {
                    $col[$opt][$subopt] = $subvalue;
                }
            }
        } else {
            $col[$opt] = $value;
        }
    }
}
```

### Common colModel_overRide Properties

| Property | Type | Description | Example |
|----------|------|-------------|---------|
| `width` | string/int | Column width in pixels | `'360'`, `200` |
| `classes` | string | CSS classes for column cells | `'bold txt18px'` |
| `searchoptions` | string (JS object) | jqGrid search configuration | `"{ sopt:['cn']}"` |
| `formatter` | string | jqGrid formatter name | `'select'`, `'date'` |
| `template` | string | Column template reference | `'colFmt.textsmall'` |
| `hidden` | boolean | Hide column by default | `true`, `false` |
| `editable` | boolean | Allow inline editing | `true`, `false` |
| `comboBoxAutoComplete` | array | Autocomplete config (see below) | `[...]` |

### Example 1: Simple Width & Styling Override

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'nombre_propiedad' => array(
            'width' => '400',
            'classes' => 'bold txt16px',
        ),
    );
    return true;
}
```

**Result**: The `nombre_propiedad` column will be 400px wide with bold 16px text.

### Example 2: Custom Search Options

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'fecha_alta' => array(
            'searchoptions' => "{ sopt:['eq','ge','le'] }", // equals, >=, <=
        ),
        'status' => array(
            'searchoptions' => "{ sopt:['eq','ne'] }",      // equals, not equals
        ),
    );
    return true;
}
```

**Available search operators** (`sopt`):
- `'eq'` - Equal
- `'ne'` - Not equal
- `'lt'` - Less than
- `'le'` - Less than or equal
- `'gt'` - Greater than
- `'ge'` - Greater than or equal
- `'bw'` - Begins with
- `'bn'` - Does not begin with
- `'in'` - In (SQL IN)
- `'ni'` - Not in
- `'ew'` - Ends with
- `'en'` - Does not end with
- `'cn'` - Contains
- `'nc'` - Does not contain

---

## Part 2: `comboBoxAutoComplete` - Adding Autocomplete to Columns

### Purpose

The `comboBoxAutoComplete` configuration object enables **type-ahead search** in jqGrid columns. As users type, matching options appear in a dropdown, improving usability and data accuracy.

### Two Modes of Operation

#### Mode 1: AJAX-based (Dynamic, fetches from database)
- Set `'isAjax' => true`
- Data fetched via `/backoffice/obtenCatalogo.php`
- Requires catalog definition in `obtenCatalogo()` function
- **Best for**: Large datasets, data that changes frequently

#### Mode 2: Static JSON (Pre-loaded)
- Set `'isAjax' => false` (or omit)
- Set `'dataUrl' => 'path/to/catalog.html'`
- Data loaded once from cached JSON file
- **Best for**: Small, stable datasets

### Complete Parameter Reference

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `isAjax` | boolean | No | `false` | `true` = AJAX mode, `false` = static JSON |
| `dataUrl` | string | For static | `''` | Path to JSON file (relative to backoffice) |
| `campos` | string | For AJAX | (field name) | Comma-separated DB columns to search |
| `catalogo` | string | For AJAX | (field - '_id') | Table/catalog name for `obtenCatalogo()` |
| `label` | string | For AJAX | (field name) | Column to display in dropdown |
| `pk` | string | For AJAX | (catalogo + '_id') | Primary key column name |
| `strict` | boolean | No | `true` | `true` = must select from list, `false` = allow freeform |
| `autoFocus` | boolean | No | `true` | Auto-focus first match in dropdown |
| `fixedWidth` | int | No | (calculated) | Fixed width in pixels for dropdown |
| `order_by` | string | No | (label) | ORDER BY clause for SQL query |
| `group_by` | string | No | `''` | GROUP BY clause for SQL query |
| `busqueda_relajada` | boolean | No | `false` | Relaxed search matching |

### Example 3: AJAX Autocomplete (Your Case)

**File**: `/app/app_propietario.php`

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'propietario' => array(
            'width' => '360',
            'classes' => 'bold txt18px',
            'searchoptions' => "{ sopt:['cn']}",
            'comboBoxAutoComplete' => [
                'isAjax' => true,              // Use AJAX mode
                'dataUrl' => '',               // Empty = use default endpoint
                'campos' => "propietario",     // Search in 'propietario' column
                'catalogo' => 'propietario',   // Catalog name (must match obtenCatalogo case)
                'label' => 'propietario',      // Display 'propietario' in dropdown
                'pk' => "propietario",         // Primary key (unusual: not propietario_id)
                'strict' => false,             // Allow typing values not in list
                'autoFocus' => true,           // Auto-focus first match
                'fixedWidth' => 350,           // Dropdown width
            ],
        ),
    );
    return true;
}
```

**What happens**:
1. User types in the "Propietario" search field
2. JavaScript sends AJAX request: `/backoffice/obtenCatalogo.php?catalogo=propietario`
3. `obtenCatalogo()` returns JSON: `[{value:"prop1",label:"Adolfo Zavala"}, ...]`
4. Dropdown shows matching names
5. User selects → search executes with selected value

### Example 4: Static JSON Autocomplete

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'tipo_nota' => array(
            'comboBoxAutoComplete' => [
                'dataUrl' => '../../backoffice/json/bodega_tipo_cash_nota.html',
            ],
        ),
    );
    return true;
}
```

**What happens**:
1. On page load, JavaScript fetches `/backoffice/json/bodega_tipo_cash_nota.html`
2. JSON is cached in browser
3. Autocomplete dropdown uses cached data (instant, no server requests)

### Example 5: Multi-field Search

Search across multiple columns:

```php
'comboBoxAutoComplete' => [
    'isAjax' => true,
    'campos' => "propietario, departamento, responsable", // Search in 3 fields
    'catalogo' => 'propietario',
    'label' => 'propietario',  // Display propietario
    'pk' => "propietario_id",
    'strict' => false,
],
```

SQL generated by AJAX endpoint will search:
```sql
WHERE propietario LIKE '%search%'
   OR departamento LIKE '%search%'
   OR responsable LIKE '%search%'
```

---

## Part 3: `obtenCatalogo()` - Backend Catalog Generation

### Purpose

The `obtenCatalogo()` function generates JSON data for autocomplete dropdowns. It:
1. Queries the database for catalog data
2. Formats results as JSON: `[{value, label, real_data}, ...]`
3. Caches results to `/backoffice/json/[catalog].html`
4. Returns cached data on subsequent requests (unless regeneration forced)

### Function Signature

**File**: `/cobranza/inc/vitex_catalogos.php`

```php
function obtenCatalogo(
    $nombreCatalogo = '',  // Catalog name (matches 'catalogo' in comboBoxAutoComplete)
    $genera = false,       // Force regeneration (ignore cache)
    $extraId = '',         // Extra identifier for filename (e.g., category-specific catalogs)
    $printFiles = false,   // Debug: print file operations
    $categoria_id = -1,    // Optional: filter by category
    $debug = false         // Debug mode
)
```

### How It Works

1. **Cache check**: Looks for `/backoffice/json/{nombreCatalogo}{extraId}.html`
2. **Cache hit**: Returns cached JSON (fast)
3. **Cache miss or $genera=true**:
   - Executes SQL query specific to the catalog
   - Converts result to JSON array
   - Saves to cache file via `jsonFile()`
   - Returns JSON

### JSON Structure

Each catalog returns an array of objects with these fields:

```json
[
  {
    "propietario_id": "uuid-123",     // Original field (same as field name)
    "propietario": "Adolfo Zavala",   // Display field
    "real_data": "uuid-123",          // Primary key value (for form submission)
    "label": "Adolfo Zavala",         // Display label (for dropdown)
    "value": "uuid-123"               // Value to store (same as real_data)
  },
  // ... more items
]
```

**Required fields**:
- `label` - Displayed in dropdown
- `value` - Stored when selected
- `real_data` - Original PK value

### Adding a New Catalog

**Step 1**: Add case to `obtenCatalogo()` switch statement

**File**: `/cobranza/inc/vitex_catalogos.php`

```php
function obtenCatalogo($nombreCatalogo ='', $genera=false, ...) {
    // ... existing code ...

    switch($nombreCatalogo) {

        // EXISTING CATALOGS...

        case 'propietario':  // ← Add your case here
            if(!$genera && file_exists($fileName))
                $jsonData = getSafeDataFromFile($fileName);

            if(strlen($jsonData) === 0) {
                // Define catalog query
                $tabla = 'propietario';
                $fieldName = 'propietario_id';
                $fieldDesc = 'propietario';

                $sql = "SELECT /*$sqlComment*/
                    $fieldName,
                    $fieldDesc,
                    $fieldName as real_data,
                    $fieldDesc as label,
                    $fieldName as value
                FROM $tabla
                WHERE 1
                ORDER BY 2";

                $array = ia_sqlArrayIndx($sql);
                jsonFile($fileName, $array);  // Cache to file
                $jsonData = json_encode($array);
            }
            break;

        // ... more cases ...
    }

    return $jsonData;
}
```

**Step 2**: Register catalog in `catalogosDefault()`

```php
function catalogosDefault() {
    return array(
        'propietario' => array('propietario'),
        // ... other catalogs ...
    );
}
```

**Step 3**: (Optional) Pre-generate catalog

From PHP or via URL:

```php
generaCatalogos(['propietario']);
```

Or visit in browser (as admin):
```
https://dev-app.filemonprime.net/quantix/backoffice/obtenCatalogo.php?catalogo=propietario&genera=1
```

### Catalog SQL Patterns

#### Simple catalog (single table)
```php
case 'departamento':
    $sql = "SELECT
        departamento_id,
        nombre_departamento,
        departamento_id as real_data,
        nombre_departamento as label,
        departamento_id as value
    FROM departamento
    WHERE activo = 1
    ORDER BY nombre_departamento";
    // ... execute and cache ...
    break;
```

#### Catalog with JOIN
```php
case 'usuario_completo':
    $sql = "SELECT
        u.usuario_id,
        CONCAT(u.nombre, ' - ', d.nombre) as nombre_completo,
        u.usuario_id as real_data,
        CONCAT(u.nombre, ' - ', d.nombre) as label,
        u.usuario_id as value
    FROM usuarios u
    LEFT JOIN departamentos d ON u.departamento_id = d.departamento_id
    WHERE u.activo = 1
    ORDER BY u.nombre";
    // ... execute and cache ...
    break;
```

#### Catalog with category filter
```php
case 'producto_por_categoria':
    $where = $categoria_id > 0 ? " AND categoria_id = " . (int)$categoria_id : "";
    $sql = "SELECT
        producto_id,
        nombre_producto,
        producto_id as real_data,
        nombre_producto as label,
        producto_id as value
    FROM productos
    WHERE activo = 1 $where
    ORDER BY nombre_producto";
    // ... execute and cache ...
    break;
```

### Cache Management

**Cache location**: `/backoffice/json/`

**File naming**: `{nombreCatalogo}{extraId}.html`

Examples:
- `propietario.html`
- `producto_por_categoria123.html` (with extraId)

**Force regeneration**:
1. Via URL parameter: `?genera=1`
2. Via PHP: `obtenCatalogo('propietario', true)`
3. Delete cache file manually

**When to regenerate**:
- Data changed (new records added/updated)
- Column definitions changed
- Testing catalog modifications

---

## Part 4: Complete Working Example

### Scenario: Add autocomplete search to "Propietario" column

Let's walk through adding autocomplete to the `propietario` field in the `app_propietario` class.

#### Step 1: Define the catalog in `obtenCatalogo()`

**File**: `/cobranza/inc/vitex_catalogos.php`

```php
function obtenCatalogo($nombreCatalogo ='', $genera=false, $extraId='', $printFiles = false, $categoria_id = -1, $debug = false) {
    // ... setup code ...

    switch($nombreCatalogo) {

        case 'propietario':
            if(!$genera && file_exists($fileName))
                $jsonData = getSafeDataFromFile($fileName);

            if(strlen($jsonData) === 0) {
                $tabla = 'propietario';
                $fieldName = 'propietario_id';
                $fieldDesc = 'propietario';

                $sql = "SELECT /*$sqlComment*/
                    $fieldName,
                    $fieldDesc,
                    $fieldName as real_data,
                    $fieldDesc as label,
                    $fieldName as value
                FROM $tabla
                WHERE 1
                ORDER BY 2";

                $array = ia_sqlArrayIndx($sql);
                jsonFile($fileName, $array);
                $jsonData = json_encode($array);
            }
            break;

        // ... other cases ...
    }

    return $jsonData;
}
```

#### Step 2: Register catalog in `catalogosDefault()`

**File**: `/cobranza/inc/vitex_catalogos.php`

```php
function catalogosDefault() {
    return array(
        'propietario' => array('propietario'),
        // ... other catalogs ...
    );
}
```

#### Step 3: Configure `colModel_overRide` in app class

**File**: `/app/app_propietario.php`

```php
class app_propietario extends iacase_base {

    function listme_pre($grid) {
        $this->colModel_overRide = array(
            'propietario' => array(
                'width' => '360',
                'classes' => 'bold txt18px',
                'searchoptions' => "{ sopt:['cn']}",
                'comboBoxAutoComplete' => [
                    'isAjax' => true,
                    'dataUrl' => '',
                    'campos' => "propietario",
                    'catalogo' => 'propietario',
                    'label' => 'propietario',
                    'pk' => "propietario",
                    'strict' => false,
                    'autoFocus' => true,
                    'fixedWidth' => 350,
                ],
            ),
        );

        return true;
    }
}
```

#### Step 4: Test the implementation

1. **Generate the catalog** (visit as admin):
   ```
   https://dev-app.filemonprime.net/quantix/backoffice/obtenCatalogo.php?catalogo=propietario&genera=1
   ```

2. **Verify cache file created**:
   ```
   /backoffice/json/propietario.html
   ```

3. **Open the grid page**:
   ```
   https://dev-app.filemonprime.net/quantix/backoffice/propietario.php
   ```

4. **Test autocomplete**:
   - Click in the "Propietario" search field
   - Type a few characters
   - Dropdown should appear with matching names
   - Select an item → grid filters to that propietario

#### Step 5: Debug if not working

**Check browser console** (F12):
```javascript
// Should see AJAX request:
GET /quantix/backoffice/obtenCatalogo.php?catalogo=propietario
// Response should be JSON array
```

**Check PHP error log** for SQL errors or missing functions.

**Verify colModel generated** (browser console):
```javascript
$("#propietario_grid").jqGrid('getGridParam', 'colModel')
// Find 'propietario' column, check comboBoxAutoComplete exists
```

---

## Part 5: Advanced Examples

### Example 6: Autocomplete with Custom Formatting

Combine autocomplete with custom cell formatting:

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'cliente_id' => array(
            'width' => '300',
            'classes' => 'cliente-cell',
            'formatter' => 'clienteFormatter',  // Custom JavaScript formatter
            'comboBoxAutoComplete' => [
                'isAjax' => true,
                'campos' => "razon_social, rfc",
                'catalogo' => 'cliente',
                'label' => 'razon_social',
                'pk' => "cliente_id",
            ],
        ),
    );
    return true;
}
```

Then define JavaScript formatter:

```javascript
function clienteFormatter(cellvalue, options, rowObject) {
    return '<strong>' + rowObject.razon_social + '</strong><br>' +
           '<small>' + rowObject.rfc + '</small>';
}
```

### Example 7: Multiple Autocomplete Columns

```php
function listme_pre($grid) {
    $this->colModel_overRide = array(
        'propietario' => array(
            'comboBoxAutoComplete' => [
                'isAjax' => true,
                'catalogo' => 'propietario',
                'label' => 'propietario',
                'pk' => "propietario_id",
            ],
        ),
        'departamento' => array(
            'comboBoxAutoComplete' => [
                'isAjax' => true,
                'catalogo' => 'departamento',
                'label' => 'nombre_departamento',
                'pk' => "departamento_id",
            ],
        ),
        'responsable' => array(
            'comboBoxAutoComplete' => [
                'isAjax' => true,
                'catalogo' => 'usuario',
                'label' => 'nombre_usuario',
                'pk' => "usuario_id",
            ],
        ),
    );
    return true;
}
```

### Example 8: Conditional Catalog (Category-filtered)

**obtenCatalogo() with category support**:

```php
case 'producto_categoria':
    $where = $categoria_id > 0 ? " AND c.categoria_id = " . (int)$categoria_id : "";
    $sql = "SELECT
        p.producto_id,
        CONCAT(p.nombre, ' (', c.nombre_categoria, ')') as nombre_completo,
        p.producto_id as real_data,
        CONCAT(p.nombre, ' (', c.nombre_categoria, ')') as label,
        p.producto_id as value
    FROM productos p
    LEFT JOIN categorias c ON p.categoria_id = c.categoria_id
    WHERE p.activo = 1 $where
    ORDER BY p.nombre";

    $array = ia_sqlArrayIndx($sql);
    jsonFile($fileName, $array);
    $jsonData = json_encode($array);
    break;
```

**Use different catalog per category**:

```php
'comboBoxAutoComplete' => [
    'isAjax' => true,
    'dataUrl' => 'obtenCatalogo.php?catalogo=producto_categoria&cat=' . $categoria_id,
    'catalogo' => 'producto_categoria',
    'label' => 'nombre_completo',
    'pk' => "producto_id",
],
```

---

## Part 6: Reference Tables

### comboBoxAutoComplete Parameters (Complete)

| Parameter | Type | Required | Default | AJAX Mode | Static Mode | Description |
|-----------|------|----------|---------|-----------|-------------|-------------|
| `isAjax` | bool | No | `false` | ✓ | — | Enable AJAX mode (vs static JSON) |
| `dataUrl` | string | For static | `''` | ✓ | ✓ | URL to fetch data (empty = default) |
| `campos` | string | For AJAX | field name | ✓ | — | Comma-separated columns to search |
| `catalogo` | string | For AJAX | field - '_id' | ✓ | — | Catalog name (case in obtenCatalogo) |
| `label` | string | For AJAX | field name | ✓ | — | Column to display in dropdown |
| `pk` | string | For AJAX | catalogo + '_id' | ✓ | — | Primary key column |
| `strict` | bool | No | `true` | ✓ | ✓ | Must select from list? |
| `autoFocus` | bool | No | `true` | ✓ | ✓ | Auto-focus first match |
| `fixedWidth` | int | No | calculated | ✓ | ✓ | Dropdown width (px) |
| `order_by` | string | No | label | ✓ | — | SQL ORDER BY clause |
| `group_by` | string | No | `''` | ✓ | — | SQL GROUP BY clause |
| `accion` | string | No | `'buscar_valor'` | ✓ | — | AJAX action parameter |
| `busqueda_relajada` | bool | No | `false` | ✓ | ✓ | Relaxed matching |
| `onEmptyValSliceNumber` | int | No | `25` | ✓ | ✓ | Max results when empty |
| `json_var` | string | No | — | — | ✓ | Global JS variable with data |
| `data` | array | No | — | — | ✓ | Inline data array |

### Common colModel Properties

| Property | Type | Example | Description |
|----------|------|---------|-------------|
| `name` | string | `'propietario'` | Field name (matches DB column) |
| `index` | string | `'propietario'` | SQL ORDER BY column |
| `label` | string | `'Propietario'` | Column header text |
| `width` | int/string | `360`, `'25%'` | Column width |
| `align` | string | `'left'`, `'center'`, `'right'` | Text alignment |
| `sortable` | bool | `true`, `false` | Allow sorting |
| `search` | bool | `true`, `false` | Allow searching |
| `hidden` | bool | `true`, `false` | Hide column |
| `editable` | bool | `true`, `false` | Inline editing |
| `classes` | string | `'bold txt18px'` | CSS classes for cells |
| `formatter` | string | `'select'`, `'date'`, `'currency'` | Cell formatter |
| `searchoptions` | object (string) | `"{ sopt:['cn'] }"` | Search configuration |
| `editoptions` | object | `{size:30, maxlength:50}` | Edit field options |
| `template` | string | `'colFmt.textsmall'` | Column template reference |
| `comboBoxAutoComplete` | object | `[...]` | Autocomplete configuration |

### Search Operator Reference (`sopt`)

| Operator | Code | SQL Translation | Use Case |
|----------|------|-----------------|----------|
| Equal | `'eq'` | `field = 'value'` | Exact match |
| Not equal | `'ne'` | `field != 'value'` | Exclusion |
| Less than | `'lt'` | `field < 'value'` | Numeric/date comparisons |
| Less or equal | `'le'` | `field <= 'value'` | Range searches |
| Greater than | `'gt'` | `field > 'value'` | Numeric/date comparisons |
| Greater or equal | `'ge'` | `field >= 'value'` | Range searches |
| Begins with | `'bw'` | `field LIKE 'value%'` | Name/code prefixes |
| Does not begin with | `'bn'` | `field NOT LIKE 'value%'` | Negative prefix |
| Ends with | `'ew'` | `field LIKE '%value'` | Suffixes |
| Does not end with | `'en'` | `field NOT LIKE '%value'` | Negative suffix |
| Contains | `'cn'` | `field LIKE '%value%'` | **Most common** - text search |
| Does not contain | `'nc'` | `field NOT LIKE '%value%'` | Negative text search |
| In | `'in'` | `field IN (...)` | Multiple values |
| Not in | `'ni'` | `field NOT IN (...)` | Exclude multiple |

---

## Part 7: Troubleshooting Guide

### Problem: Autocomplete doesn't appear

**Symptom**: Typing in search field shows no dropdown.

**Diagnosis**:

1. **Check browser console (F12)**:
   - Look for JavaScript errors
   - Check AJAX request to `obtenCatalogo.php`
   - Verify response is valid JSON

2. **Check colModel configuration**:
   ```javascript
   var cm = $("#grid_id").jqGrid('getGridParam', 'colModel');
   console.log(cm);
   // Find your column, verify comboBoxAutoComplete exists
   ```

3. **Check catalog endpoint**:
   Visit directly: `/backoffice/obtenCatalogo.php?catalogo=your_catalog`
   - Should return JSON array
   - Should not show errors

**Solutions**:

- **Missing comboBoxAutoComplete**: Check `listme_pre()` is defined and returns `true`
- **Wrong catalog name**: Ensure `'catalogo'` matches case in `obtenCatalogo()`
- **Empty JSON**: Check SQL query in `obtenCatalogo()`, verify table has data
- **Permission denied**: Ensure user has permission to access catalog endpoint

### Problem: Autocomplete shows but search doesn't work

**Symptom**: Dropdown appears, user selects item, but grid doesn't filter.

**Diagnosis**:

1. **Check grid postData**:
   ```javascript
   var postData = $("#grid_id").jqGrid('getGridParam', 'postData');
   console.log(postData.filters);
   ```

2. **Verify field name** in `comboBoxAutoComplete.pk` matches actual column name

3. **Check search operators**: Ensure `searchoptions` includes valid operators

**Solutions**:

- **Mismatched field names**: Verify `'pk'` is correct column name
- **Wrong search operator**: Use `'cn'` (contains) for text fields
- **Strict mode blocking**: Set `'strict' => false` to allow freeform input

### Problem: Catalog not updating after data changes

**Symptom**: New records don't appear in autocomplete.

**Diagnosis**:

1. **Check cache file**: `/backoffice/json/your_catalog.html`
   - Is it old (timestamp)?
   - Delete it manually

2. **Force regeneration**:
   ```
   /backoffice/obtenCatalogo.php?catalogo=your_catalog&genera=1
   ```

**Solutions**:

- **Clear cache**: Delete JSON cache file
- **Force regeneration**: Add `&genera=1` to URL
- **Disable caching during dev**: Modify `obtenCatalogo()` to always regenerate:
  ```php
  // Temporary - remove before production
  $genera = true;
  ```

### Problem: AJAX errors / 500 responses

**Symptom**: Console shows HTTP 500 error on autocomplete request.

**Diagnosis**:

1. **Check PHP error log**: Look for SQL errors, undefined functions
2. **Test catalog directly**: Visit `/backoffice/obtenCatalogo.php?catalogo=your_catalog`
3. **Check SQL query**: Copy SQL from `obtenCatalogo()`, run in MySQL shell

**Solutions**:

- **SQL syntax error**: Fix query in `obtenCatalogo()`
- **Missing table**: Create table or adjust SQL
- **Permission error**: Check user has SELECT permission on table
- **Missing function**: Ensure `ia_sqlArrayIndx()` is available

### Problem: Autocomplete too slow

**Symptom**: Dropdown takes several seconds to appear.

**Diagnosis**:

1. **Check catalog size**: How many rows?
   ```sql
   SELECT COUNT(*) FROM your_table;
   ```

2. **Check query performance**:
   ```sql
   EXPLAIN SELECT ... FROM your_table WHERE ...;
   ```

**Solutions**:

- **Add index**: Create index on search columns
  ```sql
  ALTER TABLE your_table ADD INDEX idx_search (search_column);
  ```
- **Use static mode**: Pre-generate catalog, use `'isAjax' => false`
- **Limit results**: Add `LIMIT 100` to SQL query
- **Add WHERE clause**: Filter out inactive records

### Problem: Dropdown appears in wrong position

**Symptom**: Autocomplete dropdown appears far from input field.

**Solutions**:

- **Set appendTo**: In JavaScript customization, set `appendTo` to grid container
- **Check CSS**: Conflicting `position` or `z-index` styles
- **Resize grid**: Trigger resize after grid initialization

---

## Best Practices

### 1. Always define catalogs before using

**Bad**:
```php
// In app class, references 'producto' catalog
'comboBoxAutoComplete' => ['catalogo' => 'producto', ...]
```
// But no 'producto' case in `obtenCatalogo()`!

**Good**:
```php
// First, add to obtenCatalogo():
case 'producto':
    // ... SQL query ...
    break;

// Then use in app class:
'comboBoxAutoComplete' => ['catalogo' => 'producto', ...]
```

### 2. Use meaningful catalog names

**Bad**: `'catalogo' => 'cat1'`

**Good**: `'catalogo' => 'propietario'`, `'catalogo' => 'cliente_activo'`

### 3. Always include ORDER BY in catalog SQL

**Bad**:
```sql
SELECT * FROM propietario WHERE 1
```

**Good**:
```sql
SELECT * FROM propietario WHERE 1 ORDER BY propietario ASC
```

### 4. Cache static catalogs, use AJAX for dynamic data

**Static (rarely changes)**: Country list, status types → Use `'isAjax' => false`

**Dynamic (changes often)**: Clients, products, users → Use `'isAjax' => true`

### 5. Set appropriate `strict` mode

**Strict (`true`)**: User must select from list (good for foreign keys)

**Non-strict (`false`)**: User can type freeform text (good for flexible search)

### 6. Use `campos` for multi-field search

**Single field**: `'campos' => 'propietario'`

**Multiple fields**: `'campos' => 'propietario, departamento, correo'`

This allows searching across multiple columns simultaneously.

### 7. Pre-generate catalogs for production

During deployment, pre-generate all catalogs:

```php
generaCatalogos('QWERTY'); // Generates all registered catalogs
```

Avoids first-user slowness and ensures cache is fresh.

### 8. Monitor catalog file sizes

Large catalogs (>1MB) should be optimized:
- Add WHERE clauses to filter data
- Use AJAX mode instead of static
- Consider pagination for very large datasets

### 9. Document custom catalogs

Add comments to `obtenCatalogo()` cases:

```php
case 'propietario':
    // PROPIETARIO CATALOG
    // Used in: app_propietario.php, app_propiedad.php
    // Columns: propietario_id (PK), propietario (name)
    // Filters: None (shows all)

    $sql = "SELECT ...";
    // ...
    break;
```

### 10. Test with empty search

Double-click on search field to trigger empty search (`#@@#`). This shows all available options. Ensure:
- Results are reasonable (not thousands of rows)
- Ordering is correct
- Labels are formatted properly

---

## Summary

The `colModel_overRide` + `obtenCatalogo` + `comboBoxAutoComplete` system provides a powerful way to enhance jqGrid columns with autocomplete search:

1. **Define catalog** in `obtenCatalogo()` switch statement (SQL query)
2. **Register catalog** in `catalogosDefault()`
3. **Configure autocomplete** in `listme_pre()` using `colModel_overRide`
4. **Test and debug** using browser console and direct catalog URLs

This three-layer approach allows:
- **Auto-generation**: Start with automatic grids from DB schema
- **Customization**: Override specific columns as needed
- **Performance**: Cached catalogs for speed
- **Flexibility**: AJAX for dynamic data, static for stable data

By understanding these components and their interactions, you can create rich, user-friendly data grids with minimal code.

### File Organization

**DO NOT EDIT**:
- `/inc/appRelateBase.php` - Auto-generated from DB
- `/vendor/*` - Composer dependencies

**Configuration Files**:
- `/inc/config.php` - Main configuration (DB, session, paths)
- `phpunit.xml` - Test suite configuration
- `eslint.config.mjs` - JavaScript linting rules
- `composer.json` - PHP dependencies

**Entry Points**:
- `/backoffice/index.php` - Main dashboard
- `/cobranza/index.php` - Collections module
- `/mobile/index.php` - Mobile interface
- `websocket_server.php` - WebSocket server daemon

### Security Considerations

**SQL Injection Prevention**:
- Use `ia_insert()`, `ia_update()` helpers (auto-escape)
- Manual queries: Use `mysqli_real_escape_string()` or prepared statements
- **Never** concatenate user input directly into SQL

**Session Security**:
- Regenerate session ID on login: `session_regenerate_id(true)`
- Timeout enforcement via `sessionTimeOut()`
- SameSite=Strict cookies

**CSRF Protection**:
- Built into iaCase framework
- Token validation on form submissions

**Permission Checks**:
- Always verify in iaCase constructor and action handlers
- Check user type: `usuarioTipoRony()`, `usuarioSupervisaBodega()`
- Respect field-level permissions from `iac_field_permission`

**Input Validation**:
- iaCase auto-validates based on DB schema (type, length, nullable)
- Custom validation: Override `validate()` method in app class

### Troubleshooting

**Tests failing**:
- Ensure `/wamp/www/showErrors.vin` exists
- Check PHPUnit running from `/wamp/www/vitex` directory
- Verify session user_id=1 in bootstrap.php

**WebSocket async not working**:
- Check Workerman server running: `php websocket_server.php status`
- Test WebSocket connection: `ws://localhost:2000`
- Falls back to HTTP async automatically if WebSocket unavailable

**jqGrid not loading data**:
- Check AJAX endpoint: `/backoffice/ajax/jqgrid_read.php`
- Verify SQL query in app class constructor
- Inspect browser console for JavaScript errors

**Permission denied**:
- Check user type: `SELECT tipo FROM iac_usr WHERE id = ?`
- Verify table permissions: `permiso_insert`, `permiso_update`, `permiso_delete`
- Check field permissions: `SELECT * FROM iac_field_permission WHERE user_id = ?`

**Schema changes not reflected**:
- Regenerate `appRelateBase.php` after ALTER TABLE
- Clear opcache if in production
- Hard refresh browser (Ctrl+F5) to clear cached JavaScript

**Database query returns empty array / "No data loaded"**:
- ✓ Check you're using the correct function:
  - `ia_sqlArrayIndx($sql)` - For simple loops (most common)
  - `ia_sqlArray($sql, 'id')` - For keyed access (requires 2nd parameter!)
- ✓ Verify SQL query syntax: `echo` the SQL and test in MySQL shell
- ✓ Check database connection is working: `ia_singleton("SELECT 1")`
- ✓ Ensure user is logged in (session required for web scripts)
- ✓ Verify table has data: `SELECT COUNT(*) FROM tablename`
- ✓ Common mistake: Using `ia_sqlArray()` without the key parameter will fail!

**Helper scripts show login page instead of running**:
- Scripts in `/backoffice/helper/` require authenticated session
- Access via browser where you're already logged into Quantix
- **Cannot** use curl without session cookies
- **Cannot** run via CLI without modifying path dependencies
- Solution: Open script URL in logged-in browser session
