<?php

use JetBrains\PhpStorm\NoReturn;

/**
 * Clase para acciones AJAX
 * @author José Juan del Prado
 * @version 1.2
 *
 * 2021-10-27 $this->response['content_type'] por default 'json' de poner 'html' el html va en $this->response[html]
 *    o $this->response[content] o $this->response[content_html]
 * 2021-10-? $this->setDebug() o tener $gDebugging=true antes de crear, agrega en la respuesta sql errors, sql trace y last php error
 */

/**
 * Si response, tipo json, tiene selector_NOMBRE_FUNCTION_JQUERY ejemplo selector_html, selector_val, selector_addClass etc
 * Entonces para todos los selectors ("#id", ".clase", etc) corre la function con el parametro
 * Y en el done del ajax se llama autoAjaxHelper(data)
 *   ejemplo $.ajax({url:'xx_acciones.php'}).done(function(data){autoAjaxHelper(data);});
 *
 * @example data {selector_html:{"#nombreGato":"felix"}} corre $("#nombreGato").html("felix")
 */

abstract class AjaxHelper
{
    public const ERROR_CAPTURA = 400;
    public const ERROR_SIN_PERMISO = 403;

    /**
     * @var array $response Array de respuesta ['code' => 200, 'status' => true, 'message' => '', 'keysNecesariosPorAccion'=>mixed, ...]
     *
     */
    public array $response = ['code' => 200, 'status' => true, 'message' => '' , 'content_type' => 'json'];

    /**
     * @var array|string[] Array de headers para content-type
     */
    protected array $headerContentType = ['json' => 'application/json', 'html' => 'text/html; charset=utf-8'];

    /**
     * @var string $accion Nombre del la funcion que se va a ejecutar. Se obtiene del request con nombre 'accion'
     */
    protected string $accion = '';

    protected bool $debug = false;

    /**
     * @param array $opciones - [accion=> 'string de accion a ejecutar', 'execute' => true|false]
     */
    function __construct(array $opciones = [])
    {
        global $gDebugging;
        if($gDebugging) {
            $this->setDebug();
            iaTimer::init();
        }
        $this->accion = $opciones['accion']??param('accion');
        $execute = $opciones['execute']??true;

        if (empty($this->accion)) {
            $this->response = ['code' => 400, 'status' => false, 'message' => 'Solicitud errónea! no se solicito la acción'];
            $this->response();
        }

        if (in_array($this->accion, get_class_methods($this))) {
            if($this->debug) iaTimer::start(get_class($this) . '->' . "puedeHacerAccion()");
            if($this->puedeHacerAccion()) {
                if($this->debug) iaTimer::end(get_class($this) . '->' . "puedeHacerAccion()");
                if ($execute) {
                    if($this->debug) iaTimer::start(get_class($this) . '->' . $this->accion.'()');
                    $this->{$this->accion}();
                    if($this->debug) iaTimer::end(get_class($this) . '->' . $this->accion.'()');
                }
            }
            else {
                if($this->debug) iaTimer::end(get_class($this) . '->' . "puedeHacerAccion()");
                if (!isset($this->response['message'])  || empty($this->response['message']))
                    $this->response = ['code' => 403, 'status' => false, 'message' => 'No tienes permisos para esta acción'];
            }
        } else {
            $this->response = ['code' => 400, 'status' => false, 'message' => "La acción solicitada es invalida '$this->accion'", 'accion' => $this->accion];
        }
        if ($execute)
            $this->response();
    }

    protected function setDebug() {
        $this->debug = true;
        global $gSqlClass;
        $gSqlClass->traceOn = true;
    }

    /**
     * Regresa true si el usuario tiene permiso de hacer $this->accion, false del contrario
     *
     * @return bool true puede hacer la accion, false no tiene permiso
     *
     * @example
      protected function puedeHacerAccion() {
         switch($this->accion) {
           case 'obtenProductoColor':
               return puede_consultar_productos();
           default:
               return false;
         }
      }
     */
    abstract protected function puedeHacerAccion():bool;

    /**
     *  Manda header 200 o de error, $this->response y Muere el script
     *  El header y contenido segun $this->response['content_type']:
     *    en default o $this->response['content_type']='json' manda json
     *    en html
     */
    #[NoReturn]
    private function response() {
        if(empty($this->response['code']))
            $this->response['code'] = 500;
        if ($this->response['code'] == 200 ) {
            if(($this->response['content_type'] ?? 'json') === 'json') {
                header('Content-Type: ' . $this->headerContentType['json']);
            }
        } else {
            $responseCode = intval( $this->response['code'] ?? 500 );
            http_response_code( $responseCode === 0 ? 500 : $responseCode );
        }

        if($this->debug)
            $this->quePexReport();

        if (($this->response['content_type'] ?? 'json') === 'html') {
            header('Content-Type: ' . $this->headerContentType['html']);
            echo $this->response['content_html'] ?? $this->response['content'] ?? $this->response['html'];
        }
        else {
            unset($this->response['content_type']);
            echo json_encode($this->response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }
        ia_errores_a_dime();
        die();
    }

    public function quePexReport() {
        global $gSqlClass, $gIApath, $gDime;

        $reportResponse = [];
        foreach($this->response as $k => $v)
            if(is_array($v))
                $reportResponse[$k] = 'Array';
            elseif(is_string($v))
                $reportResponse[$k] = substr(htmlentities($v), 0, 100) .
                    (strlen($v) > 100 ? " ..." : "");
            else
                $reportResponse[$k] = $v;

                $sqlErr = $gSqlClass->errorLog_get();
        if (!empty($sqlErr))
            $this->response['sqlErrors'] = $sqlErr;
        $this->response['sqlTrace'] = $gSqlClass->trace_get();
        $explain = [];
        foreach($this->response['sqlTrace'] as $id => $sql) {
            $s = ltrim(strtolower($sql));
            if( !empty($sql) &&
                !str_starts_with($s, 'set') && !str_starts_with($s, 'call') &&
                !str_starts_with($s, 'commit') && !str_starts_with($s, 'rollback') &&
                !str_starts_with($s, 'truncate') &&
                !str_contains($s, 'FROM aa_tmp')
            )
                $explain[$id] = "<li><pre>".strip_tags($sql)."</pre><pre>" . iaTableIt::getTableIt(ia_sqlArrayIndx("EXPLAIN $sql")) . "</pre>";
            else
                $explain[$id] = "<li><pre>$sql</pre>";
        }

        $phpLastError = error_get_last();
        if (!empty($phpLastError))
            $this->response['phpLastError'] = $phpLastError;

        $named = get_class($this) . ":" . ($_REQUEST['accion'] ?? '');
        $date = Date('d/M/y H:i');
        $css = iaTimer::cssClases() . iaTableIt::getCssClases();
        $fontFamily = 'font-family: "Source Code Pro", "SFMono-Regular", Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace';
        $fileName = __DIR__ . "/../txt/quePex_ajax_" .get_class($this) . "_" . ($_REQUEST['accion'] ?? '') . ".html";

        file_put_contents($fileName,
            "<html><head><meta charset='utf-8'><title>q!Pex: $named</title><style>
                BODY{margin:1em;padding:1em;$fontFamily} 
                PRE {margin:1em;padding:1em;border:1px blue solid;$fontFamily;width:fit-content}
                $css
                FIELDSET {width:fit-content}        
                TABLE {white-space: normal}  
                </style>
                </head><body><h1>q!Pex: $named</h1><br>$date $_SESSION[usuario] url:$_SERVER[REQUEST_URI]<pre><fieldset>" .

            "<fieldset><legend>\$_REQUEST</legend>" .
                ToCode::variable('Request',  $_REQUEST) .
            "</fieldset>" .

            "<fieldset><legend>\$gDime</legend>" .
                ToCode::variable('gDime',  empty($gDime) ? [] : $gDime) . "\r\n" .
            "</fieldset>" .

            "<fieldset><legend>Response</legend>" .
                ToCode::variable('response',  $reportResponse) . "\r\n" .
            "</fieldset>" .

            "<fieldset><legend>Detected Errors</legend>" .
                ToCode::variable('PHP Last Error',  $phpLastError) . "\r\n" .
                ToCode::variable('PHP Last JsonError',  json_last_error_msg()) . "\r\n" .
                ToCode::variable('PHP Last RegExpError',  preg_last_error_msg()) . "\r\n" .
                ToCode::variable('Sql_Errors', $sqlErr) . "\r\n" .
            "</fieldset>" .

            "<fieldset><legend>Queries</legend>\r\n\r\n<ol>" . implode("\r\n\r\n", $explain) .
                "\r\n\r\n</ol>\r\n\r\n</fieldset>" . "\r\n" .
            "</fieldset>" .

            "<fieldset><legend>Timing</legend>" .
                iaTimer::table(null, true, true, true) .
            "</fieldset>" .

            "</fieldset></pre></body></html>"

        );
    }
}
