<?php
/**
 * PropertyRevenueLib - Unified Property Revenue Aggregation Library
 * 
 * Provides unified access to Cloudbeds and Hostify financial data
 * Normalizes schema differences for consistent reporting
 * 
 * Data Sources:
 *   - cloudbeds_reserva: 6 financial fields (grand_total, accommodation_total, amount_paid, etc.)
 *   - hostify_reserva: 1 financial field (total_price)
 * 
 * Usage:
 *   $lib = new PropertyRevenueLib();
 *   $data = $lib->getUnifiedReservations(['fecha_inicio' => '2025-01-01', 'fecha_fin' => '2025-01-31']);
 *   $grouped = $lib->aggregateByProperty($data);
 *   $summary = $lib->calculateFinancialSummary($grouped);
 * 
 * @package    MrW-Reporte-Propietarios
 * @subpackage FinancialReporting
 * @author     Development Team
 * @created    2026-01-08
 * @version    1.0
 */

class PropertyRevenueLib {
    
    /**
     * Default currency for reports
     */
    const DEFAULT_CURRENCY = 'MXN';
    
    /**
     * Source system identifiers
     */
    const SOURCE_CLOUDBEDS = 'Cloudbeds';
    const SOURCE_HOSTIFY = 'Hostify';
    
    /**
     * Fetch unified reservations from Cloudbeds and Hostify
     * 
     * Normalizes data from both PMS systems into a consistent format:
     * - Cloudbeds: Uses detailed financial breakdown
     * - Hostify: Maps total_price to all financial fields
     * 
     * @param array $filters Optional filters:
     *   - fecha_inicio (string): Start date YYYY-MM-DD (default: first day of current month)
     *   - fecha_fin (string): End date YYYY-MM-DD (default: last day of current month)
     *   - propiedad_id (string): Filter by specific property UUID
     *   - propietario_id (string): Filter by property owner UUID
     *   - sistema (string): Filter by source system ('Cloudbeds' or 'Hostify')
     *   - canal (string): Filter by booking channel
     *   - include_canceled (bool): Include canceled reservations (default: true)
     * 
     * @return array Unified reservation data, sorted by property name and check-in date
     */
    public function getUnifiedReservations($filters = []) {
        // Extract filters with defaults
        $fecha_inicio = $filters['fecha_inicio'] ?? date('Y-m-01');
        $fecha_fin = $filters['fecha_fin'] ?? date('Y-m-t');
        $propiedad_id = $filters['propiedad_id'] ?? '';
        $propietario_id = $filters['propietario_id'] ?? '';
        $sistema = $filters['sistema'] ?? '';
        $canal = $filters['canal'] ?? '';
        $include_canceled = $filters['include_canceled'] ?? true;
        
        // Sanitize dates
        $fecha_inicio = $this->sanitizeDate($fecha_inicio);
        $fecha_fin = $this->sanitizeDate($fecha_fin);
        
        $results = [];
        
        // Fetch Cloudbeds data (unless filtered to Hostify only)
        if (empty($sistema) || $sistema === self::SOURCE_CLOUDBEDS) {
            $cloudbeds_data = $this->fetchCloudbedsReservations(
                $fecha_inicio, $fecha_fin, $propiedad_id, $propietario_id, $canal, $include_canceled
            );
            $results = array_merge($results, $cloudbeds_data);
        }
        
        // Fetch Hostify data (unless filtered to Cloudbeds only)
        if (empty($sistema) || $sistema === self::SOURCE_HOSTIFY) {
            $hostify_data = $this->fetchHostifyReservations(
                $fecha_inicio, $fecha_fin, $propiedad_id, $propietario_id, $canal
            );
            $results = array_merge($results, $hostify_data);
        }
        
        // Sort by property name, then by check-in date
        usort($results, function($a, $b) {
            $prop_cmp = strcmp($a['nombre_propiedad'], $b['nombre_propiedad']);
            if ($prop_cmp !== 0) return $prop_cmp;
            return strcmp($a['fecha_inicio'], $b['fecha_inicio']);
        });
        
        return $results;
    }
    
    /**
     * Aggregate reservations by property
     * 
     * Groups reservations by propiedad_id and calculates subtotals for each property.
     * 
     * @param array $reservations Unified reservation data from getUnifiedReservations()
     * @return array Grouped data keyed by propiedad_id:
     *   [
     *     'uuid-123' => [
     *       'propiedad_id' => 'uuid-123',
     *       'nombre_propiedad' => 'Medellin 148 - 01',
     *       'propietario' => 'Juan Perez',
     *       'propietario_id' => 'uuid-owner',
     *       'reservations' => [...],
     *       'totals' => ['count' => 5, 'noches' => 15, 'ingreso_bruto' => 45000, ...]
     *     ]
     *   ]
     */
    public function aggregateByProperty($reservations) {
        $grouped = [];
        
        foreach ($reservations as $res) {
            $prop_id = $res['propiedad_id'];
            
            if (!isset($grouped[$prop_id])) {
                $grouped[$prop_id] = [
                    'propiedad_id' => $prop_id,
                    'nombre_propiedad' => $res['nombre_propiedad'],
                    'propietario' => $res['propietario'],
                    'propietario_id' => $res['propietario_id'] ?? null,
                    'colonia' => $res['colonia'] ?? '',
                    'reservations' => [],
                    'totals' => $this->initTotals(),
                    'by_system' => [
                        self::SOURCE_CLOUDBEDS => $this->initTotals(),
                        self::SOURCE_HOSTIFY => $this->initTotals()
                    ],
                    'by_channel' => []
                ];
            }
            
            $grouped[$prop_id]['reservations'][] = $res;
            $this->updateTotals($grouped[$prop_id]['totals'], $res);
            $this->updateTotals($grouped[$prop_id]['by_system'][$res['sistema']], $res);
            
            // Track by channel
            $channel = $res['canal'] ?: 'Direct';
            if (!isset($grouped[$prop_id]['by_channel'][$channel])) {
                $grouped[$prop_id]['by_channel'][$channel] = $this->initTotals();
            }
            $this->updateTotals($grouped[$prop_id]['by_channel'][$channel], $res);
        }
        
        // Sort grouped data by property name
        uasort($grouped, function($a, $b) {
            return strcmp($a['nombre_propiedad'], $b['nombre_propiedad']);
        });
        
        return $grouped;
    }
    
    /**
     * Aggregate reservations by owner (propietario)
     * 
     * Groups reservations by propietario_id, aggregating all their properties.
     * 
     * @param array $reservations Unified reservation data
     * @return array Grouped data keyed by propietario_id
     */
    public function aggregateByOwner($reservations) {
        $grouped = [];
        
        foreach ($reservations as $res) {
            $owner_id = $res['propietario_id'] ?? 'sin_propietario';
            
            if (!isset($grouped[$owner_id])) {
                $grouped[$owner_id] = [
                    'propietario_id' => $owner_id,
                    'propietario' => $res['propietario'] ?: 'Sin Propietario Asignado',
                    'properties' => [],
                    'reservations' => [],
                    'totals' => $this->initTotals()
                ];
            }
            
            // Track unique properties
            $prop_id = $res['propiedad_id'];
            if (!isset($grouped[$owner_id]['properties'][$prop_id])) {
                $grouped[$owner_id]['properties'][$prop_id] = $res['nombre_propiedad'];
            }
            
            $grouped[$owner_id]['reservations'][] = $res;
            $this->updateTotals($grouped[$owner_id]['totals'], $res);
        }
        
        // Convert properties array to count
        foreach ($grouped as &$owner) {
            $owner['property_count'] = count($owner['properties']);
            $owner['property_names'] = array_values($owner['properties']);
            unset($owner['properties']);
        }
        
        return $grouped;
    }
    
    /**
     * Aggregate reservations by booking channel
     * 
     * @param array $reservations Unified reservation data
     * @return array Grouped data keyed by channel name
     */
    public function aggregateByChannel($reservations) {
        $grouped = [];
        
        foreach ($reservations as $res) {
            $channel = $res['canal'] ?: 'Direct';
            
            if (!isset($grouped[$channel])) {
                $grouped[$channel] = [
                    'canal' => $channel,
                    'reservations' => [],
                    'totals' => $this->initTotals(),
                    'properties' => []
                ];
            }
            
            $grouped[$channel]['reservations'][] = $res;
            $this->updateTotals($grouped[$channel]['totals'], $res);
            
            // Track unique properties
            $prop_id = $res['propiedad_id'];
            if (!isset($grouped[$channel]['properties'][$prop_id])) {
                $grouped[$channel]['properties'][$prop_id] = $res['nombre_propiedad'];
            }
        }
        
        // Calculate property counts and sort by revenue
        foreach ($grouped as &$channel) {
            $channel['property_count'] = count($channel['properties']);
            unset($channel['properties']);
        }
        
        uasort($grouped, function($a, $b) {
            return $b['totals']['ingreso_bruto'] <=> $a['totals']['ingreso_bruto'];
        });
        
        return $grouped;
    }
    
    /**
     * Calculate financial summary across all properties
     * 
     * @param array $grouped_data Grouped property data from aggregateByProperty()
     * @return array Grand totals:
     *   [
     *     'propiedades' => 45,
     *     'count' => 327,
     *     'noches' => 1240,
     *     'ingreso_bruto' => 487350.00,
     *     'ingreso_recibido' => 485000.00,
     *     'saldo_pendiente' => 2350.00,
     *     'avg_nightly_rate' => 392.86,
     *     'avg_stay_length' => 3.79
     *   ]
     */
    public function calculateFinancialSummary($grouped_data) {
        $grand_totals = $this->initTotals();
        $grand_totals['propiedades'] = count($grouped_data);
        
        foreach ($grouped_data as $prop) {
            $grand_totals['count'] += $prop['totals']['count'];
            $grand_totals['noches'] += $prop['totals']['noches'];
            $grand_totals['ingreso_bruto'] += $prop['totals']['ingreso_bruto'];
            $grand_totals['ingreso_alojamiento'] += $prop['totals']['ingreso_alojamiento'];
            $grand_totals['ingreso_recibido'] += $prop['totals']['ingreso_recibido'];
            $grand_totals['saldo_pendiente'] += $prop['totals']['saldo_pendiente'];
            $grand_totals['deposito'] += $prop['totals']['deposito'];
            $grand_totals['cargo_cancelacion'] += $prop['totals']['cargo_cancelacion'];
        }
        
        // Calculate averages
        if ($grand_totals['noches'] > 0) {
            $grand_totals['avg_nightly_rate'] = round($grand_totals['ingreso_bruto'] / $grand_totals['noches'], 2);
        } else {
            $grand_totals['avg_nightly_rate'] = 0;
        }
        
        if ($grand_totals['count'] > 0) {
            $grand_totals['avg_stay_length'] = round($grand_totals['noches'] / $grand_totals['count'], 2);
        } else {
            $grand_totals['avg_stay_length'] = 0;
        }
        
        return $grand_totals;
    }
    
    /**
     * Get available properties for filter dropdown
     * 
     * @param bool $only_with_reservations Only show properties that have linked reservations
     * @return array Property list [propiedad_id => nombre_propiedad]
     */
    public function getPropertyList($only_with_reservations = false) {
        if ($only_with_reservations) {
            $sql = "
                SELECT DISTINCT p.propiedad_id, p.nombre_propiedad
                FROM propiedad p
                WHERE p.propiedad_id IN (
                    SELECT DISTINCT propiedad_id FROM cloudbeds_reserva WHERE propiedad_id IS NOT NULL
                    UNION
                    SELECT DISTINCT propiedad_id FROM hostify_reserva WHERE propiedad_id IS NOT NULL
                )
                ORDER BY p.nombre_propiedad
            ";
        } else {
            $sql = "SELECT propiedad_id, nombre_propiedad FROM propiedad ORDER BY nombre_propiedad";
        }
        
        $result = ia_sqlArrayIndx($sql);
        $list = [];
        foreach ($result as $row) {
            $list[$row['propiedad_id']] = $row['nombre_propiedad'];
        }
        return $list;
    }
    
    /**
     * Get available owners for filter dropdown
     * 
     * @param bool $only_with_properties Only show owners that have properties
     * @return array Owner list [propietario_id => propietario]
     */
    public function getOwnerList($only_with_properties = true) {
        if ($only_with_properties) {
            $sql = "
                SELECT DISTINCT pr.propietario_id, pr.propietario
                FROM propietario pr
                INNER JOIN propiedad p ON pr.propietario_id = p.propietario_id
                ORDER BY pr.propietario
            ";
        } else {
            $sql = "SELECT propietario_id, propietario FROM propietario ORDER BY propietario";
        }
        
        $result = ia_sqlArrayIndx($sql);
        $list = [];
        foreach ($result as $row) {
            $list[$row['propietario_id']] = $row['propietario'];
        }
        return $list;
    }
    
    /**
     * Get available booking channels
     * 
     * @return array Unique channel names from both systems
     */
    public function getChannelList() {
        $sql = "
            SELECT DISTINCT source as canal FROM cloudbeds_reserva WHERE source IS NOT NULL AND source != ''
            UNION
            SELECT DISTINCT channel as canal FROM hostify_reserva WHERE channel IS NOT NULL AND channel != ''
            ORDER BY canal
        ";
        
        $result = ia_sqlArrayIndx($sql);
        $list = [];
        foreach ($result as $row) {
            $list[] = $row['canal'];
        }
        return $list;
    }
    
    /**
     * Export data to CSV
     * 
     * Outputs CSV directly to browser for download.
     * 
     * @param array $grouped_data Grouped property data
     * @param string $filename Output filename (default: auto-generated with timestamp)
     * @param bool $include_details Include individual reservations (true) or summary only (false)
     */
    public function exportToCSV($grouped_data, $filename = null, $include_details = true) {
        if ($filename === null) {
            $filename = 'ingresos_propiedad_' . date('Ymd_His') . '.csv';
        }
        
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // BOM for Excel UTF-8 compatibility
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        if ($include_details) {
            // Detailed CSV with all reservations
            fputcsv($output, [
                'Propiedad', 'Propietario', 'Colonia', 'Sistema', 'Fecha Entrada', 'Fecha Salida',
                'Noches', 'Canal', 'Estado', 'Ingreso Bruto', 'Ingreso Alojamiento', 
                'Ingreso Recibido', 'Saldo Pendiente', 'Deposito', 'Huesped'
            ]);
            
            foreach ($grouped_data as $prop) {
                foreach ($prop['reservations'] as $res) {
                    fputcsv($output, [
                        $prop['nombre_propiedad'],
                        $prop['propietario'],
                        $prop['colonia'],
                        $res['sistema'],
                        $res['fecha_inicio'],
                        $res['fecha_fin'],
                        $res['noches'],
                        $res['canal'],
                        $res['estado'],
                        $res['ingreso_bruto'],
                        $res['ingreso_alojamiento'],
                        $res['ingreso_recibido'],
                        $res['saldo_pendiente'],
                        $res['deposito'],
                        $res['huesped']
                    ]);
                }
            }
        } else {
            // Summary CSV - one row per property
            fputcsv($output, [
                'Propiedad', 'Propietario', 'Colonia', 'Reservaciones', 'Noches',
                'Ingreso Bruto', 'Ingreso Recibido', 'Saldo Pendiente', 'Tarifa Promedio/Noche'
            ]);
            
            foreach ($grouped_data as $prop) {
                $avg_rate = $prop['totals']['noches'] > 0 
                    ? round($prop['totals']['ingreso_bruto'] / $prop['totals']['noches'], 2) 
                    : 0;
                    
                fputcsv($output, [
                    $prop['nombre_propiedad'],
                    $prop['propietario'],
                    $prop['colonia'],
                    $prop['totals']['count'],
                    $prop['totals']['noches'],
                    $prop['totals']['ingreso_bruto'],
                    $prop['totals']['ingreso_recibido'],
                    $prop['totals']['saldo_pendiente'],
                    $avg_rate
                ]);
            }
        }
        
        fclose($output);
        exit;
    }
    
    /**
     * Get statistics about data sources
     * 
     * @return array Statistics about Cloudbeds and Hostify data
     */
    public function getDataSourceStats() {
        $sql_cb = "
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as linked,
                MIN(check_in_date) as min_date,
                MAX(check_in_date) as max_date,
                SUM(grand_total) as total_revenue
            FROM cloudbeds_reserva
        ";
        
        $sql_hf = "
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as linked,
                MIN(check_in) as min_date,
                MAX(check_in) as max_date,
                SUM(total_price) as total_revenue
            FROM hostify_reserva
        ";
        
        $cb_stats = ia_sqlArrayIndx($sql_cb)[0];
        $hf_stats = ia_sqlArrayIndx($sql_hf)[0];
        
        return [
            'cloudbeds' => [
                'total_reservations' => (int)$cb_stats['total'],
                'linked_to_property' => (int)$cb_stats['linked'],
                'link_rate' => $cb_stats['total'] > 0 ? round(100 * $cb_stats['linked'] / $cb_stats['total'], 2) : 0,
                'date_range' => [$cb_stats['min_date'], $cb_stats['max_date']],
                'total_revenue' => (float)$cb_stats['total_revenue']
            ],
            'hostify' => [
                'total_reservations' => (int)$hf_stats['total'],
                'linked_to_property' => (int)$hf_stats['linked'],
                'link_rate' => $hf_stats['total'] > 0 ? round(100 * $hf_stats['linked'] / $hf_stats['total'], 2) : 0,
                'date_range' => [$hf_stats['min_date'], $hf_stats['max_date']],
                'total_revenue' => (float)$hf_stats['total_revenue']
            ]
        ];
    }
    
    // =========================================================================
    // PRIVATE METHODS
    // =========================================================================
    
    /**
     * Fetch and normalize Cloudbeds reservations
     */
    private function fetchCloudbedsReservations($fecha_inicio, $fecha_fin, $propiedad_id, $propietario_id, $canal, $include_canceled) {
        $where = "cb.propiedad_id IS NOT NULL AND cb.check_in_date BETWEEN '$fecha_inicio' AND '$fecha_fin'";
        
        if (!empty($propiedad_id)) {
            $where .= " AND cb.propiedad_id = " . strit($propiedad_id);
        }
        
        if (!empty($propietario_id)) {
            $where .= " AND p.propietario_id = " . strit($propietario_id);
        }
        
        if (!empty($canal)) {
            $where .= " AND cb.source = " . strit($canal);
        }
        
        if (!$include_canceled) {
            $where .= " AND (cb.status IS NULL OR cb.status NOT IN ('Canceled', 'Cancelled', 'canceled', 'cancelled'))";
        }
        
        $sql = "
            SELECT
                cb.cloudbeds_reserva_id as reserva_id,
                '" . self::SOURCE_CLOUDBEDS . "' as sistema,
                cb.propiedad_id,
                p.nombre_propiedad,
                p.colonia,
                prop.propietario_id,
                prop.propietario,
                cb.check_in_date as fecha_inicio,
                cb.check_out_date as fecha_fin,
                cb.nights as noches,
                cb.source as canal,
                cb.status as estado,
                COALESCE(cb.grand_total, 0) as ingreso_bruto,
                COALESCE(cb.accommodation_total, 0) as ingreso_alojamiento,
                COALESCE(cb.amount_paid, 0) as ingreso_recibido,
                COALESCE(cb.balance_due, 0) as saldo_pendiente,
                COALESCE(cb.deposit, 0) as deposito,
                COALESCE(cb.cancelation_fee, 0) as cargo_cancelacion,
                cb.name as huesped,
                cb.reservation_date as fecha_reservacion,
                cb.room_type as tipo_habitacion,
                (cb.adults + cb.children) as num_huespedes
            FROM cloudbeds_reserva cb
            INNER JOIN propiedad p ON cb.propiedad_id = p.propiedad_id
            LEFT JOIN propietario prop ON p.propietario_id = prop.propietario_id
            WHERE $where
            ORDER BY p.nombre_propiedad, cb.check_in_date
        ";
        
        return ia_sqlArrayIndx($sql);
    }
    
    /**
     * Fetch and normalize Hostify reservations
     */
    private function fetchHostifyReservations($fecha_inicio, $fecha_fin, $propiedad_id, $propietario_id, $canal) {
        $where = "hf.propiedad_id IS NOT NULL AND hf.check_in BETWEEN '$fecha_inicio' AND '$fecha_fin'";
        
        if (!empty($propiedad_id)) {
            $where .= " AND hf.propiedad_id = " . strit($propiedad_id);
        }
        
        if (!empty($propietario_id)) {
            $where .= " AND p.propietario_id = " . strit($propietario_id);
        }
        
        if (!empty($canal)) {
            $where .= " AND hf.channel = " . strit($canal);
        }
        
        $sql = "
            SELECT
                hf.hostify_reserva_id as reserva_id,
                '" . self::SOURCE_HOSTIFY . "' as sistema,
                hf.propiedad_id,
                p.nombre_propiedad,
                p.colonia,
                prop.propietario_id,
                prop.propietario,
                hf.check_in as fecha_inicio,
                hf.check_out as fecha_fin,
                hf.nights as noches,
                hf.channel as canal,
                'Active' as estado,
                COALESCE(hf.total_price, 0) as ingreso_bruto,
                COALESCE(hf.total_price, 0) as ingreso_alojamiento,
                COALESCE(hf.total_price, 0) as ingreso_recibido,
                0 as saldo_pendiente,
                0 as deposito,
                0 as cargo_cancelacion,
                hf.guest_name as huesped,
                NULL as fecha_reservacion,
                NULL as tipo_habitacion,
                0 as num_huespedes
            FROM hostify_reserva hf
            INNER JOIN propiedad p ON hf.propiedad_id = p.propiedad_id
            LEFT JOIN propietario prop ON p.propietario_id = prop.propietario_id
            WHERE $where
            ORDER BY p.nombre_propiedad, hf.check_in
        ";
        
        return ia_sqlArrayIndx($sql);
    }
    
    /**
     * Initialize totals array
     */
    private function initTotals() {
        return [
            'count' => 0,
            'noches' => 0,
            'ingreso_bruto' => 0,
            'ingreso_alojamiento' => 0,
            'ingreso_recibido' => 0,
            'saldo_pendiente' => 0,
            'deposito' => 0,
            'cargo_cancelacion' => 0,
            'propiedades' => 0
        ];
    }
    
    /**
     * Update totals with reservation data
     */
    private function updateTotals(&$totals, $res) {
        $totals['count']++;
        $totals['noches'] += (int)$res['noches'];
        $totals['ingreso_bruto'] += (float)$res['ingreso_bruto'];
        $totals['ingreso_alojamiento'] += (float)$res['ingreso_alojamiento'];
        $totals['ingreso_recibido'] += (float)$res['ingreso_recibido'];
        $totals['saldo_pendiente'] += (float)$res['saldo_pendiente'];
        $totals['deposito'] += (float)$res['deposito'];
        $totals['cargo_cancelacion'] += (float)$res['cargo_cancelacion'];
    }
    
    /**
     * Sanitize date input
     */
    private function sanitizeDate($date) {
        // Validate date format
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return $date;
        }
        // Try to parse and format
        $timestamp = strtotime($date);
        if ($timestamp !== false) {
            return date('Y-m-d', $timestamp);
        }
        // Default to today
        return date('Y-m-d');
    }
}
