# Property Revenue Report System

## Quick Start

```
URL: /backoffice/helper/reporte_ingresos_propiedad.php
```

## Overview

Unified income reporting system that normalizes data from **Cloudbeds** and **Hostify** PMS systems into a single view. Groups transactions by property with financial totals.

## Data Sources

| System | Reservations | Financial Fields |
|--------|--------------|------------------|
| Cloudbeds | 750 linked | grand_total, accommodation_total, amount_paid, deposit, balance_due, cancelation_fee |
| Hostify | 2,223 linked | total_price (mapped to all financial fields) |

## Features

### Report Views
- **By Property**: Each property with all its reservations and subtotals
- **By Owner**: Grouped by propietario with all their properties
- **By Channel**: Performance comparison across booking platforms (Airbnb, Booking, etc.)

### Filters
- Date range (check-in dates)
- Specific property
- Property owner
- PMS system (Cloudbeds/Hostify)
- Booking channel

### Exports
- **CSV Detailed**: All individual reservations
- **CSV Summary**: One row per property with totals

## Library Usage

```php
require_once("PropertyRevenueLib.php");

$lib = new PropertyRevenueLib();

// Fetch unified data
$filters = [
    'fecha_inicio' => '2025-01-01',
    'fecha_fin' => '2025-01-31',
    'propiedad_id' => '',      // Optional: filter by property
    'propietario_id' => '',    // Optional: filter by owner
    'sistema' => '',           // Optional: 'Cloudbeds' or 'Hostify'
    'canal' => ''              // Optional: filter by channel
];

$reservations = $lib->getUnifiedReservations($filters);

// Aggregate by property
$by_property = $lib->aggregateByProperty($reservations);

// Calculate grand totals
$totals = $lib->calculateFinancialSummary($by_property);

// Export to CSV
$lib->exportToCSV($by_property, 'report.csv');
```

## Library Methods

| Method | Returns | Description |
|--------|---------|-------------|
| `getUnifiedReservations($filters)` | array | Normalized reservations from both systems |
| `aggregateByProperty($reservations)` | array | Grouped by propiedad_id with subtotals |
| `aggregateByOwner($reservations)` | array | Grouped by propietario_id |
| `aggregateByChannel($reservations)` | array | Grouped by booking channel |
| `calculateFinancialSummary($grouped)` | array | Grand totals |
| `getPropertyList($only_with_reservations)` | array | Properties for dropdown |
| `getOwnerList($only_with_properties)` | array | Owners for dropdown |
| `getChannelList()` | array | Available channels |
| `getDataSourceStats()` | array | Statistics about linked reservations |
| `exportToCSV($data, $filename, $details)` | void | Download CSV file |

## Data Normalization

### Unified Reservation Format
```php
[
    'reserva_id' => 'uuid',
    'sistema' => 'Cloudbeds|Hostify',
    'propiedad_id' => 'uuid',
    'nombre_propiedad' => 'Medellin 148 - 01',
    'propietario' => 'Juan Perez',
    'fecha_inicio' => '2025-01-05',
    'fecha_fin' => '2025-01-08',
    'noches' => 3,
    'canal' => 'Airbnb',
    'estado' => 'Confirmed',
    'ingreso_bruto' => 4500.00,
    'ingreso_alojamiento' => 4000.00,
    'ingreso_recibido' => 4500.00,
    'saldo_pendiente' => 0.00,
    'deposito' => 1000.00,
    'cargo_cancelacion' => 0.00,
    'huesped' => 'John Doe'
]
```

### Field Mapping

| Normalized Field | Cloudbeds Source | Hostify Source |
|------------------|------------------|----------------|
| ingreso_bruto | grand_total | total_price |
| ingreso_alojamiento | accommodation_total | total_price |
| ingreso_recibido | amount_paid | total_price |
| saldo_pendiente | balance_due | 0 |
| deposito | deposit | 0 |
| cargo_cancelacion | cancelation_fee | 0 |
| fecha_inicio | check_in_date | check_in |
| fecha_fin | check_out_date | check_out |
| canal | source | channel |
| estado | status | 'Active' |

## Files

```
/backoffice/helper/
├── reporte_ingresos_propiedad.php   <- Main report UI
├── PropertyRevenueLib.php            <- Reusable library class
└── README_PROPERTY_REVENUE.md        <- This documentation
```

## Future Extensions

This library is designed for expansion into a full financial module:

- Occupancy rate calculations
- OTA performance comparison
- Revenue forecasting
- Seasonal trend analysis
- Owner statements generation

## Version History

| Version | Date | Changes |
|---------|------|---------|
| 1.0 | 2026-01-08 | Initial release with property/owner/channel views |
