<?php
/**
 * Property Insertion UI - Iteration 8.5
 *
 * Adds the 3 missing properties identified in Iteration 8 analysis:
 * - Ver Unit 4 (7 reservations)
 * - Ver PH 7 (11 reservations)
 * - Casa Ana (3 reservations)
 *
 * Expected Impact: +21 matches, 97.21% → 98.19% match rate
 */

require_once("../../inc/config.php");

// Define the 3 missing properties with defaults
$missing_properties = [
    'ver_unit_4' => [
        'label' => 'Ver Unit 4',
        'impact' => 7,
        'defaults' => [
            'nombre_propiedad' => 'Tigre 4 | Veracruz 26 | 4',
            'direccion' => 'Veracruz 26, Roma Norte',
            'departamento' => 'Tigre 4',
            'numero_unidad' => '4',
            'tipo_unidad' => 'Departamento',
            'vale' => 'Active',
            'codigo_postal' => '06700',
            'colonia' => 'ROMA NORTE',
            'estado' => 'DIF',
            'estado_descripcion' => 'CIUDAD DE MEXICO',
            'municipio' => '015',
            'municipio_descripcion' => 'CUAUHTEMOC',
            'num_deptos' => 1,
        ],
        'anuncios' => ['Ver 4 | RoS2BKQS -B - Bcom', 'Ver 4 | RoS2BKQS -B']
    ],
    'ver_ph_7' => [
        'label' => 'Ver PH 7',
        'impact' => 11,
        'defaults' => [
            'nombre_propiedad' => 'Tigre PH 7 | Veracruz 26 | PH 7',
            'direccion' => 'Veracruz 26, Roma Norte',
            'departamento' => 'Tigre PH 7',
            'numero_unidad' => 'PH 7',
            'tipo_unidad' => 'Penthouse',
            'vale' => 'Active',
            'codigo_postal' => '06700',
            'colonia' => 'ROMA NORTE',
            'estado' => 'DIF',
            'estado_descripcion' => 'CIUDAD DE MEXICO',
            'municipio' => '015',
            'municipio_descripcion' => 'CUAUHTEMOC',
            'num_deptos' => 1,
        ],
        'anuncios' => ['Ver PH 7 | RoSLKS -BT', 'Ver PH 7 | RoSLKS -BT - Bcom']
    ],
    'casa_ana' => [
        'label' => 'Casa Ana',
        'impact' => 3,
        'defaults' => [
            'nombre_propiedad' => 'Casa Ana',
            'tipo_unidad' => 'Casa',
            'vale' => 'Active',
            'num_deptos' => 1,
            // Address fields left blank for user input
            'direccion' => '',
            'departamento' => 'Casa Ana',
            'codigo_postal' => '',
            'colonia' => '',
            'estado' => '',
            'estado_descripcion' => '',
            'municipio' => '',
            'municipio_descripcion' => '',
        ],
        'anuncios' => ['Campeche Ana | CoG3BKKII -B']
    ]
];

// === HELPER FUNCTIONS ===

/**
 * Check if property already exists
 */
function propertyExists($nombre_propiedad) {
    $sql = "SELECT COUNT(*) FROM propiedad
            WHERE nombre_propiedad = " . strit($nombre_propiedad);
    return ia_singleton($sql) > 0;
}

/**
 * Validate property data
 */
function validateProperty($data) {
    $required = [
        'nombre_propiedad', 'direccion', 'departamento', 'vale',
        'codigo_postal', 'colonia', 'estado', 'estado_descripcion',
        'municipio', 'municipio_descripcion', 'num_deptos'
    ];

    foreach ($required as $field) {
        if (!isset($data[$field]) || trim($data[$field]) === '') {
            return "Missing required field: $field";
        }
    }

    // Validate field lengths
    $max_lengths = [
        'nombre_propiedad' => 50,
        'direccion' => 100,
        'departamento' => 200,
        'codigo_postal' => 5,
        'colonia' => 100,
        'estado' => 5,
        'estado_descripcion' => 100,
        'municipio' => 5,
        'municipio_descripcion' => 100,
    ];

    foreach ($max_lengths as $field => $max) {
        if (isset($data[$field]) && strlen($data[$field]) > $max) {
            return "Field '$field' exceeds maximum length of $max characters";
        }
    }

    // Validate enums
    $valid_vale = ['Active', 'Inactive'];
    if (isset($data['vale']) && !in_array($data['vale'], $valid_vale)) {
        return "Invalid value for 'vale': must be Active or Inactive";
    }

    $valid_tipo_unidad = ['Departamento', 'Casa', 'Estudio', 'Penthouse', 'Loft'];
    if (isset($data['tipo_unidad']) && !in_array($data['tipo_unidad'], $valid_tipo_unidad)) {
        return "Invalid value for 'tipo_unidad': must be one of " . implode(', ', $valid_tipo_unidad);
    }

    return true;
}

/**
 * Generate UUID
 */
function generateUUID() {
    $sql = "SELECT UUID() as new_id";
    return ia_singleton($sql);
}

/**
 * Build INSERT SQL statement
 */
function buildInsertSQL($data, $preview_only = false) {
    $fields = [];
    $values = [];

    // Add UUID
    $uuid = generateUUID();
    $fields[] = 'propiedad_id';
    $values[] = strit($uuid);

    // Add user-provided fields
    foreach ($data as $field => $value) {
        if ($value !== '' && $value !== null) {
            $fields[] = $field;
            $values[] = is_numeric($value) ? $value : strit($value);
        }
    }

    // Add system fields
    $fields[] = 'alta_por';
    $values[] = strit('iteration8_ui');

    $sql = "INSERT INTO propiedad (" . implode(', ', $fields) . ")
            VALUES (" . implode(', ', $values) . ")";

    if ($preview_only) {
        return ['sql' => $sql, 'uuid' => $uuid];
    }

    return $sql;
}

// === AJAX HANDLERS ===

// Handle SQL Preview
if (isset($_GET['action']) && $_GET['action'] === 'preview' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    $property_key = $_GET['property'] ?? '';
    if (!isset($missing_properties[$property_key])) {
        echo json_encode(['error' => 'Invalid property key']);
        exit;
    }

    // Merge defaults with user input
    $data = $missing_properties[$property_key]['defaults'];
    if (isset($_POST[$property_key])) {
        foreach ($_POST[$property_key] as $field => $value) {
            $data[$field] = $value;
        }
    }

    $result = buildInsertSQL($data, true);
    echo json_encode(['sql' => $result['sql'], 'uuid' => $result['uuid']]);
    exit;
}

// Handle Insert Execution
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'insert') {
    header('Content-Type: application/json');

    $results = [];

    foreach ($missing_properties as $key => $prop) {
        if (isset($_POST["add_$key"]) && $_POST["add_$key"] === '1') {
            // Merge defaults with user input
            $data = $prop['defaults'];
            if (isset($_POST[$key])) {
                foreach ($_POST[$key] as $field => $value) {
                    $data[$field] = $value;
                }
            }

            // Validate
            $validation = validateProperty($data);
            if ($validation !== true) {
                $results[] = [
                    'success' => false,
                    'property' => $prop['label'],
                    'message' => $validation
                ];
                continue;
            }

            // Check if exists
            if (propertyExists($data['nombre_propiedad'])) {
                $results[] = [
                    'success' => false,
                    'property' => $prop['label'],
                    'message' => 'Property already exists with name: ' . $data['nombre_propiedad']
                ];
                continue;
            }

            // Build and execute INSERT
            $sql = buildInsertSQL($data);

            try {
                ia_query($sql);

                // Get the inserted property ID
                $sql_id = "SELECT propiedad_id FROM propiedad
                          WHERE nombre_propiedad = " . strit($data['nombre_propiedad']);
                $prop_id = ia_singleton($sql_id);

                $results[] = [
                    'success' => true,
                    'property' => $prop['label'],
                    'message' => 'Successfully created property',
                    'propiedad_id' => $prop_id
                ];
            } catch (Exception $e) {
                $results[] = [
                    'success' => false,
                    'property' => $prop['label'],
                    'message' => 'Database error: ' . $e->getMessage()
                ];
            }
        }
    }

    echo json_encode($results);
    exit;
}

// === HTML UI ===
?>
<!DOCTYPE html>
<html>
<head>
    <title>Add Missing Properties - Iteration 8.5</title>
    <style>
        * {
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            background: #f5f5f5;
        }

        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
        }

        .header h1 {
            margin: 0 0 10px 0;
            font-size: 2em;
        }

        .header .subtitle {
            font-size: 1.2em;
            opacity: 0.9;
        }

        .property-card {
            border: 2px solid #ddd;
            padding: 25px;
            margin: 20px 0;
            border-radius: 10px;
            background: white;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }

        .property-card.enabled {
            border-color: #4caf50;
        }

        .property-card h3 {
            margin: 0 0 20px 0;
            color: #333;
            font-size: 1.5em;
            display: flex;
            align-items: center;
        }

        .property-card h3 input[type="checkbox"] {
            margin-right: 15px;
            width: 20px;
            height: 20px;
            cursor: pointer;
        }

        .impact {
            color: #4caf50;
            font-weight: bold;
            font-size: 0.9em;
            margin-left: auto;
        }

        .anuncios {
            background: #f9f9f9;
            padding: 10px 15px;
            border-left: 4px solid #2196f3;
            margin-bottom: 20px;
            font-size: 0.9em;
        }

        .anuncios strong {
            display: block;
            margin-bottom: 5px;
            color: #2196f3;
        }

        .anuncios ul {
            margin: 5px 0;
            padding-left: 20px;
        }

        .form-row {
            display: grid;
            grid-template-columns: 200px 1fr;
            gap: 15px;
            margin: 15px 0;
            align-items: center;
        }

        .form-row label {
            font-weight: bold;
            color: #555;
        }

        .form-row label.required::after {
            content: " *";
            color: red;
        }

        .form-row input,
        .form-row select {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }

        .form-row input:focus,
        .form-row select:focus {
            outline: none;
            border-color: #4caf50;
        }

        .form-row input[readonly] {
            background: #f5f5f5;
            cursor: not-allowed;
        }

        .form-section {
            margin: 20px 0;
        }

        .form-section h4 {
            color: #666;
            border-bottom: 2px solid #eee;
            padding-bottom: 10px;
            margin: 20px 0 15px 0;
        }

        .warning {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 15px;
            border-radius: 5px;
            margin: 15px 0;
            color: #856404;
        }

        .warning strong {
            display: block;
            margin-bottom: 5px;
        }

        .actions {
            margin: 40px 0;
            text-align: center;
            padding: 30px;
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }

        .actions button {
            padding: 15px 30px;
            margin: 0 10px;
            font-size: 16px;
            cursor: pointer;
            border: none;
            border-radius: 5px;
            transition: all 0.3s;
        }

        .btn-preview {
            background: #2196f3;
            color: white;
        }

        .btn-preview:hover {
            background: #0b7dda;
        }

        .btn-execute {
            background: #4caf50;
            color: white;
            font-weight: bold;
        }

        .btn-execute:hover {
            background: #45a049;
        }

        .btn-execute:disabled {
            background: #ccc;
            cursor: not-allowed;
        }

        #results {
            margin: 30px 0;
            padding: 25px;
            background: #e8f5e9;
            border: 2px solid #4caf50;
            border-radius: 10px;
            display: none;
        }

        #results.error {
            background: #ffebee;
            border-color: #f44336;
        }

        #results h3 {
            margin-top: 0;
            color: #2e7d32;
        }

        #results.error h3 {
            color: #c62828;
        }

        #results ul {
            list-style: none;
            padding: 0;
        }

        #results li {
            padding: 10px;
            margin: 5px 0;
            background: white;
            border-radius: 5px;
            border-left: 4px solid #4caf50;
        }

        #results.error li {
            border-left-color: #f44336;
        }

        .matcher-link {
            display: inline-block;
            margin-top: 20px;
            padding: 12px 25px;
            background: #673ab7;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            font-weight: bold;
        }

        .matcher-link:hover {
            background: #5e35b1;
        }

        .loading {
            text-align: center;
            padding: 20px;
        }

        .loading::after {
            content: '...';
            animation: dots 1.5s steps(4, end) infinite;
        }

        @keyframes dots {
            0%, 20% { content: '.'; }
            40% { content: '..'; }
            60%, 100% { content: '...'; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>Add Missing Properties - Iteration 8.5</h1>
        <div class="subtitle">Expected Impact: +21 matches (97.21% → 98.19%)</div>
    </div>

    <form id="add-properties-form" method="post">
        <input type="hidden" name="action" value="insert">

        <?php foreach ($missing_properties as $key => $prop): ?>
        <div class="property-card" id="card_<?php echo $key; ?>">
            <h3>
                <input type="checkbox"
                       name="add_<?php echo $key; ?>"
                       value="1"
                       id="check_<?php echo $key; ?>"
                       <?php echo ($key !== 'casa_ana') ? 'checked' : ''; ?>
                       onchange="toggleProperty('<?php echo $key; ?>')">
                <span><?php echo $prop['label']; ?></span>
                <span class="impact">+<?php echo $prop['impact']; ?> matches</span>
            </h3>

            <div class="anuncios">
                <strong>Will match these unmatched reservations:</strong>
                <ul>
                    <?php foreach ($prop['anuncios'] as $anuncio): ?>
                    <li><?php echo htmlspecialchars($anuncio); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>

            <?php if ($key === 'casa_ana'): ?>
            <div class="warning">
                <strong>⚠️ Address Information Required</strong>
                Please fill in the address fields below before adding this property.
            </div>
            <?php endif; ?>

            <div class="form-section">
                <h4>Required Fields</h4>

                <div class="form-row">
                    <label class="required">Property Name:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[nombre_propiedad]"
                           value="<?php echo htmlspecialchars($prop['defaults']['nombre_propiedad']); ?>"
                           maxlength="50"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Address:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[direccion]"
                           value="<?php echo htmlspecialchars($prop['defaults']['direccion']); ?>"
                           maxlength="100"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Department:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[departamento]"
                           value="<?php echo htmlspecialchars($prop['defaults']['departamento']); ?>"
                           maxlength="200"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Postal Code:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[codigo_postal]"
                           value="<?php echo htmlspecialchars($prop['defaults']['codigo_postal']); ?>"
                           maxlength="5"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Neighborhood:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[colonia]"
                           value="<?php echo htmlspecialchars($prop['defaults']['colonia']); ?>"
                           maxlength="100"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">State Code:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[estado]"
                           value="<?php echo htmlspecialchars($prop['defaults']['estado']); ?>"
                           maxlength="5"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">State:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[estado_descripcion]"
                           value="<?php echo htmlspecialchars($prop['defaults']['estado_descripcion']); ?>"
                           maxlength="100"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Municipality Code:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[municipio]"
                           value="<?php echo htmlspecialchars($prop['defaults']['municipio']); ?>"
                           maxlength="5"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Municipality:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[municipio_descripcion]"
                           value="<?php echo htmlspecialchars($prop['defaults']['municipio_descripcion']); ?>"
                           maxlength="100"
                           required>
                </div>

                <div class="form-row">
                    <label class="required">Status:</label>
                    <select name="<?php echo $key; ?>[vale]" required>
                        <option value="Active" selected>Active</option>
                        <option value="Inactive">Inactive</option>
                    </select>
                </div>

                <div class="form-row">
                    <label class="required">Number of Units:</label>
                    <input type="number"
                           name="<?php echo $key; ?>[num_deptos]"
                           value="<?php echo $prop['defaults']['num_deptos']; ?>"
                           readonly>
                </div>
            </div>

            <div class="form-section">
                <h4>Optional Fields</h4>

                <div class="form-row">
                    <label>Unit Number:</label>
                    <input type="text"
                           name="<?php echo $key; ?>[numero_unidad]"
                           value="<?php echo htmlspecialchars($prop['defaults']['numero_unidad'] ?? ''); ?>"
                           maxlength="10">
                </div>

                <div class="form-row">
                    <label>Unit Type:</label>
                    <select name="<?php echo $key; ?>[tipo_unidad]">
                        <option value="">-- Select --</option>
                        <?php
                        $tipos = ['Departamento', 'Casa', 'Estudio', 'Penthouse', 'Loft'];
                        $selected_tipo = $prop['defaults']['tipo_unidad'] ?? '';
                        foreach ($tipos as $tipo):
                        ?>
                        <option value="<?php echo $tipo; ?>"
                                <?php echo ($tipo === $selected_tipo) ? 'selected' : ''; ?>>
                            <?php echo $tipo; ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
        </div>
        <?php endforeach; ?>

        <div class="actions">
            <button type="button" class="btn-preview" onclick="previewAllSQL()">
                Preview SQL Statements
            </button>
            <button type="submit" class="btn-execute" id="btn-execute">
                Execute Selected Inserts
            </button>
        </div>
    </form>

    <div id="results"></div>

    <script>
        function toggleProperty(key) {
            const checkbox = document.getElementById('check_' + key);
            const card = document.getElementById('card_' + key);

            if (checkbox.checked) {
                card.classList.add('enabled');
            } else {
                card.classList.remove('enabled');
            }

            updateExecuteButton();
        }

        function updateExecuteButton() {
            const checked = document.querySelectorAll('input[type=checkbox]:checked').length;
            const btn = document.getElementById('btn-execute');
            btn.disabled = (checked === 0);
            btn.textContent = checked === 0 ? 'Select at least one property' :
                              `Execute ${checked} Insert${checked === 1 ? '' : 's'}`;
        }

        function previewAllSQL() {
            const checkboxes = document.querySelectorAll('input[type=checkbox]:checked');
            if (checkboxes.length === 0) {
                alert('Please select at least one property to preview');
                return;
            }

            let previews = [];
            let completed = 0;

            checkboxes.forEach(checkbox => {
                const key = checkbox.name.replace('add_', '');
                const formData = new FormData(document.getElementById('add-properties-form'));

                fetch('add_missing_properties.php?action=preview&property=' + key, {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.sql) {
                        previews.push('-- ' + key.replace(/_/g, ' ').toUpperCase() + '\\n' + data.sql);
                    }
                    completed++;

                    if (completed === checkboxes.length) {
                        alert('SQL PREVIEW:\\n\\n' + previews.join('\\n\\n'));
                    }
                });
            });
        }

        document.getElementById('add-properties-form').addEventListener('submit', function(e) {
            e.preventDefault();

            const checked = document.querySelectorAll('input[type=checkbox]:checked').length;
            if (checked === 0) {
                alert('Please select at least one property to insert');
                return;
            }

            if (!confirm(`Are you sure you want to add ${checked} propert${checked === 1 ? 'y' : 'ies'}?\\n\\nThis will INSERT new records into the database.`)) {
                return;
            }

            const formData = new FormData(this);
            const resultsDiv = document.getElementById('results');

            resultsDiv.innerHTML = '<p class="loading">Inserting properties</p>';
            resultsDiv.style.display = 'block';
            resultsDiv.classList.remove('error');

            fetch('add_missing_properties.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                let html = '<h3>Insert Results:</h3><ul>';
                let hasSuccess = false;
                let hasError = false;

                data.forEach(result => {
                    const status = result.success ? '✅' : '❌';
                    html += `<li>${status} <strong>${result.property}:</strong> ${result.message}`;
                    if (result.propiedad_id) {
                        html += ` (ID: ${result.propiedad_id})`;
                    }
                    html += '</li>';

                    if (result.success) hasSuccess = true;
                    if (!result.success) hasError = true;
                });

                html += '</ul>';

                if (hasSuccess) {
                    html += '<p><a href="iteration8_matcher.php" class="matcher-link">🔄 Re-run Matcher to Verify New Matches</a></p>';
                }

                resultsDiv.innerHTML = html;
                if (hasError && !hasSuccess) {
                    resultsDiv.classList.add('error');
                }
            })
            .catch(error => {
                resultsDiv.innerHTML = '<h3>Error</h3><p>An error occurred: ' + error.message + '</p>';
                resultsDiv.classList.add('error');
            });
        });

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            <?php foreach ($missing_properties as $key => $prop): ?>
            toggleProperty('<?php echo $key; ?>');
            <?php endforeach; ?>
            updateExecuteButton();
        });
    </script>
</body>
</html>
