<?php
/**
 * CFDI Matcher API - AJAX Backend
 *
 * Provides JSON endpoints for the HERMES ALGORITHM UI
 *
 * Actions:
 * - get_matches: Fetch matched invoice-deposit pairs
 * - get_unmatched: Fetch unmatched invoices
 * - get_iterations: Fetch iteration history
 * - get_analytics: Fetch analytics data
 * - bulk_approve: Approve multiple matches
 * - export_matches: Export matches to CSV
 */

// Enable error reporting for debugging
if (file_exists('/wamp/www/showErrors.vin')) {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}

// === STANDALONE BOOTSTRAP - NO VITEX.PHP DEPENDENCY ===
// Initialize globals BEFORE loading ia_utilerias.php
global $gIAParametros, $gAppRelate, $gIAsql, $gIAsql_link;

// Database configuration (must be set before requiring ia_utilerias.php)
$gIAsql = [
    'host' => 'localhost',
    'port' => '3306',
    'user' => 'root',
    'pwd' => 'M@chiavell1',
    'dbname' => 'quantix',
    'socket' => '/lamp/mysql/mysql.sock',
    'set_autocommit' => null,
    'pconnect' => false,
    'trace' => false,
    'sql_trace' => [],
    'link' => null,
    'err' => ''
];

$gIAParametros = [];
$gAppRelate = null;

// Now load utilities
require_once(__DIR__ . "/../../inc/ia_utilerias.php");

// Establish database connection
$gIAsql_link = mysqli_connect(
    $gIAsql['host'],
    $gIAsql['user'],
    $gIAsql['pwd'],
    $gIAsql['dbname'],
    $gIAsql['port'],
    $gIAsql['socket']
);

if (!$gIAsql_link) {
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

mysqli_set_charset($gIAsql_link, 'utf8mb4');

// Session management
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Define missing constants
if (!defined('IACHARSET')) define('IACHARSET', 'UTF-8');

header('Content-Type: application/json');

// Permission check
if (!isset($_SESSION['usuario_id'])) {
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {

    case 'get_matches':
        get_matches();
        break;

    case 'get_unmatched':
        get_unmatched();
        break;

    case 'get_iterations':
        get_iterations();
        break;

    case 'get_analytics':
        get_analytics();
        break;

    case 'bulk_approve':
        bulk_approve();
        break;

    case 'export_matches':
        export_matches();
        break;

    case 'preview_matches':
        preview_matches();
        break;

    case 'run_iteration':
        run_iteration();
        break;

    case 'manual_match':
        manual_match();
        break;

    case 'reject_match':
        reject_match();
        break;

    default:
        echo json_encode(['error' => 'Invalid action']);
        break;
}

/**
 * Get all matched invoice-deposit pairs from latest iteration
 */
function get_matches() {
    global $gIAsql_link;

    // Get latest iteration ID
    $latest_iter_sql = "SELECT iteration_id, iteration_number
                        FROM cfdi_matcher_iterations
                        ORDER BY iteration_id DESC LIMIT 1";
    $latest_iter = ia_sqlArrayIndx($latest_iter_sql)[0] ?? null;

    if (!$latest_iter) {
        echo json_encode(['matches' => [], 'message' => 'No iterations found']);
        return;
    }

    $iteration_id = $latest_iter['iteration_id'];

    // Fetch all matches with invoice and deposit details
    $sql = "SELECT
        r.result_id,
        r.invoice_id,
        r.deposit_id,
        r.match_tier as tier,
        r.match_confidence as confidence,
        r.match_pattern as pattern,
        r.days_between_invoice_deposit as days_between,
        r.amount_difference_percent as amount_diff_pct,
        r.invoice_amount,
        r.deposit_amount,

        -- Invoice details (FIXED: correct column names)
        i.Fecha_Emision as invoice_date,
        i.Total as invoice_total,
        i.Nombre_Receptor as client_name,
        i.RFC_Receptor as client_rfc,
        i.Estado_de_Cuenta as estado,

        -- Deposit details
        d.fecha as deposit_date,
        d.deposit as deposit_total,
        d.banco_cuenta_id,
        CASE
            WHEN d.banco_cuenta_id = 2 THEN 'BBVA'
            WHEN d.banco_cuenta_id = 3 THEN 'SANTANDER'
            ELSE 'Other'
        END as bank_name,

        -- Tier naming
        CASE
            WHEN r.match_tier = 0 THEN 'Tier 0 (Exact)'
            WHEN r.match_tier = 0.5 THEN 'Tier 0.5 (Estado)'
            WHEN r.match_tier = 1 THEN 'Tier 1 (Week)'
            WHEN r.match_tier = 2 THEN 'Tier 2 (Month)'
            WHEN r.match_tier = 3 THEN 'Tier 3 (Quarter)'
            ELSE 'Other'
        END as tier_name

    FROM cfdi_matcher_results r
    LEFT JOIN eleyeme_cfdi_emitidos i ON r.invoice_id COLLATE utf8mb4_unicode_ci = i.eleyeme_cfdi_emitido_id COLLATE utf8mb4_unicode_ci
    LEFT JOIN banco_cuenta_mov d ON r.deposit_id COLLATE utf8mb4_unicode_ci = d.banco_cuenta_mov_id COLLATE utf8mb4_unicode_ci
    WHERE r.iteration_id = ?
    ORDER BY r.match_confidence DESC, r.match_tier ASC";

    $stmt = $gIAsql_link->prepare($sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $matches = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    echo json_encode([
        'matches' => $matches,
        'iteration_number' => $latest_iter['iteration_number'],
        'count' => count($matches)
    ]);
}

/**
 * Get unmatched invoices from latest iteration
 */
function get_unmatched() {
    global $gIAsql_link;

    // Get latest iteration ID
    $latest_iter_sql = "SELECT iteration_id FROM cfdi_matcher_iterations
                        ORDER BY iteration_id DESC LIMIT 1";
    $latest_iter = ia_sqlArrayIndx($latest_iter_sql)[0] ?? null;

    if (!$latest_iter) {
        echo json_encode(['invoices' => []]);
        return;
    }

    $iteration_id = $latest_iter['iteration_id'];

    // Fetch unmatched invoices (FIXED: correct column names)
    $sql = "SELECT
        i.eleyeme_cfdi_emitido_id as invoice_id,
        i.Fecha_Emision as invoice_date,
        i.Total as amount,
        i.Nombre_Receptor as client_name,
        i.RFC_Receptor as client_rfc,
        i.Estado_de_Cuenta as estado,
        CONCAT(COALESCE(i.Serie, ''), '-', COALESCE(i.Folio, '')) as folio,

        -- Check if Estado exists
        CASE
            WHEN i.Estado_de_Cuenta IS NOT NULL AND i.Estado_de_Cuenta != '' THEN 1
            ELSE 0
        END as has_estado

    FROM eleyeme_cfdi_emitidos i
    WHERE i.Fecha_Emision >= '2024-01-01'
    AND i.eleyeme_cfdi_emitido_id NOT IN (
        SELECT invoice_id FROM cfdi_matcher_results
        WHERE iteration_id = ?
    )
    ORDER BY i.Fecha_Emision DESC";

    $stmt = $gIAsql_link->prepare($sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $invoices = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    echo json_encode([
        'invoices' => $invoices,
        'count' => count($invoices),
        'with_estado' => count(array_filter($invoices, fn($inv) => $inv['has_estado'] == 1)),
        'without_estado' => count(array_filter($invoices, fn($inv) => $inv['has_estado'] == 0))
    ]);
}

/**
 * Get iteration history with comparison metrics
 */
function get_iterations() {
    $sql = "SELECT
        iteration_id,
        iteration_number,
        test_date as run_timestamp,
        matched_count,
        match_rate_percent,
        tier0_count,
        tier0_5_count,
        tier1_count,
        tier2_count,
        tier3_count,
        high_confidence_count,
        avg_confidence,
        total_matched_amount,
        reconciliation_gap

    FROM cfdi_matcher_iterations
    ORDER BY iteration_id DESC
    LIMIT 20";

    $iterations = ia_sqlArrayIndx($sql);

    // Calculate improvements between iterations
    for ($i = 0; $i < count($iterations) - 1; $i++) {
        $current = $iterations[$i];
        $previous = $iterations[$i + 1];

        $iterations[$i]['match_rate_improvement'] = round(
            $current['match_rate_percent'] - $previous['match_rate_percent'],
            2
        );

        $iterations[$i]['amount_improvement'] = round(
            (($current['total_matched_amount'] - $previous['total_matched_amount']) /
            $previous['total_matched_amount']) * 100,
            1
        );
    }

    echo json_encode([
        'iterations' => $iterations,
        'count' => count($iterations)
    ]);
}

/**
 * Get analytics data for charts
 */
function get_analytics() {
    global $gIAsql_link;

    // Get latest iteration
    $latest_iter_sql = "SELECT iteration_id FROM cfdi_matcher_iterations
                        ORDER BY iteration_id DESC LIMIT 1";
    $latest_iter = ia_sqlArrayIndx($latest_iter_sql)[0] ?? null;

    if (!$latest_iter) {
        echo json_encode(['error' => 'No data']);
        return;
    }

    $iteration_id = $latest_iter['iteration_id'];

    // Confidence distribution (histogram buckets)
    $confidence_dist_sql = "SELECT
        CASE
            WHEN match_confidence >= 95 THEN '95-100%'
            WHEN match_confidence >= 90 THEN '90-94%'
            WHEN match_confidence >= 80 THEN '80-89%'
            WHEN match_confidence >= 70 THEN '70-79%'
            WHEN match_confidence >= 60 THEN '60-69%'
            ELSE '<60%'
        END as bucket,
        COUNT(*) as count
    FROM cfdi_matcher_results
    WHERE iteration_id = ?
    GROUP BY bucket
    ORDER BY bucket DESC";

    $stmt = $gIAsql_link->prepare($confidence_dist_sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $confidence_dist = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Payment delay analysis (scatter plot data)
    $delay_analysis_sql = "SELECT
        days_between_invoice_deposit as delay_days,
        invoice_amount as amount,
        match_tier as tier,
        match_confidence as confidence
    FROM cfdi_matcher_results
    WHERE iteration_id = ?
    AND days_between_invoice_deposit IS NOT NULL
    ORDER BY delay_days ASC
    LIMIT 500";

    $stmt = $gIAsql_link->prepare($delay_analysis_sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $delay_data = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Top clients by match count
    $top_clients_sql = "SELECT
        i.Receptor as client_name,
        COUNT(*) as match_count,
        SUM(r.invoice_amount) as total_amount,
        AVG(r.match_confidence) as avg_confidence
    FROM cfdi_matcher_results r
    LEFT JOIN eleyeme_cfdi_emitidos i ON r.invoice_id = i.eleyeme_cfdi_emitidos_id
    WHERE r.iteration_id = ?
    GROUP BY i.Receptor
    ORDER BY match_count DESC
    LIMIT 10";

    $stmt = $gIAsql_link->prepare($top_clients_sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $top_clients = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Amount reconciliation breakdown
    $amount_breakdown_sql = "SELECT
        SUM(invoice_amount) as total_matched,
        (SELECT SUM(Total) FROM eleyeme_cfdi_emitidos
         WHERE Fecha_Emision >= '2025-01-01') as total_invoices,
        SUM(CASE WHEN match_tier = 0.5 THEN invoice_amount ELSE 0 END) as estado_amount,
        SUM(CASE WHEN match_tier = 1 THEN invoice_amount ELSE 0 END) as tier1_amount,
        SUM(CASE WHEN match_tier >= 2 THEN invoice_amount ELSE 0 END) as lower_tier_amount
    FROM cfdi_matcher_results
    WHERE iteration_id = ?";

    $stmt = $gIAsql_link->prepare($amount_breakdown_sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $amount_breakdown = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    echo json_encode([
        'confidence_distribution' => $confidence_dist,
        'payment_delays' => $delay_data,
        'top_clients' => $top_clients,
        'amount_breakdown' => $amount_breakdown
    ]);
}

/**
 * Bulk approve selected matches (future functionality)
 */
function bulk_approve() {
    $match_ids = $_POST['match_ids'] ?? [];

    if (empty($match_ids)) {
        echo json_encode(['error' => 'No matches selected']);
        return;
    }

    // Future: Update invoice records with matched deposit_id
    // For now, just return success

    echo json_encode([
        'success' => true,
        'approved_count' => count($match_ids),
        'message' => 'Bulk approval feature coming soon'
    ]);
}

/**
 * Export matches to CSV
 */
function export_matches() {
    global $gIAsql_link;

    // Get latest iteration
    $latest_iter_sql = "SELECT iteration_id FROM cfdi_matcher_iterations
                        ORDER BY iteration_id DESC LIMIT 1";
    $latest_iter = ia_sqlArrayIndx($latest_iter_sql)[0] ?? null;

    if (!$latest_iter) {
        echo json_encode(['error' => 'No data to export']);
        return;
    }

    $iteration_id = $latest_iter['iteration_id'];

    // Fetch all matches
    $sql = "SELECT
        r.invoice_id,
        r.deposit_id,
        r.match_tier,
        r.match_confidence,
        r.match_pattern,
        r.days_between_invoice_deposit,
        r.amount_difference_percent,
        r.invoice_amount,
        r.deposit_amount,
        i.Fecha_Emision,
        i.Receptor,
        i.Receptor_RFC,
        d.fecha as deposit_date,
        d.banco_cuenta_id
    FROM cfdi_matcher_results r
    LEFT JOIN eleyeme_cfdi_emitidos i ON r.invoice_id = i.eleyeme_cfdi_emitidos_id
    LEFT JOIN banco_cuenta_mov d ON r.deposit_id = d.banco_cuenta_mov_id
    WHERE r.iteration_id = ?
    ORDER BY r.match_confidence DESC";

    $stmt = $gIAsql_link->prepare($sql);
    $stmt->bind_param('i', $iteration_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $matches = $result->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Generate CSV filename
    $filename = 'cfdi_matches_' . date('Y-m-d_His') . '.csv';
    $filepath = '/tmp/' . $filename;

    // Write CSV
    $fp = fopen($filepath, 'w');

    // Header row
    fputcsv($fp, [
        'Invoice ID',
        'Deposit ID',
        'Tier',
        'Confidence',
        'Pattern',
        'Days Between',
        'Amount Diff %',
        'Invoice Amount',
        'Deposit Amount',
        'Invoice Date',
        'Client',
        'RFC',
        'Deposit Date',
        'Bank Account'
    ]);

    // Data rows
    foreach ($matches as $match) {
        fputcsv($fp, [
            $match['invoice_id'],
            $match['deposit_id'],
            $match['match_tier'],
            $match['match_confidence'],
            $match['match_pattern'],
            $match['days_between_invoice_deposit'],
            $match['amount_difference_percent'],
            $match['invoice_amount'],
            $match['deposit_amount'],
            $match['Fecha_Emision'],
            $match['Receptor'],
            $match['Receptor_RFC'],
            $match['deposit_date'],
            $match['banco_cuenta_id']
        ]);
    }

    fclose($fp);

    echo json_encode([
        'success' => true,
        'filename' => $filename,
        'filepath' => $filepath,
        'download_url' => 'cfdi_matcher_download.php?file=' . urlencode($filename),
        'row_count' => count($matches)
    ]);
}

/**
 * Preview matches without saving to database
 * Runs matcher in memory and returns proposed matches
 */
function preview_matches() {
    global $gIAsql_link;

    // Get parameters
    $date_start = $_GET['date_start'] ?? '2024-01-01';
    $date_end = $_GET['date_end'] ?? null;
    $min_confidence = intval($_GET['min_confidence'] ?? 40);
    $amount_tolerance = floatval($_GET['amount_tolerance'] ?? 2.0);

    // Load matching library
    require_once(__DIR__ . '/cfdi_matcher_lib.php');

    // Load invoices
    $sql_invoices = "SELECT * FROM eleyeme_cfdi_emitidos
                     WHERE Fecha_Emision >= ? ";
    if ($date_end) {
        $sql_invoices .= " AND Fecha_Emision <= ? ";
    }
    $sql_invoices .= " ORDER BY Fecha_Emision DESC LIMIT 255";

    $stmt = $gIAsql_link->prepare($sql_invoices);
    if ($date_end) {
        $stmt->bind_param('ss', $date_start, $date_end);
    } else {
        $stmt->bind_param('s', $date_start);
    }
    $stmt->execute();
    $invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Load deposits
    $sql_deposits = "SELECT * FROM banco_cuenta_mov
                     WHERE deposit > 0
                     AND fecha >= ? ";
    if ($date_end) {
        $sql_deposits .= " AND fecha <= ? ";
    }
    $sql_deposits .= " ORDER BY fecha DESC";

    $stmt = $gIAsql_link->prepare($sql_deposits);
    if ($date_end) {
        $stmt->bind_param('ss', $date_start, $date_end);
    } else {
        $stmt->bind_param('s', $date_start);
    }
    $stmt->execute();
    $deposits = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Run matching algorithm (in memory, don't save)
    $matches = [];
    $matched_deposit_ids = [];

    foreach ($invoices as $invoice) {
        $best_match = null;
        $best_confidence = 0;
        $best_deposit = null;

        // Try Tier 0.5 first (Estado-guided)
        $result = match_invoice_to_all_deposits($invoice, $deposits);

        if ($result['match'] && $result['confidence'] >= $min_confidence) {
            $best_match = $result;
            $best_confidence = $result['confidence'];
            $best_deposit = $result['matched_deposit'];
        } else {
            // Try traditional matching
            foreach ($deposits as $deposit) {
                if (isset($matched_deposit_ids[$deposit['banco_cuenta_mov_id']])) {
                    continue;
                }

                $result = match_invoice_to_deposit($invoice, $deposit);

                if ($result['match'] && $result['confidence'] > $best_confidence && $result['confidence'] >= $min_confidence) {
                    $best_match = $result;
                    $best_confidence = $result['confidence'];
                    $best_deposit = $deposit;
                }
            }
        }

        // Store match if found
        if ($best_match && $best_confidence >= $min_confidence) {
            $matches[] = [
                'invoice_id' => $invoice['eleyeme_cfdi_emitido_id'],
                'deposit_id' => $best_deposit['banco_cuenta_mov_id'],
                'invoice_client' => $invoice['Nombre_Receptor'],
                'invoice_amount' => $invoice['Total'],
                'invoice_date' => $invoice['Fecha_Emision'],
                'deposit_amount' => $best_deposit['deposit'],
                'deposit_date' => $best_deposit['fecha'],
                'deposit_bank' => $best_deposit['banco_cuenta_id'],
                'tier' => $best_match['tier'],
                'confidence' => $best_match['confidence'],
                'explanation' => $best_match['pattern'] ?? 'Matched based on amount and date proximity',
            ];

            $matched_deposit_ids[$best_deposit['banco_cuenta_mov_id']] = true;
        }
    }

    // Calculate stats
    $high_conf = count(array_filter($matches, fn($m) => $m['confidence'] >= 80));
    $med_conf = count(array_filter($matches, fn($m) => $m['confidence'] >= 50 && $m['confidence'] < 80));
    $low_conf = count(array_filter($matches, fn($m) => $m['confidence'] < 50));

    echo json_encode([
        'matches' => array_slice($matches, 0, 50), // Limit to first 50 for preview
        'total_matches' => count($matches),
        'high_confidence' => $high_conf,
        'medium_confidence' => $med_conf,
        'low_confidence' => $low_conf,
        'total_invoices' => count($invoices),
        'total_deposits' => count($deposits)
    ]);
}

/**
 * Run iteration and save approved matches to database
 */
function run_iteration() {
    global $gIAsql_link;

    // Start output buffering to catch any stray output
    ob_start();

    try {
        // Get parameters
        $config = json_decode($_POST['config'], true);
        $approved_matches = $config['approved_matches'] ?? [];

        if (empty($approved_matches)) {
            ob_clean();
            echo json_encode(['error' => 'No approved matches']);
            return;
        }

        // Load matching library
        require_once(__DIR__ . '/cfdi_matcher_lib.php');

        // Parse approved match IDs (format: "invoice_id_deposit_id")
        $approved_pairs = [];
        foreach ($approved_matches as $matchId) {
            $parts = explode('_', $matchId, 2); // Split only on first underscore
            if (count($parts) >= 2) {
                $approved_pairs[] = [
                    'invoice_id' => $parts[0],
                    'deposit_id' => $parts[1]
                ];
            }
        }

        // Get next iteration number
        $iteration_number = get_next_cfdi_iteration_number();

        // Create iteration record
        $iteration_id = create_cfdi_iteration($iteration_number, 'both', $config['date_start'], $config['date_end'] ?? null);

        // Save only approved matches AND create/manage links
        $saved_count = 0;
        $total_invoice_amount = 0;
        $total_deposit_amount = 0;
        $links_created = 0;
        $links_updated = 0;
        $links_superseded = 0;

        foreach ($approved_pairs as $pair) {
            // Fetch full invoice and deposit data
            $invoice = ia_sqlArrayIndx("SELECT * FROM eleyeme_cfdi_emitidos WHERE eleyeme_cfdi_emitido_id = '{$pair['invoice_id']}'")[0] ?? null;
            $deposit = ia_sqlArrayIndx("SELECT * FROM banco_cuenta_mov WHERE banco_cuenta_mov_id = '{$pair['deposit_id']}'")[0] ?? null;

            if ($invoice && $deposit) {
                // Re-run matching to get match details
                $match_result = match_invoice_to_deposit($invoice, $deposit);

                if ($match_result['match']) {
                    // 1. Log match result to cfdi_matcher_results (history)
                    log_cfdi_match_result($iteration_id, $invoice, $deposit, $match_result);
                    $saved_count++;
                    $total_invoice_amount += $invoice['Total'];
                    $total_deposit_amount += $deposit['deposit'];

                    // 2. Manage production link (banco_cuenta_mov_link + cfdi_matcher_links)
                    $existing_link = get_active_invoice_link($invoice['eleyeme_cfdi_emitido_id']);

                    if ($existing_link) {
                        // Invoice already has an active link
                        if ($existing_link['deposit_id'] === $deposit['banco_cuenta_mov_id']) {
                            // Same deposit - just confirm the link
                            update_cfdi_link($existing_link['cfdi_matcher_link_id'], $iteration_id);
                            $links_updated++;
                        } else {
                            // Different deposit - supersede old link with new one
                            supersede_cfdi_link(
                                $existing_link['cfdi_matcher_link_id'],
                                $iteration_id,
                                $iteration_number,
                                $invoice,
                                $deposit,
                                $match_result
                            );
                            $links_superseded++;
                        }
                    } else {
                        // No existing link - create new one
                        create_cfdi_link($iteration_id, $iteration_number, $invoice, $deposit, $match_result);
                        $links_created++;
                    }
                }
            }
        }

        // 3. Handle invoices that HAD links but are NOT in this iteration's matches (delete links)
        $links_deleted = 0;
        $sql_previously_linked = "SELECT invoice_id, cfdi_matcher_link_id
                                   FROM cfdi_matcher_links
                                   WHERE link_status = 'active'
                                     AND invoice_id NOT IN (
                                         SELECT invoice_id FROM cfdi_matcher_results WHERE iteration_id = $iteration_id
                                     )";
        $previously_linked = ia_sqlArrayIndx($sql_previously_linked);

        foreach ($previously_linked as $old_link) {
            delete_cfdi_link($old_link['cfdi_matcher_link_id'], $iteration_id);
            $links_deleted++;
        }

        // Calculate match rate
        $total_invoices_sql = "SELECT COUNT(*) FROM eleyeme_cfdi_emitidos WHERE Fecha_Emision >= '{$config['date_start']}'";
        $total_invoices = ia_singleton($total_invoices_sql);
        $match_rate = round(($saved_count / $total_invoices) * 100, 2);

        // Update iteration stats
        update_cfdi_iteration_stats($iteration_id, [
            'matched_count' => $saved_count,
            'match_rate_percent' => $match_rate,
            'total_invoices' => $total_invoices,
            'matched_invoice_amount' => $total_invoice_amount,
            'matched_deposit_amount' => $total_deposit_amount,
            'execution_time_seconds' => 1
        ]);

        // Clean output buffer before JSON
        ob_clean();

        echo json_encode([
            'success' => true,
            'iteration_id' => $iteration_id,
            'iteration_number' => $iteration_number,
            'matched_count' => $saved_count,
            'match_rate' => $match_rate,
            'links' => [
                'created' => $links_created,
                'updated' => $links_updated,
                'superseded' => $links_superseded,
                'deleted' => $links_deleted,
                'total' => $links_created + $links_updated
            ]
        ]);

    } catch (Exception $e) {
        // Clean buffer and return error
        ob_clean();
        echo json_encode([
            'error' => 'Iteration failed: ' . $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    } finally {
        ob_end_flush();
    }
}
