<?php
/**
 * CFDI Matcher CLI - Invoice-Payment Reconciliation
 *
 * Purpose: Match invoices (eleyeme_cfdi_emitidos) with bank deposits (banco_cuenta_mov)
 * Author: Claude Code (Invoice-Payment Matcher)
 * Date: 2026-01-14
 *
 * Usage:
 *   php cfdi_matcher_lamp.php
 *   Or via CLI: /lamp/php/bin/php /lamp/www/quantix/backoffice/helper/cfdi_matcher_lamp.php
 *
 * Based on PMS Matcher architecture
 */

if (!defined('MYSQLI_ASSOC')) define('MYSQLI_ASSOC', 1);

// Determine path based on execution context
if (php_sapi_name() === 'cli') {
    // CLI execution - use minimal bootstrap to avoid vitex.php dependency
    require_once(__DIR__ . "/cfdi_matcher_bootstrap.php");
} else {
    // Web execution
    require_once("../../inc/config.php");
    require_once("cfdi_matcher_lib.php");
}

echo "═══════════════════════════════════════════════════════════════\n";
echo "  CFDI MATCHER - Invoice-Payment Reconciliation\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

// ============================================================================
// CONFIGURATION
// ============================================================================

$HIGH_CONFIDENCE_THRESHOLD = 80;
$DATE_FILTER_START = '2024-01-01'; // Only match invoices/deposits from 2024 onwards
$DATE_FILTER_END = null; // null = no end date filter

// ============================================================================
// STEP 1: Load Data
// ============================================================================

echo "[1/5] Loading invoices and deposits...\n";

// Load invoices
$sql_invoices = "SELECT * FROM eleyeme_cfdi_emitidos
                 WHERE Fecha_Emision >= '{$DATE_FILTER_START}'";
if ($DATE_FILTER_END) {
    $sql_invoices .= " AND Fecha_Emision <= '{$DATE_FILTER_END}'";
}
$sql_invoices .= " ORDER BY Fecha_Emision DESC";

$invoices = ia_sqlArrayIndx($sql_invoices);
echo "  ✓ Loaded " . count($invoices) . " invoices\n";

// Load deposits
$sql_deposits = "SELECT * FROM banco_cuenta_mov
                 WHERE deposit > 0
                 AND fecha >= '{$DATE_FILTER_START}'";
if ($DATE_FILTER_END) {
    $sql_deposits .= " AND fecha <= '{$DATE_FILTER_END}'";
}
$sql_deposits .= " ORDER BY fecha DESC";

$deposits = ia_sqlArrayIndx($sql_deposits);
echo "  ✓ Loaded " . count($deposits) . " deposits\n\n";

// ============================================================================
// STEP 2: Initialize Statistics
// ============================================================================

$stats = [
    'total_invoices' => count($invoices),
    'total_deposits' => count($deposits),
    'matched_count' => 0,
    'unmatched_invoices' => 0,
    'unmatched_deposits' => 0,
    'tier0_count' => 0,
    'tier0_5_count' => 0,  // NEW: Estado-guided matches
    'tier1_count' => 0,
    'tier2_count' => 0,
    'tier3_count' => 0,
    'tier4_count' => 0,
    'high_confidence_count' => 0,
    'medium_confidence_count' => 0,
    'low_confidence_count' => 0,
    'total_invoice_amount' => 0,
    'total_deposit_amount' => 0,
    'matched_invoice_amount' => 0,
    'matched_deposit_amount' => 0,
];

// Calculate totals
foreach ($invoices as $inv) {
    $stats['total_invoice_amount'] += $inv['Total'];
}
foreach ($deposits as $dep) {
    $stats['total_deposit_amount'] += $dep['deposit'];
}

// ============================================================================
// STEP 3: Matching Phase
// ============================================================================

echo "[2/5] Matching invoices to deposits...\n";

$matches = []; // Store successful matches
$matched_deposit_ids = []; // Track which deposits have been matched
$matched_invoice_ids = []; // Track which invoices have been matched

$start_time = time();

foreach ($invoices as $idx => $invoice) {
    // Progress indicator (every 10%)
    if ($idx % max(1, floor(count($invoices) / 10)) === 0) {
        $pct = round(($idx / count($invoices)) * 100);
        echo "  Progress: {$pct}% ({$idx}/" . count($invoices) . ")\n";
    }

    $best_match = null;
    $best_confidence = 0;
    $best_deposit = null;

    // NEW in Iteration 2: Try Tier 0.5 first (Estado-guided)
    // This function handles all deposits internally and returns best match
    $result = match_invoice_to_all_deposits($invoice, $deposits);

    if ($result['match'] && isset($result['matched_deposit'])) {
        // Tier 0.5 found a match!
        $best_match = $result;
        $best_confidence = $result['confidence'];
        $best_deposit = $result['matched_deposit'];
    } else {
        // Tier 0.5 didn't match, try traditional matching
        // Try to match against all deposits one by one
        foreach ($deposits as $deposit) {
            // Skip already matched deposits
            if (isset($matched_deposit_ids[$deposit['banco_cuenta_mov_id']])) {
                continue;
            }

            $result = match_invoice_to_deposit($invoice, $deposit);

            if ($result['match'] && $result['confidence'] > $best_confidence) {
                $best_match = $result;
                $best_confidence = $result['confidence'];
                $best_deposit = $deposit;

                // If we got a perfect match (100%), stop searching
                if ($best_confidence >= 100) {
                    break;
                }
            }
        }
    }

    // Store the match if found
    if ($best_match && $best_confidence >= 40) { // Minimum threshold
        $matches[] = [
            'invoice' => $invoice,
            'deposit' => $best_deposit,
            'match' => $best_match,
        ];

        // Mark as matched
        $matched_invoice_ids[$invoice['eleyeme_cfdi_emitido_id']] = true;
        $matched_deposit_ids[$best_deposit['banco_cuenta_mov_id']] = true;

        // Update stats
        $stats['matched_count']++;
        $tier = $best_match['tier'];

        // Handle decimal tier (0.5)
        if ($tier == 0.5) {
            $stats['tier0_5_count']++;
        } else {
            $stats["tier{$tier}_count"]++;
        }

        if ($best_confidence >= 80) {
            $stats['high_confidence_count']++;
        } elseif ($best_confidence >= 50) {
            $stats['medium_confidence_count']++;
        } else {
            $stats['low_confidence_count']++;
        }

        $stats['matched_invoice_amount'] += $invoice['Total'];
        $stats['matched_deposit_amount'] += $best_deposit['deposit'];
    }
}

$execution_time = time() - $start_time;

$stats['unmatched_invoices'] = $stats['total_invoices'] - $stats['matched_count'];
$stats['unmatched_deposits'] = $stats['total_deposits'] - $stats['matched_count'];
$stats['match_rate_percent'] = $stats['total_invoices'] > 0
    ? round(($stats['matched_count'] / $stats['total_invoices']) * 100, 2)
    : 0;
$stats['avg_confidence'] = $stats['matched_count'] > 0
    ? round(array_sum(array_column(array_column($matches, 'match'), 'confidence')) / $stats['matched_count'], 2)
    : 0;
$stats['reconciliation_gap'] = $stats['matched_deposit_amount'] - $stats['matched_invoice_amount'];

echo "  ✓ Matched " . count($matches) . " invoice-deposit pairs\n\n";

// ============================================================================
// STEP 4: Display Statistics
// ============================================================================

echo "[3/7] Match Statistics\n";
echo "───────────────────────────────────────────────────────────────\n";

$match_rate = $stats['match_rate_percent'];
echo "INVOICES: {$stats['matched_count']}/{$stats['total_invoices']} ({$match_rate}%)\n";
echo "  Tier 0 (Exact): {$stats['tier0_count']}\n";
echo "  Tier 0.5 (Estado-Guided): {$stats['tier0_5_count']} ⭐ NEW!\n";
echo "  Tier 1 (Strong): {$stats['tier1_count']}\n";
echo "  Tier 2 (Probable): {$stats['tier2_count']}\n";
echo "  Tier 3 (Possible): {$stats['tier3_count']}\n";
echo "  Tier 4 (Weak): {$stats['tier4_count']}\n\n";

echo "CONFIDENCE DISTRIBUTION:\n";
echo "  High (≥80%): {$stats['high_confidence_count']}\n";
echo "  Medium (50-79%): {$stats['medium_confidence_count']}\n";
echo "  Low (<50%): {$stats['low_confidence_count']}\n\n";

echo "FINANCIAL RECONCILIATION:\n";
echo "  Total Invoices: $" . number_format($stats['total_invoice_amount'], 2) . "\n";
echo "  Total Deposits: $" . number_format($stats['total_deposit_amount'], 2) . "\n";
echo "  Matched Invoices: $" . number_format($stats['matched_invoice_amount'], 2) . "\n";
echo "  Matched Deposits: $" . number_format($stats['matched_deposit_amount'], 2) . "\n";
echo "  Reconciliation Gap: $" . number_format($stats['reconciliation_gap'], 2) . "\n\n";

echo "UNMATCHED:\n";
echo "  Invoices: {$stats['unmatched_invoices']}\n";
echo "  Deposits: {$stats['unmatched_deposits']}\n\n";

echo "Average Confidence: {$stats['avg_confidence']}%\n";
echo "Execution Time: {$execution_time} seconds\n";
echo "───────────────────────────────────────────────────────────────\n\n";

// ============================================================================
// STEP 5: Save Results to Database
// ============================================================================

echo "[5/7] Saving results to database...\n";

global $gIAsql_link;

// Test database connection
$test_result = ia_singleton("SELECT 1");
if ($test_result === null) {
    echo "  ✗ Database connection lost!\n";
    exit(1);
}
echo "  ✓ Database connection OK\n";

// Determine iteration number
echo "  - Getting next iteration number...\n";
$iter_number_sql = "SELECT COALESCE(MAX(iteration_number), 0) + 1 as next_iteration
                    FROM cfdi_matcher_iterations";
$iter_result = ia_sqlArrayIndx($iter_number_sql);
$iteration_number = $iter_result[0]['next_iteration'] ?? 1;
echo "  - Next iteration: #{$iteration_number}\n";

// Insert iteration record using ia_insert
$iteration_data = [
    'iteration_number' => $iteration_number,
    'total_invoices' => $stats['total_invoices'],
    'total_deposits' => $stats['total_deposits'],
    'matched_count' => $stats['matched_count'],
    'unmatched_invoices' => $stats['unmatched_invoices'],
    'unmatched_deposits' => $stats['unmatched_deposits'],
    'match_rate_percent' => $stats['match_rate_percent'],
    'tier0_count' => $stats['tier0_count'],
    'tier0_5_count' => $stats['tier0_5_count'],
    'tier1_count' => $stats['tier1_count'],
    'tier2_count' => $stats['tier2_count'],
    'tier3_count' => $stats['tier3_count'],
    'tier4_count' => $stats['tier4_count'],
    'high_confidence_count' => $stats['high_confidence_count'],
    'avg_confidence' => $stats['avg_confidence'],
    'total_invoice_amount' => $stats['total_invoice_amount'],
    'total_deposit_amount' => $stats['total_deposit_amount'],
    'matched_invoice_amount' => $stats['matched_invoice_amount'],
    'matched_deposit_amount' => $stats['matched_deposit_amount'],
    'reconciliation_gap' => $stats['reconciliation_gap'],
    'execution_time_seconds' => $execution_time,
    'created_by' => 'cfdi_matcher_v2'
];

echo "  - Preparing iteration data...\n";

// DEBUG: Print all stats values
echo "  - Debug: total_invoices={$stats['total_invoices']}, matched={$stats['matched_count']}\n";
echo "  - Debug: tier0_5_count={$stats['tier0_5_count']}, high_conf={$stats['high_confidence_count']}\n";

echo "  - Inserting into database...\n";
flush();

// Use raw SQL instead of ia_insert to avoid timeout issues
$sql = "INSERT INTO cfdi_matcher_iterations (
    iteration_number, total_invoices, total_deposits, matched_count,
    unmatched_invoices, unmatched_deposits, match_rate_percent,
    tier0_count, tier0_5_count, tier1_count, tier2_count, tier3_count, tier4_count,
    high_confidence_count, avg_confidence,
    total_invoice_amount, total_deposit_amount,
    matched_invoice_amount, matched_deposit_amount, reconciliation_gap,
    execution_time_seconds, created_by
) VALUES (
    {$iteration_number}, {$stats['total_invoices']}, {$stats['total_deposits']}, {$stats['matched_count']},
    {$stats['unmatched_invoices']}, {$stats['unmatched_deposits']}, {$stats['match_rate_percent']},
    {$stats['tier0_count']}, {$stats['tier0_5_count']}, {$stats['tier1_count']}, {$stats['tier2_count']}, {$stats['tier3_count']}, {$stats['tier4_count']},
    {$stats['high_confidence_count']}, {$stats['avg_confidence']},
    {$stats['total_invoice_amount']}, {$stats['total_deposit_amount']},
    {$stats['matched_invoice_amount']}, {$stats['matched_deposit_amount']}, {$stats['reconciliation_gap']},
    {$execution_time}, 'cfdi_matcher_v2'
)";

// Execute insert directly with mysqli
$result = mysqli_query($gIAsql_link, $sql);

// Check for errors
if (!$result) {
    echo "  ✗ MySQL Error: " . mysqli_error($gIAsql_link) . "\n";
    echo "  SQL: " . substr($sql, 0, 300) . "...\n";
    exit(1);
}

// Get inserted ID
$iteration_id = mysqli_insert_id($gIAsql_link);
$affected_rows = mysqli_affected_rows($gIAsql_link);

echo "  ✓ Insert executed (affected rows: {$affected_rows}, insert_id: {$iteration_id})\n";

if ($iteration_id > 0) {
    echo "  ✓ Saved iteration #{$iteration_number} (ID: {$iteration_id})\n";
} else {
    echo "  ✗ Insert failed - no ID returned!\n";
    exit(1);
}

// Insert match results
echo "  - Saving {$stats['matched_count']} match results...\n";
$saved_count = 0;

// Process in batches to avoid timeouts
$batch_size = 50;
$total_matches = count($matches);

for ($i = 0; $i < $total_matches; $i += $batch_size) {
    $batch_matches = array_slice($matches, $i, $batch_size);

    foreach ($batch_matches as $m) {
        $invoice = $m['invoice'];
        $deposit = $m['deposit'];
        $match = $m['match'];

        // Use direct SQL insert for reliability
        $inv_id = mysqli_real_escape_string($gIAsql_link, $invoice['eleyeme_cfdi_emitido_id']);
        $dep_id = mysqli_real_escape_string($gIAsql_link, $deposit['banco_cuenta_mov_id']);
        $pattern = mysqli_real_escape_string($gIAsql_link, $match['pattern'] ?? 'unknown');
        $days_between = days_between($invoice['Fecha_Emision'], $deposit['fecha']);
        $amount_diff = abs(amount_difference_percent($invoice['Total'], $deposit['deposit']));

        $insert_sql = "INSERT INTO cfdi_matcher_results (
            iteration_id, invoice_id, deposit_id, match_tier, match_confidence, match_pattern,
            days_between_invoice_deposit, amount_difference_percent,
            invoice_amount, deposit_amount, invoice_date, deposit_date
        ) VALUES (
            {$iteration_id}, '{$inv_id}', '{$dep_id}', {$match['tier']}, {$match['confidence']}, '{$pattern}',
            {$days_between}, {$amount_diff},
            {$invoice['Total']}, {$deposit['deposit']}, '{$invoice['Fecha_Emision']}', '{$deposit['fecha']}'
        )";

        mysqli_query($gIAsql_link, $insert_sql);
        $saved_count++;
    }

    if ($saved_count % $batch_size == 0) {
        echo "    - Saved {$saved_count}/{$total_matches}...\n";
        flush();
    }
}

echo "  ✓ Saved {$saved_count} match results\n\n";

// ============================================================================
// STEP 6: Sample Matches Display
// ============================================================================

echo "[6/7] Sample Matches (Top 10 by Confidence)\n";
echo "───────────────────────────────────────────────────────────────\n";

// Sort matches by confidence
usort($matches, function($a, $b) {
    return $b['match']['confidence'] - $a['match']['confidence'];
});

$sample_count = min(10, count($matches));
for ($i = 0; $i < $sample_count; $i++) {
    $m = $matches[$i];
    $inv = $m['invoice'];
    $dep = $m['deposit'];
    $match = $m['match'];

    echo "\n[" . ($i + 1) . "] Tier {$match['tier']} - {$match['confidence']}% - {$match['pattern']}\n";
    echo "    Invoice: {$inv['UUID']} ({$inv['Fecha_Emision']}) - $" . number_format($inv['Total'], 2) . "\n";
    echo "    Client: {$inv['Nombre_Receptor']}\n";
    echo "    Deposit: " . substr($dep['banco_cuenta_mov_id'], 0, 8) . "... ({$dep['fecha']}) - $" . number_format($dep['deposit'], 2) . "\n";
    echo "    Ref: " . substr($dep['numero'], 0, 60) . "...\n";
    echo "    Days: {$match['days_diff']} | Amount Diff: $" . number_format($match['amount_diff'], 2) . " ({$match['amount_diff_pct']}%)\n";
}

echo "\n───────────────────────────────────────────────────────────────\n\n";

// ============================================================================
// STEP 6: Summary & Next Steps
// ============================================================================

echo "[5/5] Summary & Recommendations\n";
echo "───────────────────────────────────────────────────────────────\n";

if ($stats['high_confidence_count'] > 0) {
    echo "✓ {$stats['high_confidence_count']} high-confidence matches ready for auto-linking\n";
}

if ($stats['medium_confidence_count'] > 0) {
    echo "⚠ {$stats['medium_confidence_count']} medium-confidence matches need review\n";
}

if ($stats['unmatched_invoices'] > 0) {
    echo "❌ {$stats['unmatched_invoices']} invoices remain unmatched (potential collection issues)\n";
}

if ($stats['unmatched_deposits'] > 0) {
    echo "❌ {$stats['unmatched_deposits']} deposits remain unmatched (identify mystery payments)\n";
}

echo "\nNEXT STEPS:\n";
echo "1. Review high-confidence matches for accuracy\n";
echo "2. Manually verify medium-confidence matches\n";
echo "3. Investigate unmatched invoices (overdue payments?)\n";
echo "4. Investigate unmatched deposits (identify unknown clients)\n";
echo "5. Run test runner to log results: cfdi_matcher_test_runner.php\n";

echo "\n═══════════════════════════════════════════════════════════════\n";
echo "  Matching Complete!\n";
echo "═══════════════════════════════════════════════════════════════\n";

?>
