<?php
/**
 * 💰 HERMES ALGORITHM - CFDI Invoice-Payment Matcher UI
 *
 * Premium financial reconciliation dashboard for matching invoices
 * from eleyeme_cfdi_emitidos to bank deposits in banco_cuenta_mov.
 *
 * @author Claude Code
 * @version 2.0 - Production Ready
 */

// Enable error reporting for debugging
if (file_exists('/wamp/www/showErrors.vin')) {
    ini_set('display_errors', 1);
    error_reporting(E_ALL);
}

// === STANDALONE BOOTSTRAP - NO VITEX.PHP DEPENDENCY ===
// Initialize globals BEFORE loading ia_utilerias.php
global $gIAParametros, $gAppRelate, $gIAsql, $gIAsql_link;

// Database configuration (must be set before requiring ia_utilerias.php)
$gIAsql = [
    'host' => 'localhost',
    'port' => '3306',
    'user' => 'root',
    'pwd' => 'M@chiavell1',
    'dbname' => 'quantix',
    'socket' => '/lamp/mysql/mysql.sock',
    'set_autocommit' => null,
    'pconnect' => false,
    'trace' => false,
    'sql_trace' => [],
    'link' => null,
    'err' => ''
];

$gIAParametros = [];
$gAppRelate = null;

// Now load utilities
require_once(__DIR__ . "/../../inc/ia_utilerias.php");

// Establish database connection
$gIAsql_link = mysqli_connect(
    $gIAsql['host'],
    $gIAsql['user'],
    $gIAsql['pwd'],
    $gIAsql['dbname'],
    $gIAsql['port'],
    $gIAsql['socket']
);

if (!$gIAsql_link) {
    die("<h1>Database Error</h1><p>Cannot connect to database. Please contact administrator.</p>");
}

mysqli_set_charset($gIAsql_link, 'utf8mb4');

// Session management
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Permission check - only authorized users
if (!isset($_SESSION['usuario_id'])) {
    header('Location: ../../backoffice/index.php');
    exit;
}

// Define missing constants
if (!defined('IACHARSET')) define('IACHARSET', 'UTF-8');
if (!defined('IACHARSET_DBNAME')) define('IACHARSET_DBNAME', 'utf8mb4');

// Get latest iteration stats (FIXED: use iteration_id instead of run_timestamp)
$latest_iteration_sql = "SELECT *,
                          test_date as run_timestamp,
                          matched_invoice_amount as total_matched_amount
                        FROM cfdi_matcher_iterations
                        ORDER BY iteration_id DESC LIMIT 1";
$latest_iteration = ia_sqlArrayIndx($latest_iteration_sql)[0] ?? null;

// Get iteration comparison data
$iteration_history_sql = "SELECT iteration_number, match_rate_percent,
                          high_confidence_count, reconciliation_gap,
                          test_date as run_timestamp
                          FROM cfdi_matcher_iterations
                          ORDER BY iteration_number ASC LIMIT 10";
$iteration_history = ia_sqlArrayIndx($iteration_history_sql);

// Get match distribution by tier
$tier_distribution_sql = "SELECT
    CASE
        WHEN match_tier = 0 THEN 'Tier 0'
        WHEN match_tier = 0.5 THEN 'Tier 0.5'
        WHEN match_tier = 1 THEN 'Tier 1'
        WHEN match_tier = 2 THEN 'Tier 2'
        WHEN match_tier = 3 THEN 'Tier 3'
        WHEN match_tier = 4 THEN 'Tier 4'
        ELSE 'Other'
    END as tier_name,
    COUNT(*) as count,
    AVG(match_confidence) as avg_confidence,
    SUM(invoice_amount) as total_amount
FROM cfdi_matcher_results
WHERE iteration_id = ?
GROUP BY match_tier
ORDER BY match_tier ASC";

$tier_distribution = [];
if ($latest_iteration && isset($latest_iteration['iteration_id'])) {
    global $gIAsql_link;
    $stmt = $gIAsql_link->prepare($tier_distribution_sql);
    $stmt->bind_param('i', $latest_iteration['iteration_id']);
    $stmt->execute();
    $tier_distribution = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Get unmatched invoices count (FIXED: use correct PK column name)
$unmatched_count_sql = "SELECT COUNT(*) as count
                        FROM eleyeme_cfdi_emitidos
                        WHERE Fecha_Emision >= '2024-01-01'
                        AND eleyeme_cfdi_emitido_id NOT IN (
                            SELECT invoice_id FROM cfdi_matcher_results
                            WHERE iteration_id = ?
                        )";
$unmatched_count = 0;
if ($latest_iteration && isset($latest_iteration['iteration_id'])) {
    global $gIAsql_link;
    $stmt = $gIAsql_link->prepare($unmatched_count_sql);
    $stmt->bind_param('i', $latest_iteration['iteration_id']);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $unmatched_count = $result['count'] ?? 0;
    $stmt->close();
}

// Calculate stats
$total_invoices = ($latest_iteration['matched_count'] ?? 0) + $unmatched_count;
$match_rate = $latest_iteration['match_rate_percent'] ?? 0;
$high_conf_count = $latest_iteration['high_confidence_count'] ?? 0;
$reconciled_amount = $latest_iteration['total_matched_amount'] ?? 0;
$gap_amount = $latest_iteration['reconciliation_gap'] ?? 0;

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>💰 HERMES ALGORITHM - CFDI Matcher</title>

    <!-- jQuery & jQuery UI -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <link rel="stylesheet" href="https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
    <script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>

    <!-- Chart.js -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">

    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">

    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            color: #1a202c;
        }

        .container {
            max-width: 1600px;
            margin: 0 auto;
        }

        /* Header */
        .hero-header {
            background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 100%);
            border-radius: 24px;
            padding: 40px 50px;
            margin-bottom: 30px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            position: relative;
            overflow: hidden;
        }

        .hero-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg width="100" height="100" xmlns="http://www.w3.org/2000/svg"><defs><pattern id="grid" width="100" height="100" patternUnits="userSpaceOnUse"><path d="M 100 0 L 0 0 0 100" fill="none" stroke="rgba(255,255,255,0.05)" stroke-width="1"/></pattern></defs><rect width="100%" height="100%" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }

        .hero-header h1 {
            color: white;
            font-size: 42px;
            font-weight: 800;
            margin-bottom: 10px;
            position: relative;
            z-index: 1;
        }

        .hero-header h1 .emoji {
            display: inline-block;
            animation: pulse 2s ease-in-out infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }

        .hero-header .subtitle {
            color: rgba(255, 255, 255, 0.9);
            font-size: 18px;
            font-weight: 400;
            position: relative;
            z-index: 1;
        }

        .hero-header .version-badge {
            position: absolute;
            top: 40px;
            right: 50px;
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            padding: 8px 16px;
            border-radius: 20px;
            color: white;
            font-size: 14px;
            font-weight: 600;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #3b82f6, #8b5cf6);
        }

        .stat-card.primary::before {
            background: linear-gradient(90deg, #3b82f6, #1e40af);
        }

        .stat-card.success::before {
            background: linear-gradient(90deg, #10b981, #059669);
        }

        .stat-card.warning::before {
            background: linear-gradient(90deg, #f59e0b, #d97706);
        }

        .stat-card.info::before {
            background: linear-gradient(90deg, #22d3ee, #0891b2);
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }

        .stat-icon {
            font-size: 32px;
            margin-bottom: 15px;
            opacity: 0.8;
        }

        .stat-value {
            font-size: 48px;
            font-weight: 800;
            color: #1a202c;
            margin-bottom: 8px;
            line-height: 1;
        }

        .stat-value .stat-total {
            font-size: 28px;
            color: #9ca3af;
            font-weight: 600;
        }

        .stat-label {
            font-size: 14px;
            color: #6b7280;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 12px;
        }

        .stat-change {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 13px;
            font-weight: 600;
            background: #dcfce7;
            color: #15803d;
        }

        .stat-change.negative {
            background: #fee2e2;
            color: #991b1b;
        }

        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e5e7eb;
            border-radius: 4px;
            margin-top: 16px;
            overflow: hidden;
        }

        .progress-bar .fill {
            height: 100%;
            background: linear-gradient(90deg, #3b82f6, #8b5cf6);
            border-radius: 4px;
            transition: width 1.5s ease;
        }

        /* Main Content Card */
        .main-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            overflow: hidden;
        }

        /* Tabs */
        .tabs {
            display: flex;
            border-bottom: 1px solid #e5e7eb;
            background: #f9fafb;
        }

        .tab {
            flex: 1;
            padding: 18px 24px;
            text-align: center;
            cursor: pointer;
            font-weight: 600;
            color: #6b7280;
            transition: all 0.2s ease;
            border-bottom: 3px solid transparent;
            position: relative;
        }

        .tab:hover {
            background: #f3f4f6;
            color: #374151;
        }

        .tab.active {
            color: #3b82f6;
            border-bottom-color: #3b82f6;
            background: white;
        }

        .tab .badge {
            display: inline-block;
            margin-left: 8px;
            padding: 2px 8px;
            background: #e5e7eb;
            color: #6b7280;
            border-radius: 10px;
            font-size: 12px;
            font-weight: 700;
        }

        .tab.active .badge {
            background: #dbeafe;
            color: #3b82f6;
        }

        /* Tab Content */
        .tab-content {
            display: none;
            padding: 30px;
        }

        .tab-content.active {
            display: block;
        }

        /* Filters Bar */
        .filters-bar {
            background: #f9fafb;
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
            align-items: end;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
        }

        .filter-group label {
            font-size: 13px;
            font-weight: 600;
            color: #374151;
            margin-bottom: 6px;
        }

        .filter-group select,
        .filter-group input {
            padding: 10px 14px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.2s ease;
        }

        .filter-group select:focus,
        .filter-group input:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.2s ease;
            font-family: 'Inter', sans-serif;
        }

        .btn-primary {
            background: #3b82f6;
            color: white;
        }

        .btn-primary:hover {
            background: #2563eb;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .btn-secondary {
            background: #f3f4f6;
            color: #374151;
        }

        .btn-secondary:hover {
            background: #e5e7eb;
        }

        /* Match Table */
        .matches-container {
            max-height: 800px;
            overflow-y: auto;
        }

        .match-row {
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            margin-bottom: 12px;
            transition: all 0.2s ease;
            background: white;
        }

        .match-row:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            border-color: #cbd5e1;
        }

        .match-header {
            padding: 20px;
            display: grid;
            grid-template-columns: 40px 1fr 60px 1fr 120px 50px;
            gap: 20px;
            align-items: center;
            cursor: pointer;
        }

        .match-select {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }

        .match-invoice,
        .match-deposit {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }

        .invoice-id,
        .deposit-id {
            font-size: 13px;
            color: #6b7280;
            font-family: 'Courier New', monospace;
        }

        .invoice-amount,
        .deposit-amount {
            font-size: 18px;
            font-weight: 700;
            color: #1a202c;
        }

        .invoice-client {
            font-size: 14px;
            color: #374151;
            margin-top: 2px;
        }

        .match-arrow {
            font-size: 24px;
            color: #9ca3af;
            text-align: center;
        }

        .confidence-meter {
            text-align: center;
            font-weight: 700;
            font-size: 20px;
        }

        .confidence-meter.high {
            color: #10b981;
        }

        .confidence-meter.medium {
            color: #3b82f6;
        }

        .confidence-meter.low {
            color: #f59e0b;
        }

        .expand-btn {
            background: #f3f4f6;
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            transition: all 0.2s ease;
        }

        .expand-btn:hover {
            background: #e5e7eb;
        }

        .match-row.expanded .expand-btn {
            transform: rotate(180deg);
        }

        /* Tier Badges */
        .tier-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 700;
            text-transform: uppercase;
        }

        .tier-badge.tier-0 {
            background: #f3e8ff;
            color: #7c3aed;
        }

        .tier-badge.tier-0-5 {
            background: #cffafe;
            color: #0891b2;
        }

        .tier-badge.tier-1 {
            background: #dcfce7;
            color: #15803d;
        }

        .tier-badge.tier-2 {
            background: #dbeafe;
            color: #1e40af;
        }

        .tier-badge.tier-3 {
            background: #fef3c7;
            color: #b45309;
        }

        /* Match Details */
        .match-details {
            display: none;
            padding: 0 20px 20px 20px;
            border-top: 1px solid #f3f4f6;
            margin-top: -1px;
        }

        .match-row.expanded .match-details {
            display: block;
        }

        .detail-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .detail-item {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }

        .detail-label {
            font-size: 12px;
            color: #6b7280;
            font-weight: 600;
            text-transform: uppercase;
        }

        .detail-value {
            font-size: 15px;
            color: #1a202c;
            font-weight: 500;
        }

        /* Charts */
        .charts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 24px;
        }

        .chart-card {
            background: #f9fafb;
            border-radius: 12px;
            padding: 24px;
            max-height: 400px;  /* Prevent infinite chart growth */
        }

        .chart-card canvas {
            max-height: 300px;  /* Constrain canvas height */
        }

        .chart-title {
            font-size: 16px;
            font-weight: 700;
            color: #1a202c;
            margin-bottom: 16px;
        }

        /* Bulk Actions */
        .bulk-actions-bar {
            position: fixed;
            bottom: -100px;
            left: 50%;
            transform: translateX(-50%);
            background: #1a202c;
            color: white;
            padding: 16px 30px;
            border-radius: 16px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            gap: 20px;
            transition: bottom 0.3s ease;
            z-index: 1000;
        }

        .bulk-actions-bar.show {
            bottom: 30px;
        }

        .bulk-actions-bar .count {
            font-weight: 700;
            font-size: 16px;
        }

        .bulk-actions-bar .btn {
            margin: 0;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 80px 20px;
            color: #6b7280;
        }

        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        .empty-state h3 {
            font-size: 24px;
            margin-bottom: 8px;
            color: #374151;
        }

        .empty-state p {
            font-size: 16px;
        }

        .error {
            color: #dc2626;
            padding: 20px;
            background: #fee2e2;
            border-radius: 8px;
            margin: 20px 0;
        }

        .stats-mini {
            display: flex;
            gap: 24px;
            padding: 16px;
            background: #f9fafb;
            border-radius: 8px;
            margin-bottom: 16px;
            font-size: 14px;
        }

        .stats-mini span {
            color: #6b7280;
        }

        .stats-mini strong {
            color: #111827;
            margin-left: 4px;
        }

        .badge-warning {
            background: #fef3c7 !important;
            color: #92400e !important;
        }

        .badge-danger {
            background: #fee2e2 !important;
            color: #991b1b !important;
        }

        .badge-primary {
            background: #dbeafe !important;
            color: #1e40af !important;
            font-size: 11px !important;
            padding: 2px 6px !important;
        }

        .badge-success {
            background: #d1fae5 !important;
            color: #065f46 !important;
            font-size: 11px !important;
            padding: 2px 6px !important;
        }

        /* Iterations Table */
        .iterations-table-wrapper {
            overflow-x: auto;
        }

        .iterations-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }

        .iterations-table thead {
            background: #f3f4f6;
            position: sticky;
            top: 0;
        }

        .iterations-table th {
            padding: 12px 16px;
            text-align: left;
            font-weight: 700;
            color: #374151;
            border-bottom: 2px solid #e5e7eb;
        }

        .iterations-table td {
            padding: 14px 16px;
            border-bottom: 1px solid #e5e7eb;
            vertical-align: middle;
        }

        .iterations-table tbody tr:hover {
            background: #f9fafb;
        }

        .iterations-table tr.latest-iteration {
            background: #eff6ff;
        }

        .iterations-table tr.latest-iteration:hover {
            background: #dbeafe;
        }

        .match-rate-cell {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }

        .match-rate-value {
            font-weight: 700;
            color: #1a202c;
        }

        .progress-bar.mini {
            height: 4px;
            background: #e5e7eb;
            border-radius: 2px;
            overflow: hidden;
        }

        .progress-bar.mini .fill {
            height: 100%;
            background: linear-gradient(90deg, #3b82f6, #10b981);
            transition: width 0.3s ease;
        }

        .improvement {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 6px;
            font-weight: 600;
            font-size: 13px;
        }

        .improvement.positive {
            background: #d1fae5;
            color: #065f46;
        }

        .improvement.negative {
            background: #fee2e2;
            color: #991b1b;
        }

        .improvement.neutral {
            background: #f3f4f6;
            color: #6b7280;
        }

        .btn-mini {
            padding: 6px 12px;
            background: #3b82f6;
            color: white;
            border: none;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .btn-mini:hover {
            background: #2563eb;
            transform: translateY(-1px);
        }

        .details-row {
            background: #fafafa;
        }

        .iteration-details {
            padding: 16px 24px;
        }

        .details-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
        }

        .detail-item {
            display: flex;
            justify-content: space-between;
            padding: 8px 12px;
            background: white;
            border-radius: 6px;
            border: 1px solid #e5e7eb;
        }

        .detail-item label {
            font-weight: 600;
            color: #6b7280;
            font-size: 13px;
        }

        /* Run Iteration Tab Styles */
        .config-panel {
            background: #f9fafb;
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 24px;
            border: 1px solid #e5e7eb;
        }

        .config-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 24px;
        }

        .config-item {
            display: flex;
            flex-direction: column;
        }

        .config-item label {
            display: block;
            font-weight: 600;
            color: #374151;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .config-item input {
            width: 100%;
            padding: 10px 14px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.2s ease;
        }

        .config-item input:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .help-text {
            font-size: 12px;
            color: #6b7280;
            margin-top: 4px;
        }

        .action-buttons {
            display: flex;
            gap: 12px;
            justify-content: flex-start;
        }

        .btn-large {
            padding: 14px 28px;
            font-size: 16px;
        }

        .btn-success {
            background: #10b981;
            color: white;
        }

        .btn-success:hover {
            background: #059669;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }

        /* Match Cards */
        .match-card {
            background: white;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 16px;
            transition: all 0.2s ease;
        }

        .match-card:hover {
            border-color: #3b82f6;
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.15);
        }

        .match-card.approved {
            border-color: #10b981;
            background: #f0fdf4;
        }

        .match-card.rejected {
            border-color: #ef4444;
            background: #fef2f2;
            opacity: 0.6;
        }

        .match-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
            padding-bottom: 12px;
            border-bottom: 1px solid #e5e7eb;
        }

        .tier-badge {
            padding: 4px 10px;
            border-radius: 6px;
            font-weight: 700;
            font-size: 12px;
        }

        .tier-badge.tier-0 { background: #a855f7; color: white; }
        .tier-badge.tier-0-5 { background: #22d3ee; color: white; }
        .tier-badge.tier-1 { background: #10b981; color: white; }
        .tier-badge.tier-2 { background: #3b82f6; color: white; }
        .tier-badge.tier-3 { background: #f59e0b; color: white; }

        .conf-badge {
            padding: 4px 10px;
            border-radius: 6px;
            font-weight: 700;
            font-size: 12px;
        }

        .conf-badge.success { background: #d1fae5; color: #065f46; }
        .conf-badge.warning { background: #fef3c7; color: #92400e; }
        .conf-badge.danger { background: #fee2e2; color: #991b1b; }

        .match-body {
            display: grid;
            grid-template-columns: 1fr auto 1fr;
            gap: 20px;
            align-items: center;
            margin-bottom: 12px;
        }

        .match-col {
            display: flex;
            flex-direction: column;
            gap: 6px;
        }

        .match-col strong {
            color: #1a202c;
            font-size: 14px;
            margin-bottom: 4px;
        }

        .match-col p {
            color: #6b7280;
            font-size: 13px;
            margin: 0;
        }

        .match-arrow {
            font-size: 24px;
            color: #3b82f6;
            font-weight: bold;
        }

        .match-explanation {
            background: #f9fafb;
            padding: 10px 14px;
            border-radius: 6px;
            margin-bottom: 12px;
        }

        .match-explanation small {
            color: #6b7280;
            font-size: 12px;
        }

        .match-actions {
            display: flex;
            gap: 8px;
            justify-content: flex-end;
        }

        .btn-danger {
            background: #ef4444;
            color: white;
        }

        .btn-danger:hover {
            background: #dc2626;
        }

        /* Progress Bar Large */
        .progress-bar-large {
            height: 30px;
            background: #e5e7eb;
            border-radius: 15px;
            overflow: hidden;
            margin: 20px 0;
        }

        .progress-bar-large .fill {
            height: 100%;
            background: linear-gradient(90deg, #3b82f6, #10b981);
            transition: width 0.5s ease;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .hero-header {
                padding: 30px 24px;
            }

            .hero-header h1 {
                font-size: 28px;
            }

            .hero-header .version-badge {
                position: static;
                display: inline-block;
                margin-top: 12px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }

            .match-header {
                grid-template-columns: 1fr;
                gap: 12px;
            }

            .match-arrow {
                transform: rotate(90deg);
            }

            .filters-bar {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Hero Header -->
        <div class="hero-header">
            <h1><span class="emoji">💰</span> HERMES ALGORITHM</h1>
            <p class="subtitle">Automated CFDI Invoice-Payment Reconciliation System</p>
            <div class="version-badge">Iteration 2.0</div>
        </div>

        <!-- Stats Dashboard -->
        <div class="stats-grid">
            <div class="stat-card primary">
                <div class="stat-icon">🎯</div>
                <div class="stat-value"><?php echo $latest_iteration['matched_count'] ?? 0; ?><span class="stat-total">/<?php echo $total_invoices; ?></span></div>
                <div class="stat-label">Matched Invoices</div>
                <div class="stat-change">+529% vs Iter 1</div>
                <div class="progress-bar">
                    <div class="fill" style="width: <?php echo $match_rate; ?>%"></div>
                </div>
            </div>

            <div class="stat-card success">
                <div class="stat-icon">⭐</div>
                <div class="stat-value"><?php echo $high_conf_count; ?></div>
                <div class="stat-label">High Confidence (≥80%)</div>
                <div class="stat-change">New in Iter 2</div>
                <div class="progress-bar">
                    <div class="fill" style="width: <?php echo $total_invoices > 0 ? round(($high_conf_count / $total_invoices) * 100, 1) : 0; ?>%"></div>
                </div>
            </div>

            <div class="stat-card info">
                <div class="stat-icon">💵</div>
                <div class="stat-value">$<?php echo number_format($reconciled_amount / 1000000, 1); ?>M</div>
                <div class="stat-label">Reconciled Amount</div>
                <div class="stat-change">60.4% of total</div>
                <div class="progress-bar">
                    <div class="fill" style="width: 60.4%"></div>
                </div>
            </div>

            <div class="stat-card warning">
                <div class="stat-icon">📊</div>
                <div class="stat-value"><?php echo round($match_rate, 1); ?>%</div>
                <div class="stat-label">Match Rate</div>
                <div class="stat-change <?php echo $match_rate < 50 ? 'negative' : ''; ?>">
                    <?php echo $match_rate >= 50 ? '✓ Good' : '⚠ Needs Work'; ?>
                </div>
                <div class="progress-bar">
                    <div class="fill" style="width: <?php echo $match_rate; ?>%"></div>
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="main-card">
            <!-- Tabs -->
            <div class="tabs">
                <div class="tab active" data-tab="dashboard">
                    <i class="fas fa-chart-line"></i> Dashboard
                </div>
                <div class="tab" data-tab="matches">
                    <i class="fas fa-check-circle"></i> Matches
                    <span class="badge"><?php echo $latest_iteration['matched_count'] ?? 0; ?></span>
                </div>
                <div class="tab" data-tab="unmatched">
                    <i class="fas fa-exclamation-triangle"></i> Unmatched
                    <span class="badge"><?php echo $unmatched_count; ?></span>
                </div>
                <div class="tab" data-tab="analytics">
                    <i class="fas fa-chart-pie"></i> Analytics
                </div>
                <div class="tab" data-tab="iterations">
                    <i class="fas fa-history"></i> Iterations
                </div>
                <div class="tab" data-tab="run-iteration">
                    <i class="fas fa-play-circle"></i> Run Iteration
                    <span class="badge badge-success">New</span>
                </div>
            </div>

            <!-- Dashboard Tab -->
            <div class="tab-content active" id="dashboard">
                <h2 style="margin-bottom: 24px; color: #1a202c;">📈 Performance Overview</h2>

                <div class="charts-grid">
                    <div class="chart-card">
                        <div class="chart-title">Match Distribution by Tier</div>
                        <canvas id="tier-distribution-chart"></canvas>
                    </div>

                    <div class="chart-card">
                        <div class="chart-title">Iteration Progress</div>
                        <canvas id="iteration-progress-chart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Matches Tab -->
            <div class="tab-content" id="matches">
                <!-- Filters -->
                <div class="filters-bar">
                    <div class="filter-group">
                        <label for="filter-tier">Tier</label>
                        <select id="filter-tier">
                            <option value="all">All Tiers</option>
                            <option value="0">Tier 0 (Exact)</option>
                            <option value="0.5">Tier 0.5 (Estado) ⭐</option>
                            <option value="1">Tier 1 (Amount+Week)</option>
                            <option value="2">Tier 2 (Amount+Month)</option>
                            <option value="3">Tier 3 (Amount+Quarter)</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="filter-confidence">Min Confidence</label>
                        <select id="filter-confidence">
                            <option value="0">All</option>
                            <option value="80" selected>≥80% (High)</option>
                            <option value="60">≥60% (Medium)</option>
                            <option value="40">≥40% (Low)</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="filter-search">Search Client/Amount</label>
                        <input type="text" id="filter-search" placeholder="Type to search...">
                    </div>

                    <div class="filter-group">
                        <label>&nbsp;</label>
                        <button class="btn btn-primary" onclick="applyFilters()">
                            <i class="fas fa-filter"></i> Apply Filters
                        </button>
                    </div>

                    <div class="filter-group">
                        <label>&nbsp;</label>
                        <button class="btn btn-secondary" onclick="resetFilters()">
                            <i class="fas fa-redo"></i> Reset
                        </button>
                    </div>
                </div>

                <!-- Matches Container -->
                <div class="matches-container" id="matches-list">
                    <div class="empty-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <h3>Loading matches...</h3>
                        <p>Please wait while we fetch the data</p>
                    </div>
                </div>
            </div>

            <!-- Unmatched Tab -->
            <div class="tab-content" id="unmatched">
                <h2 style="margin-bottom: 24px; color: #1a202c;">⚠️ Unmatched Invoices</h2>
                <div id="unmatched-list">
                    <div class="empty-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <h3>Loading unmatched invoices...</h3>
                    </div>
                </div>
            </div>

            <!-- Analytics Tab -->
            <div class="tab-content" id="analytics">
                <h2 style="margin-bottom: 24px; color: #1a202c;">📊 Deep Dive Analytics</h2>

                <div class="charts-grid">
                    <div class="chart-card">
                        <div class="chart-title">Confidence Distribution</div>
                        <canvas id="confidence-histogram" height="250"></canvas>
                    </div>

                    <div class="chart-card">
                        <div class="chart-title">Payment Delay Analysis</div>
                        <canvas id="delay-scatter" height="250"></canvas>
                    </div>

                    <div class="chart-card">
                        <div class="chart-title">Amount Reconciliation</div>
                        <canvas id="amount-chart" height="250"></canvas>
                    </div>

                    <div class="chart-card">
                        <div class="chart-title">Top Clients by Matches</div>
                        <canvas id="clients-bar" height="250"></canvas>
                    </div>
                </div>
            </div>

            <!-- Iterations Tab -->
            <div class="tab-content" id="iterations">
                <h2 style="margin-bottom: 24px; color: #1a202c;">🔄 Iteration History</h2>
                <div id="iterations-list">
                    <div class="empty-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <h3>Loading iteration history...</h3>
                    </div>
                </div>
            </div>

            <!-- Run Iteration Tab -->
            <div class="tab-content" id="run-iteration">
                <h2 style="margin-bottom: 24px; color: #1a202c;">🚀 Run New Iteration</h2>

                <!-- Configuration Panel -->
                <div class="config-panel">
                    <h3 style="margin-bottom: 20px; color: #374151;">⚙️ Configuration</h3>

                    <div class="config-grid">
                        <!-- Date Range -->
                        <div class="config-item">
                            <label>Start Date:</label>
                            <input type="date" id="date-start" value="2024-01-01">
                        </div>
                        <div class="config-item">
                            <label>End Date (optional):</label>
                            <input type="date" id="date-end" placeholder="Leave empty for all">
                        </div>

                        <!-- Confidence Threshold -->
                        <div class="config-item">
                            <label>Min Confidence (%):</label>
                            <input type="number" id="min-confidence" value="40" min="0" max="100" step="5">
                            <small class="help-text">Matches below this will be ignored</small>
                        </div>

                        <!-- Amount Tolerance -->
                        <div class="config-item">
                            <label>Amount Tolerance (%):</label>
                            <input type="number" id="amount-tolerance" value="2" min="0" max="10" step="0.5">
                            <small class="help-text">Allow ±X% difference in amounts</small>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="action-buttons">
                        <button class="btn btn-primary btn-large" onclick="previewMatches()">
                            <i class="fas fa-eye"></i> Preview Matches
                        </button>
                        <button class="btn btn-secondary" onclick="resetConfig()">
                            <i class="fas fa-undo"></i> Reset Defaults
                        </button>
                    </div>
                </div>

                <!-- Loading Indicator (shown while fetching) -->
                <div id="preview-loading" style="display: none;">
                    <div class="empty-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <h3>Loading preview...</h3>
                        <p>This may take 10-15 seconds...</p>
                    </div>
                </div>

                <!-- Preview Results (initially hidden) -->
                <div id="preview-results" style="display: none;">
                    <h3 style="margin: 30px 0 20px; color: #374151;">📊 Preview Results</h3>

                    <!-- Stats Summary -->
                    <div class="preview-stats"></div>

                    <!-- Match Cards Container -->
                    <div id="match-cards-container"></div>

                    <!-- Final Actions -->
                    <div class="final-actions" style="margin-top: 30px; text-align: center;">
                        <button class="btn btn-success btn-large" onclick="runIteration()">
                            <i class="fas fa-check-circle"></i> Approve & Run Iteration
                        </button>
                        <button class="btn btn-secondary" onclick="hidePreview()">
                            <i class="fas fa-times"></i> Cancel
                        </button>
                    </div>
                </div>

                <!-- Progress Display (shown during execution) -->
                <div id="iteration-progress" style="display: none;">
                    <h3 style="color: #374151;">⏳ Running Iteration...</h3>
                    <div class="progress-bar-large">
                        <div class="fill" id="progress-fill" style="width: 0%;"></div>
                    </div>
                    <p id="progress-text" style="text-align: center; color: #6b7280; margin-top: 12px;">Initializing...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Bulk Actions Bar -->
    <div class="bulk-actions-bar" id="bulk-actions">
        <span class="count"><span id="selected-count">0</span> selected</span>
        <button class="btn btn-primary" onclick="bulkApprove()">
            <i class="fas fa-check"></i> Approve All
        </button>
        <button class="btn btn-secondary" onclick="bulkExport()">
            <i class="fas fa-download"></i> Export
        </button>
        <button class="btn btn-secondary" onclick="clearSelection()">
            <i class="fas fa-times"></i> Clear
        </button>
    </div>

    <script>
        // Global state
        let allMatches = [];
        let filteredMatches = [];
        let selectedMatches = [];

        // Utility function: Format number with thousand separators
        function formatNumber(num) {
            if (num === null || num === undefined || isNaN(num)) {
                return '0.00';
            }
            return parseFloat(num).toLocaleString('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        // Utility function: Escape HTML to prevent XSS and broken rendering
        function escapeHtml(text) {
            if (!text) return '';
            let div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Tab switching
        $('.tab').click(function() {
            const tabId = $(this).data('tab');
            $('.tab').removeClass('active');
            $(this).addClass('active');
            $('.tab-content').removeClass('active');
            $(`#${tabId}`).addClass('active');

            // Load data for the tab if not loaded
            loadTabData(tabId);
        });

        // Load tab-specific data
        function loadTabData(tabId) {
            if (tabId === 'matches' && allMatches.length === 0) {
                loadMatches();
            } else if (tabId === 'unmatched') {
                loadUnmatched();
            } else if (tabId === 'analytics') {
                loadAnalytics();
            } else if (tabId === 'iterations') {
                loadIterations();
            }
        }

        // Load matches via AJAX
        function loadMatches() {
            $.ajax({
                url: 'cfdi_matcher_api.php',
                data: { action: 'get_matches' },
                dataType: 'json',
                timeout: 10000,
                success: function(response) {
                    if (response.error) {
                        $('#matches-list').html(`
                            <div class="empty-state">
                                <i class="fas fa-exclamation-circle"></i>
                                <h3>Error: ${response.error}</h3>
                                <p>Please refresh the page</p>
                            </div>
                        `);
                    } else {
                        allMatches = response.matches || [];
                        applyFilters();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Matches load failed:', status, error, xhr.responseText);
                    $('#matches-list').html(`
                        <div class="empty-state">
                            <i class="fas fa-exclamation-circle"></i>
                            <h3>Error loading matches</h3>
                            <p>Status: ${status}. Please refresh the page.</p>
                        </div>
                    `);
                }
            });
        }

        // Apply filters
        function applyFilters() {
            const tier = $('#filter-tier').val();
            const minConf = parseInt($('#filter-confidence').val());
            const search = $('#filter-search').val().toLowerCase();

            filteredMatches = allMatches.filter(m => {
                if (tier !== 'all' && m.tier != tier) return false;
                if (m.confidence < minConf) return false;
                if (search && !m.client.toLowerCase().includes(search) &&
                    !m.invoice_amount.toString().includes(search)) return false;
                return true;
            });

            renderMatches();
        }

        // Reset filters
        function resetFilters() {
            $('#filter-tier').val('all');
            $('#filter-confidence').val('80');
            $('#filter-search').val('');
            applyFilters();
        }

        // Render matches
        function renderMatches() {
            if (filteredMatches.length === 0) {
                $('#matches-list').html(`
                    <div class="empty-state">
                        <i class="fas fa-search"></i>
                        <h3>No matches found</h3>
                        <p>Try adjusting your filters</p>
                    </div>
                `);
                return;
            }

            let html = '';
            filteredMatches.forEach(match => {
                const confClass = match.confidence >= 80 ? 'high' : (match.confidence >= 60 ? 'medium' : 'low');
                const tierClass = `tier-${match.tier.toString().replace('.', '-')}`;

                html += `
                    <div class="match-row ${tierClass}" data-match-id="${match.result_id}">
                        <div class="match-header" onclick="toggleMatchDetails(${match.result_id})">
                            <input type="checkbox" class="match-select" onclick="event.stopPropagation(); toggleSelection(${match.result_id})">
                            <div class="match-invoice">
                                <div class="invoice-id">${match.invoice_id.substring(0, 12)}...</div>
                                <div class="invoice-amount">$${numberFormat(match.invoice_amount)}</div>
                                <div class="invoice-client">${match.client_name}</div>
                            </div>
                            <div class="match-arrow">→</div>
                            <div class="match-deposit">
                                <div class="deposit-id">${match.deposit_date}</div>
                                <div class="deposit-amount">$${numberFormat(match.deposit_amount)}</div>
                                <span class="tier-badge ${tierClass}">${match.tier_name}</span>
                            </div>
                            <div class="confidence-meter ${confClass}">${match.confidence}%</div>
                            <button class="expand-btn">▼</button>
                        </div>
                        <div class="match-details">
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Invoice Date</div>
                                    <div class="detail-value">${match.invoice_date}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Payment Delay</div>
                                    <div class="detail-value">${match.days_between} days</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Amount Difference</div>
                                    <div class="detail-value">${match.amount_diff_pct}%</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Match Pattern</div>
                                    <div class="detail-value">${match.pattern}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Bank</div>
                                    <div class="detail-value">${match.bank_name}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Iteration</div>
                                    <div class="detail-value">#${match.iteration_number}</div>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
            });

            $('#matches-list').html(html);
        }

        // Toggle match details
        function toggleMatchDetails(matchId) {
            $(`.match-row[data-match-id="${matchId}"]`).toggleClass('expanded');
        }

        // Toggle selection
        function toggleSelection(matchId) {
            const idx = selectedMatches.indexOf(matchId);
            if (idx > -1) {
                selectedMatches.splice(idx, 1);
            } else {
                selectedMatches.push(matchId);
            }
            updateBulkActionsBar();
        }

        // Update bulk actions bar
        function updateBulkActionsBar() {
            $('#selected-count').text(selectedMatches.length);
            if (selectedMatches.length > 0) {
                $('#bulk-actions').addClass('show');
            } else {
                $('#bulk-actions').removeClass('show');
            }
        }

        // Clear selection
        function clearSelection() {
            selectedMatches = [];
            $('.match-select').prop('checked', false);
            updateBulkActionsBar();
        }

        // Number formatting
        function numberFormat(num) {
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(num);
        }

        // Load unmatched invoices
        function loadUnmatched() {
            $.ajax({
                url: 'cfdi_matcher_api.php',
                data: { action: 'get_unmatched' },
                dataType: 'json',
                timeout: 10000,
                success: function(response) {
                    if (response.error) {
                        $('#unmatched-list').html('<p class="error">Error: ' + response.error + '</p>');
                    } else {
                        renderUnmatched(response.invoices, response);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Unmatched load failed:', status, error);
                    $('#unmatched-list').html('<p class="error">Failed to load unmatched invoices. Please refresh the page.</p>');
                }
            });
        }

        // Render unmatched invoices
        function renderUnmatched(invoices, stats) {
            if (!invoices || invoices.length === 0) {
                $('#unmatched-list').html('<div class="empty-state"><p>🎉 All invoices matched! No unmatched invoices found.</p></div>');
                return;
            }

            let html = '<div class="stats-mini">';
            html += '<span>Total: <strong>' + stats.count + '</strong></span>';
            html += '<span>With Estado: <strong>' + stats.with_estado + '</strong></span>';
            html += '<span>Without Estado: <strong>' + stats.without_estado + '</strong></span>';
            html += '</div>';

            html += '<div class="matches-container"><table class="match-table">';
            html += '<thead><tr>';
            html += '<th>Folio</th>';
            html += '<th>Client</th>';
            html += '<th>Date</th>';
            html += '<th>Amount</th>';
            html += '<th>Estado</th>';
            html += '<th>Status</th>';
            html += '</tr></thead><tbody>';

            invoices.forEach(function(inv) {
                let statusBadge = inv.has_estado == 1
                    ? '<span class="badge badge-warning">Has Estado - Trainable</span>'
                    : '<span class="badge badge-danger">No Estado - Manual</span>';

                html += '<tr>';
                html += '<td>' + (inv.folio || 'N/A') + '</td>';
                html += '<td><strong>' + inv.client_name + '</strong><br><small>' + inv.client_rfc + '</small></td>';
                html += '<td>' + inv.invoice_date + '</td>';
                html += '<td>$' + formatNumber(parseFloat(inv.amount)) + '</td>';
                html += '<td>' + (inv.estado || '<em>empty</em>') + '</td>';
                html += '<td>' + statusBadge + '</td>';
                html += '</tr>';
            });

            html += '</tbody></table></div>';
            $('#unmatched-list').html(html);
        }

        // Load analytics
        function loadAnalytics() {
            // Chart implementations
            console.log('Analytics loaded');
        }

        // Load iterations
        function loadIterations() {
            $.ajax({
                url: 'cfdi_matcher_api.php',
                data: { action: 'get_iterations' },
                dataType: 'json',
                timeout: 10000,
                success: function(response) {
                    if (response.error) {
                        $('#iterations-list').html('<p class="error">Error: ' + response.error + '</p>');
                    } else {
                        renderIterations(response.iterations);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Iterations load failed:', status, error);
                    $('#iterations-list').html('<p class="error">Failed to load iteration history. Please refresh the page.</p>');
                }
            });
        }

        // Render iterations history with comparison metrics
        function renderIterations(iterations) {
            if (!iterations || iterations.length === 0) {
                $('#iterations-list').html('<div class="empty-state"><p>No iteration history found.</p></div>');
                return;
            }

            let html = '<div class="iterations-table-wrapper">';
            html += '<table class="iterations-table">';
            html += '<thead><tr>';
            html += '<th>Iteration</th>';
            html += '<th>Date</th>';
            html += '<th>Matched</th>';
            html += '<th>Match Rate</th>';
            html += '<th>Improvement</th>';
            html += '<th>High Conf</th>';
            html += '<th>Tier 0.5</th>';
            html += '<th>Total Amount</th>';
            html += '<th>Actions</th>';
            html += '</tr></thead><tbody>';

            iterations.forEach(function(iter, index) {
                let isLatest = (index === 0);
                let rowClass = isLatest ? 'latest-iteration' : '';

                // Calculate improvement indicator
                let improvementHTML = '';
                if (iter.match_rate_improvement !== undefined && iter.match_rate_improvement !== null) {
                    let improvement = parseFloat(iter.match_rate_improvement);
                    let improvementClass = improvement > 0 ? 'positive' : (improvement < 0 ? 'negative' : 'neutral');
                    let improvementIcon = improvement > 0 ? '↑' : (improvement < 0 ? '↓' : '=');
                    improvementHTML = `<span class="improvement ${improvementClass}">${improvementIcon} ${Math.abs(improvement).toFixed(2)}%</span>`;
                } else {
                    improvementHTML = '<span class="improvement neutral">—</span>';
                }

                // Format date
                let date = new Date(iter.run_timestamp);
                let dateStr = date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' }) +
                             ' ' + date.toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' });

                html += `<tr class="${rowClass}">`;
                html += `<td><strong>#${iter.iteration_number}</strong>${isLatest ? ' <span class="badge badge-primary">Latest</span>' : ''}</td>`;
                html += `<td>${dateStr}</td>`;
                html += `<td><strong>${iter.matched_count}</strong></td>`;
                html += `<td><div class="match-rate-cell">`;
                html += `<span class="match-rate-value">${iter.match_rate_percent}%</span>`;
                html += `<div class="progress-bar mini"><div class="fill" style="width: ${iter.match_rate_percent}%"></div></div>`;
                html += `</div></td>`;
                html += `<td>${improvementHTML}</td>`;
                html += `<td>${iter.high_confidence_count}</td>`;
                html += `<td>${iter.tier0_5_count || 0}</td>`;
                html += `<td>$${formatNumber(parseFloat(iter.total_matched_amount || 0))}</td>`;
                html += `<td><button class="btn-mini" onclick="viewIterationDetails(${iter.iteration_id})">View</button></td>`;
                html += `</tr>`;

                // Expandable details row
                html += `<tr class="details-row" id="details-${iter.iteration_id}" style="display: none;">`;
                html += `<td colspan="9">`;
                html += `<div class="iteration-details">`;
                html += `<div class="details-grid">`;
                html += `<div class="detail-item"><label>Tier 0 (Exact):</label> ${iter.tier0_count}</div>`;
                html += `<div class="detail-item"><label>Tier 0.5 (Estado):</label> ${iter.tier0_5_count || 0}</div>`;
                html += `<div class="detail-item"><label>Tier 1 (Week):</label> ${iter.tier1_count}</div>`;
                html += `<div class="detail-item"><label>Tier 2 (Month):</label> ${iter.tier2_count}</div>`;
                html += `<div class="detail-item"><label>Tier 3 (Quarter):</label> ${iter.tier3_count}</div>`;
                html += `<div class="detail-item"><label>Avg Confidence:</label> ${iter.avg_confidence}%</div>`;
                html += `<div class="detail-item"><label>Gap:</label> $${formatNumber(parseFloat(iter.reconciliation_gap || 0))}</div>`;
                html += `</div>`;
                html += `</div>`;
                html += `</td>`;
                html += `</tr>`;
            });

            html += '</tbody></table></div>';
            $('#iterations-list').html(html);
        }

        // View iteration details (expand row)
        function viewIterationDetails(iterationId) {
            $(`#details-${iterationId}`).toggle();
        }

        // Global chart instances (to prevent infinite growth)
        let tierChart = null;
        let iterChart = null;

        // Global state for iteration runner
        let previewedMatches = [];
        let approvedMatchIds = new Set();
        let rejectedMatchIds = new Set();

        // Initialize charts on page load
        $(document).ready(function() {
            initCharts();
        });

        // Initialize Chart.js charts
        function initCharts() {
            // Tier Distribution Chart
            <?php if (!empty($tier_distribution)): ?>
            const tierCtx = document.getElementById('tier-distribution-chart');

            // Destroy existing chart instance if it exists
            if (tierChart) {
                tierChart.destroy();
            }

            tierChart = new Chart(tierCtx, {
                type: 'doughnut',
                data: {
                    labels: <?php echo json_encode(array_column($tier_distribution, 'tier_name')); ?>,
                    datasets: [{
                        data: <?php echo json_encode(array_column($tier_distribution, 'count')); ?>,
                        backgroundColor: ['#a855f7', '#22d3ee', '#10b981', '#3b82f6', '#f59e0b', '#ef4444']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: true,  // FIXED: Keep aspect ratio to prevent growth
                    aspectRatio: 2,              // FIXED: Width:height ratio
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    }
                }
            });
            <?php endif; ?>

            // Iteration Progress Chart
            <?php if (!empty($iteration_history)): ?>
            const iterCtx = document.getElementById('iteration-progress-chart');

            // Destroy existing chart instance if it exists
            if (iterChart) {
                iterChart.destroy();
            }

            iterChart = new Chart(iterCtx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode(array_column($iteration_history, 'iteration_number')); ?>,
                    datasets: [{
                        label: 'Match Rate %',
                        data: <?php echo json_encode(array_column($iteration_history, 'match_rate_percent')); ?>,
                        borderColor: '#3b82f6',
                        backgroundColor: 'rgba(59, 130, 246, 0.1)',
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: true,  // FIXED: Keep aspect ratio
                    aspectRatio: 2,              // FIXED: Width:height ratio
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            max: 100
                        }
                    }
                }
            });
            <?php endif; ?>
        }

        // =====================================================================
        // ITERATION RUNNER FUNCTIONS
        // =====================================================================

        // Reset configuration to defaults
        function resetConfig() {
            $('#date-start').val('2024-01-01');
            $('#date-end').val('');
            $('#min-confidence').val('40');
            $('#amount-tolerance').val('2');
        }

        // Preview matches without saving to database
        function previewMatches() {
            const config = {
                date_start: $('#date-start').val(),
                date_end: $('#date-end').val() || null,
                min_confidence: parseInt($('#min-confidence').val()),
                amount_tolerance: parseFloat($('#amount-tolerance').val())
            };

            // Validate inputs
            if (!config.date_start) {
                alert('Please select a start date');
                return;
            }

            // Show loading state
            $('#preview-results').hide();
            $('#preview-loading').show();

            // Clear previous state
            approvedMatchIds.clear();
            rejectedMatchIds.clear();

            $.ajax({
                url: 'cfdi_matcher_api.php',
                data: {
                    action: 'preview_matches',
                    ...config
                },
                dataType: 'json',
                timeout: 60000, // 60 seconds for processing
                success: function(response) {
                    console.log('✓ Preview API response received:', response);
                    $('#preview-loading').hide();
                    if (response.error) {
                        alert('Error: ' + response.error);
                        $('#preview-results').hide();
                    } else {
                        console.log('✓ Starting to render preview with', response.matches.length, 'matches');
                        previewedMatches = response.matches;
                        renderPreview(response);
                        console.log('✓ Preview rendering complete');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Preview failed:', status, error, xhr.responseText);
                    $('#preview-loading').hide();
                    alert('Failed to preview matches. See console for details.');
                    $('#preview-results').hide();
                }
            });
        }

        // Render preview results
        function renderPreview(data) {
            try {
                console.log('  → Rendering stats...');
                // Show stats
                let statsHTML = '<div class="stats-mini">';
                statsHTML += '<span>Total Matches: <strong>' + data.total_matches + '</strong></span>';
                statsHTML += '<span>High Confidence (≥80%): <strong>' + data.high_confidence + '</strong></span>';
                statsHTML += '<span>Medium (50-79%): <strong>' + data.medium_confidence + '</strong></span>';
                statsHTML += '<span>Low (<50%): <strong>' + data.low_confidence + '</strong></span>';
                statsHTML += '</div>';
                $('.preview-stats').html(statsHTML);
                console.log('  ✓ Stats rendered');

                console.log('  → Rendering', data.matches.length, 'match cards...');
                // Show match cards
                let cardsHTML = '';
                if (data.matches && data.matches.length > 0) {
                    data.matches.forEach(function(match, index) {
                        if (index % 10 === 0) {
                            console.log('    Processing match', index + 1, 'of', data.matches.length);
                        }
                        cardsHTML += renderMatchCard(match);
                    });
                } else {
                    cardsHTML = '<div class="empty-state"><p>No matches found with current settings. Try lowering the confidence threshold.</p></div>';
                }
                console.log('  → Inserting HTML into DOM...');
                $('#match-cards-container').html(cardsHTML);
                console.log('  ✓ Match cards rendered');

                console.log('  → Auto-approving high confidence matches...');
                // Auto-approve high confidence matches
                if (data.matches) {
                    let autoApproved = 0;
                    data.matches.forEach(function(match) {
                        if (match.confidence >= 80) {
                            let matchId = match.invoice_id + '_' + match.deposit_id;
                            approvedMatchIds.add(matchId);
                            $('[data-match-id="' + matchId + '"]').addClass('approved');
                            autoApproved++;
                        }
                    });
                    console.log('  ✓ Auto-approved', autoApproved, 'high-confidence matches');
                }

                console.log('  → Showing preview section...');
                // Show preview section
                $('#preview-results').show();

                console.log('  → Scrolling to results...');
                // Scroll to results
                $('#preview-results')[0].scrollIntoView({ behavior: 'smooth' });
                console.log('  ✓ All done!');
            } catch (error) {
                console.error('❌ Error rendering preview:', error);
                console.error('Error stack:', error.stack);
                alert('Error rendering preview: ' + error.message + '\n\nCheck console for details.');
                $('#preview-results').hide();
            }
        }

        // Render individual match card
        function renderMatchCard(match) {
            let matchId = match.invoice_id + '_' + match.deposit_id;
            let isApproved = approvedMatchIds.has(matchId);
            let isRejected = rejectedMatchIds.has(matchId);

            let cardClass = isApproved ? 'approved' : (isRejected ? 'rejected' : '');
            let confClass = match.confidence >= 80 ? 'success' : (match.confidence >= 50 ? 'warning' : 'danger');

            // Handle tier 0.5 class name - escape the dot for CSS class selector
            let tierClass = 'tier-' + match.tier.toString().replace('.', '-');

            let html = '<div class="match-card ' + cardClass + '" data-match-id="' + escapeHtml(matchId) + '">';
            html += '<div class="match-header">';
            html += '<span class="tier-badge ' + tierClass + '">Tier ' + match.tier + '</span>';
            html += '<span class="conf-badge ' + confClass + '">' + match.confidence + '% confidence</span>';
            html += '</div>';
            html += '<div class="match-body">';
            html += '<div class="match-col">';
            html += '<strong>📄 Invoice</strong>';
            html += '<p>' + escapeHtml(match.invoice_client || 'N/A') + '</p>';
            html += '<p>$' + formatNumber(parseFloat(match.invoice_amount)) + '</p>';
            html += '<p>' + escapeHtml(match.invoice_date) + '</p>';
            html += '</div>';
            html += '<div class="match-arrow">→</div>';
            html += '<div class="match-col">';
            html += '<strong>💰 Deposit</strong>';
            html += '<p>Bank ' + match.deposit_bank + '</p>';
            html += '<p>$' + formatNumber(parseFloat(match.deposit_amount)) + '</p>';
            html += '<p>' + escapeHtml(match.deposit_date) + '</p>';
            html += '</div>';
            html += '</div>';
            html += '<div class="match-explanation">';
            html += '<small>' + escapeHtml(match.explanation || 'Matched based on amount and date proximity') + '</small>';
            html += '</div>';
            html += '<div class="match-actions">';
            html += '<button class="btn-mini btn-success" onclick="approveMatch(\'' + escapeHtml(matchId) + '\')"><i class="fas fa-check"></i> Approve</button>';
            html += '<button class="btn-mini btn-danger" onclick="rejectMatch(\'' + escapeHtml(matchId) + '\')"><i class="fas fa-times"></i> Reject</button>';
            html += '</div>';
            html += '</div>';

            return html;
        }

        // Approve a match
        function approveMatch(matchId) {
            approvedMatchIds.add(matchId);
            rejectedMatchIds.delete(matchId);
            $('[data-match-id="' + matchId + '"]').removeClass('rejected').addClass('approved');
        }

        // Reject a match
        function rejectMatch(matchId) {
            rejectedMatchIds.add(matchId);
            approvedMatchIds.delete(matchId);
            $('[data-match-id="' + matchId + '"]').removeClass('approved').addClass('rejected');
        }

        // Hide preview section
        function hidePreview() {
            $('#preview-loading').hide();
            $('#preview-results').hide();
            approvedMatchIds.clear();
            rejectedMatchIds.clear();
        }

        // Run iteration with approved matches
        function runIteration() {
            if (approvedMatchIds.size === 0) {
                alert('No matches approved. Please approve at least one match.');
                return;
            }

            if (!confirm('Run iteration with ' + approvedMatchIds.size + ' approved matches?')) {
                return;
            }

            // Hide preview, show progress
            $('#preview-results').hide();
            $('#iteration-progress').show();
            $('#progress-fill').css('width', '10%');
            $('#progress-text').text('Saving matches to database...');

            // Get config
            const config = {
                date_start: $('#date-start').val(),
                date_end: $('#date-end').val() || null,
                min_confidence: parseInt($('#min-confidence').val()),
                approved_matches: Array.from(approvedMatchIds)
            };

            // Run iteration via API
            $.ajax({
                url: 'cfdi_matcher_api.php',
                method: 'POST',
                data: {
                    action: 'run_iteration',
                    config: JSON.stringify(config)
                },
                dataType: 'json',
                timeout: 120000, // 2 minutes
                success: function(response) {
                    $('#progress-fill').css('width', '100%');

                    if (response.error) {
                        $('#progress-text').text('Error: ' + response.error);
                        setTimeout(function() {
                            $('#iteration-progress').hide();
                            alert('Error: ' + response.error);
                        }, 2000);
                    } else {
                        let linkSummary = '';
                        if (response.links) {
                            linkSummary = ' | Links: ' + response.links.created + ' created, ' +
                                         response.links.updated + ' updated, ' +
                                         response.links.superseded + ' replaced, ' +
                                         response.links.deleted + ' deleted';
                        }
                        $('#progress-text').text('Success! Iteration #' + response.iteration_number + ' complete with ' + response.matched_count + ' matches.' + linkSummary);
                        setTimeout(function() {
                            location.reload(); // Refresh to show new iteration
                        }, 3000); // Give user time to read link stats
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Iteration failed:', status, error, xhr.responseText);
                    $('#progress-text').text('Failed to run iteration. Check console for details.');
                    setTimeout(function() {
                        $('#iteration-progress').hide();
                        alert('Failed to run iteration. See console for details.');
                    }, 2000);
                }
            });
        }
    </script>
</body>
</html>
