<?php
// Debug the normalize and match functions

function normalize_text(string $text): string {
    if (empty($text)) return '';
    $text = strtolower(trim($text));
    $accents = ['á', 'à', 'ä', 'â', 'ã', 'él', 'è', 'ë', 'ê', 'í', 'ì', 'ï', 'î', 'ó', 'ò', 'ö', 'ô', 'õ', 'ú', 'ù', 'ü', 'û', 'ñ', 'ý', 'ÿ', 'ç', 'š', 'ž', 'ć', 'č'];
    $replacements = ['a', 'a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'n', 'y', 'y', 'c', 's', 'z', 'c', 'c'];
    $text = str_replace($accents, $replacements, $text);
    $text = preg_replace('/\s+/', ' ', $text);
    return $text;
}

function extract_street_name(string $text): string {
    if (empty($text)) return '';
    $text = normalize_text($text);
    preg_match('/(\d)/', $text, $matches, PREG_OFFSET_CAPTURE);
    if (!empty($matches[0][1])) {
        return trim(substr($text, 0, $matches[0][1]));
    }
    return $text;
}

function extract_unit_number(string $text): string {
    if (empty($text)) return '';
    $text = normalize_text($text);
    $patterns = [
        '/\b(\d{1,3})\b/',
        '/\b(ph\s*\d{0,2}[a-z]?)\b/i',
        '/\b(unidad\s*\d+)\b/i',
        '/\b(depto\.?\s*\d+)\b/i',
        '/\b(apt\.?\s*\d+)\b/i',
        '/\b([a-z])\b(?!\s*\d)/',
    ];
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) {
            return trim($matches[1]);
        }
    }
    return '';
}

function similar_text_pct(string $str1, string $str2): float {
    similar_text($str1, $str2, $percent);
    return round($percent, 2);
}

$pdo = new PDO("mysql:host=localhost;unix_socket=/lamp/mysql/mysql.sock;dbname=quantix;charset=utf8mb4", "root", "M@chiavell1");

// Get a Sinaloa reservation
$res = $pdo->query("
    SELECT anuncio, hostify_reserva_id
    FROM hostify_reserva
    WHERE anuncio LIKE '%Sinaloa 51%'
    AND check_in >= '2025-01-01'
    LIMIT 1
")->fetch();

// Get Sinaloa properties
$props = $pdo->query("
    SELECT nombre_propiedad, propiedad_id
    FROM propiedad
    WHERE nombre_propiedad LIKE '%Sinaloa 51%'
")->fetchAll();

echo "Reservation: " . $res["anuncio"] . "\n\n";

$normAnuncio = normalize_text($res["anuncio"]);
$anuncioStreet = extract_street_name($res["anuncio"]);
$anuncioUnit = extract_unit_number($res["anuncio"]);

echo "Normalized anuncio: '$normAnuncio'\n";
echo "Anuncio street: '$anuncioStreet'\n";
echo "Anuncio unit: '$anuncioUnit'\n\n";

foreach ($props as $prop) {
    $normProp = normalize_text($prop["nombre_propiedad"]);
    $propStreet = extract_street_name($prop["nombre_propiedad"]);
    $propUnit = extract_unit_number($prop["nombre_propiedad"]);

    $streetScore = similar_text_pct($anuncioStreet, $propStreet);
    $unitScore = $anuncioUnit === $propUnit ? 100 : 0;

    echo "Property: " . $prop["nombre_propiedad"] . "\n";
    echo "  Norm: '$normProp'\n";
    echo "  Street: '$propStreet' (score: $streetScore%)\n";
    echo "  Unit: '$propUnit' (score: $unitScore%)\n";

    $confidence = round(($streetScore * 0.7) + ($unitScore * 0.3));
    echo "  Overall confidence: $confidence%\n\n";
}
