<?php
/**
 * Iteration 5 - Diagnose Why Patterns Aren't Matching
 */

require_once("../../inc/config.php");

echo "<!DOCTYPE html><html><head><title>Iteration 5 Diagnostics</title></head><body>";
echo "<h1>Iteration 5 - Failure Diagnosis</h1>";
echo "<hr>";

// Test cases that SHOULD match but don't
$test_cases = [
    'VS146 - 102 - Karen Kling | CoB2BQQ - Bcom',
    'VS146 - 101 | CoB2BQQ',
    'VS146 - GH1 - Male Jimenez | CoB2BQQ -P - Bcom',
    'P.E.21 - #701 | CoG2BrKK-B',
    'Ver PH 7 | RoSLKS -BT',
    'Ver 4 | RoS2BKQS -B - Bcom',
    'Ibsen - 3'
];

// Load properties
$sql = "SELECT nombre_propiedad FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql);

echo "<h2>Pattern Testing Results:</h2>";

foreach ($test_cases as $text) {
    echo "<div style='border: 1px solid #ccc; padding: 15px; margin: 15px 0; background: #f9f9f9;'>";
    echo "<h3>Testing: \"$text\"</h3>";

    // Normalize
    $norm = strtolower(trim(preg_replace('/[^a-z0-9\s\-]/i', ' ', $text)));
    $norm_clean = mb_strtolower(trim(preg_replace('/\s+/', ' ', $norm)), 'UTF-8');

    echo "<p><strong>Normalized:</strong> '$norm_clean'</p>";

    // Test Pattern 0.3: ABBR## - #unit
    if (preg_match('/^([a-z]{2,4})(\d{2,3})\s+\-\s+\#?([a-z]*\d+)/i', $norm_clean, $matches)) {
        echo "<p style='color: green;'>✓ <strong>Pattern 0.3 MATCHED!</strong></p>";
        echo "<ul>";
        echo "<li>Abbr: {$matches[1]}</li>";
        echo "<li>Building: {$matches[2]}</li>";
        echo "<li>Unit: {$matches[3]}</li>";
        echo "</ul>";

        // Check abbreviation mapping
        $abbr_map = [
            'slp' => 'san luis potosi',
            'vs' => 'versalles',  // WRONG! Should be Vicente Suárez
            'ver' => 'veracruz',
            'pe' => 'paseo de la reforma',
            'p' => 'paseo de la reforma'
        ];

        $abbr = strtolower($matches[1]);
        $mapped = $abbr_map[$abbr] ?? $abbr;

        echo "<p><strong>Abbreviation Mapping:</strong> '$abbr' → '$mapped'</p>";

        if ($abbr == 'vs') {
            echo "<p style='color: red;'>❌ <strong>ERROR FOUND!</strong> 'vs' maps to 'versalles' but should be 'vicente suarez'!</p>";
        }
        if ($abbr == 'pe' || $abbr == 'p') {
            echo "<p style='color: orange;'>⚠️  'pe' maps to 'paseo de la reforma' - need to verify this is correct</p>";
        }

        // Search for property
        $search_street = $mapped;
        $search_unit = $matches[3];

        echo "<p><strong>Searching for:</strong> Street containing '$search_street' AND Unit '$search_unit'</p>";

        $found = false;
        foreach ($propiedades as $prop) {
            $prop_norm = mb_strtolower($prop['nombre_propiedad'], 'UTF-8');

            // Remove accents for comparison
            $prop_norm = strtr($prop_norm, [
                'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
                'ñ'=>'n', 'ü'=>'u'
            ]);

            if (strpos($prop_norm, $search_street) !== false && strpos($prop_norm, $search_unit) !== false) {
                echo "<p style='color: blue;'>🎯 <strong>FOUND MATCH:</strong> {$prop['nombre_propiedad']}</p>";
                $found = true;
                break;
            }
        }

        if (!$found) {
            echo "<p style='color: red;'>❌ No matching property found in database</p>";

            // Show similar properties
            echo "<p><strong>Properties with similar street:</strong></p><ul>";
            $similar_count = 0;
            foreach ($propiedades as $prop) {
                $prop_norm = mb_strtolower($prop['nombre_propiedad'], 'UTF-8');
                $prop_norm = strtr($prop_norm, ['á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u']);

                if (strpos($prop_norm, $search_street) !== false) {
                    echo "<li>{$prop['nombre_propiedad']}</li>";
                    $similar_count++;
                    if ($similar_count >= 5) break;
                }
            }
            echo "</ul>";
        }

    } else {
        echo "<p style='color: red;'>❌ Pattern 0.3 did NOT match</p>";

        // Try Pattern 0.5: ABBR # or ABBR PH #
        if (preg_match('/^([a-z]{2,4})\s+(ph\s+)?(\d+)/i', $norm_clean, $matches)) {
            echo "<p style='color: green;'>✓ <strong>Pattern 0.5 MATCHED!</strong></p>";
            echo "<ul>";
            echo "<li>Abbr: {$matches[1]}</li>";
            echo "<li>Is PH: " . (!empty($matches[2]) ? 'Yes' : 'No') . "</li>";
            echo "<li>Unit: {$matches[3]}</li>";
            echo "</ul>";
        } else {
            echo "<p style='color: orange;'>⚠️  No pattern matched</p>";
        }
    }

    echo "</div>";
}

echo "<hr>";
echo "<h2>🔍 Key Findings:</h2>";
echo "<ul>";
echo "<li><strong>VS abbreviation is WRONG!</strong> Maps to 'versalles' but should be 'vicente suarez' (with accent Vicente Suárez)</li>";
echo "<li><strong>Accent handling needed:</strong> Database has 'Vicente Suárez' with accent, normalization removes it</li>";
echo "<li><strong>P.E. abbreviation unclear:</strong> Need to verify if it maps to Paseo de la Reforma or something else</li>";
echo "<li><strong>Some properties don't exist:</strong> Campeche Frida, Ibsen - 3 may be invalid listings</li>";
echo "</ul>";

echo "</body></html>";
?>
