<?php
/**
 * ============================================================================
 * Propiedad Expansion & Analytics Tool - Learning Web UI
 * ============================================================================
 *
 * Purpose: Analyze PMS data, suggest improvements, expand multi-unit properties
 * Author: Claude Code (Thoth's Algorithm v3 - Learning Edition)
 * Date: 2026-01-05
 *
 * Features:
 *   - Learns from each iteration (stores analytics per run)
 *   - Auto-applies high-confidence name suggestions (>=90%)
 *   - Discovers unit patterns from Cloudbeds + Hostify reservations
 *   - Expands multi-unit properties intelligently
 *   - Backs up original names before modification
 *
 * Usage:
 *   - Preview:     ?action=preview (default)
 *   - Analyze:     ?action=analyze (scan PMS, update analytics - creates new iteration)
 *   - Apply names: ?action=apply_names&confidence=90 (auto high-conf)
 *   - Apply single: ?action=apply_name&propiedad_id=xxx
 *   - Expand:      ?action=expand&propiedad_id=xxx
 *   - Expand all:  ?action=expand_all
 *
 * ============================================================================
 */

require_once("../../inc/config.php");

// ============================================================================
// SECTION 1: CONFIGURATION & SETUP
// ============================================================================

// Action handler
$action = $_GET['action'] ?? 'preview';
$target_id = $_GET['propiedad_id'] ?? null;
$confidence_threshold = intval($_GET['confidence'] ?? 90);
$debug = isset($_GET['debug']);

// Iteration viewing - allows viewing historical iterations
$view_iteration = isset($_GET['iteration']) ? intval($_GET['iteration']) : null;

// High confidence threshold for auto-apply
$HIGH_CONFIDENCE_THRESHOLD = 90;

// Get current user
$current_user_id = $_SESSION['usuario_id'] ?? '1';

?>
<!DOCTYPE html>
<html>
<head>
    <title>Propiedad Analytics & Expansion Tool</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: #f5f5f5;
        }
        .container {
            max-width: 1800px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 3px solid #9b59b6;
            padding-bottom: 10px;
        }
        h2 {
            color: #34495e;
            border-bottom: 2px solid #ecf0f1;
            padding-bottom: 8px;
            margin-top: 30px;
        }

        /* Tabs */
        .tabs {
            display: flex;
            gap: 10px;
            margin: 20px 0;
            border-bottom: 2px solid #ecf0f1;
            flex-wrap: wrap;
        }
        .tab {
            padding: 12px 24px;
            background: #ecf0f1;
            border: none;
            border-radius: 5px 5px 0 0;
            cursor: pointer;
            font-size: 1em;
            transition: all 0.3s;
        }
        .tab:hover {
            background: #d4dce0;
        }
        .tab.active {
            background: #9b59b6;
            color: white;
            font-weight: bold;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }

        .stats {
            background: #ecf0f1;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
            gap: 15px;
        }
        .stat-box {
            background: white;
            padding: 15px;
            border-radius: 5px;
            text-align: center;
            border-left: 4px solid #9b59b6;
        }
        .stat-box h3 { margin: 0 0 5px 0; font-size: 1.8em; color: #9b59b6; }
        .stat-box p { margin: 0; color: #7f8c8d; font-size: 0.85em; }
        .stat-box.success { border-color: #28a745; }
        .stat-box.success h3 { color: #28a745; }
        .stat-box.warning { border-color: #ffc107; }
        .stat-box.warning h3 { color: #ffc107; }
        .stat-box.info { border-color: #17a2b8; }
        .stat-box.info h3 { color: #17a2b8; }
        .stat-box.danger { border-color: #dc3545; }
        .stat-box.danger h3 { color: #dc3545; }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 0.8em;
        }
        th {
            background: #34495e;
            color: white;
            padding: 10px 8px;
            text-align: left;
            position: sticky;
            top: 0;
            font-size: 0.85em;
        }
        td {
            padding: 8px;
            border-bottom: 1px solid #ecf0f1;
            vertical-align: top;
        }
        tr:hover { background: #f8f9fa; }

        .confidence-high { background-color: #d4edda; border-left: 4px solid #28a745; }
        .confidence-medium { background-color: #fff3cd; border-left: 4px solid #ffc107; }
        .confidence-low { background-color: #f8d7da; border-left: 4px solid #dc3545; }
        .confidence-none { background-color: #f8f9fa; border-left: 4px solid #6c757d; }

        .badge {
            display: inline-block;
            padding: 3px 10px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: bold;
            color: white;
        }
        .badge-success { background: #28a745; }
        .badge-warning { background: #ffc107; color: #333; }
        .badge-danger { background: #dc3545; }
        .badge-info { background: #17a2b8; }
        .badge-secondary { background: #6c757d; }
        .badge-purple { background: #9b59b6; }

        .actions {
            margin: 20px 0;
            padding: 20px;
            background: #f3e5f5;
            border-radius: 5px;
            border-left: 4px solid #9b59b6;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            margin: 5px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.95em;
            text-decoration: none;
            transition: all 0.3s;
        }
        .btn-sm {
            padding: 5px 12px;
            font-size: 0.8em;
            margin: 2px;
        }
        .btn-primary { background: #3498db; color: white; }
        .btn-primary:hover { background: #2980b9; }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-warning:hover { background: #e0a800; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; }
        .btn-purple { background: #9b59b6; color: white; }
        .btn-purple:hover { background: #8e44ad; }
        .btn-secondary { background: #6c757d; color: white; }
        .btn-secondary:hover { background: #5a6268; }

        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            border-left: 4px solid;
        }
        .alert-success { background: #d4edda; border-color: #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border-color: #17a2b8; color: #0c5460; }
        .alert-warning { background: #fff3cd; border-color: #ffc107; color: #856404; }
        .alert-danger { background: #f8d7da; border-color: #dc3545; color: #721c24; }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }

        .detail-text {
            font-size: 0.75em;
            color: #7f8c8d;
        }

        .learning-badge {
            display: inline-block;
            background: linear-gradient(135deg, #9b59b6, #3498db);
            color: white;
            padding: 4px 12px;
            border-radius: 15px;
            font-size: 0.85em;
            font-weight: bold;
            margin-left: 10px;
        }

        .iteration-box {
            display: inline-block;
            background: #9b59b6;
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 1.2em;
            font-weight: bold;
            margin: 10px 0;
        }

        .trend-up { color: #28a745; }
        .trend-down { color: #dc3545; }
        .trend-neutral { color: #6c757d; }

        .preview-box {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
            margin: 10px 0;
        }

        .unit-list {
            display: flex;
            flex-wrap: wrap;
            gap: 5px;
            margin: 5px 0;
        }
        .unit-chip {
            display: inline-block;
            background: #e9ecef;
            padding: 3px 8px;
            border-radius: 10px;
            font-size: 0.8em;
        }
        .unit-chip.pms { background: #d4edda; }
        .unit-chip.generated { background: #fff3cd; }

        .discrepancy-warning {
            background: #fff3cd;
            border: 1px solid #ffc107;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }

        .scroll-table {
            max-height: 600px;
            overflow-y: auto;
        }

        /* Modal styles */
        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.6);
            z-index: 1000;
            overflow-y: auto;
        }
        .modal-overlay.active {
            display: flex;
            justify-content: center;
            align-items: flex-start;
            padding: 40px 20px;
        }
        .modal-content {
            background: white;
            border-radius: 10px;
            max-width: 1100px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 10px 50px rgba(0,0,0,0.4);
            animation: modalSlideIn 0.3s ease;
        }
        @keyframes modalSlideIn {
            from { opacity: 0; transform: translateY(-30px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .modal-header {
            background: linear-gradient(135deg, #9b59b6, #8e44ad);
            color: white;
            padding: 18px 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 10;
            border-radius: 10px 10px 0 0;
        }
        .modal-header.iteration-header {
            background: linear-gradient(135deg, #34495e, #2c3e50);
        }
        .modal-header h3 { margin: 0; font-size: 1.3em; }
        .modal-close {
            background: rgba(255,255,255,0.2);
            border: none;
            color: white;
            font-size: 1.5em;
            cursor: pointer;
            width: 35px;
            height: 35px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background 0.2s;
        }
        .modal-close:hover { background: rgba(255,255,255,0.3); }
        .modal-body { padding: 25px; }
        .modal-section { margin-bottom: 25px; }
        .modal-section h4 {
            color: #2c3e50;
            border-bottom: 2px solid #ecf0f1;
            padding-bottom: 8px;
            margin-bottom: 15px;
        }
        
        .match-card {
            background: #f8f9fa;
            border-left: 4px solid #9b59b6;
            padding: 12px 15px;
            margin: 8px 0;
            border-radius: 0 6px 6px 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 10px;
        }
        .match-card.cloudbeds { border-color: #9b59b6; }
        .match-card.hostify { border-color: #3498db; }
        .match-card .match-info { flex: 1; min-width: 200px; }
        .match-card .match-info strong { color: #2c3e50; }
        .match-card .match-meta { font-size: 0.85em; color: #7f8c8d; }
        
        .clickable-row { cursor: pointer; transition: background 0.15s; }
        .clickable-row:hover { background: #e8f4fc !important; }
        
        .mini-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 12px;
            margin-bottom: 20px;
        }
        .mini-stat {
            background: #f8f9fa;
            padding: 12px;
            border-radius: 6px;
            text-align: center;
            border-top: 3px solid #9b59b6;
        }
        .mini-stat.cb { border-color: #9b59b6; }
        .mini-stat.hf { border-color: #3498db; }
        .mini-stat.success { border-color: #28a745; }
        .mini-stat h4 { margin: 0 0 5px 0; font-size: 1.6em; color: #2c3e50; }
        .mini-stat p { margin: 0; font-size: 0.8em; color: #7f8c8d; }
        
        .suggestion-box {
            background: linear-gradient(135deg, #d1ecf1, #bee5eb);
            border: 1px solid #17a2b8;
            border-radius: 8px;
            padding: 15px 20px;
            margin: 15px 0;
        }
        .suggestion-box .new-name {
            font-size: 1.1em;
            font-weight: bold;
            color: #0c5460;
            margin-bottom: 8px;
        }
        
        .viewing-indicator {
            background: #34495e;
            color: white;
            padding: 8px 15px;
            border-radius: 5px;
            display: inline-block;
            margin-right: 10px;
        }
    </style>
</head>
<body>
<div class="container">
    <h1>Propiedad Analytics & Expansion Tool <span class="learning-badge">LEARNING AI</span></h1>
    <p style="color: #7f8c8d; font-style: italic;">
        Thoth's Algorithm v3 - Each iteration makes the system smarter
    </p>

<?php

// ============================================================================
// SECTION 2: HELPER FUNCTIONS
// ============================================================================

/**
 * Safe htmlspecialchars that handles NULL values
 */
function esc($value) {
    return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Normalize text for comparison
 */
function normalize_text($text) {
    if (empty($text)) return '';

    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);
    $text = mb_strtolower($text, 'UTF-8');
    $text = trim(preg_replace('/\s+/', ' ', $text));

    return $text;
}

/**
 * Extract street name (text before first digit)
 */
function extract_street_name($text) {
    $normalized = normalize_text($text);

    if (preg_match('/\d/', $normalized, $matches, PREG_OFFSET_CAPTURE)) {
        $pos = $matches[0][1];
        return trim(substr($normalized, 0, $pos));
    }

    return $normalized;
}

/**
 * Extract unit number from text
 */
function extract_unit_number($text) {
    $normalized = normalize_text($text);

    // Pattern 1: Pure numbers (103, 401, 302)
    if (preg_match('/\b(\d{2,4})\b/', $normalized, $matches)) {
        return $matches[1];
    }

    // Pattern 2: Penthouse (PH1, PH2, ph chico)
    if (preg_match('/ph\s*(\w+)/i', $normalized, $matches)) {
        return 'ph' . normalize_text($matches[1]);
    }

    // Pattern 3: Suite (SU4, Suite 4)
    if (preg_match('/su(?:ite)?\s*(\w+)/i', $normalized, $matches)) {
        return 'su' . normalize_text($matches[1]);
    }

    // Pattern 4: Single letter suffix (A, B, C)
    if (preg_match('/\b([A-Z])\b/', $text, $matches)) {
        return strtolower($matches[1]);
    }

    return '';
}

/**
 * Advanced unit extraction
 */
function extract_unit_advanced($text) {
    $norm = normalize_text($text);
    
    // SU1(1) format
    if (preg_match('/\bSU(\d+)\(\d+\)/i', $text, $matches)) {
        return 'su' . $matches[1];
    }

    // PH Chico/Grande
    if (preg_match('/\bPH\s+(Chico|Grande)/i', $text, $matches)) {
        return 'ph' . strtolower($matches[1]);
    }

    // 2PH(1) format
    if (preg_match('/(\d)PH\(\d+\)/i', $text, $matches)) {
        return 'ph' . $matches[1];
    }

    // Suite 10
    if (preg_match('/\bsuite\s+(\d+)/i', $norm, $matches)) {
        return 'suite' . $matches[1];
    }

    // PH1, PH2
    if (preg_match('/\bPH(\d+)\b/i', $text, $matches)) {
        return 'ph' . $matches[1];
    }

    // Letter units: - A, - H
    if (preg_match('/[-\s]([A-J])\b/', $text, $matches)) {
        return strtolower($matches[1]);
    }

    // Two-digit: - 01, - 23
    if (preg_match('/[-\s](\d{2})\b/', $text, $matches)) {
        if (strpos($text, '- ' . $matches[1]) !== false) {
            return $matches[1];
        }
    }

    // Three-digit room numbers: 502, 302
    if (preg_match('/\b([1-5]\d{2})\b/', $norm, $matches)) {
        return $matches[1];
    }

    return extract_unit_number($text);
}

/**
 * Extract unit suffix from property name (ONLY from AFTER separator)
 * This is the key function that differentiates unit numbers from street numbers.
 * 
 * Examples:
 *   "Queretaro 121 - 101" → "101" (unit after separator)
 *   "Amsterdam 210 - A" → "a" (letter unit)
 *   "Queretaro 121 | 503 PH" → "ph503" (pipe separator with PH)
 *   "Mineria 102" → null (no separator = no unit suffix)
 *   "Queretaro 121" → null (no separator)
 */
function extract_unit_suffix($property_name) {
    $text = trim($property_name);
    
    // Pattern 1: " - UNIT" with dash separator (most common in our data)
    // Matches: "Queretaro 121 - 101", "Amsterdam 210 - A", "Mineria 13 - 13"
    if (preg_match('/\s+-\s+([A-Za-z0-9]+(?:\s*PH)?)\s*$/i', $text, $matches)) {
        $unit = trim($matches[1]);
        // Handle PH suffix
        if (preg_match('/^(\d+)\s*PH$/i', $unit, $ph_matches)) {
            return 'ph' . $ph_matches[1];
        }
        if (preg_match('/^PH\s*(\d*)$/i', $unit, $ph_matches)) {
            return 'ph' . ($ph_matches[1] ?: '');
        }
        return strtolower($unit);
    }
    
    // Pattern 2: " | UNIT" pipe separator
    // Matches: "Queretaro 121 | 503 PH", "Street 10 | 201"
    if (preg_match('/\s*\|\s*([A-Za-z0-9]+(?:\s*PH)?)\s*$/i', $text, $matches)) {
        $unit = trim($matches[1]);
        if (preg_match('/^(\d+)\s*PH$/i', $unit, $ph_matches)) {
            return 'ph' . $ph_matches[1];
        }
        if (preg_match('/^PH\s*(\d*)$/i', $unit, $ph_matches)) {
            return 'ph' . ($ph_matches[1] ?: '');
        }
        return strtolower($unit);
    }
    
    // Pattern 3: Standalone "PH" or "PH1" at very end (no separator but clear unit marker)
    // Matches: "Building Name PH", "Building Name PH2"
    if (preg_match('/\s+(PH\s*\d*)\s*$/i', $text, $matches)) {
        $unit = preg_replace('/\s+/', '', strtolower($matches[1]));
        return $unit;
    }
    
    // No clear unit suffix found - property name doesn't have a separator-based unit
    return null;
}

/**
 * Extract street/building number from property name (the base number, NOT the unit)
 * 
 * Examples:
 *   "Queretaro 121 - 101" → "121" (street number before separator)
 *   "Amsterdam 210" → "210" (only number = street number)
 *   "Mineria 13 - 13" → "13" (first number before separator)
 *   "Calle Sin Numero - 5" → null (no street number)
 */
function extract_street_number($property_name) {
    $text = trim($property_name);
    
    // Get the base name (everything before any separator like - or |)
    $base = preg_replace('/\s*[-|].*$/', '', $text);
    
    // Also remove (N deptos) suffix if present
    $base = preg_replace('/\s*\(\d+\s*deptos?\)/i', '', $base);
    
    // Extract the last number from the base (usually the street/building number)
    // "Queretaro 121" → "121"
    // "Calle 5 de Mayo 100" → "100"
    if (preg_match('/(\d+)\s*$/', trim($base), $matches)) {
        return $matches[1];
    }
    
    // Fallback: get first number if no number at end
    if (preg_match('/(\d+)/', $base, $matches)) {
        return $matches[1];
    }
    
    return null;
}

/**
 * Extract unit from PMS data (room_number or anuncio), filtering out street numbers
 * 
 * @param string $text The PMS room number or anuncio text
 * @param string|null $property_street_number The property's street number to filter out
 * @return string|null The extracted unit, or null if none found
 */
function extract_unit_from_pms($text, $property_street_number = null) {
    if (empty($text)) {
        return null;
    }
    
    // First, use the advanced extraction
    $unit = extract_unit_advanced($text);
    
    if ($unit === null) {
        return null;
    }
    
    // Filter out if the extracted "unit" is actually the street number
    if ($property_street_number !== null) {
        $norm_unit = strtolower(trim($unit));
        $norm_street = strtolower(trim($property_street_number));
        
        // Direct match - this is the street number, not a unit
        if ($norm_unit === $norm_street) {
            return null;
        }
        
        // Also check if unit is just street number with leading zeros
        // e.g., street "121" should filter out unit "121" or "0121"
        if (ltrim($norm_unit, '0') === ltrim($norm_street, '0')) {
            return null;
        }
    }
    
    return $unit;
}

/**
 * Expand combo anuncio patterns
 */
function expand_combo_anuncio($text) {
    $norm = normalize_text($text);

    // Doble X y Y
    if (preg_match('/doble\s+(\d+)\s+y\s+(\d+)/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'doble_y',
            'street' => extract_street_name($text)
        ];
    }

    // Suite 1, Suite 4, Suite 10
    if (preg_match_all('/suite\s+(\d+)/i', $norm, $matches)) {
        if (count($matches[1]) > 1) {
            return [
                'units' => $matches[1],
                'type' => 'multi_suite',
                'street' => extract_street_name($text)
            ];
        }
    }

    // SU2(1), SU1(1)
    if (preg_match_all('/su(\d+)\(\d+\)/i', $norm, $matches)) {
        if (count($matches[1]) > 1) {
            return [
                'units' => $matches[1],
                'type' => 'multi_su',
                'street' => extract_street_name($text)
            ];
        }
    }

    // Comma-separated: 204, 103, 401
    if (strpos($text, ',') !== false) {
        if (preg_match_all('/\b(\d{2,4})\b/', $norm, $matches)) {
            if (count($matches[1]) >= 2) {
                return [
                    'units' => $matches[1],
                    'type' => 'comma_separated',
                    'street' => extract_street_name($text)
                ];
            }
        }
    }

    // X y Y
    if (preg_match('/(\d{2,4})\s+y\s+(\d{2,4})/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'y_connector',
            'street' => extract_street_name($text)
        ];
    }

    // X | Y
    if (preg_match('/(\d{2,4})\s*\|\s*(\d{2,4})/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'pipe_separator',
            'street' => extract_street_name($text)
        ];
    }

    return [];
}

// ============================================================================
// SECTION 3: ANALYTICS FUNCTIONS
// ============================================================================

/**
 * Get current iteration number
 */
function get_current_iteration() {
    $sql = "SELECT COALESCE(MAX(iteration), 0) as max_iter FROM propiedad_pms_analytics";
    $result = ia_singleton($sql);
    // ia_singleton returns an array, extract the value
    return intval($result['max_iter'] ?? 0);
}

/**
 * Get next iteration number
 */
function get_next_iteration() {
    return get_current_iteration() + 1;
}

/**
 * Get previous analytics for a property
 */
function get_previous_analytics($propiedad_id, $iteration = null) {
    if ($iteration === null) {
        $iteration = get_current_iteration();
    }
    
    $sql = "SELECT * FROM propiedad_pms_analytics 
            WHERE propiedad_id = " . strit($propiedad_id) . "
            AND iteration = " . intval($iteration) . "
            LIMIT 1";
    $result = ia_sqlArrayIndx($sql);
    return !empty($result) ? $result[0] : null;
}

/**
 * Get latest analytics for a property
 */
function get_latest_analytics($propiedad_id) {
    $sql = "SELECT * FROM propiedad_pms_analytics 
            WHERE propiedad_id = " . strit($propiedad_id) . "
            ORDER BY iteration DESC
            LIMIT 1";
    $result = ia_sqlArrayIndx($sql);
    return !empty($result) ? $result[0] : null;
}

/**
 * Save analytics for a property
 */
function save_analytics($propiedad_id, $data, $iteration, $user_id) {
    $analytics_id = ia_guid('analytics');
    
    // Calculate delta from previous iteration
    $prev = get_previous_analytics($propiedad_id, $iteration - 1);
    $delta_confidence = null;
    $delta_matches = null;
    $delta_notes = '';
    
    if ($prev) {
        $prev_conf = floatval($prev['match_confidence_avg'] ?? 0);
        $curr_conf = floatval($data['match_confidence_avg'] ?? 0);
        $delta_confidence = $curr_conf - $prev_conf;
        
        $prev_matches = intval($prev['total_reservations'] ?? 0);
        $curr_matches = intval($data['total_reservations'] ?? 0);
        $delta_matches = $curr_matches - $prev_matches;
        
        $notes = [];
        if ($delta_confidence > 0) {
            $notes[] = "Confidence +" . round($delta_confidence, 1) . "%";
        } elseif ($delta_confidence < 0) {
            $notes[] = "Confidence " . round($delta_confidence, 1) . "%";
        }
        if ($delta_matches > 0) {
            $notes[] = "+" . $delta_matches . " matches";
        } elseif ($delta_matches < 0) {
            $notes[] = $delta_matches . " matches";
        }
        $delta_notes = implode(", ", $notes);
    }
    
    $values = [
        'analytics_id' => $analytics_id,
        'propiedad_id' => $propiedad_id,
        'iteration' => $iteration,
        'discovered_units' => json_encode($data['discovered_units'] ?? []),
        'cloudbeds_matches' => json_encode($data['cloudbeds_matches'] ?? []),
        'hostify_matches' => json_encode($data['hostify_matches'] ?? []),
        'match_confidence_avg' => $data['match_confidence_avg'] ?? null,
        'match_confidence_max' => $data['match_confidence_max'] ?? null,
        'match_confidence_min' => $data['match_confidence_min'] ?? null,
        'match_tier_best' => $data['match_tier_best'] ?? null,
        'match_count_cloudbeds' => $data['match_count_cloudbeds'] ?? 0,
        'match_count_hostify' => $data['match_count_hostify'] ?? 0,
        'total_reservations' => $data['total_reservations'] ?? 0,
        'suggested_name' => $data['suggested_name'] ?? null,
        'suggested_name_confidence' => $data['suggested_name_confidence'] ?? null,
        'suggested_name_reason' => $data['suggested_name_reason'] ?? null,
        'suggested_expansion' => json_encode($data['suggested_expansion'] ?? null),
        'delta_confidence' => $delta_confidence,
        'delta_matches' => $delta_matches,
        'delta_notes' => $delta_notes,
        'analyzed_by' => $user_id,
    ];
    
    $sql = ia_insert('propiedad_pms_analytics', $values);
    ia_query($sql);
    
    return $analytics_id;
}

/**
 * Get iteration summary statistics
 */
function get_iteration_summary($iteration = null) {
    if ($iteration === null) {
        $iteration = get_current_iteration();
    }
    
    $sql = "SELECT 
                COUNT(*) as properties_analyzed,
                AVG(match_confidence_avg) as avg_confidence,
                MAX(match_confidence_max) as max_confidence,
                SUM(match_count_cloudbeds) as total_cb_matches,
                SUM(match_count_hostify) as total_hf_matches,
                SUM(total_reservations) as total_reservations,
                SUM(CASE WHEN suggested_name IS NOT NULL THEN 1 ELSE 0 END) as suggestions_count,
                SUM(suggestion_applied) as suggestions_applied,
                SUM(expansion_applied) as expansions_applied,
                AVG(delta_confidence) as avg_delta_confidence,
                MIN(analyzed_at) as started_at,
                MAX(analyzed_at) as completed_at
            FROM propiedad_pms_analytics
            WHERE iteration = " . intval($iteration);
    
    $result = ia_sqlArrayIndx($sql);
    return !empty($result) ? $result[0] : null;
}

/**
 * Get all iteration summaries
 */
function get_all_iteration_summaries() {
    $sql = "SELECT 
                iteration,
                COUNT(*) as properties_analyzed,
                ROUND(AVG(match_confidence_avg), 1) as avg_confidence,
                SUM(total_reservations) as total_reservations,
                SUM(suggestion_applied) as suggestions_applied,
                SUM(expansion_applied) as expansions_applied,
                ROUND(AVG(delta_confidence), 1) as avg_delta,
                MIN(analyzed_at) as started_at
            FROM propiedad_pms_analytics
            GROUP BY iteration
            ORDER BY iteration DESC";
    
    return ia_sqlArrayIndx($sql);
}

// ============================================================================
// SECTION 4: PMS DISCOVERY FUNCTIONS
// ============================================================================

/**
 * Discover units from Cloudbeds reservations for a property
 * 
 * UPDATED: Now uses extract_unit_from_pms() to filter out street numbers
 * from discovered units. This prevents confusing "Queretaro 121" street number
 * with unit numbers like "101".
 */
function discover_units_from_cloudbeds($propiedad, $all_cloudbeds) {
    $matches = [];
    $discovered_units = [];
    
    $prop_name = $propiedad['nombre_propiedad'];
    $prop_dir = $propiedad['direccion'] ?? '';
    $prop_street = extract_street_name($prop_name);
    
    // Extract the property's street number to filter it out from unit discovery
    $prop_street_number = extract_street_number($prop_name);
    
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($prop_dir);
    }
    
    foreach ($all_cloudbeds as $reserva) {
        $cb_property = $reserva['property'] ?? '';
        $cb_room = $reserva['room_number'] ?? '';
        $cb_street = extract_street_name($cb_property);
        
        // Check if streets match
        $street_match = false;
        if (strlen($prop_street) >= 4 && strlen($cb_street) >= 4) {
            if (strpos($prop_street, $cb_street) !== false || 
                strpos($cb_street, $prop_street) !== false) {
                $street_match = true;
            } else {
                similar_text($prop_street, $cb_street, $percent);
                if ($percent >= 70) {
                    $street_match = true;
                }
            }
        }
        
        if ($street_match) {
            // Extract unit from room number, filtering out street number
            $unit = extract_unit_from_pms($cb_room, $prop_street_number);
            if (empty($unit)) {
                // Fallback: try extracting from property name
                $unit = extract_unit_from_pms($cb_property, $prop_street_number);
            }
            
            if (!empty($unit) && !in_array($unit, $discovered_units)) {
                $discovered_units[] = $unit;
            }
            
            // Calculate match confidence
            $building_score = 100; // Street matched
            $unit_score = 0;
            
            // Use extract_unit_suffix for the property (only unit AFTER separator)
            $prop_unit = extract_unit_suffix($prop_name);
            
            if (!empty($prop_unit) && !empty($unit)) {
                if ($prop_unit === $unit) {
                    $unit_score = 100;
                } else {
                    similar_text($prop_unit, $unit, $percent);
                    $unit_score = $percent;
                }
            }
            
            $confidence = round(($building_score * 0.6) + ($unit_score * 0.4));
            
            if ($confidence >= 40) {
                $matches[] = [
                    'property' => $cb_property,
                    'room_number' => $cb_room,
                    'confidence' => $confidence,
                    'tier' => ($confidence >= 95) ? 1 : (($confidence >= 80) ? 2 : (($confidence >= 65) ? 3 : 4)),
                    'discovered_unit' => $unit,
                    'reserva_id' => $reserva['cloudbeds_reserva_id'] ?? null
                ];
            }
        }
    }
    
    // Sort units naturally
    sort($discovered_units, SORT_NATURAL);
    
    return [
        'matches' => $matches,
        'discovered_units' => $discovered_units
    ];
}

/**
 * Discover units from Hostify reservations for a property
 * 
 * UPDATED: Now uses extract_unit_from_pms() to filter out street numbers
 * from discovered units, and extract_unit_suffix() for property unit comparison.
 */
function discover_units_from_hostify($propiedad, $all_hostify) {
    $matches = [];
    $discovered_units = [];
    
    $prop_name = $propiedad['nombre_propiedad'];
    $prop_dir = $propiedad['direccion'] ?? '';
    $prop_street = extract_street_name($prop_name);
    
    // Extract the property's street number to filter it out from unit discovery
    $prop_street_number = extract_street_number($prop_name);
    
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($prop_dir);
    }
    
    foreach ($all_hostify as $reserva) {
        $anuncio = $reserva['anuncio'] ?? '';
        $anuncio_street = extract_street_name($anuncio);
        
        // Check for combo anuncios
        $combo = expand_combo_anuncio($anuncio);
        
        // Check if streets match
        $street_match = false;
        if (strlen($prop_street) >= 4 && strlen($anuncio_street) >= 4) {
            if (strpos($prop_street, $anuncio_street) !== false || 
                strpos($anuncio_street, $prop_street) !== false) {
                $street_match = true;
            } else {
                similar_text($prop_street, $anuncio_street, $percent);
                if ($percent >= 70) {
                    $street_match = true;
                }
            }
        }
        
        if ($street_match) {
            // Discover units from combo or single, filtering out street numbers
            if (!empty($combo['units'])) {
                foreach ($combo['units'] as $combo_unit) {
                    $unit = normalize_text($combo_unit);
                    // Filter out if it matches the street number
                    if (!empty($unit) && $unit !== $prop_street_number && 
                        ltrim($unit, '0') !== ltrim($prop_street_number ?? '', '0')) {
                        if (!in_array($unit, $discovered_units)) {
                            $discovered_units[] = $unit;
                        }
                    }
                }
            } else {
                // Use the new extract_unit_from_pms which filters street numbers
                $unit = extract_unit_from_pms($anuncio, $prop_street_number);
                if (!empty($unit) && !in_array($unit, $discovered_units)) {
                    $discovered_units[] = $unit;
                }
            }
            
            // Calculate match confidence
            // Use extract_unit_suffix for property (only unit AFTER separator)
            $prop_unit = extract_unit_suffix($prop_name);
            // Use extract_unit_from_pms for anuncio (filters out street number)
            $anuncio_unit = extract_unit_from_pms($anuncio, $prop_street_number);
            
            $confidence = 70; // Base for street match
            $tier = 3;
            $combo_type = null;
            
            // Check if this is a combo match
            if (!empty($combo['units'])) {
                foreach ($combo['units'] as $combo_unit) {
                    $norm_combo = normalize_text($combo_unit);
                    // Skip if combo unit is actually street number
                    if ($norm_combo === $prop_street_number || 
                        ltrim($norm_combo, '0') === ltrim($prop_street_number ?? '', '0')) {
                        continue;
                    }
                    if ($prop_unit === $norm_combo) {
                        $confidence = 90;
                        $tier = 0;
                        $combo_type = $combo['type'];
                        break;
                    }
                    // Number to letter conversion (5 -> e)
                    if (is_numeric($norm_combo) && $norm_combo >= 1 && $norm_combo <= 26) {
                        $letter = chr(96 + intval($norm_combo));
                        if ($prop_unit === $letter) {
                            $confidence = 88;
                            $tier = 0;
                            $combo_type = $combo['type'] . '_num2letter';
                            break;
                        }
                    }
                }
            } elseif (!empty($prop_unit) && !empty($anuncio_unit)) {
                if ($prop_unit === $anuncio_unit) {
                    $confidence = 100;
                    $tier = 1;
                } else {
                    similar_text($prop_unit, $anuncio_unit, $percent);
                    if ($percent >= 80) {
                        $confidence = 90;
                        $tier = 2;
                    }
                }
            }
            
            if ($confidence >= 50) {
                $matches[] = [
                    'anuncio' => $anuncio,
                    'confidence' => $confidence,
                    'tier' => $tier,
                    'combo_type' => $combo_type,
                    'discovered_unit' => $anuncio_unit ?: ($combo['units'][0] ?? null),
                    'reserva_id' => $reserva['hostify_reserva_id'] ?? null
                ];
            }
        }
    }
    
    // Sort units naturally
    sort($discovered_units, SORT_NATURAL);
    
    return [
        'matches' => $matches,
        'discovered_units' => $discovered_units
    ];
}

/**
 * Merge discovered units from both sources
 */
function merge_discovered_units($cb_units, $hf_units) {
    $all = array_merge($cb_units, $hf_units);
    $all = array_unique($all);
    sort($all, SORT_NATURAL);
    return $all;
}

/**
 * Generate name suggestion for a property
 * 
 * IMPORTANT: This function now uses extract_unit_suffix() to distinguish between
 * street/building numbers (e.g., "121" in "Queretaro 121") and unit numbers
 * (e.g., "101" in "Queretaro 121 - 101").
 * 
 * The old logic was broken because it used extract_unit_advanced() which would
 * extract the FIRST number found, which is often the street number.
 */
function generate_name_suggestion($propiedad, $discovered_data) {
    $current_name = $propiedad['nombre_propiedad'];
    $num_deptos = intval($propiedad['num_deptos'] ?? 1);
    
    // Use the NEW functions that properly distinguish street vs unit numbers
    $current_unit_suffix = extract_unit_suffix($current_name);
    $street_number = extract_street_number($current_name);
    
    // Get matches from both PMS sources
    $cb_matches = $discovered_data['cloudbeds_matches'] ?? [];
    $hf_matches = $discovered_data['hostify_matches'] ?? [];
    $total_matches = count($cb_matches) + count($hf_matches);
    
    // Need minimum matches to make any suggestion
    if ($total_matches < 2) {
        return null;
    }
    
    // Count unit occurrences, EXCLUDING the street number
    // This is the key fix - we filter out discovered "units" that are actually street numbers
    $unit_counts = [];
    $all_matches = array_merge($cb_matches, $hf_matches);
    
    foreach ($all_matches as $m) {
        $u = $m['discovered_unit'] ?? null;
        if (empty($u)) {
            continue;
        }
        
        // Normalize for comparison
        $norm_u = strtolower(trim($u));
        $norm_street = $street_number !== null ? strtolower(trim($street_number)) : null;
        
        // Skip if this "unit" is actually the street number
        if ($norm_street !== null) {
            if ($norm_u === $norm_street) {
                continue; // e.g., "121" discovered when street is "121"
            }
            // Also check with leading zeros stripped
            if (ltrim($norm_u, '0') === ltrim($norm_street, '0')) {
                continue;
            }
        }
        
        // Valid unit - count it
        $unit_counts[$norm_u] = ($unit_counts[$norm_u] ?? 0) + 1;
    }
    
    // No valid units found after filtering
    if (empty($unit_counts)) {
        return null;
    }
    
    // Get most common unit (must have at least 2 occurrences for confidence)
    arsort($unit_counts);
    $most_common_unit = array_key_first($unit_counts);
    $most_common_count = $unit_counts[$most_common_unit];
    
    if ($most_common_count < 2) {
        return null; // Not enough confidence
    }
    
    // Normalize current unit suffix for comparison
    $norm_current_suffix = $current_unit_suffix !== null ? strtolower(trim($current_unit_suffix)) : null;
    
    // CASE 1: Property already has the correct unit suffix - no suggestion needed
    if ($norm_current_suffix !== null && $norm_current_suffix === $most_common_unit) {
        return null; // Already correct
    }
    
    // CASE 2: Property has a unit suffix but it's different from PMS data
    if ($norm_current_suffix !== null && $norm_current_suffix !== $most_common_unit) {
        // Only suggest if the most_common is significantly more common than current
        $current_count = $unit_counts[$norm_current_suffix] ?? 0;
        
        if ($most_common_count <= $current_count) {
            return null; // Current suffix might be correct, PMS data might be for other units
        }
        
        // Build suggested name by replacing the suffix
        // Remove existing unit suffix (everything after the last separator)
        $base_name = preg_replace('/\s*[-|]\s*[A-Za-z0-9]+(?:\s*PH)?\s*$/i', '', $current_name);
        $base_name = trim($base_name);
        
        // Format the unit for display (uppercase, handle PH)
        $display_unit = format_unit_for_display($most_common_unit);
        $suggested = $base_name . ' - ' . $display_unit;
        
        // Confidence based on how much more common the suggested unit is
        $confidence = min(90, 70 + ($most_common_count * 3));
        
        return [
            'suggested_name' => $suggested,
            'confidence' => $confidence,
            'reason' => "Unit mismatch: current '" . strtoupper($current_unit_suffix) . 
                       "' but PMS shows '" . strtoupper($most_common_unit) . 
                       "' ($most_common_count times vs $current_count)"
        ];
    }
    
    // CASE 3: Property has NO unit suffix (single unit, no separator in name)
    if ($norm_current_suffix === null && $num_deptos == 1) {
        // Only suggest adding unit if we're very confident (3+ occurrences)
        if ($most_common_count >= 3) {
            // Clean the name (remove "(N deptos)" if present, trim trailing separators)
            $clean_name = preg_replace('/\s*\(\d+\s*deptos?\)/i', '', $current_name);
            $clean_name = trim(preg_replace('/\s*[-|]\s*$/', '', $clean_name));
            
            // Format the unit for display
            $display_unit = format_unit_for_display($most_common_unit);
            $suggested = $clean_name . ' - ' . $display_unit;
            
            $confidence = min(90, 65 + ($most_common_count * 3));
            
            return [
                'suggested_name' => $suggested,
                'confidence' => $confidence,
                'reason' => "Add unit '$display_unit' found in $most_common_count PMS reservations"
            ];
        }
    }
    
    // CASE 4: Multi-unit property without suffix - don't suggest individual unit
    // (These should be handled by the expansion queue instead)
    
    return null;
}

/**
 * Format a unit string for display in property names
 * e.g., "101" → "101", "ph3" → "PH3", "a" → "A"
 */
function format_unit_for_display($unit) {
    $unit = strtoupper(trim($unit));
    
    // Handle PH prefix
    if (preg_match('/^PH\s*(\d*)$/i', $unit, $matches)) {
        return 'PH' . ($matches[1] ?? '');
    }
    
    return $unit;
}

/**
 * Generate expansion suggestion for multi-unit property
 */
function generate_expansion_suggestion($propiedad, $discovered_units) {
    $num_deptos = intval($propiedad['num_deptos'] ?? 1);
    
    if ($num_deptos <= 1) {
        return null;
    }
    
    $pms_count = count($discovered_units);
    $discrepancy = $num_deptos - $pms_count;
    
    // Build suggested units list
    $suggested_units = [];
    $source = [];
    
    // Add PMS-discovered units first
    foreach ($discovered_units as $unit) {
        $suggested_units[] = $unit;
        $source[$unit] = 'pms';
    }
    
    // Fill missing with generated units
    if ($discrepancy > 0) {
        $generated_count = 0;
        $i = 1;
        while ($generated_count < $discrepancy && $i <= 100) {
            $gen_unit = generateUnitNumber($i, $num_deptos);
            if (!in_array($gen_unit, $suggested_units)) {
                $suggested_units[] = $gen_unit;
                $source[$gen_unit] = 'generated';
                $generated_count++;
            }
            $i++;
        }
    }
    
    return [
        'num_units' => $num_deptos,
        'pms_found' => $pms_count,
        'discrepancy' => $discrepancy,
        'units' => $suggested_units,
        'source' => $source
    ];
}

// ============================================================================
// SECTION 5: EXPANSION FUNCTIONS (from CLI script)
// ============================================================================

/**
 * Clean property name (remove hints like "(12 Deptos)")
 */
function cleanPropertyName($nombre_propiedad) {
    $cleaned = preg_replace('/\s*\(\d+\s*deptos?\)/i', '', $nombre_propiedad);
    $cleaned = preg_replace('/\s*-\s*$/', '', $cleaned);
    return trim($cleaned);
}

/**
 * Extract floor from unit number
 */
function extractFloorFromUnit($numero_unidad) {
    if (empty($numero_unidad)) return null;

    // 3-digit format: 101, 201, 301 (first digit = floor)
    if (preg_match('/^(\d)(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1];
    }

    // 4-digit format: 1001, 2001 (first 2 digits = floor)
    if (preg_match('/^(\d{2})(\d{2})$/', $numero_unidad, $matches)) {
        return (int)$matches[1];
    }

    return null;
}

/**
 * Generate unit number based on index and total count
 */
function generateUnitNumber($index, $total_units) {
    // For small buildings (<=10 units), use letters: A, B, C, etc.
    if ($total_units <= 10 && $total_units > 1) {
        return chr(64 + $index); // A=65, B=66, etc.
    }

    // For larger buildings, use zero-padded numbers
    $padding = strlen((string)$total_units);
    return str_pad($index, max(2, $padding), '0', STR_PAD_LEFT);
}

/**
 * Determine unit type based on property name
 */
function determineUnitType($nombre_propiedad) {
    $lower = mb_strtolower($nombre_propiedad, 'UTF-8');

    if (str_contains($lower, 'casa')) return 'Casa';
    if (str_contains($lower, 'penthouse') || str_contains($lower, 'ph')) return 'Penthouse';
    if (str_contains($lower, 'loft')) return 'Loft';
    if (str_contains($lower, 'estudio')) return 'Estudio';

    return 'Departamento';
}

/**
 * Expand a multi-unit property into individual units
 */
function expand_propiedad($propiedad_id, $suggested_units = null) {
    // Load property
    $sql = "SELECT * FROM propiedad WHERE propiedad_id = " . strit($propiedad_id);
    $result = ia_sqlArrayIndx($sql);
    if (empty($result)) {
        return ['success' => false, 'error' => 'Property not found'];
    }
    $propiedad = $result[0];
    
    $num_deptos = intval($propiedad['num_deptos'] ?? 1);
    if ($num_deptos <= 1) {
        return ['success' => false, 'error' => 'Property is not multi-unit'];
    }
    
    $base_name = cleanPropertyName($propiedad['nombre_propiedad']);
    $tipo_unidad = determineUnitType($base_name);
    
    // Determine units to create
    $units_to_create = [];
    if ($suggested_units && !empty($suggested_units['units'])) {
        $units_to_create = $suggested_units['units'];
    } else {
        // Generate default units
        for ($i = 1; $i <= $num_deptos; $i++) {
            $units_to_create[] = generateUnitNumber($i, $num_deptos);
        }
    }
    
    // Limit to num_deptos
    $units_to_create = array_slice($units_to_create, 0, $num_deptos);
    
    $created_ids = [];
    $report = [];
    
    foreach ($units_to_create as $index => $unit) {
        $new_name = "$base_name - " . strtoupper($unit);
        $piso = extractFloorFromUnit($unit);
        
        if ($index === 0) {
            // Update original row
            // Backup original name if not already backed up
            if (empty($propiedad['nombre_propiedad_original'])) {
                $sql = "UPDATE propiedad SET nombre_propiedad_original = nombre_propiedad WHERE propiedad_id = " . strit($propiedad_id);
                ia_query($sql);
            }
            
            $update = ia_update('propiedad', [
                'num_deptos' => 1,
                'nombre_propiedad' => $new_name,
                'numero_unidad' => $unit,
                'piso' => $piso,
                'tipo_unidad' => $tipo_unidad,
            ], ['propiedad_id' => $propiedad_id]);
            ia_query($update);
            
            $created_ids[] = $propiedad_id;
            $report[] = [
                'id' => $propiedad_id,
                'name' => $new_name,
                'unit' => $unit,
                'action' => 'updated'
            ];
        } else {
            // Insert new row
            $new_id = ia_guid('propiedad_exp');
            
            $insert = ia_insert('propiedad', [
                'propiedad_id' => $new_id,
                'num_deptos' => 1,
                'nombre_propiedad' => $new_name,
                'numero_unidad' => $unit,
                'piso' => $piso,
                'propietario_id' => $propiedad['propietario_id'],
                'direccion' => $propiedad['direccion'],
                'vale' => $propiedad['vale'],
                'codigo_postal' => $propiedad['codigo_postal'],
                'colonia' => $propiedad['colonia'],
                'estado' => $propiedad['estado'],
                'estado_descripcion' => $propiedad['estado_descripcion'],
                'municipio' => $propiedad['municipio'],
                'municipio_descripcion' => $propiedad['municipio_descripcion'],
                'tipo_unidad' => $tipo_unidad,
                'nombre_propiedad_original' => $propiedad['nombre_propiedad'],
                'alta_por' => 'expansion_webui',
            ]);
            ia_query($insert);
            
            $created_ids[] = $new_id;
            $report[] = [
                'id' => $new_id,
                'name' => $new_name,
                'unit' => $unit,
                'action' => 'inserted'
            ];
        }
    }
    
    // Mark expansion as applied in analytics
    $sql = "UPDATE propiedad_pms_analytics 
            SET expansion_applied = 1, expansion_applied_at = NOW() 
            WHERE propiedad_id = " . strit($propiedad_id) . "
            AND iteration = " . get_current_iteration();
    ia_query($sql);
    
    return [
        'success' => true,
        'created_ids' => $created_ids,
        'report' => $report,
        'count' => count($created_ids)
    ];
}

// ============================================================================
// SECTION 6: ACTION HANDLERS
// ============================================================================

/**
 * Handle analyze action - scan PMS data and create new iteration
 */
function handle_analyze($user_id) {
    $iteration = get_next_iteration();
    
    // Load all data
    $propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad");
    $cloudbeds = ia_sqlArrayIndx("SELECT * FROM cloudbeds_reserva");
    $hostify = ia_sqlArrayIndx("SELECT * FROM hostify_reserva");
    
    $stats = [
        'analyzed' => 0,
        'with_matches' => 0,
        'suggestions_generated' => 0,
        'expansion_candidates' => 0
    ];
    
    foreach ($propiedades as $propiedad) {
        $propiedad_id = $propiedad['propiedad_id'];
        
        // Discover from both sources
        $cb_data = discover_units_from_cloudbeds($propiedad, $cloudbeds);
        $hf_data = discover_units_from_hostify($propiedad, $hostify);
        
        // Merge units
        $all_units = merge_discovered_units($cb_data['discovered_units'], $hf_data['discovered_units']);
        
        // Calculate confidence metrics
        $all_matches = array_merge($cb_data['matches'], $hf_data['matches']);
        $confidences = array_column($all_matches, 'confidence');
        $tiers = array_column($all_matches, 'tier');
        
        $analytics_data = [
            'discovered_units' => $all_units,
            'cloudbeds_matches' => $cb_data['matches'],
            'hostify_matches' => $hf_data['matches'],
            'match_confidence_avg' => !empty($confidences) ? round(array_sum($confidences) / count($confidences), 2) : null,
            'match_confidence_max' => !empty($confidences) ? max($confidences) : null,
            'match_confidence_min' => !empty($confidences) ? min($confidences) : null,
            'match_tier_best' => !empty($tiers) ? min($tiers) : null,
            'match_count_cloudbeds' => count($cb_data['matches']),
            'match_count_hostify' => count($hf_data['matches']),
            'total_reservations' => count($all_matches),
        ];
        
        // Generate name suggestion
        $name_suggestion = generate_name_suggestion($propiedad, [
            'cloudbeds_matches' => $cb_data['matches'],
            'hostify_matches' => $hf_data['matches']
        ]);
        
        if ($name_suggestion) {
            $analytics_data['suggested_name'] = $name_suggestion['suggested_name'];
            $analytics_data['suggested_name_confidence'] = $name_suggestion['confidence'];
            $analytics_data['suggested_name_reason'] = $name_suggestion['reason'];
            $stats['suggestions_generated']++;
        }
        
        // Generate expansion suggestion for multi-unit
        $num_deptos = intval($propiedad['num_deptos'] ?? 1);
        if ($num_deptos > 1) {
            $expansion = generate_expansion_suggestion($propiedad, $all_units);
            $analytics_data['suggested_expansion'] = $expansion;
            $stats['expansion_candidates']++;
        }
        
        // Save analytics
        save_analytics($propiedad_id, $analytics_data, $iteration, $user_id);
        
        $stats['analyzed']++;
        if (count($all_matches) > 0) {
            $stats['with_matches']++;
        }
    }
    
    return [
        'success' => true,
        'iteration' => $iteration,
        'stats' => $stats
    ];
}

/**
 * Handle apply names action
 */
function handle_apply_names($confidence_threshold, $propiedad_id = null) {
    $iteration = get_current_iteration();
    
    $where = "iteration = $iteration AND suggestion_applied = 0 AND suggested_name IS NOT NULL AND suggested_name_confidence >= $confidence_threshold";
    if ($propiedad_id) {
        $where .= " AND propiedad_id = " . strit($propiedad_id);
    }
    
    $sql = "SELECT a.*, p.nombre_propiedad as current_name, p.nombre_propiedad_original 
            FROM propiedad_pms_analytics a
            JOIN propiedad p ON a.propiedad_id = p.propiedad_id
            WHERE $where";
    $suggestions = ia_sqlArrayIndx($sql);
    
    $applied = 0;
    foreach ($suggestions as $s) {
        $pid = $s['propiedad_id'];
        $new_name = $s['suggested_name'];
        
        // Backup original name if not already backed up
        if (empty($s['nombre_propiedad_original'])) {
            $sql = "UPDATE propiedad SET nombre_propiedad_original = nombre_propiedad WHERE propiedad_id = " . strit($pid);
            ia_query($sql);
        }
        
        // Apply new name
        $sql = "UPDATE propiedad SET nombre_propiedad = " . strit($new_name) . " WHERE propiedad_id = " . strit($pid);
        ia_query($sql);
        
        // Mark as applied
        $sql = "UPDATE propiedad_pms_analytics SET suggestion_applied = 1, suggestion_applied_at = NOW() WHERE analytics_id = " . strit($s['analytics_id']);
        ia_query($sql);
        
        $applied++;
    }
    
    return [
        'success' => true,
        'applied' => $applied
    ];
}

// ============================================================================
// SECTION 6B: AJAX HANDLERS (JSON responses for modals)
// ============================================================================

// AJAX: Get property detail for modal
if ($action === 'get_property_detail') {
    header('Content-Type: application/json; charset=utf-8');
    $analytics_id = $_GET['analytics_id'] ?? '';
    
    $sql = "SELECT a.*, p.nombre_propiedad, p.direccion, p.num_deptos 
            FROM propiedad_pms_analytics a
            JOIN propiedad p ON a.propiedad_id = p.propiedad_id
            WHERE a.analytics_id = " . strit($analytics_id);
    $data = ia_sqlArrayIndx($sql);
    
    if (!empty($data)) {
        $row = $data[0];
        $response = [
            'success' => true,
            'nombre_propiedad' => $row['nombre_propiedad'],
            'direccion' => $row['direccion'],
            'num_deptos' => intval($row['num_deptos'] ?? 1),
            'iteration' => intval($row['iteration']),
            'match_confidence_avg' => floatval($row['match_confidence_avg'] ?? 0),
            'match_confidence_max' => floatval($row['match_confidence_max'] ?? 0),
            'match_confidence_min' => floatval($row['match_confidence_min'] ?? 0),
            'match_tier_best' => $row['match_tier_best'],
            'match_count_cloudbeds' => intval($row['match_count_cloudbeds'] ?? 0),
            'match_count_hostify' => intval($row['match_count_hostify'] ?? 0),
            'total_reservations' => intval($row['total_reservations'] ?? 0),
            'discovered_units' => json_decode($row['discovered_units'] ?? '[]', true),
            'cloudbeds_matches' => json_decode($row['cloudbeds_matches'] ?? '[]', true),
            'hostify_matches' => json_decode($row['hostify_matches'] ?? '[]', true),
            'suggested_name' => $row['suggested_name'],
            'suggested_name_confidence' => intval($row['suggested_name_confidence'] ?? 0),
            'suggested_name_reason' => $row['suggested_name_reason'],
            'suggestion_applied' => intval($row['suggestion_applied'] ?? 0),
            'suggested_expansion' => json_decode($row['suggested_expansion'] ?? 'null', true),
            'delta_confidence' => floatval($row['delta_confidence'] ?? 0),
            'delta_notes' => $row['delta_notes'],
            'analyzed_at' => $row['analyzed_at'],
        ];
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode(['success' => false, 'error' => 'Analytics not found']);
    }
    exit;
}

// AJAX: Get iteration detail for modal
if ($action === 'get_iteration_detail') {
    header('Content-Type: application/json; charset=utf-8');
    $iteration = intval($_GET['iteration'] ?? 0);
    
    $summary = get_iteration_summary($iteration);
    
    if (!$summary) {
        echo json_encode(['success' => false, 'error' => 'Iteration not found']);
        exit;
    }
    
    // Get top properties by confidence
    $top_properties = ia_sqlArrayIndx("
        SELECT a.propiedad_id, a.match_confidence_avg, a.match_count_cloudbeds, a.match_count_hostify,
               a.total_reservations, a.suggested_name, p.nombre_propiedad
        FROM propiedad_pms_analytics a
        JOIN propiedad p ON a.propiedad_id = p.propiedad_id
        WHERE a.iteration = $iteration
        ORDER BY a.match_confidence_avg DESC
        LIMIT 15
    ");
    
    // Get unmatched properties (low confidence)
    $unmatched = ia_sqlArrayIndx("
        SELECT a.propiedad_id, a.match_confidence_avg, p.nombre_propiedad
        FROM propiedad_pms_analytics a
        JOIN propiedad p ON a.propiedad_id = p.propiedad_id
        WHERE a.iteration = $iteration AND (a.match_confidence_avg < 50 OR a.match_confidence_avg IS NULL)
        ORDER BY a.match_confidence_avg ASC
        LIMIT 10
    ");
    
    // Get suggestions pending
    $suggestions_pending = ia_sqlArrayIndx("
        SELECT a.propiedad_id, a.suggested_name, a.suggested_name_confidence, p.nombre_propiedad
        FROM propiedad_pms_analytics a
        JOIN propiedad p ON a.propiedad_id = p.propiedad_id
        WHERE a.iteration = $iteration AND a.suggested_name IS NOT NULL AND a.suggestion_applied = 0
        ORDER BY a.suggested_name_confidence DESC
        LIMIT 10
    ");
    
    $response = [
        'success' => true,
        'iteration' => $iteration,
        'properties_analyzed' => intval($summary['properties_analyzed'] ?? 0),
        'avg_confidence' => floatval($summary['avg_confidence'] ?? 0),
        'max_confidence' => floatval($summary['max_confidence'] ?? 0),
        'total_cb_matches' => intval($summary['total_cb_matches'] ?? 0),
        'total_hf_matches' => intval($summary['total_hf_matches'] ?? 0),
        'total_reservations' => intval($summary['total_reservations'] ?? 0),
        'suggestions_count' => intval($summary['suggestions_count'] ?? 0),
        'suggestions_applied' => intval($summary['suggestions_applied'] ?? 0),
        'expansions_applied' => intval($summary['expansions_applied'] ?? 0),
        'started_at' => $summary['started_at'],
        'completed_at' => $summary['completed_at'],
        'top_properties' => $top_properties,
        'unmatched' => $unmatched,
        'suggestions_pending' => $suggestions_pending,
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit;
}

// ============================================================================
// SECTION 7: MAIN EXECUTION
// ============================================================================

// Load current data
$current_iteration = get_current_iteration();
$propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad");
$all_iterations = get_all_iteration_summaries();

// Determine which iteration to display (current or historical view)
$display_iteration = $view_iteration ?? $current_iteration;
$is_viewing_history = ($view_iteration !== null && $view_iteration != $current_iteration);

// Load summary for display iteration
$iteration_summary = $display_iteration > 0 ? get_iteration_summary($display_iteration) : null;

// Process actions
$action_result = null;

if ($action === 'analyze') {
    $action_result = handle_analyze($current_user_id);
    $current_iteration = get_current_iteration(); // Refresh
    $display_iteration = $current_iteration; // Switch to new iteration
    $iteration_summary = get_iteration_summary($current_iteration);
    $all_iterations = get_all_iteration_summaries();
    $is_viewing_history = false;
}

if ($action === 'apply_names') {
    $action_result = handle_apply_names($confidence_threshold, $target_id);
}

if ($action === 'apply_name' && $target_id) {
    $action_result = handle_apply_names(0, $target_id); // Apply regardless of confidence
}

if ($action === 'expand' && $target_id) {
    // Get suggested expansion from analytics
    $analytics = get_latest_analytics($target_id);
    $suggested = null;
    if ($analytics && !empty($analytics['suggested_expansion'])) {
        $suggested = json_decode($analytics['suggested_expansion'], true);
    }
    $action_result = expand_propiedad($target_id, $suggested);
    $propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad"); // Refresh
}

if ($action === 'expand_all') {
    $multi_units = ia_sqlArrayIndx("SELECT propiedad_id FROM propiedad WHERE num_deptos > 1");
    $total_expanded = 0;
    foreach ($multi_units as $p) {
        $analytics = get_latest_analytics($p['propiedad_id']);
        $suggested = null;
        if ($analytics && !empty($analytics['suggested_expansion'])) {
            $suggested = json_decode($analytics['suggested_expansion'], true);
        }
        $result = expand_propiedad($p['propiedad_id'], $suggested);
        if ($result['success']) {
            $total_expanded += $result['count'];
        }
    }
    $action_result = ['success' => true, 'total_expanded' => $total_expanded];
    $propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad"); // Refresh
}

// Display action results
if ($action_result) {
    if ($action === 'analyze' && $action_result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>Analysis Complete - Iteration #{$action_result['iteration']}</strong><br>";
        echo "Analyzed: {$action_result['stats']['analyzed']} properties<br>";
        echo "With matches: {$action_result['stats']['with_matches']}<br>";
        echo "Suggestions generated: {$action_result['stats']['suggestions_generated']}<br>";
        echo "Expansion candidates: {$action_result['stats']['expansion_candidates']}";
        echo "</div>";
    }
    
    if ($action === 'apply_names' && $action_result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>Applied {$action_result['applied']} name changes</strong>";
        if ($action_result['applied'] > 0) {
            echo "<br><span class='badge badge-warning'>Re-matching recommended</span> - Run link_pms_propiedades.php to update reservation links";
        }
        echo "</div>";
    }
    
    if ($action === 'apply_name' && $action_result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>Applied 1 name change</strong>";
        echo "</div>";
    }
    
    if ($action === 'expand' && $action_result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>Expanded property into {$action_result['count']} units</strong><br>";
        echo "<ul>";
        foreach ($action_result['report'] as $r) {
            echo "<li>{$r['name']} (unit: {$r['unit']}, {$r['action']})</li>";
        }
        echo "</ul>";
        echo "<span class='badge badge-warning'>Re-matching recommended</span>";
        echo "</div>";
    }
    
    if ($action === 'expand_all' && $action_result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>Expanded all multi-unit properties: {$action_result['total_expanded']} total units created</strong>";
        echo "<br><span class='badge badge-warning'>Re-matching recommended</span>";
        echo "</div>";
    }
}

// Show iteration info
if ($is_viewing_history) {
    echo "<span class='viewing-indicator'>Viewing Historical Iteration</span>";
    echo "<div class='iteration-box' style='background:#34495e;'>Iteration #$display_iteration</div>";
    echo " <a href='?' class='btn btn-sm btn-secondary'>Back to Current (#$current_iteration)</a>";
} else {
    echo "<div class='iteration-box'>Iteration #$current_iteration</div>";
}

if ($current_iteration == 0) {
    echo "<div class='alert alert-info'>";
    echo "<strong>No analysis has been run yet.</strong> Click 'Run Analysis' to scan PMS data and create the first iteration.";
    echo "</div>";
}

// ============================================================================
// SECTION 8: UI OUTPUT - TABS
// ============================================================================

echo "<div class='tabs'>";
echo "<button class='tab active' onclick='showTab(\"dashboard\")'>Dashboard</button>";
echo "<button class='tab' onclick='showTab(\"analytics\")'>Properties Analytics</button>";
echo "<button class='tab' onclick='showTab(\"suggestions\")'>Name Suggestions</button>";
echo "<button class='tab' onclick='showTab(\"expansion\")'>Expansion Queue</button>";
echo "<button class='tab' onclick='showTab(\"history\")'>Learning History</button>";
echo "</div>";

// ============================================================================
// TAB 1: DASHBOARD
// ============================================================================

echo "<div id='tab-dashboard' class='tab-content active'>";
echo "<h2>Dashboard</h2>";

echo "<div class='actions'>";
echo "<h3>Actions</h3>";
echo "<a href='?action=analyze' class='btn btn-purple' onclick='return confirm(\"Run new analysis? This will create iteration #" . ($current_iteration + 1) . "\")'>Run Analysis (Create Iteration #" . ($current_iteration + 1) . ")</a>";

if ($iteration_summary) {
    $suggestions_pending = intval($iteration_summary['suggestions_count'] ?? 0) - intval($iteration_summary['suggestions_applied'] ?? 0);
    if ($suggestions_pending > 0) {
        echo "<a href='?action=apply_names&confidence=90' class='btn btn-success' onclick='return confirm(\"Apply all name suggestions with >=90% confidence?\")'>Apply High-Confidence Names (>=90%)</a>";
    }
}

$multi_count = 0;
foreach ($propiedades as $p) {
    if (intval($p['num_deptos'] ?? 1) > 1) $multi_count++;
}
if ($multi_count > 0) {
    echo "<a href='?action=expand_all' class='btn btn-warning' onclick='return confirm(\"Expand ALL $multi_count multi-unit properties?\")'>Expand All Multi-Unit ($multi_count)</a>";
}

echo "</div>";

// Stats
if ($iteration_summary) {
    echo "<div class='stats'>";
    echo "<div class='stat-box'><h3>" . intval($iteration_summary['properties_analyzed']) . "</h3><p>Properties Analyzed</p></div>";
    echo "<div class='stat-box info'><h3>" . round($iteration_summary['avg_confidence'] ?? 0, 1) . "%</h3><p>Avg Confidence</p></div>";
    echo "<div class='stat-box success'><h3>" . intval($iteration_summary['total_reservations']) . "</h3><p>Total Matches</p></div>";
    echo "<div class='stat-box'><h3>" . intval($iteration_summary['total_cb_matches']) . "</h3><p>Cloudbeds</p></div>";
    echo "<div class='stat-box'><h3>" . intval($iteration_summary['total_hf_matches']) . "</h3><p>Hostify</p></div>";
    echo "<div class='stat-box warning'><h3>" . intval($iteration_summary['suggestions_count']) . "</h3><p>Suggestions</p></div>";
    
    $delta = floatval($iteration_summary['avg_delta_confidence'] ?? 0);
    $trend_class = $delta > 0 ? 'success' : ($delta < 0 ? 'danger' : '');
    $trend_icon = $delta > 0 ? '+' : '';
    echo "<div class='stat-box $trend_class'><h3>{$trend_icon}" . round($delta, 1) . "%</h3><p>Delta from Previous</p></div>";
    
    echo "</div>";
} else {
    echo "<div class='alert alert-warning'>No iteration data available. Run analysis first.</div>";
}

// Quick summary table
echo "<h3>Property Summary</h3>";
$single_count = count($propiedades) - $multi_count;
echo "<table style='width: auto;'>";
echo "<tr><td><strong>Total Properties:</strong></td><td>" . count($propiedades) . "</td></tr>";
echo "<tr><td><strong>Single-Unit:</strong></td><td>$single_count</td></tr>";
echo "<tr><td><strong>Multi-Unit (pending expansion):</strong></td><td>$multi_count</td></tr>";
echo "</table>";

echo "</div>"; // End dashboard tab

// ============================================================================
// TAB 2: PROPERTIES ANALYTICS
// ============================================================================

echo "<div id='tab-analytics' class='tab-content'>";
echo "<h2>Properties Analytics</h2>";

if ($display_iteration > 0) {
    // Load analytics for display iteration
    $analytics_data = ia_sqlArrayIndx("
        SELECT a.*, p.nombre_propiedad, p.num_deptos, p.direccion
        FROM propiedad_pms_analytics a
        JOIN propiedad p ON a.propiedad_id = p.propiedad_id
        WHERE a.iteration = $display_iteration
        ORDER BY a.match_confidence_avg DESC
    ");
    
    echo "<p>Showing " . count($analytics_data) . " properties for iteration #$display_iteration";
    if ($is_viewing_history) {
        echo " <span class='badge badge-secondary'>Historical View</span>";
    }
    echo " <small class='detail-text'>(Click any row for details)</small></p>";
    
    echo "<div class='scroll-table'>";
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>Property</th>";
    echo "<th>Units</th>";
    echo "<th>CB Matches</th>";
    echo "<th>HF Matches</th>";
    echo "<th>Confidence</th>";
    echo "<th>Best Tier</th>";
    echo "<th>Discovered Units</th>";
    echo "<th>Suggestion</th>";
    echo "<th>Delta</th>";
    echo "</tr></thead>";
    echo "<tbody>";
    
    foreach ($analytics_data as $a) {
        $conf = floatval($a['match_confidence_avg'] ?? 0);
        $row_class = $conf >= 80 ? 'confidence-high' : ($conf >= 50 ? 'confidence-medium' : ($conf > 0 ? 'confidence-low' : 'confidence-none'));
        
        $tier = $a['match_tier_best'];
        $tier_badge = $tier !== null ? "<span class='badge badge-" . ($tier <= 1 ? 'success' : ($tier <= 2 ? 'info' : 'warning')) . "'>T$tier</span>" : '-';
        
        $discovered = json_decode($a['discovered_units'] ?? '[]', true);
        $units_html = '';
        if (!empty($discovered)) {
            $units_html = "<div class='unit-list'>";
            foreach (array_slice($discovered, 0, 8) as $u) {
                $units_html .= "<span class='unit-chip pms'>$u</span>";
            }
            if (count($discovered) > 8) {
                $units_html .= "<span class='unit-chip'>+" . (count($discovered) - 8) . "</span>";
            }
            $units_html .= "</div>";
        }
        
        $suggestion_html = '';
        if (!empty($a['suggested_name'])) {
            $s_conf = intval($a['suggested_name_confidence']);
            $applied = $a['suggestion_applied'] ? "<span class='badge badge-success'>Applied</span>" : '';
            $suggestion_html = "<small>{$a['suggested_name']}</small><br><span class='badge badge-" . ($s_conf >= 90 ? 'success' : 'warning') . "'>$s_conf%</span> $applied";
        }
        
        $delta = floatval($a['delta_confidence'] ?? 0);
        $delta_html = $delta != 0 ? "<span class='trend-" . ($delta > 0 ? 'up' : 'down') . "'>" . ($delta > 0 ? '+' : '') . round($delta, 1) . "%</span>" : '-';
        
        // Make row clickable to show property detail modal
        $analytics_id_js = esc($a['analytics_id']);
        $propiedad_id_js = esc($a['propiedad_id']);
        $nombre_js = esc(addslashes($a['nombre_propiedad']));
        
        echo "<tr class='$row_class clickable-row' onclick=\"showPropertyDetail('$analytics_id_js', '$propiedad_id_js', '$nombre_js')\">";
        echo "<td><strong>" . esc($a['nombre_propiedad']) . "</strong><br><small class='detail-text'>" . esc($a['direccion']) . "</small></td>";
        echo "<td>" . intval($a['num_deptos']) . "</td>";
        echo "<td>" . intval($a['match_count_cloudbeds']) . "</td>";
        echo "<td>" . intval($a['match_count_hostify']) . "</td>";
        echo "<td><strong>" . round($conf, 1) . "%</strong></td>";
        echo "<td>$tier_badge</td>";
        echo "<td>$units_html</td>";
        echo "<td>$suggestion_html</td>";
        echo "<td>$delta_html</td>";
        echo "</tr>";
    }
    
    echo "</tbody></table>";
    echo "</div>";
} else {
    echo "<div class='alert alert-info'>Run analysis first to see property analytics.</div>";
}

echo "</div>"; // End analytics tab

// ============================================================================
// TAB 3: NAME SUGGESTIONS
// ============================================================================

echo "<div id='tab-suggestions' class='tab-content'>";
echo "<h2>Name Suggestions</h2>";

if ($display_iteration > 0) {
    $suggestions = ia_sqlArrayIndx("
        SELECT a.*, p.nombre_propiedad as current_name, p.nombre_propiedad_original
        FROM propiedad_pms_analytics a
        JOIN propiedad p ON a.propiedad_id = p.propiedad_id
        WHERE a.iteration = $display_iteration
        AND a.suggested_name IS NOT NULL
        ORDER BY a.suggested_name_confidence DESC, a.suggestion_applied ASC
    ");
    
    $pending = array_filter($suggestions, fn($s) => !$s['suggestion_applied']);
    $applied = array_filter($suggestions, fn($s) => $s['suggestion_applied']);
    
    echo "<div class='actions'>";
    if (count($pending) > 0) {
        $high_conf = count(array_filter($pending, fn($s) => intval($s['suggested_name_confidence']) >= 90));
        if ($high_conf > 0) {
            echo "<a href='?action=apply_names&confidence=90' class='btn btn-success'>Apply All >=90% ($high_conf)</a>";
        }
        $med_conf = count(array_filter($pending, fn($s) => intval($s['suggested_name_confidence']) >= 80));
        if ($med_conf > $high_conf) {
            echo "<a href='?action=apply_names&confidence=80' class='btn btn-warning'>Apply All >=80% ($med_conf)</a>";
        }
    }
    echo "</div>";
    
    echo "<h3>Pending Suggestions (" . count($pending) . ")</h3>";
    
    if (!empty($pending)) {
        echo "<table>";
        echo "<thead><tr>";
        echo "<th>Current Name</th>";
        echo "<th>Suggested Name</th>";
        echo "<th>Confidence</th>";
        echo "<th>Reason</th>";
        echo "<th>Action</th>";
        echo "</tr></thead>";
        echo "<tbody>";
        
        foreach ($pending as $s) {
            $conf = intval($s['suggested_name_confidence']);
            $row_class = $conf >= 90 ? 'confidence-high' : ($conf >= 70 ? 'confidence-medium' : 'confidence-low');
            
            echo "<tr class='$row_class'>";
            echo "<td><strong>" . esc($s['current_name']) . "</strong></td>";
            echo "<td><code>" . esc($s['suggested_name']) . "</code></td>";
            echo "<td><span class='badge badge-" . ($conf >= 90 ? 'success' : ($conf >= 70 ? 'warning' : 'danger')) . "'>$conf%</span></td>";
            echo "<td><small>" . esc($s['suggested_name_reason']) . "</small></td>";
            echo "<td><a href='?action=apply_name&propiedad_id=" . urlencode($s['propiedad_id']) . "' class='btn btn-sm btn-success'>Apply</a></td>";
            echo "</tr>";
        }
        
        echo "</tbody></table>";
    } else {
        echo "<p>No pending suggestions.</p>";
    }
    
    if (!empty($applied)) {
        echo "<h3>Applied Suggestions (" . count($applied) . ")</h3>";
        echo "<table>";
        echo "<thead><tr><th>Original Name</th><th>New Name</th><th>Applied At</th></tr></thead>";
        echo "<tbody>";
        foreach (array_slice($applied, 0, 20) as $s) {
            echo "<tr>";
            echo "<td><del>" . esc($s['nombre_propiedad_original'] ?: $s['current_name']) . "</del></td>";
            echo "<td><strong>" . esc($s['suggested_name']) . "</strong></td>";
            echo "<td>" . esc($s['suggestion_applied_at']) . "</td>";
            echo "</tr>";
        }
        echo "</tbody></table>";
    }
} else {
    echo "<div class='alert alert-info'>Run analysis first to generate suggestions.</div>";
}

echo "</div>"; // End suggestions tab

// ============================================================================
// TAB 4: EXPANSION QUEUE
// ============================================================================

echo "<div id='tab-expansion' class='tab-content'>";
echo "<h2>Expansion Queue</h2>";

$multi_units = array_filter($propiedades, fn($p) => intval($p['num_deptos'] ?? 1) > 1);

if (!empty($multi_units)) {
    echo "<div class='actions'>";
    echo "<a href='?action=expand_all' class='btn btn-warning' onclick='return confirm(\"Expand ALL " . count($multi_units) . " multi-unit properties? This cannot be undone.\")'>Expand All (" . count($multi_units) . " buildings)</a>";
    echo "</div>";
    
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>Building</th>";
    echo "<th>num_deptos</th>";
    echo "<th>PMS Units Found</th>";
    echo "<th>Discrepancy</th>";
    echo "<th>Suggested Units</th>";
    echo "<th>Action</th>";
    echo "</tr></thead>";
    echo "<tbody>";
    
    foreach ($multi_units as $p) {
        $analytics = get_latest_analytics($p['propiedad_id']);
        $expansion = null;
        $discovered = [];
        
        if ($analytics) {
            $expansion = json_decode($analytics['suggested_expansion'] ?? 'null', true);
            $discovered = json_decode($analytics['discovered_units'] ?? '[]', true);
        }
        
        $num_deptos = intval($p['num_deptos']);
        $pms_count = count($discovered);
        $discrepancy = $num_deptos - $pms_count;
        
        $disc_class = $discrepancy == 0 ? 'confidence-high' : ($discrepancy <= 2 ? 'confidence-medium' : 'confidence-low');
        $disc_text = $discrepancy == 0 ? '0' : ($discrepancy > 0 ? "-$discrepancy missing" : "+" . abs($discrepancy) . " extra");
        
        // Build units preview
        $units_html = "<div class='unit-list'>";
        if ($expansion && !empty($expansion['units'])) {
            foreach (array_slice($expansion['units'], 0, 10) as $u) {
                $source = $expansion['source'][$u] ?? 'unknown';
                $chip_class = $source === 'pms' ? 'pms' : 'generated';
                $units_html .= "<span class='unit-chip $chip_class'>$u</span>";
            }
            if (count($expansion['units']) > 10) {
                $units_html .= "<span class='unit-chip'>+" . (count($expansion['units']) - 10) . "</span>";
            }
        } else {
            $units_html .= "<span class='detail-text'>Run analysis to generate</span>";
        }
        $units_html .= "</div>";
        
        echo "<tr class='$disc_class'>";
        echo "<td><strong>" . esc($p['nombre_propiedad']) . "</strong></td>";
        echo "<td><strong>$num_deptos</strong></td>";
        echo "<td>$pms_count</td>";
        echo "<td>$disc_text</td>";
        echo "<td>$units_html</td>";
        echo "<td><a href='?action=expand&propiedad_id=" . urlencode($p['propiedad_id']) . "' class='btn btn-sm btn-warning' onclick='return confirm(\"Expand this building into $num_deptos units?\")'>Expand</a></td>";
        echo "</tr>";
    }
    
    echo "</tbody></table>";
    
    echo "<div style='margin-top: 15px;'>";
    echo "<span class='unit-chip pms'>PMS-discovered</span> ";
    echo "<span class='unit-chip generated'>Generated</span>";
    echo "</div>";
} else {
    echo "<div class='alert alert-success'>";
    echo "<strong>All properties are single-unit!</strong> No expansion needed.";
    echo "</div>";
}

echo "</div>"; // End expansion tab

// ============================================================================
// TAB 5: LEARNING HISTORY
// ============================================================================

echo "<div id='tab-history' class='tab-content'>";
echo "<h2>Learning History</h2>";

if (!empty($all_iterations)) {
    echo "<p class='detail-text'>Click any row to view iteration details, or click the iteration number to browse that iteration's data.</p>";
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>Iteration</th>";
    echo "<th>Date</th>";
    echo "<th>Properties</th>";
    echo "<th>Avg Confidence</th>";
    echo "<th>Total Matches</th>";
    echo "<th>Suggestions Applied</th>";
    echo "<th>Expansions</th>";
    echo "<th>Delta</th>";
    echo "<th>Actions</th>";
    echo "</tr></thead>";
    echo "<tbody>";
    
    $prev_conf = null;
    foreach (array_reverse($all_iterations) as $iter) {
        $conf = floatval($iter['avg_confidence'] ?? 0);
        $delta = $iter['avg_delta'] ?? 0;
        $iter_num = intval($iter['iteration']);
        
        $delta_html = '';
        if ($delta != 0) {
            $delta_html = "<span class='trend-" . ($delta > 0 ? 'up' : 'down') . "'>" . ($delta > 0 ? '+' : '') . round($delta, 1) . "%</span>";
        } else {
            $delta_html = "<span class='trend-neutral'>baseline</span>";
        }
        
        $is_current = ($iter_num == $current_iteration);
        $row_style = $is_current ? "background: #e8f4fc;" : "";
        
        echo "<tr class='clickable-row' style='$row_style' onclick=\"showIterationDetail($iter_num)\">";
        echo "<td><strong>#$iter_num</strong>" . ($is_current ? " <span class='badge badge-info'>Current</span>" : "") . "</td>";
        echo "<td>" . date('Y-m-d H:i', strtotime($iter['started_at'])) . "</td>";
        echo "<td>" . $iter['properties_analyzed'] . "</td>";
        echo "<td><strong>" . round($conf, 1) . "%</strong></td>";
        echo "<td>" . $iter['total_reservations'] . "</td>";
        echo "<td>" . intval($iter['suggestions_applied']) . "</td>";
        echo "<td>" . intval($iter['expansions_applied']) . "</td>";
        echo "<td>$delta_html</td>";
        echo "<td onclick='event.stopPropagation()'><a href='?iteration=$iter_num' class='btn btn-sm btn-primary'>View</a></td>";
        echo "</tr>";
        
        $prev_conf = $conf;
    }
    
    echo "</tbody></table>";
    
    // Calculate total improvement
    if (count($all_iterations) >= 2) {
        $first = end($all_iterations);
        $last = reset($all_iterations);
        $total_improvement = floatval($last['avg_confidence'] ?? 0) - floatval($first['avg_confidence'] ?? 0);
        
        echo "<div class='alert alert-" . ($total_improvement >= 0 ? 'success' : 'warning') . "'>";
        echo "<strong>Total Improvement:</strong> ";
        echo ($total_improvement >= 0 ? '+' : '') . round($total_improvement, 1) . "% confidence ";
        echo "over " . count($all_iterations) . " iterations";
        echo "</div>";
    }
} else {
    echo "<div class='alert alert-info'>No iteration history yet. Run your first analysis!</div>";
}

echo "</div>"; // End history tab

?>

<!-- Property Detail Modal -->
<div id="property-modal" class="modal-overlay" onclick="if(event.target===this)closeModal('property-modal')">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="property-modal-title">Property Details</h3>
            <button class="modal-close" onclick="closeModal('property-modal')">&times;</button>
        </div>
        <div class="modal-body" id="property-modal-body">
            <p>Loading...</p>
        </div>
    </div>
</div>

<!-- Iteration Detail Modal -->
<div id="iteration-modal" class="modal-overlay" onclick="if(event.target===this)closeModal('iteration-modal')">
    <div class="modal-content">
        <div class="modal-header iteration-header">
            <h3 id="iteration-modal-title">Iteration Details</h3>
            <button class="modal-close" onclick="closeModal('iteration-modal')">&times;</button>
        </div>
        <div class="modal-body" id="iteration-modal-body">
            <p>Loading...</p>
        </div>
    </div>
</div>

<script>
// ============================================================================
// TAB NAVIGATION
// ============================================================================

function showTab(tabName) {
    document.querySelectorAll('.tab-content').forEach(el => {
        el.classList.remove('active');
    });
    document.querySelectorAll('.tab').forEach(el => {
        el.classList.remove('active');
    });
    document.getElementById('tab-' + tabName).classList.add('active');
    event.target.classList.add('active');
}

// ============================================================================
// MODAL FUNCTIONS
// ============================================================================

function openModal(modalId) {
    document.getElementById(modalId).classList.add('active');
    document.body.style.overflow = 'hidden';
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.remove('active');
    document.body.style.overflow = 'auto';
}

// Close modals on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        document.querySelectorAll('.modal-overlay.active').forEach(m => {
            m.classList.remove('active');
        });
        document.body.style.overflow = 'auto';
    }
});

// ============================================================================
// PROPERTY DETAIL MODAL
// ============================================================================

function showPropertyDetail(analyticsId, propiedadId, nombrePropiedad) {
    openModal('property-modal');
    document.getElementById('property-modal-title').textContent = nombrePropiedad;
    document.getElementById('property-modal-body').innerHTML = '<p style="text-align:center;padding:40px;">Loading property details...</p>';
    
    fetch('?action=get_property_detail&analytics_id=' + encodeURIComponent(analyticsId))
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                renderPropertyDetail(data);
            } else {
                document.getElementById('property-modal-body').innerHTML = 
                    '<div class="alert alert-danger">Error: ' + (data.error || 'Unknown error') + '</div>';
            }
        })
        .catch(err => {
            document.getElementById('property-modal-body').innerHTML = 
                '<div class="alert alert-danger">Error loading data: ' + err.message + '</div>';
        });
}

function renderPropertyDetail(data) {
    let html = '';
    
    // Mini stats
    html += '<div class="mini-stats">';
    html += '<div class="mini-stat success"><h4>' + (data.match_confidence_avg || 0).toFixed(1) + '%</h4><p>Avg Confidence</p></div>';
    html += '<div class="mini-stat cb"><h4>' + (data.match_count_cloudbeds || 0) + '</h4><p>Cloudbeds</p></div>';
    html += '<div class="mini-stat hf"><h4>' + (data.match_count_hostify || 0) + '</h4><p>Hostify</p></div>';
    html += '<div class="mini-stat"><h4>' + (data.total_reservations || 0) + '</h4><p>Total Matches</p></div>';
    if (data.num_deptos > 1) {
        html += '<div class="mini-stat" style="border-color:#ffc107;"><h4>' + data.num_deptos + '</h4><p>Units (Multi)</p></div>';
    }
    html += '</div>';
    
    // Address
    if (data.direccion) {
        html += '<p><strong>Address:</strong> ' + escapeHtml(data.direccion) + '</p>';
    }
    
    // Delta info
    if (data.delta_notes) {
        html += '<p><strong>Change from previous:</strong> ' + escapeHtml(data.delta_notes) + '</p>';
    }
    
    // Discovered units
    if (data.discovered_units && data.discovered_units.length > 0) {
        html += '<div class="modal-section"><h4>Discovered Units (' + data.discovered_units.length + ')</h4>';
        html += '<div class="unit-list">';
        data.discovered_units.forEach(u => {
            html += '<span class="unit-chip pms">' + escapeHtml(u) + '</span>';
        });
        html += '</div></div>';
    }
    
    // Name suggestion
    if (data.suggested_name) {
        html += '<div class="modal-section"><h4>Name Suggestion</h4>';
        html += '<div class="suggestion-box">';
        html += '<div class="new-name">' + escapeHtml(data.suggested_name) + '</div>';
        html += '<span class="badge badge-' + (data.suggested_name_confidence >= 90 ? 'success' : 'warning') + '">' + data.suggested_name_confidence + '% confidence</span>';
        if (data.suggestion_applied) {
            html += ' <span class="badge badge-success">Applied</span>';
        }
        if (data.suggested_name_reason) {
            html += '<p style="margin-top:10px;margin-bottom:0;font-size:0.9em;color:#666;">' + escapeHtml(data.suggested_name_reason) + '</p>';
        }
        html += '</div></div>';
    }
    
    // Cloudbeds matches
    if (data.cloudbeds_matches && data.cloudbeds_matches.length > 0) {
        html += '<div class="modal-section"><h4>Cloudbeds Matches (' + data.cloudbeds_matches.length + ')</h4>';
        let shown = Math.min(data.cloudbeds_matches.length, 15);
        data.cloudbeds_matches.slice(0, shown).forEach(m => {
            html += '<div class="match-card cloudbeds">';
            html += '<div class="match-info">';
            html += '<strong>' + escapeHtml(m.property || '') + '</strong>';
            html += '<div class="match-meta">Room: ' + escapeHtml(m.room_number || 'N/A');
            if (m.discovered_unit) html += ' | Unit: ' + escapeHtml(m.discovered_unit);
            html += '</div></div>';
            html += '<span class="badge badge-' + (m.confidence >= 80 ? 'success' : (m.confidence >= 50 ? 'warning' : 'danger')) + '">' + m.confidence + '%</span>';
            html += '</div>';
        });
        if (data.cloudbeds_matches.length > shown) {
            html += '<p class="detail-text">...and ' + (data.cloudbeds_matches.length - shown) + ' more matches</p>';
        }
        html += '</div>';
    }
    
    // Hostify matches
    if (data.hostify_matches && data.hostify_matches.length > 0) {
        html += '<div class="modal-section"><h4>Hostify Matches (' + data.hostify_matches.length + ')</h4>';
        let shown = Math.min(data.hostify_matches.length, 15);
        data.hostify_matches.slice(0, shown).forEach(m => {
            html += '<div class="match-card hostify">';
            html += '<div class="match-info">';
            html += '<strong>' + escapeHtml(m.anuncio || '') + '</strong>';
            html += '<div class="match-meta">';
            if (m.combo_type) html += 'Combo: ' + escapeHtml(m.combo_type) + ' | ';
            if (m.discovered_unit) html += 'Unit: ' + escapeHtml(m.discovered_unit);
            html += '</div></div>';
            html += '<span class="badge badge-' + (m.confidence >= 80 ? 'success' : (m.confidence >= 50 ? 'warning' : 'danger')) + '">' + m.confidence + '%</span>';
            if (m.combo_type) {
                html += ' <span class="badge badge-purple">Combo</span>';
            }
            html += '</div>';
        });
        if (data.hostify_matches.length > shown) {
            html += '<p class="detail-text">...and ' + (data.hostify_matches.length - shown) + ' more matches</p>';
        }
        html += '</div>';
    }
    
    // Expansion suggestion
    if (data.suggested_expansion && data.suggested_expansion.units) {
        html += '<div class="modal-section"><h4>Expansion Suggestion</h4>';
        html += '<p>This property has <strong>' + data.num_deptos + '</strong> units. ';
        html += 'PMS data found <strong>' + data.suggested_expansion.pms_found + '</strong> units.</p>';
        if (data.suggested_expansion.discrepancy != 0) {
            html += '<p class="alert alert-warning">Discrepancy: ' + data.suggested_expansion.discrepancy + ' units</p>';
        }
        html += '<div class="unit-list">';
        data.suggested_expansion.units.forEach(u => {
            let source = data.suggested_expansion.source ? data.suggested_expansion.source[u] : 'unknown';
            let chipClass = source === 'pms' ? 'pms' : 'generated';
            html += '<span class="unit-chip ' + chipClass + '">' + escapeHtml(u) + '</span>';
        });
        html += '</div>';
        html += '<p class="detail-text" style="margin-top:10px;"><span class="unit-chip pms">PMS</span> = discovered from reservations, <span class="unit-chip generated">Generated</span> = auto-generated</p>';
        html += '</div>';
    }
    
    // Footer with metadata
    html += '<hr style="margin:20px 0;border-color:#ecf0f1;">';
    html += '<p class="detail-text">Iteration #' + data.iteration + ' | Analyzed: ' + escapeHtml(data.analyzed_at || 'N/A') + '</p>';
    
    document.getElementById('property-modal-body').innerHTML = html;
}

// ============================================================================
// ITERATION DETAIL MODAL
// ============================================================================

function showIterationDetail(iteration) {
    openModal('iteration-modal');
    document.getElementById('iteration-modal-title').textContent = 'Iteration #' + iteration + ' Details';
    document.getElementById('iteration-modal-body').innerHTML = '<p style="text-align:center;padding:40px;">Loading iteration details...</p>';
    
    fetch('?action=get_iteration_detail&iteration=' + iteration)
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                renderIterationDetail(data, iteration);
            } else {
                document.getElementById('iteration-modal-body').innerHTML = 
                    '<div class="alert alert-danger">Error: ' + (data.error || 'Unknown error') + '</div>';
            }
        })
        .catch(err => {
            document.getElementById('iteration-modal-body').innerHTML = 
                '<div class="alert alert-danger">Error loading data: ' + err.message + '</div>';
        });
}

function renderIterationDetail(data, iteration) {
    let html = '';
    
    // Stats
    html += '<div class="mini-stats">';
    html += '<div class="mini-stat"><h4>' + data.properties_analyzed + '</h4><p>Properties</p></div>';
    html += '<div class="mini-stat success"><h4>' + (data.avg_confidence || 0).toFixed(1) + '%</h4><p>Avg Confidence</p></div>';
    html += '<div class="mini-stat cb"><h4>' + data.total_cb_matches + '</h4><p>CB Matches</p></div>';
    html += '<div class="mini-stat hf"><h4>' + data.total_hf_matches + '</h4><p>HF Matches</p></div>';
    html += '<div class="mini-stat"><h4>' + data.total_reservations + '</h4><p>Total</p></div>';
    html += '</div>';
    
    // Suggestions summary
    html += '<p><strong>Suggestions:</strong> ' + data.suggestions_count + ' generated, ' + data.suggestions_applied + ' applied</p>';
    html += '<p><strong>Expansions:</strong> ' + data.expansions_applied + ' applied</p>';
    html += '<p class="detail-text">Analyzed: ' + escapeHtml(data.started_at || '') + ' - ' + escapeHtml(data.completed_at || '') + '</p>';
    
    // Top properties
    if (data.top_properties && data.top_properties.length > 0) {
        html += '<div class="modal-section"><h4>Top Properties by Confidence</h4>';
        html += '<table style="font-size:0.9em;"><thead><tr><th>Property</th><th>Confidence</th><th>CB</th><th>HF</th></tr></thead><tbody>';
        data.top_properties.forEach(p => {
            html += '<tr>';
            html += '<td>' + escapeHtml(p.nombre_propiedad) + '</td>';
            html += '<td><strong>' + (parseFloat(p.match_confidence_avg) || 0).toFixed(1) + '%</strong></td>';
            html += '<td>' + p.match_count_cloudbeds + '</td>';
            html += '<td>' + p.match_count_hostify + '</td>';
            html += '</tr>';
        });
        html += '</tbody></table></div>';
    }
    
    // Unmatched/low confidence
    if (data.unmatched && data.unmatched.length > 0) {
        html += '<div class="modal-section"><h4>Low Confidence Properties</h4>';
        html += '<table style="font-size:0.9em;"><thead><tr><th>Property</th><th>Confidence</th></tr></thead><tbody>';
        data.unmatched.forEach(p => {
            let conf = parseFloat(p.match_confidence_avg) || 0;
            html += '<tr class="confidence-low">';
            html += '<td>' + escapeHtml(p.nombre_propiedad) + '</td>';
            html += '<td>' + conf.toFixed(1) + '%</td>';
            html += '</tr>';
        });
        html += '</tbody></table></div>';
    }
    
    // Pending suggestions
    if (data.suggestions_pending && data.suggestions_pending.length > 0) {
        html += '<div class="modal-section"><h4>Pending Name Suggestions</h4>';
        html += '<table style="font-size:0.9em;"><thead><tr><th>Current</th><th>Suggested</th><th>Conf</th></tr></thead><tbody>';
        data.suggestions_pending.forEach(s => {
            html += '<tr>';
            html += '<td>' + escapeHtml(s.nombre_propiedad) + '</td>';
            html += '<td><code>' + escapeHtml(s.suggested_name) + '</code></td>';
            html += '<td><span class="badge badge-' + (s.suggested_name_confidence >= 90 ? 'success' : 'warning') + '">' + s.suggested_name_confidence + '%</span></td>';
            html += '</tr>';
        });
        html += '</tbody></table></div>';
    }
    
    // Action button
    html += '<div style="margin-top:25px;text-align:center;">';
    html += '<a href="?iteration=' + iteration + '" class="btn btn-primary">Browse Full Iteration Data</a>';
    html += '</div>';
    
    document.getElementById('iteration-modal-body').innerHTML = html;
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
</script>

</div>
</body>
</html>
