<?php
/**
 * Scholarship Student CSV Export Helper
 * Purpose: Export scholarship student records to CSV format with preview functionality
 * Security: SQL injection prevention via parameterized queries
 * UI: Preview workflow mirroring link_propiedades_propietarios.php styling
 */

require_once("../../inc/config.php");

// ============================================================================
// CONFIGURATION
// ============================================================================

define('EXPORT_LIMIT', 1000);
$action = $_GET['action'] ?? 'preview';
$debug = isset($_GET['debug']);

// Safe HTML escaping helper
function esc($value) {
    return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>📊 Scholarship Student CSV Export</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .container {
            max-width: 1600px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            border-bottom: 4px solid #667eea;
            padding-bottom: 15px;
            font-size: 2.2em;
            margin-top: 0;
        }
        h1 .emoji { font-size: 1.2em; }
        .subtitle {
            color: #764ba2;
            font-style: italic;
            margin-top: -10px;
            margin-bottom: 20px;
        }
        .stats {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }
        .stat-box {
            background: white;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            border-left: 4px solid #667eea;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stat-box.records { border-color: #ffc107; }
        .stat-box.exported { border-color: #28a745; }
        .stat-box h3 { margin: 0 0 5px 0; font-size: 2.2em; color: #667eea; }
        .stat-box p { margin: 0; color: #7f8c8d; font-size: 0.85em; }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 0.85em;
        }
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 8px;
            text-align: left;
            position: sticky;
            top: 0;
            font-size: 0.9em;
        }
        td {
            padding: 10px 8px;
            border-bottom: 1px solid #ecf0f1;
        }
        tr:hover { background: #f8f9fa; }

        /* Confidence-based row colors */
        .confidence-100 { background-color: #d4edda !important; border-left: 4px solid #28a745; }

        /* Tier badges */
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: bold;
            color: white;
        }
        .badge-preview { background: #ffc107; }
        .badge-exported { background: #28a745; }

        /* Action buttons */
        .actions {
            margin: 30px 0;
            padding: 20px;
            background: linear-gradient(135deg, #e8f4f8 0%, #f3e7f9 100%);
            border-radius: 8px;
            border: 2px solid #667eea;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            margin: 5px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1em;
            text-decoration: none;
            transition: all 0.3s;
            font-weight: bold;
        }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5568d3; transform: translateY(-2px); }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; transform: translateY(-2px); }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-warning:hover { background: #e0a800; transform: translateY(-2px); }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; transform: translateY(-2px); }

        /* Alert boxes */
        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success { background: #d4edda; border-color: #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border-color: #17a2b8; color: #0c5460; }
        .alert-warning { background: #fff3cd; border-color: #ffc107; color: #856404; }
        .alert-danger { background: #f8d7da; border-color: #dc3545; color: #721c24; }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }

        /* Debug mode styling */
        .debug-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            border-radius: 6px;
            font-size: 0.85em;
        }
        .debug-box pre {
            background: white;
            padding: 10px;
            border-radius: 4px;
            overflow-x: auto;
            max-height: 300px;
        }
    </style>
</head>
<body>
<div class="container">
    <h1><span class="emoji">📊</span> Scholarship Student CSV Export</h1>
    <p class="subtitle">Export scholarship student records to CSV format</p>

    <?php if (!$debug): ?>
    <div class="alert alert-info">
        💡 <strong>Tip:</strong> Add <code>?debug=1</code> to URL for detailed export diagnostics
    </div>
    <?php endif; ?>

<?php
// ============================================================================
// DATABASE QUERY FUNCTIONS WITH PARAMETERIZATION
// ============================================================================

/**
 * Fetch scholarship student records with parameterized query
 */
function fetch_student_records($limit = EXPORT_LIMIT) {
    global $db;
    
    $sql = "
        SELECT estudiante_id, beca_tipo, porcentaje_beca_colegiatura, porcentaje_beca_inscripcion
        FROM alumnos_becados";
    
    $stmt = ia_prepare($sql);
    ia_execute($stmt);
    return ia_sqlArrayIndx($stmt);
}

// ============================================================================
// MAIN EXECUTION PATHS
// ============================================================================

// Preview mode (default)
if ($action === 'preview') {
    $records = fetch_student_records();
    
    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Preview Mode:</strong> ";
    echo count($records) . " scholarship student records found";
    echo "</div>";
    
    if (!empty($records)) {
        echo "<div class='stats'>";
        echo "<div class='stat-box records'><h3>" . count($records) . "</h3><p>Records</p></div>";
        echo "</div>";
        
        echo "<h2>Scholarship Student Records</h2>";
        echo "<table>";
        echo "<thead><tr>";
        echo "<th>Estudiante</th>";
        echo "<th>Beca Tipo</th>";
        echo "<th>Colegiatura %</th>";
        echo "<th>Inscripción %</th>";
        echo "</tr></thead>";
        echo "<tbody>";
        
        foreach ($records as $record) {
            echo "<tr>";
            echo "<td>" . esc($record['estudiante_id']) . "</td>";
            echo "<td>" . esc($record['beca_tipo']) . "</td>";
            echo "<td>" . esc($record['porcentaje_beca_colegiatura']) . "</td>";
            echo "<td>" . esc($record['porcentaje_beca_inscripcion']) . "</td>";
            echo "</tr>";
        }
        
        echo "</tbody></table>";
    } else {
        echo "<div class='alert alert-success'>";
        echo "<strong>✓ Success:</strong> No scholarship student records found";
        echo "</div>";
    }
    
    echo "<div class='actions'>";
    echo "<h3>Actions</h3>";
    echo "<a href='?action=export' class='btn btn-success'>📥 Export CSV</a>";
    echo "</div>";
    
    if ($debug) {
        echo "<div class='debug-box'>";
        echo "<strong>🐛 DEBUG MODE</strong><br>";
        echo "<strong>Records:</strong><pre>";
        print_r($records);
        echo "</pre>";
        echo "</div>";
    }
    
    echo "<div class='alert alert-info'>";
    echo "<strong>💡 Export Logic:</strong><br>";
    echo "• Fetches all scholarship student records<br>";
    echo "• Formats data for CSV export<br>";
    echo "• Includes all scholarship percentage fields";
    echo "</div>";
}

// Export mode
elseif ($action === 'export') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename=scholarship_students_' . date('Ymd') . '.csv');
    
    $output = fopen('php://output', 'w');
    
    // CSV Header
    fputcsv($output, [
        'Estudiante',
        'Beca Tipo',
        'Colegiatura %',
        'Inscripción %'
    ]);
    
    // CSV Data
    $records = fetch_student_records();
    
    foreach ($records as $record) {
        fputcsv($output, [
            esc($record['estudiante_id']),
            esc($record['beca_tipo']),
            esc($record['porcentaje_beca_colegiatura']),
            esc($record['porcentaje_beca_inscripcion'])
        ]);
    }
    
    fclose($output);
    exit;
}

?>
</div>
</body>
</html>