<?php
/**
 * Iteration 6 - Fuzzy Name Matching for Edge Cases
 */

require_once("../../inc/config.php");

echo "<!DOCTYPE html><html><head><title>Iteration 6</title></head><body>";
echo "<h1>Iteration 6 - Fuzzy Name Matching</h1>";
echo "<p><strong>New Patterns Added:</strong></p>";
echo "<ul>";
echo "<li>✓ Fuzzy name matching (Casa Pani → Yautepec 24 Pani)</li>";
echo "<li>✓ Partial name matching (Av Mexico Ana → Av. Mexico)</li>";
echo "<li>✓ Single word property names (simple contains match)</li>";
echo "</ul>";
echo "<hr>";

// Normalize with accent removal
function normalize_iter6($text) {
    $accents = ['Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
                'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
                'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'];
    $text = strtr($text, $accents);
    $text = mb_strtolower($text, 'UTF-8');
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $text;
}

// Extract primary name from anuncio (before | or -)
function extract_primary_name($text) {
    // Remove codes like "CoG2BrKK-B", "RoS1BQSS" etc
    $text = preg_replace('/\b[A-Z][a-z]{0,2}[A-Z0-9]{2,}\b/', '', $text);

    // Get text before | or first -
    if (strpos($text, '|') !== false) {
        $text = substr($text, 0, strpos($text, '|'));
    }

    $text = trim($text);
    return normalize_iter6($text);
}

// Fuzzy name matcher
function match_by_name_iter6($text, &$propiedades) {
    $primary_name = extract_primary_name($text);

    if (strlen($primary_name) < 5) {
        return ['match' => false];
    }

    // Special cases first
    $special_cases = [
        'casa pani' => 'yautepec 24 pani',
        'av mexico ana' => 'av. mexico',
        'av mexico' => 'av. mexico',
    ];

    foreach ($special_cases as $search => $target) {
        if (strpos($primary_name, $search) !== false) {
            foreach ($propiedades as $prop) {
                $prop_norm = normalize_iter6($prop['nombre_propiedad']);
                if (strpos($prop_norm, $target) !== false) {
                    return ['match' => true, 'prop' => $prop, 'confidence' => 75, 'pattern' => 'name_fuzzy_special'];
                }
            }
        }
    }

    // General fuzzy matching - look for name in property
    foreach ($propiedades as $prop) {
        $prop_norm = normalize_iter6($prop['nombre_propiedad']);

        // Check if primary name appears in property name
        if (strlen($primary_name) >= 8 && strpos($prop_norm, $primary_name) !== false) {
            return ['match' => true, 'prop' => $prop, 'confidence' => 70, 'pattern' => 'name_fuzzy_contains'];
        }

        // Or if property name appears in primary name
        $prop_words = explode(' ', $prop_norm);
        $prop_main = '';
        foreach ($prop_words as $word) {
            if (strlen($word) >= 5) {  // Significant word
                $prop_main .= $word . ' ';
            }
        }
        $prop_main = trim($prop_main);

        if (strlen($prop_main) >= 8 && strpos($primary_name, $prop_main) !== false) {
            return ['match' => true, 'prop' => $prop, 'confidence' => 70, 'pattern' => 'name_fuzzy_reverse'];
        }
    }

    return ['match' => false];
}

// Load propiedades
$sql_props = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_props);

echo "<p>✓ Loaded " . count($propiedades) . " properties</p>";

// Process only unmatched
$sql_hostify = "SELECT *
                FROM hostify_reserva
                WHERE check_in >= '2025-01-01'
                  AND propiedad_id IS NULL";
$unmatched_hf = ia_sqlArrayIndx($sql_hostify);

echo "<p>✓ Found " . count($unmatched_hf) . " unmatched Hostify reservations</p>";
echo "<p><strong>Applying fuzzy name matching...</strong></p>";

$newly_matched = 0;
$pattern_counts = [];

echo "<h3>Matches Found:</h3>";
echo "<table border='1' cellpadding='5' style='border-collapse: collapse; font-size: 0.9em;'>";
echo "<tr><th>Anuncio</th><th>Pattern</th><th>Matched Property</th></tr>";

foreach ($unmatched_hf as $reserva) {
    $result = match_by_name_iter6($reserva['anuncio'], $propiedades);

    if ($result['match']) {
        $propiedad_id = $result['prop']['propiedad_id'];
        $confidence = $result['confidence'];
        $pattern = $result['pattern'];
        $reserva_id = $reserva['hostify_reserva_id'];

        $sql_update = "UPDATE hostify_reserva
                      SET propiedad_id = " . strit($propiedad_id) . ",
                          match_tier = 3,
                          match_confidence = $confidence,
                          match_pattern = " . strit($pattern) . ",
                          match_timestamp = NOW()
                      WHERE hostify_reserva_id = " . strit($reserva_id);
        ia_query($sql_update);

        $newly_matched++;
        $pattern_counts[$pattern] = ($pattern_counts[$pattern] ?? 0) + 1;

        echo "<tr>";
        echo "<td>{$reserva['anuncio']}</td>";
        echo "<td style='color: blue;'><strong>$pattern</strong></td>";
        echo "<td>{$result['prop']['nombre_propiedad']}</td>";
        echo "</tr>";
    }
}

echo "</table>";

echo "<hr>";
echo "<h2>📊 Iteration 6 Results:</h2>";
echo "<ul>";
echo "<li style='color: green;'><strong>Newly Matched:</strong> $newly_matched</li>";
echo "</ul>";

if (!empty($pattern_counts)) {
    echo "<h3>Pattern Usage:</h3>";
    echo "<table border='1' cellpadding='8'>";
    echo "<tr><th>Pattern</th><th>Matches</th></tr>";
    foreach ($pattern_counts as $pattern => $count) {
        echo "<tr><td><strong>$pattern</strong></td><td style='text-align: center;'>$count</td></tr>";
    }
    echo "</table>";
}

// Get final stats
$sql_hf = "SELECT COUNT(*) as total,
           SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as matched,
           SUM(CASE WHEN match_confidence >= 80 THEN 1 ELSE 0 END) as high_conf
           FROM hostify_reserva
           WHERE check_in >= '2025-01-01'";
$hf_stats = ia_sqlArrayIndx($sql_hf)[0];

$sql_cb = "SELECT COUNT(*) as total,
           SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as matched,
           SUM(CASE WHEN match_confidence >= 80 THEN 1 ELSE 0 END) as high_conf
           FROM cloudbeds_reserva
           WHERE check_in_date >= '2025-01-01'";
$cb_stats = ia_sqlArrayIndx($sql_cb)[0];

$total = $cb_stats['total'] + $hf_stats['total'];
$matched = $cb_stats['matched'] + $hf_stats['matched'];
$unmatched = $total - $matched;
$high_conf = $cb_stats['high_conf'] + $hf_stats['high_conf'];
$match_rate = $total > 0 ? ($matched / $total) * 100 : 0;

echo "<hr>";
echo "<h2>🎯 Final System State:</h2>";
echo "<table border='1' cellpadding='10'>";
echo "<tr><th>Metric</th><th>Value</th></tr>";
echo "<tr><td>Total Reservations</td><td>$total</td></tr>";
echo "<tr><td>Matched</td><td>$matched</td></tr>";
echo "<tr><td>Unmatched</td><td>$unmatched</td></tr>";
echo "<tr><td><strong>Match Rate</strong></td><td><strong>" . number_format($match_rate, 2) . "%</strong></td></tr>";
echo "<tr><td><strong>High Confidence (≥80%)</strong></td><td><strong>$high_conf</strong></td></tr>";
if ($newly_matched > 0) {
    echo "<tr><td style='color: green;'><strong>Gain vs Iteration 5</strong></td><td style='color: green;'><strong>+$newly_matched</strong></td></tr>";
}
echo "</table>";

echo "<h2>✅ Iteration 6 Complete!</h2>";

if ($newly_matched > 0) {
    echo "<p><strong>Fuzzy Name Matching Impact:</strong> Successfully matched $newly_matched reservations with property name similarity!</p>";
} else {
    echo "<p><strong>Analysis:</strong> Fuzzy name matching didn't find additional matches. Most remaining unmatched are invalid/missing properties.</p>";
}

echo "<p><strong>Cumulative Gain:</strong> From 420 (baseline) to $high_conf (+" . ($high_conf - 420) . ")</p>";
echo "<p><strong>Final Match Rate:</strong> " . number_format($match_rate, 2) . "% of 2,153 reservations</p>";

// Show breakdown of remaining unmatched
echo "<hr>";
echo "<h2>🔍 Remaining Unmatched Breakdown:</h2>";

$sql_remaining = "SELECT anuncio, COUNT(*) as count
                  FROM hostify_reserva
                  WHERE check_in >= '2025-01-01'
                    AND propiedad_id IS NULL
                  GROUP BY anuncio
                  ORDER BY count DESC
                  LIMIT 10";
$remaining = ia_sqlArrayIndx($sql_remaining);

if (count($remaining) > 0) {
    echo "<p><strong>Top unmatched (out of $unmatched total):</strong></p>";
    echo "<table border='1' cellpadding='8'>";
    echo "<tr><th>Anuncio</th><th>Count</th><th>Reason</th></tr>";

    foreach ($remaining as $row) {
        echo "<tr>";
        echo "<td>{$row['anuncio']}</td>";
        echo "<td>{$row['count']}</td>";
        echo "<td>";

        if (strpos($row['anuncio'], 'P.E.21') !== false) {
            echo "<span style='color: red;'>❌ Property does not exist</span>";
        } elseif (strpos($row['anuncio'], '1111 Reservas') !== false) {
            echo "<span style='color: gray;'>Dummy/test data</span>";
        } elseif (strpos($row['anuncio'], 'Ver 4') !== false || strpos($row['anuncio'], 'Ver PH 7') !== false) {
            echo "<span style='color: orange;'>⚠️ Missing from database</span>";
        } elseif (strpos($row['anuncio'], 'Campeche') !== false) {
            echo "<span style='color: red;'>❌ Property does not exist</span>";
        } else {
            echo "<span style='color: orange;'>Unknown/missing property</span>";
        }

        echo "</td>";
        echo "</tr>";
    }

    echo "</table>";
}

echo "</body></html>";
?>
