<?php
/**
 * Iteration 9 - Final Property Matching
 *
 * New Patterns:
 * - Ver → Veracruz/Tigre abbreviation mapping (Ver 4 → Tigre 4 | Veracruz 26 | 4)
 * - Enhanced Campeche → Casa direct mapping (Campeche Ana → Casa Ana)
 *
 * Expected Impact: +21 matches (97.21% → 98.19%)
 * - Ver 4 variants: +7 matches
 * - Ver PH 7 variants: +11 matches
 * - Campeche Ana: +3 matches
 */

require_once("../../inc/config.php");

echo "<!DOCTYPE html><html><head><title>Iteration 9 - Final Match</title></head><body>";
echo "<h1>Iteration 9 - Ver/Campeche Property Matching</h1>";
echo "<p><strong>New Patterns Added:</strong></p>";
echo "<ul>";
echo "<li>✓ Ver → Veracruz/Tigre abbreviation mapping (Ver 4 → Tigre 4)</li>";
echo "<li>✓ Enhanced Campeche → Casa direct name mapping (Campeche Ana → Casa Ana)</li>";
echo "<li>✓ Handles Ver PH# (Penthouse) format</li>";
echo "</ul>";
echo "<hr>";

// Normalize with accent removal
function normalize_iter9($text) {
    $accents = ['Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
                'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
                'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'];
    $text = strtr($text, $accents);
    $text = mb_strtolower($text, 'UTF-8');
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $text;
}

// Extract clean name (remove codes like CoG2BrKK-B, RoS1BQSS)
function extract_clean_name_iter9($text) {
    // Remove codes like "CoG2BrKK-B", "RoS1BQSS", etc.
    $text = preg_replace('/\b[A-Z][a-z]{0,2}[A-Z0-9]{2,}[A-Z0-9\-]*\b/', '', $text);

    // Remove trailing descriptors
    $text = preg_replace('/\s*-\s*(Bcom|Vrbo|PT|BT|B)$/', '', $text);

    // Get text before | or first -
    if (strpos($text, '|') !== false) {
        $text = substr($text, 0, strpos($text, '|'));
    }

    $text = trim($text);
    return normalize_iter9($text);
}

// Iteration 9 matcher
function match_iter9($text, &$propiedades) {
    $clean_name = extract_clean_name_iter9($text);
    $norm = normalize_iter9($text);

    // PATTERN 9.1: Ver → Veracruz/Tigre abbreviation
    // Matches: "Ver 4" → "Tigre 4 | Veracruz 26 | 4"
    // Matches: "Ver PH 7" → "Tigre PH 7 | Veracruz 26 | PH 7"
    if (preg_match('/^ver\s+(ph\s+)?(\d+)/i', $clean_name, $matches)) {
        $unit = $matches[2];  // Extract unit number (4, 7, etc.)
        $is_ph = !empty($matches[1]);  // Check if Penthouse

        foreach ($propiedades as $prop) {
            $prop_norm = normalize_iter9($prop['nombre_propiedad']);

            // Match Veracruz 26 properties with matching unit number
            if (strpos($prop_norm, 'veracruz') !== false) {
                // Check for unit number match
                $unit_match = false;

                if ($is_ph) {
                    // For Penthouse, look for "ph 7" or "ph7" pattern
                    if (preg_match('/ph\s*' . $unit . '\b/', $prop_norm)) {
                        $unit_match = true;
                    }
                } else {
                    // For regular units, look for "| 4" or similar
                    if (preg_match('/\|\s*' . $unit . '\b/', $prop_norm) ||
                        preg_match('/tigre\s+' . $unit . '\b/', $prop_norm)) {
                        $unit_match = true;
                    }
                }

                if ($unit_match) {
                    return [
                        'match' => true,
                        'prop' => $prop,
                        'confidence' => 90,
                        'pattern' => 'ver_veracruz_abbr'
                    ];
                }
            }
        }
    }

    // PATTERN 9.2: Campeche → Casa direct mapping
    // Matches: "Campeche Ana" → "Casa Ana"
    // Matches: "Campeche Frida" → "Casa Frida"
    if (preg_match('/campeche\s+(\w+)/i', $clean_name, $matches)) {
        $descriptor = mb_strtolower($matches[1], 'UTF-8');  // Extract name (ana, frida, etc.)

        foreach ($propiedades as $prop) {
            $prop_norm = normalize_iter9($prop['nombre_propiedad']);

            // Direct match: "casa ana", "casa frida", etc.
            if (preg_match('/casa\s+' . preg_quote($descriptor, '/') . '\b/', $prop_norm)) {
                return [
                    'match' => true,
                    'prop' => $prop,
                    'confidence' => 85,
                    'pattern' => 'campeche_casa_direct'
                ];
            }
        }
    }

    return ['match' => false];
}

// Load propiedades
$sql_props = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_props);

echo "<p>✓ Loaded " . count($propiedades) . " properties</p>";

// Process only unmatched
$sql_hostify = "SELECT *
                FROM hostify_reserva
                WHERE check_in >= '2025-01-01'
                  AND propiedad_id IS NULL";
$unmatched_hf = ia_sqlArrayIndx($sql_hostify);

echo "<p>✓ Found " . count($unmatched_hf) . " unmatched Hostify reservations</p>";
echo "<p><strong>Applying Ver/Campeche pattern matching...</strong></p>";

$newly_matched = 0;
$pattern_counts = [];

echo "<h3>Matches Found:</h3>";
echo "<table border='1' cellpadding='5' style='border-collapse: collapse; font-size: 0.9em;'>";
echo "<tr><th>Anuncio</th><th>Pattern</th><th>Matched Property</th><th>Confidence</th></tr>";

foreach ($unmatched_hf as $reserva) {
    $result = match_iter9($reserva['anuncio'], $propiedades);

    if ($result['match']) {
        $propiedad_id = $result['prop']['propiedad_id'];
        $confidence = $result['confidence'];
        $pattern = $result['pattern'];
        $reserva_id = $reserva['hostify_reserva_id'];

        $sql_update = "UPDATE hostify_reserva
                      SET propiedad_id = " . strit($propiedad_id) . ",
                          match_tier = 2,
                          match_confidence = $confidence,
                          match_pattern = " . strit($pattern) . ",
                          match_timestamp = NOW()
                      WHERE hostify_reserva_id = " . strit($reserva_id);
        ia_query($sql_update);

        $newly_matched++;
        $pattern_counts[$pattern] = ($pattern_counts[$pattern] ?? 0) + 1;

        echo "<tr>";
        echo "<td>{$reserva['anuncio']}</td>";
        echo "<td style='color: blue;'><strong>$pattern</strong></td>";
        echo "<td>{$result['prop']['nombre_propiedad']}</td>";
        echo "<td>$confidence%</td>";
        echo "</tr>";
    }
}

echo "</table>";

echo "<hr>";
echo "<h2>📊 Iteration 9 Results:</h2>";
echo "<ul>";
echo "<li style='color: green;'><strong>Newly Matched:</strong> $newly_matched</li>";
echo "</ul>";

if (!empty($pattern_counts)) {
    echo "<h3>Pattern Usage:</h3>";
    echo "<table border='1' cellpadding='8'>";
    echo "<tr><th>Pattern</th><th>Matches</th></tr>";
    foreach ($pattern_counts as $pattern => $count) {
        echo "<tr><td><strong>$pattern</strong></td><td style='text-align: center;'>$count</td></tr>";
    }
    echo "</table>";
}

// Get final stats
$sql_hf = "SELECT COUNT(*) as total,
           SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as matched,
           SUM(CASE WHEN match_confidence >= 80 THEN 1 ELSE 0 END) as high_conf
           FROM hostify_reserva
           WHERE check_in >= '2025-01-01'";
$hf_stats = ia_sqlArrayIndx($sql_hf)[0];

$sql_cb = "SELECT COUNT(*) as total,
           SUM(CASE WHEN propiedad_id IS NOT NULL THEN 1 ELSE 0 END) as matched,
           SUM(CASE WHEN match_confidence >= 80 THEN 1 ELSE 0 END) as high_conf
           FROM cloudbeds_reserva
           WHERE check_in_date >= '2025-01-01'";
$cb_stats = ia_sqlArrayIndx($sql_cb)[0];

$total = $cb_stats['total'] + $hf_stats['total'];
$matched = $cb_stats['matched'] + $hf_stats['matched'];
$unmatched = $total - $matched;
$high_conf = $cb_stats['high_conf'] + $hf_stats['high_conf'];
$match_rate = $total > 0 ? ($matched / $total) * 100 : 0;

echo "<hr>";
echo "<h2>🎯 Final System State:</h2>";
echo "<table border='1' cellpadding='10'>";
echo "<tr><th>Metric</th><th>Value</th></tr>";
echo "<tr><td>Total Reservations</td><td>$total</td></tr>";
echo "<tr><td>Matched</td><td>$matched</td></tr>";
echo "<tr><td>Unmatched</td><td>$unmatched</td></tr>";
echo "<tr><td><strong>Match Rate</strong></td><td><strong>" . number_format($match_rate, 2) . "%</strong></td></tr>";
echo "<tr><td><strong>High Confidence (≥80%)</strong></td><td><strong>$high_conf</strong></td></tr>";
if ($newly_matched > 0) {
    echo "<tr><td style='color: green;'><strong>Gain vs Iteration 8</strong></td><td style='color: green;'><strong>+$newly_matched</strong></td></tr>";
}
echo "</table>";

echo "<h2>✅ Iteration 9 Complete!</h2>";

if ($newly_matched > 0) {
    echo "<p style='color: green; font-size: 1.2em;'><strong>🎉 SUCCESS!</strong> Matched $newly_matched reservations with Ver/Campeche patterns!</p>";
    echo "<p><strong>Impact:</strong> Added properties (Ver Unit 4, Ver PH 7, Casa Ana) are now matching their reservations!</p>";
} else {
    echo "<p><strong>Analysis:</strong> Ver/Campeche patterns didn't find matches (properties may not be in database yet).</p>";
}

echo "<p><strong>Total Journey:</strong> From 88.16% (Iteration 1) to " . number_format($match_rate, 2) . "% (Iteration 9)</p>";
echo "<p><strong>Total Improvement:</strong> +" . ($matched - 1898) . " matches (+\" . number_format($match_rate - 88.16, 2) . \"pp)</p>";

// Show remaining unmatched breakdown
echo "<hr>";
echo "<h2>🔍 Remaining Unmatched Analysis:</h2>";

$sql_remaining = "SELECT anuncio, COUNT(*) as count
                  FROM hostify_reserva
                  WHERE check_in >= '2025-01-01'
                    AND propiedad_id IS NULL
                  GROUP BY anuncio
                  ORDER BY count DESC";
$remaining = ia_sqlArrayIndx($sql_remaining);

if (count($remaining) > 0) {
    echo "<p><strong>Remaining $unmatched unmatched, across " . count($remaining) . " unique anuncios:</strong></p>";
    echo "<table border='1' cellpadding='8' style='width: 100%;'>";
    echo "<tr><th>Anuncio</th><th>Count</th><th>Category</th></tr>";

    foreach ($remaining as $row) {
        $anuncio = $row['anuncio'];
        $count = $row['count'];

        echo "<tr>";
        echo "<td>{$anuncio}</td>";
        echo "<td style='text-align: center;'>{$count}</td>";
        echo "<td>";

        if (strpos($anuncio, 'P.E.21') !== false) {
            echo "<span style='color: red;'>❌ Invalid Data</span>";
        } elseif (strpos($anuncio, '1111 Reservas') !== false) {
            echo "<span style='color: gray;'>⚠️ Test Data</span>";
        } else {
            echo "<span style='color: orange;'>🟡 Other</span>";
        }

        echo "</td></tr>";
    }

    echo "</table>";
}

if ($match_rate >= 98.0) {
    echo "<hr>";
    echo "<div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; border-radius: 10px; text-align: center;'>";
    echo "<h2 style='margin:0; font-size: 2em;'>🏆 98%+ MATCH RATE ACHIEVED! 🏆</h2>";
    echo "<p style='font-size: 1.3em; margin: 10px 0;'>Outstanding performance!</p>";
    echo "<p>Remaining unmatched: $unmatched (primarily invalid/test data)</p>";
    echo "</div>";
}

echo "</body></html>";
?>
