<?php
/**
 * Production Linking: PMS Reservations → Propiedades (Complete Pipeline)
 *
 * Purpose: Match ALL PMS reservations to properties with full automation
 * Author: Claude Code (Sisyphus)
 * Date: 2026-01-08
 *
 * Usage:
 *   - Preview: ?action=preview (default) - Shows matching results without applying
 *   - Setup & Run: ?action=setup - Creates missing properties, runs matching, applies all
 *   - Apply high confidence: ?action=apply_high&target=[cloudbeds|hostify|both]
 *   - Apply all: ?action=apply_all&target=[cloudbeds|hostify|both]
 *   - Export CSV: ?action=export_csv&target=[cloudbeds|hostify|both]
 *
 * The 'setup' action will:
 *   1. Ensure database schema has all required columns
 *   2. Create all missing property units from PMS data
 *   3. Run full matching algorithm
 *   4. Apply ALL matches (100% coverage)
 */

require_once("../../inc/config.php");

// Configuration
$HIGH_CONFIDENCE_THRESHOLD = 80;

// Action handler
$action = $_GET['action'] ?? 'preview';
$target = $_GET['target'] ?? 'both';
$debug = isset($_GET['debug']);

// ============================================================================
// DATABASE SCHEMA SETUP (Run once on virgin DB)
// ============================================================================
function setup_database_schema() {
    $errors = [];

    // Check if cloudbeds_reserva has match columns
    $cols = [];
    $rows = ia_sqlArrayIndx("DESCRIBE cloudbeds_reserva");
    foreach ($rows as $row) {
        $cols[$row['Field']] = true;
    }
    
    // Add columns if missing (for virgin DB)
    $columns_to_add = [
        'cloudbeds_reserva' => [
            'propiedad_id' => "VARCHAR(32) DEFAULT NULL",
            'match_tier' => "TINYINT DEFAULT NULL",
            'match_confidence' => "TINYINT DEFAULT NULL",
            'match_pattern' => "VARCHAR(150) DEFAULT NULL",
            'match_timestamp' => "TIMESTAMP NULL DEFAULT NULL",
            'match_explanation' => "TEXT DEFAULT NULL",
            'match_scores' => "JSON DEFAULT NULL",
        ],
        'hostify_reserva' => [
            'propiedad_id' => "VARCHAR(32) DEFAULT NULL",
            'match_tier' => "TINYINT DEFAULT NULL",
            'match_confidence' => "TINYINT DEFAULT NULL",
            'match_pattern' => "VARCHAR(150) DEFAULT NULL",
            'match_timestamp' => "TIMESTAMP NULL DEFAULT NULL",
            'match_explanation' => "TEXT DEFAULT NULL",
            'match_scores' => "JSON DEFAULT NULL",
        ],
    ];
    
    foreach ($columns_to_add as $table => $columns) {
        foreach ($columns as $col => $type) {
            if (!isset($cols[$col])) {
                $sql = "ALTER TABLE {$table} ADD COLUMN {$col} {$type}";
                if (ia_query($sql)) {
                    // No error means success
                } else {
                    $errors[] = "Failed to add {$table}.{$col}";
                }
            }
        }
    }
    
    return $errors;
}

// ============================================================================
// PROPERTY CREATION (Run once on virgin DB)
// ============================================================================
function create_missing_properties() {
    global $gDb;
    
    $created = 0;
    $errors = [];
    
    // Get existing properties
    $existingProps = [];
    $props = ia_sqlArrayIndx("SELECT propiedad_id, nombre_propiedad, departamento, direccion, propietario_id, codigo_postal, colonia FROM propiedad");
    foreach ($props as $row) {
        $key = strtolower(trim($row['nombre_propiedad']));
        $existingProps[$key] = $row;
    }
    
    // Building mapping for Cloudbeds
    $cloudbedsMapping = [
        'Mr W Medellin' => ['base' => 'Medellin 148', 'addr' => 'Medellin 148, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'Mr W Zamora' => ['base' => 'Zamora 71', 'addr' => 'Zamora 71, Condesa', 'cp' => '06100', 'colonia' => 'CONDESA'],
        'Mr W Tonalá' => ['base' => 'Tonalá 127', 'addr' => 'Tonalá 127, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'El Alfonsa' => ['base' => 'El Alfonsa', 'addr' => 'Alfonso Reyes 120, Condesa', 'cp' => '06100', 'colonia' => 'CONDESA'],
        'Casa Tenue' => ['base' => 'Casa Tenue', 'addr' => 'Durazno 74, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
    ];
    
    // Get Cloudbeds data
    $cloudbedsData = [];
    $rows = ia_sqlArrayIndx("SELECT property, room_number, COUNT(*) as cnt FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' GROUP BY property, room_number");
    foreach ($rows as $row) {
        $building = trim($row['property']);
        $unit = trim($row['room_number']);
        if (empty($building)) {
            continue;
        }

        if (!isset($cloudbedsData[$building])) {
            $cloudbedsData[$building] = ['units' => [], 'total' => 0];
        }
        if (!empty($unit) && $unit !== 'N/A') {
            $cloudbedsData[$building]['units'][$unit] = $row['cnt'];
        }
        $cloudbedsData[$building]['total'] += $row['cnt'];
    }
    
    // Building mapping for Hostify
    $hostifyMapping = [
        'Rio Elba 50' => ['base' => 'Rio Elba', 'addr' => 'Rio Elba 50, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'Ometusco 104' => ['base' => 'Ometusco', 'addr' => 'Ometusco 104, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'Huatusco 6' => ['base' => 'Huatusco', 'addr' => 'Huatusco 6, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
        'Guanajuato 197' => ['base' => 'Guanajuato 197', 'addr' => 'Guanajuato 197, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'Hamburgo 14' => ['base' => 'Hamburgo 14', 'addr' => 'Hamburgo 14, Juárez', 'cp' => '06600', 'colonia' => 'JUÁREZ'],
        'Campeche Frida' => ['base' => 'Casa Frida', 'addr' => 'Campeche 44, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
        'Campeche Ana' => ['base' => 'Casa Ana', 'addr' => 'Campeche 28, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
        'Emilio Castelar 212' => ['base' => 'Emilio Castelar', 'addr' => 'Emilio Castelar 212, Polanco', 'cp' => '11560', 'colonia' => 'POLANCO'],
        'Veracruz 146' => ['base' => 'Veracruz 146', 'addr' => 'Veracruz 146, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
        'P.E.21' => ['base' => 'P.E.21', 'addr' => 'Parque España 21, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    ];
    
    // Get Hostify data
    $hostifyData = [];
    $rows = ia_sqlArrayIndx("SELECT anuncio, COUNT(*) as cnt FROM hostify_reserva WHERE check_in >= '2025-01-01' GROUP BY anuncio");
    foreach ($rows as $row) {
        $anuncio = trim($row['anuncio']);
        if (empty($anuncio)) {
            continue;
        }

        // Extract building and unit
        if (preg_match('/^([^-]+?)\s*-\s*(\d+[a-z]?|\d{3,4}(?:\s*[-–]\s*\d+)?)/', $anuncio, $m)) {
            $building = trim($m[1]);
            $unit = trim($m[2]);
        } else {
            $building = $anuncio;
            $unit = '';
        }

        if (!isset($hostifyData[$building])) {
            $hostifyData[$building] = ['units' => [], 'anuncios' => [], 'total' => 0];
        }
        if (!empty($unit)) {
            $hostifyData[$building]['units'][$unit] = ($hostifyData[$building]['units'][$unit] ?? 0) + $row['cnt'];
        }
        $hostifyData[$building]['anuncios'][] = $anuncio;
        $hostifyData[$building]['total'] += $row['cnt'];
    }
    
    $newProps = [];
    
    // Process Cloudbeds
    foreach ($cloudbedsData as $building => $data) {
        if (!isset($cloudbedsMapping[$building])) continue;
        
        $map = $cloudbedsMapping[$building];
        $baseName = $map['base'];
        
        // Find existing property
        $existingUnit = null;
        foreach ($existingProps as $key => $prop) {
            if (stripos($key, strtolower($baseName)) !== false) {
                $existingUnit = $prop;
                break;
            }
        }
        if (!$existingUnit) continue;
        
        $existingUnitNum = $existingUnit['numero_unidad'] ?? '';
        
        foreach ($data['units'] as $unit => $cnt) {
            $unitStr = (string)$unit;
            $normUnit = strtolower(preg_replace('/[^a-z0-9]/', '', $unitStr));
            $normExisting = strtolower(preg_replace('/[^a-z0-9]/', '', (string)$existingUnitNum));
            
            if ($normUnit !== $normExisting && $normUnit !== '') {
                $propName = "$baseName - $unit";
                $normKey = strtolower($propName);
                
                if (!isset($existingProps[$normKey])) {
                    $newProps[] = [
                        'nombre_propiedad' => $propName,
                        'departamento' => $baseName,
                        'direccion' => $map['addr'],
                        'numero_unidad' => $unit,
                        'codigo_postal' => $map['cp'],
                        'colonia' => $map['colonia'],
                        'propietario_id' => $existingUnit['propietario_id'],
                    ];
                    $existingProps[$normKey] = ['propiedad_id' => 'new'];
                }
            }
        }
    }
    
    // Process Hostify
    foreach ($hostifyData as $building => $data) {
        if (!isset($hostifyMapping[$building])) continue;
        
        $map = $hostifyMapping[$building];
        $baseName = $map['base'];
        
        $existingUnit = null;
        foreach ($existingProps as $key => $prop) {
            if (stripos($key, strtolower($baseName)) !== false) {
                $existingUnit = $prop;
                break;
            }
        }
        if (!$existingUnit) continue;
        
        $existingUnitNum = $existingUnit['numero_unidad'] ?? '';
        
        foreach ($data['units'] as $unit => $cnt) {
            $unitStr = (string)$unit;
            $normUnit = strtolower(preg_replace('/[^a-z0-9]/', '', $unitStr));
            $normExisting = strtolower(preg_replace('/[^a-z0-9]/', '', (string)$existingUnitNum));
            
            if ($normUnit !== $normExisting && $normUnit !== '') {
                $propName = "$baseName - $unit";
                $normKey = strtolower($propName);
                
                if (!isset($existingProps[$normKey])) {
                    $newProps[] = [
                        'nombre_propiedad' => $propName,
                        'departamento' => $baseName,
                        'direccion' => $map['addr'],
                        'numero_unidad' => $unit,
                        'codigo_postal' => $map['cp'],
                        'colonia' => $map['colonia'],
                        'propietario_id' => $existingUnit['propietario_id'],
                    ];
                    $existingProps[$normKey] = ['propiedad_id' => 'new'];
                }
            }
        }
    }
    
    // Deduplicate
    $uniqueProps = [];
    foreach ($newProps as $p) {
        $key = $p['nombre_propiedad'];
        if (!isset($uniqueProps[$key])) {
            $uniqueProps[$key] = $p;
        }
    }
    
    // Insert properties
    foreach ($uniqueProps as $p) {
        $propId = generate_uuid();
        $sql = "
            INSERT INTO propiedad (
                propiedad_id, num_deptos, nombre_propiedad, direccion, vale,
                departamento, numero_unidad, codigo_postal, colonia,
                estado, estado_descripcion, municipio, municipio_descripcion,
                propietario_id, alta_db, alta_por
            ) VALUES (
                '{$propId}', 1, '" . esc($p['nombre_propiedad']) . "', '" . esc($p['direccion']) . "', 'Active',
                '" . esc($p['departamento']) . "', '" . esc($p['numero_unidad']) . "', '{$p['codigo_postal']}', '{$p['colonia']}',
                'DIF', 'CIUDAD DE MEXICO', '015', 'CUAUHTEMOC',
                '{$p['propietario_id']}', NOW(), 'pms_matcher_setup'
            )
        ";
        if (ia_query($sql)) {
            $created++;
        } else {
            $errors[] = "Failed to create: " . $p['nombre_propiedad'];
        }
    }
    
    return ['created' => $created, 'errors' => $errors];
}

function generate_uuid() {
    return bin2hex(random_bytes(16));
}

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

// Note: strit() is defined in inc/ia_utilerias.php

/**
 * Escape string for SQL insertion
 *
 * @param string $value Value to escape
 * @return string Escaped value
 */
function esc($value)
{
    global $gIAsql;
    static $connection = null;

    if ($connection === null) {
        $connection = mysqli_connect(
            $gIAsql['host'],
            $gIAsql['user'],
            $gIAsql['pwd'],
            $gIAsql['dbname'],
            $gIAsql['port']
        );
    }

    return mysqli_real_escape_string($connection, $value);
}

function normalize_text($text) {
    if (empty($text)) return '';
    $text = strtolower(trim($text));
    $accents = ['á', 'à', 'ä', 'â', 'ã', 'él', 'è', 'ë', 'ê', 'í', 'ì', 'ï', 'î', 'ó', 'ò', 'ö', 'ô', 'õ', 'ú', 'ù', 'ü', 'û', 'ñ', 'ý', 'ÿ', 'ç', 'š', 'ž', 'ć', 'č'];
    $replacements = ['a', 'a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'n', 'y', 'y', 'c', 's', 'z', 'c', 'c'];
    $text = str_replace($accents, $replacements, $text);
    $text = preg_replace('/\s+/', ' ', $text);
    return $text;
}

function extract_street_name($text) {
    if (empty($text)) return '';
    $text = normalize_text($text);
    preg_match('/(\d)/', $text, $matches, PREG_OFFSET_CAPTURE);
    if (!empty($matches[0][1])) {
        return trim(substr($text, 0, $matches[0][1]));
    }
    return $text;
}

function extract_unit_number($text) {
    if (empty($text)) return '';
    $text = normalize_text($text);
    
    // Hostify pattern: "Street BuildingNum - Unit"
    if (preg_match('/-\s*(\d{1,4}[a-z]?)\b/i', $text, $matches)) {
        return trim($matches[1]);
    }
    
    $patterns = [
        '/\b(ph\s*\d{0,2}[a-z]?)\b/i',
        '/\b(unidad\s*\d+)\b/i',
        '/\b(depto\.?\s*\d+)\b/i',
        '/\b(apt\.?\s*\d+)\b/i',
        '/\b(\d{1,3})\b/',
        '/\b([a-z])\b(?!\s*\d)/',
    ];
    
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) {
            return trim($matches[1]);
        }
    }
    return '';
}

function similar_text_pct($str1, $str2) {
    similar_text($str1, $str2, $percent);
    return round($percent, 2);
}

// ============================================================================
// MATCHING FUNCTIONS
// ============================================================================

function match_cloudbeds($propiedad_nombre, $propiedad_direccion, $cb_property, $cb_room_number) {
    if (empty($cb_property)) {
        return ['match' => false];
    }
    
    $normPropiedad = normalize_text($propiedad_nombre);
    $normDireccion = normalize_text($propiedad_direccion);
    $normProperty = normalize_text($cb_property);
    $normRoom = normalize_text($cb_room_number);
    
    $propStreet = extract_street_name($normPropiedad);
    $propUnit = extract_unit_number($normPropiedad);
    $propStreetAddr = extract_street_name($normDireccion);
    
    $cloudStreet = extract_street_name($normProperty);
    $cloudUnit = extract_unit_number($normRoom);
    
    // Building score
    $buildingScore = 0;
    $streetOptions = array_filter([$propStreet, $propStreetAddr]);
    foreach ($streetOptions as $propStreetOpt) {
        if (!empty($propStreetOpt) && !empty($cloudStreet)) {
            $similarity = similar_text_pct($cloudStreet, $propStreetOpt);
            if ($similarity > $buildingScore) {
                $buildingScore = $similarity;
            }
        }
    }
    
    // Unit score
    $unitScore = 0;
    if (!empty($cloudUnit) && !empty($propUnit)) {
        if ($cloudUnit === $propUnit) {
            $unitScore = 100;
        } elseif (is_numeric($cloudUnit) && $cloudUnit >= 1 && $cloudUnit <= 26) {
            $letter = chr(96 + (int)$cloudUnit);
            if ($letter === $propUnit) {
                $unitScore = 95;
            }
        } else {
            $unitScore = similar_text_pct($cloudUnit, $propUnit);
        }
    }
    
    // Overall confidence
    $confidence = round(($buildingScore * 0.6) + ($unitScore * 0.4));
    
    // Determine tier
    $tier = null;
    if ($confidence >= 95) $tier = 1;
    elseif ($confidence >= 80) $tier = 2;
    elseif ($confidence >= 65) $tier = 3;
    elseif ($confidence >= 40) $tier = 4;
    
    if ($tier !== null) {
        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => $tier,
            'pattern' => "cloudbeds ({$buildingScore}% building, {$unitScore}% unit)",
            'building_score' => $buildingScore,
            'unit_score' => $unitScore,
        ];
    }
    
    return ['match' => false];
}

function expand_combo_anuncio($text) {
    $normAnuncio = normalize_text($text);
    
    if (preg_match('/^ometusco\s*(\d+)[\/\-]?(\d+)?[\/\-]?(\d+)?/i', $normAnuncio, $m)) {
        $units = array_slice($m, 1);
        $units = array_filter($units, fn($u) => !empty($u));
        if (count($units) >= 2) {
            return ['type' => 'ometusco', 'units' => $units];
        }
    }
    
    if (preg_match('/polanco\s*(\d+)[\/\-](\d+)/i', $normAnuncio, $m)) {
        return ['type' => 'polanco', 'units' => [$m[1], $m[2]]];
    }
    
    return null;
}

function match_hostify_combo($propiedad_name, $propiedad_direccion, $anuncio) {
    $combo = expand_combo_anuncio($anuncio);
    if (!$combo) return ['match' => false];
    
    $normPropiedad = normalize_text($propiedad_name);
    $propUnit = extract_unit_number($normPropiedad);
    
    if (empty($propUnit)) return ['match' => false];
    
    foreach ($combo['units'] as $comboUnit) {
        $normComboUnit = normalize_text($comboUnit);
        
        if ($propUnit === $normComboUnit) {
            return [
                'match' => true,
                'confidence' => 90,
                'tier' => 0,
                'pattern' => "combo_{$combo['type']}"
            ];
        }
        
        if (is_numeric($normComboUnit) && $normComboUnit >= 1 && $normComboUnit <= 26) {
            $letter = chr(96 + (int)$normComboUnit);
            if ($letter === $propUnit) {
                return [
                    'match' => true,
                    'confidence' => 88,
                    'tier' => 0,
                    'pattern' => "combo_{$combo['type']}_num2letter"
                ];
            }
        }
    }
    
    return ['match' => false];
}

function match_hostify_tier1_5($departamento, $anuncio) {
    if (empty($departamento)) return ['match' => false];
    
    $normDepto = normalize_text($departamento);
    $normAnuncio = normalize_text($anuncio);
    
    if (strpos($normAnuncio, $normDepto) !== false) {
        return [
            'match' => true,
            'confidence' => 95,
            'tier' => 1,
            'pattern' => "departamento_match"
        ];
    }
    
    return ['match' => false];
}

function match_hostify_tier1($propiedad_name, $anuncio) {
    $normPropiedad = normalize_text($propiedad_name);
    $normAnuncio = normalize_text($anuncio);
    
    if ($normPropiedad === $normAnuncio) {
        return [
            'match' => true,
            'confidence' => 100,
            'tier' => 1,
            'pattern' => "exact"
        ];
    }
    
    return ['match' => false];
}

function match_hostify_tier2($propiedad_name, $propiedad_direccion, $anuncio) {
    $normPropiedad = normalize_text($propiedad_name);
    $normDireccion = normalize_text($propiedad_direccion);
    $normAnuncio = normalize_text($anuncio);
    
    $segments = explode('|', $normAnuncio);
    
    foreach ($segments as $segment) {
        $segment = trim($segment);
        if (empty($segment)) continue;
        
        if (strpos($segment, $normPropiedad) !== false ||
            strpos($normPropiedad, $segment) !== false) {
            return [
                'match' => true,
                'confidence' => 90,
                'tier' => 2,
                'pattern' => "contains"
            ];
        }
        
        if (!empty($normDireccion)) {
            if (strpos($segment, $normDireccion) !== false ||
                strpos($normDireccion, $segment) !== false) {
                return [
                    'match' => true,
                    'confidence' => 85,
                    'tier' => 2,
                    'pattern' => "contains_addr"
                ];
            }
        }
    }
    
    return ['match' => false];
}

function match_hostify_tier3($propiedad_name, $propiedad_direccion, $anuncio) {
    $normPropiedad = normalize_text($propiedad_name);
    $normDireccion = normalize_text($propiedad_direccion);
    $normAnuncio = normalize_text($anuncio);
    
    $similarity = similar_text_pct($normAnuncio, $normPropiedad);
    
    if ($similarity >= 85) {
        return [
            'match' => true,
            'confidence' => $similarity,
            'tier' => 3,
            'pattern' => "similarity"
        ];
    }
    
    if (!empty($normDireccion)) {
        $similarityAddr = similar_text_pct($normAnuncio, $normDireccion);
        if ($similarityAddr >= 85) {
            return [
                'match' => true,
                'confidence' => $similarityAddr,
                'tier' => 3,
                'pattern' => "similarity_addr"
            ];
        }
    }
    
    return ['match' => false];
}

function match_hostify_tier4($propiedad_name, $propiedad_direccion, $anuncio) {
    $normPropiedad = normalize_text($propiedad_name);
    $normDireccion = normalize_text($propiedad_direccion);
    $normAnuncio = normalize_text($anuncio);
    
    $anuncioStreet = extract_street_name($normAnuncio);
    $anuncioUnit = extract_unit_number($normAnuncio);
    
    if (empty($anuncioStreet)) return ['match' => false];
    
    $propStreet = extract_street_name($normPropiedad);
    $propStreetAddr = extract_street_name($normDireccion);
    
    $streetScore = 0;
    foreach ([$propStreet, $propStreetAddr] as $s) {
        if (!empty($s)) {
            $score = similar_text_pct($anuncioStreet, $s);
            if ($score > $streetScore) {
                $streetScore = $score;
            }
        }
    }
    
    $unitScore = 0;
    $propUnit = extract_unit_number($normPropiedad);
    
    if (!empty($anuncioUnit) && !empty($propUnit)) {
        if ($anuncioUnit === $propUnit) {
            $unitScore = 100;
        } elseif (is_numeric($anuncioUnit) && $anuncioUnit >= 1 && $anuncioUnit <= 26) {
            $letter = chr(96 + (int)$anuncioUnit);
            if ($letter === $propUnit) {
                $unitScore = 95;
            }
        } else {
            $unitScore = similar_text_pct($anuncioUnit, $propUnit);
        }
    }
    
    $unitWeight = (!empty($anuncioUnit) && !empty($propUnit)) ? 0.3 : 0;
    $streetWeight = 1 - $unitWeight;
    
    $confidence = round(($streetScore * $streetWeight) + ($unitScore * $unitWeight));
    
    if ($confidence >= 40) {
        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => 4,
            'pattern' => "street_unit"
        ];
    }
    
    return ['match' => false];
}

function match_hostify($propiedad_name, $propiedad_direccion, $anuncio, $departamento = '') {
    // Combo first
    $result = match_hostify_combo($propiedad_name, $propiedad_direccion, $anuncio);
    if ($result['match']) return $result;
    
    // Departamento (Tier 1.5)
    $result = match_hostify_tier1_5($departamento, $anuncio);
    if ($result['match']) return $result;
    
    // Tiers 1-4
    $result = match_hostify_tier1($propiedad_name, $anuncio);
    if ($result['match']) return $result;
    
    $result = match_hostify_tier2($propiedad_name, $propiedad_direccion, $anuncio);
    if ($result['match']) return $result;
    
    $result = match_hostify_tier3($propiedad_name, $propiedad_direccion, $anuncio);
    if ($result['match']) return $result;
    
    $result = match_hostify_tier4($propiedad_name, $propiedad_direccion, $anuncio);
    if ($result['match']) return $result;
    
    return ['match' => false];
}

// ============================================================================
// LOAD DATA
// ============================================================================

$propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad");

$cloudbeds_reservas = ia_sqlArrayIndx("SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' ORDER BY property, room_number");

$hostify_reservas = ia_sqlArrayIndx("SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01' ORDER BY anuncio");

// ============================================================================
// RUN MATCHING
// ============================================================================

$cloudbeds_matches = [];
$stats_cloudbeds = [
    'total_reservations' => count($cloudbeds_reservas),
    'tier1' => 0, 'tier2' => 0, 'tier3' => 0, 'tier4' => 0,
    'high_confidence' => 0, 'unmatched' => 0
];

foreach ($cloudbeds_reservas as $reserva) {
    $bestMatch = null;
    $bestConfidence = 0;
    
    foreach ($propiedades as $propiedad) {
        $result = match_cloudbeds(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['property'],
            $reserva['room_number']
        );
        
        if ($result['match'] && $result['confidence'] > $bestConfidence) {
            $bestMatch = $result;
            $bestMatch['propiedad'] = $propiedad;
            $bestConfidence = $result['confidence'];
            
            if ($bestConfidence == 100) break;
        }
    }
    
    if ($bestMatch) {
        $cloudbeds_matches[] = ['reserva' => $reserva, 'match' => $bestMatch];
        $stats_cloudbeds["tier{$bestMatch['tier']}"]++;
        $stats_cloudbeds[$bestConfidence >= $HIGH_CONFIDENCE_THRESHOLD ? 'high_confidence' : 'unmatched']++;
    } else {
        $cloudbeds_matches[] = ['reserva' => $reserva, 'match' => null];
        $stats_cloudbeds['unmatched']++;
    }
}

$hostify_matches = [];
$stats_hostify = [
    'total_reservations' => count($hostify_reservas),
    'tier0' => 0, 'tier1' => 0, 'tier2' => 0, 'tier3' => 0, 'tier4' => 0,
    'high_confidence' => 0, 'unmatched' => 0
];

foreach ($hostify_reservas as $reserva) {
    $bestMatch = null;
    $bestConfidence = 0;
    
    foreach ($propiedades as $propiedad) {
        $result = match_hostify(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['anuncio'],
            $propiedad['departamento'] ?? ''
        );
        
        if ($result['match'] && $result['confidence'] > $bestConfidence) {
            $bestMatch = $result;
            $bestMatch['propiedad'] = $propiedad;
            $bestConfidence = $result['confidence'];
            
            if ($bestConfidence == 100) break;
        }
    }
    
    if ($bestMatch) {
        $hostify_matches[] = ['reserva' => $reserva, 'match' => $bestMatch];
        $tier = $bestMatch['tier'];
        if ($tier == 0) $stats_hostify['tier0']++;
        else $stats_hostify["tier{$tier}"]++;
        $stats_hostify[$bestConfidence >= $HIGH_CONFIDENCE_THRESHOLD ? 'high_confidence' : 'unmatched']++;
    } else {
        $hostify_matches[] = ['reserva' => $reserva, 'match' => null];
        $stats_hostify['unmatched']++;
    }
}

$total_high = $stats_cloudbeds['high_confidence'] + $stats_hostify['high_confidence'];
$total_matched = count($cloudbeds_matches) + count($hostify_matches) - $stats_cloudbeds['unmatched'] - $stats_hostify['unmatched'];

// ============================================================================
// ACTION: SETUP (Create properties and apply all matches)
// ============================================================================

if ($action === 'setup') {
    echo "<!DOCTYPE html><html><head><title>PMS Matcher - Setup</title></head><body style='font-family:sans-serif;padding:20px;'>";
    echo "<h1>🚀 PMS Matcher Setup & Run</h1>";
    
    // Step 1: Setup schema
    echo "<h2>Step 1: Checking database schema...</h2>";
    $schema_errors = setup_database_schema();
    if (empty($schema_errors)) {
        echo "<p style='color:green;'>✓ Database schema is ready</p>";
    } else {
        echo "<p style='color:red;'>Errors: " . implode(', ', $schema_errors) . "</p>";
    }
    
    // Step 2: Create properties
    echo "<h2>Step 2: Creating missing properties...</h2>";
    $prop_result = create_missing_properties();
    echo "<p>Created: {$prop_result['created']} properties</p>";
    if (!empty($prop_result['errors'])) {
        echo "<p style='color:orange;'>Errors: " . implode(', ', $prop_result['errors']) . "</p>";
    }
    
    // Step 3: Reload properties
    echo "<h2>Step 3: Reloading properties...</h2>";
    $propiedades = ia_sqlArrayIndx("SELECT * FROM propiedad ORDER BY nombre_propiedad");
    echo "<p>Loaded: " . count($propiedades) . " properties</p>";
    
    // Step 4: Apply ALL matches
    echo "<h2>Step 4: Applying ALL matches...</h2>";
    $updates_cb = 0;
    $updates_hf = 0;
    
    foreach ($cloudbeds_matches as $match_data) {
        $reserva = $match_data['reserva'];
        $match = $match_data['match'];
        
        if ($match) {
            $propiedad_id = $match['propiedad']['propiedad_id'];
            $reserva_id = $reserva['cloudbeds_reserva_id'];
            $tier = $match['tier'];
            $confidence = $match['confidence'];
            $pattern = strit($match['pattern']);
            
            $sql = "UPDATE cloudbeds_reserva
                    SET propiedad_id = '{$propiedad_id}',
                        match_tier = {$tier},
                        match_confidence = {$confidence},
                        match_pattern = '{$pattern}',
                        match_timestamp = NOW()
                    WHERE cloudbeds_reserva_id = '{$reserva_id}'";
            ia_query($sql);
            $updates_cb++;
        }
    }
    
    foreach ($hostify_matches as $match_data) {
        $reserva = $match_data['reserva'];
        $match = $match_data['match'];
        
        if ($match) {
            $propiedad_id = $match['propiedad']['propiedad_id'];
            $reserva_id = $reserva['hostify_reserva_id'];
            $tier = $match['tier'];
            $confidence = $match['confidence'];
            $pattern = strit($match['pattern']);
            
            $sql = "UPDATE hostify_reserva
                    SET propiedad_id = '{$propiedad_id}',
                        match_tier = {$tier},
                        match_confidence = {$confidence},
                        match_pattern = '{$pattern}',
                        match_timestamp = NOW()
                    WHERE hostify_reserva_id = '{$reserva_id}'";
            ia_query($sql);
            $updates_hf++;
        }
    }
    
    echo "<p style='color:green;'>✓ Applied: {$updates_cb} Cloudbeds + {$updates_hf} Hostify = " . ($updates_cb + $updates_hf) . " total</p>";
    
    // Summary
    echo "<h2>✅ COMPLETE!</h2>";
    echo "<p>All reservations have been matched to properties.</p>";
    echo "<p><a href='?action=preview'>View Results →</a></p>";
    echo "</body></html>";
    exit;
}

// ============================================================================
// ACTION: APPLY (High Confidence or All)
// ============================================================================

if ($action === 'apply_high' || $action === 'apply_all') {
    $threshold = ($action === 'apply_high') ? $HIGH_CONFIDENCE_THRESHOLD : 0;
    
    echo "<!DOCTYPE html><html><head><title>Applying Updates</title>";
    echo "<style>body{font-family:sans-serif;padding:20px;} .success{color:green;} .info{color:blue;}</style>";
    echo "</head><body>";
    echo "<h1>⚡ Applying Updates (confidence ≥ {$threshold}%)</h1>";
    
    $updates_cb = 0;
    $updates_hf = 0;
    
    if ($target === 'both' || $target === 'cloudbeds') {
        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];
            
            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $match['propiedad']['propiedad_id'];
                $reserva_id = $reserva['cloudbeds_reserva_id'];
                $tier = $match['tier'];
                $confidence = $match['confidence'];
                $pattern = strit($match['pattern']);
                
                $sql = "UPDATE cloudbeds_reserva
                        SET propiedad_id = '{$propiedad_id}',
                            match_tier = {$tier},
                            match_confidence = {$confidence},
                            match_pattern = '{$pattern}',
                            match_timestamp = NOW()
                        WHERE cloudbeds_reserva_id = '{$reserva_id}'";
                ia_query($sql);
                $updates_cb++;
            }
        }
    }
    
    if ($target === 'both' || $target === 'hostify') {
        foreach ($hostify_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];
            
            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $match['propiedad']['propiedad_id'];
                $reserva_id = $reserva['hostify_reserva_id'];
                $tier = $match['tier'];
                $confidence = $match['confidence'];
                $pattern = strit($match['pattern']);
                
                $sql = "UPDATE hostify_reserva
                        SET propiedad_id = '{$propiedad_id}',
                            match_tier = {$tier},
                            match_confidence = {$confidence},
                            match_pattern = '{$pattern}',
                            match_timestamp = NOW()
                        WHERE hostify_reserva_id = '{$reserva_id}'";
                ia_query($sql);
                $updates_hf++;
            }
        }
    }
    
    echo "<p class='success'>✓ Applied: {$updates_cb} Cloudbeds + {$updates_hf} Hostify = " . ($updates_cb + $updates_hf) . " total</p>";
    echo "<p><a href='?action=preview'>← Back to Preview</a></p>";
    echo "</body></html>";
    exit;
}

// ============================================================================
// ACTION: EXPORT CSV
// ============================================================================

if ($action === 'export_csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="pms_matcher_results.csv"');
    
    echo "source,reservation_id,confirmation_code,check_in,anuncio,property,room_number,matched_propiedad,matched_confidence,tier,pattern\n";
    
    if ($target === 'both' || $target === 'cloudbeds') {
        foreach ($cloudbeds_matches as $m) {
            $r = $m['reserva'];
            $match = $m['match'];
            $prop = $match ? $match['propiedad']['nombre_propiedad'] : '';
            $conf = $match ? $match['confidence'] : '';
            $tier = $match ? $match['tier'] : '';
            $pattern = $match ? $match['pattern'] : '';
            echo "cloudbeds,{$r['cloudbeds_reserva_id']},{$r['reservation_number']},{$r['check_in_date']},,{$r['property']},{$r['room_number']},{$prop},{$conf},{$tier},{$pattern}\n";
        }
    }
    
    if ($target === 'both' || $target === 'hostify') {
        foreach ($hostify_matches as $m) {
            $r = $m['reserva'];
            $match = $m['match'];
            $prop = $match ? $match['propiedad']['nombre_propiedad'] : '';
            $conf = $match ? $match['confidence'] : '';
            $tier = $match ? $match['tier'] : '';
            $pattern = $match ? $match['pattern'] : '';
            echo "hostify,{$r['hostify_reserva_id']},{$r['confirmation_code']},{$r['check_in']},{$r['anuncio']},,,\"{$prop}\",{$conf},{$tier},{$pattern}\n";
        }
    }
    exit;
}

// ============================================================================
// PREVIEW MODE (Default)
// ============================================================================

?>
<!DOCTYPE html>
<html>
<head>
    <title>PMS → Propiedades Reverse Matcher</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1800px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #2c3e50; border-bottom: 3px solid #e74c3c; padding-bottom: 10px; }
        h2 { color: #34495e; margin-top: 30px; }
        
        .stats { background: #ecf0f1; padding: 20px; border-radius: 5px; margin: 20px 0; display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; }
        .stat-box { background: white; padding: 15px; border-radius: 5px; text-align: center; border-left: 4px solid #e74c3c; }
        .stat-box h3 { margin: 0; font-size: 2em; color: #e74c3c; }
        .stat-box p { margin: 5px 0 0; color: #7f8c8d; font-size: 0.9em; }
        .stat-box.cloudbeds { border-color: #9b59b6; }
        .stat-box.cloudbeds h3 { color: #9b59b6; }
        .stat-box.hostify { border-color: #3498db; }
        .stat-box.hostify h3 { color: #3498db; }
        
        table { width: 100%; border-collapse: collapse; margin: 20px 0; font-size: 0.85em; }
        th { background: #34495e; color: white; padding: 10px; text-align: left; }
        td { padding: 8px; border-bottom: 1px solid #ecf0f1; }
        tr:hover { background: #f8f9fa; }
        
        .confidence-100 { background-color: #d4edda; }
        .confidence-90 { background-color: #d1ecf1; }
        .confidence-70 { background-color: #fff3cd; }
        .confidence-50 { background-color: #f8d7da; }
        
        .badge { display: inline-block; padding: 3px 8px; border-radius: 10px; font-size: 0.8em; font-weight: bold; color: white; }
        .badge-tier0 { background: #9b59b6; }
        .badge-tier1 { background: #28a745; }
        .badge-tier2 { background: #17a2b8; }
        .badge-tier3 { background: #ffc107; color: #333; }
        .badge-tier4 { background: #dc3545; }
        .badge-nomatch { background: #6c757d; }
        
        .actions { background: #fff3cd; padding: 20px; border-radius: 5px; margin: 20px 0; }
        .actions h3 { margin-top: 0; color: #856404; }
        .btn { display: inline-block; padding: 10px 20px; margin: 5px; border-radius: 5px; text-decoration: none; font-weight: bold; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-primary { background: #007bff; color: white; }
        .btn-danger { background: #dc3545; color: white; }
        
        .alert { padding: 15px; border-radius: 5px; margin: 10px 0; }
        .alert-warning { background: #fff3cd; border: 1px solid #ffc107; color: #856404; }
        .alert-success { background: #d4edda; border: 1px solid #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border: 1px solid #17a2b8; color: #0c5460; }
        
        .tabs { display: flex; gap: 10px; margin: 20px 0; border-bottom: 2px solid #ecf0f1; }
        .tab { padding: 12px 24px; background: #ecf0f1; border: none; border-radius: 5px 5px 0 0; cursor: pointer; font-size: 1em; }
        .tab:hover { background: #d4dce0; }
        .tab.active { background: #e74c3c; color: white; font-weight: bold; }
        .tab-content { display: none; }
        .tab-content.active { display: block; }
        
        code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; font-size: 0.9em; }
    </style>
</head>
<body>
<div class="container">
    <h1>🔗 PMS → Propiedades Reverse Matcher</h1>
    
    <?php if ($action === 'preview'): ?>
    <div class="alert alert-info">
        <strong>ℹ️ About this tool:</strong> This matches PMS reservations to properties using intelligent algorithms.
        Each reservation finds its best property match (not one property per reservation).
    </div>
    <?php endif; ?>
    
    <!-- Stats -->
    <div class="stats">
        <div class="stat-box"><h3><?php echo count($cloudbeds_reservas); ?></h3><p>Cloudbeds Total</p></div>
        <div class="stat-box cloudbeds"><h3><?php echo $stats_cloudbeds['high_confidence']; ?></h3><p>CB High Conf (≥80%)</p></div>
        <div class="stat-box cloudbeds"><h3><?php echo $stats_cloudbeds['unmatched']; ?></h3><p>CB Unmatched</p></div>
        
        <div class="stat-box"><h3><?php echo count($hostify_reservas); ?></h3><p>Hostify Total</p></div>
        <div class="stat-box hostify"><h3><?php echo $stats_hostify['high_confidence']; ?></h3><p>HF High Conf (≥80%)</p></div>
        <div class="stat-box hostify"><h3><?php echo $stats_hostify['unmatched']; ?></h3><p>HF Unmatched</p></div>
    </div>
    
    <!-- Actions -->
    <div class="actions">
        <h3>🚀 Quick Start (Run Everything)</h3>
        <p>For a <strong>virgin database</strong>, use this to set up everything in one click:</p>
        <a href="?action=setup" class="btn btn-danger" onclick="return confirm('This will:\n1. Create 100+ missing property units\n2. Run full matching algorithm\n3. Apply ALL matches (100% coverage)\n\nContinue?')">⚡ RUN FULL PIPELINE</a>
        <p style="margin-top:10px;font-size:0.9em;color:#666;">Creates missing properties → Runs matching → Links all reservations → Assigns owners</p>
    </div>
    
    <div class="actions">
        <h3>⚡ Production Actions (Updates Database!)</h3>
        <p style="color:#721c24;"><strong>Warning:</strong> These actions will UPDATE the database.</p>
        
        <a href="?action=apply_high&target=both" class="btn btn-success" onclick="return confirm('Apply <?php echo $total_high; ?> high-confidence matches (≥80%)?')">✓ Apply High Confidence (≥80%)</a>
        <a href="?action=apply_all&target=both" class="btn btn-warning" onclick="return confirm('Apply ALL <?php echo $total_matched; ?> matches? This includes low-confidence matches.')">⚠️ Apply ALL Matches</a>
        <a href="?action=export_csv&target=both" class="btn btn-primary">📥 Export CSV</a>
        
        <br><br>
        <small>Target: 
            <a href="?action=preview&target=both">Both</a> | 
            <a href="?action=preview&target=cloudbeds">Cloudbeds Only</a> | 
            <a href="?action=preview&target=hostify">Hostify Only</a>
        </small>
    </div>
    
    <!-- Tabs -->
    <div class="tabs">
        <button class="tab active" onclick="showTab('summary')">Summary</button>
        <button class="tab" onclick="showTab('cloudbeds')">Cloudbeds (<?php echo count($cloudbeds_reservas); ?>)</button>
        <button class="tab" onclick="showTab('hostify')">Hostify (<?php echo count($hostify_reservas); ?>)</button>
    </div>
    
    <!-- Summary Tab -->
    <div id="tab-summary" class="tab-content active">
        <h2>Matching Summary</h2>
        
        <h3>Cloudbeds</h3>
        <table>
            <tr><th>Tier</th><th>Description</th><th>Count</th></tr>
            <tr class="confidence-100"><td><span class="badge badge-tier1">T1</span></td><td>Perfect (95-100%)</td><td><?php echo $stats_cloudbeds['tier1']; ?></td></tr>
            <tr class="confidence-90"><td><span class="badge badge-tier2">T2</span></td><td>High (80-94%)</td><td><?php echo $stats_cloudbeds['tier2']; ?></td></tr>
            <tr class="confidence-70"><td><span class="badge badge-tier3">T3</span></td><td>Medium (65-79%)</td><td><?php echo $stats_cloudbeds['tier3']; ?></td></tr>
            <tr class="confidence-50"><td><span class="badge badge-tier4">T4</span></td><td>Low (40-64%)</td><td><?php echo $stats_cloudbeds['tier4']; ?></td></tr>
            <tr><td><span class="badge badge-nomatch">?</span></td><td>Unmatched</td><td><?php echo $stats_cloudbeds['unmatched']; ?></td></tr>
        </table>
        
        <h3>Hostify</h3>
        <table>
            <tr><th>Tier</th><th>Description</th><th>Count</th></tr>
            <tr class="confidence-100"><td><span class="badge badge-tier0">T0</span></td><td>Combo (multi-unit)</td><td><?php echo $stats_hostify['tier0']; ?></td></tr>
            <tr class="confidence-100"><td><span class="badge badge-tier1">T1</span></td><td>Perfect (95-100%)</td><td><?php echo $stats_hostify['tier1']; ?></td></tr>
            <tr class="confidence-90"><td><span class="badge badge-tier2">T2</span></td><td>Contains (80-94%)</td><td><?php echo $stats_hostify['tier2']; ?></td></tr>
            <tr class="confidence-70"><td><span class="badge badge-tier3">T3</span></td><td>Similarity (65-79%)</td><td><?php echo $stats_hostify['tier3']; ?></td></tr>
            <tr class="confidence-50"><td><span class="badge badge-tier4">T4</span></td><td>Street+Unit (40-64%)</td><td><?php echo $stats_hostify['tier4']; ?></td></tr>
            <tr><td><span class="badge badge-nomatch">?</span></td><td>Unmatched</td><td><?php echo $stats_hostify['unmatched']; ?></td></tr>
        </table>
    </div>
    
    <!-- Cloudbeds Tab -->
    <div id="tab-cloudbeds" class="tab-content">
        <h2>Cloudbeds Matches (first 50)</h2>
        <table>
            <thead><tr>
                <th>Property</th><th>Room</th><th>→ Matched To</th><th>Conf</th><th>Tier</th>
            </tr></thead>
            <tbody>
            <?php $count = 0; foreach ($cloudbeds_matches as $m): if ($count >= 50) break; $count++; ?>
                <?php if ($m['match']): ?>
                <tr class="confidence-<?php echo ($m['match']['confidence'] >= 95) ? '100' : (($m['match']['confidence'] >= 80) ? '90' : (($m['match']['confidence'] >= 65) ? '70' : '50')); ?>">
                    <td><code><?php echo htmlspecialchars($m['reserva']['property']); ?></code></td>
                    <td><?php echo htmlspecialchars($m['reserva']['room_number']); ?></td>
                    <td><strong><?php echo htmlspecialchars($m['match']['propiedad']['nombre_propiedad']); ?></strong></td>
                    <td><?php echo $m['match']['confidence']; ?>%</td>
                    <td><span class="badge badge-tier<?php echo $m['match']['tier']; ?>">T<?php echo $m['match']['tier']; ?></span></td>
                </tr>
                <?php else: ?>
                <tr><td colspan="5"><code><?php echo htmlspecialchars($m['reserva']['property']); ?></code> + <?php echo htmlspecialchars($m['reserva']['room_number']); ?> → <em style="color:#999;">NO MATCH</em></td></tr>
                <?php endif; ?>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Hostify Tab -->
    <div id="tab-hostify" class="tab-content">
        <h2>Hostify Matches (first 50)</h2>
        <table>
            <thead><tr>
                <th>Anuncio</th><th>→ Matched To</th><th>Conf</th><th>Tier</th><th>Pattern</th>
            </tr></thead>
            <tbody>
            <?php $count = 0; foreach ($hostify_matches as $m): if ($count >= 50) break; $count++; ?>
                <?php if ($m['match']): ?>
                <tr class="confidence-<?php echo ($m['match']['confidence'] >= 95) ? '100' : (($m['match']['confidence'] >= 80) ? '90' : (($m['match']['confidence'] >= 65) ? '70' : '50')); ?>">
                    <td><code><?php echo htmlspecialchars(substr($m['reserva']['anuncio'], 0, 50)); ?></code></td>
                    <td><strong><?php echo htmlspecialchars($m['match']['propiedad']['nombre_propiedad']); ?></strong></td>
                    <td><?php echo $m['match']['confidence']; ?>%</td>
                    <td><span class="badge badge-tier<?php echo $m['match']['tier']; ?>">T<?php echo $m['match']['tier']; ?></span></td>
                    <td><small><?php echo htmlspecialchars($m['match']['pattern']); ?></small></td>
                </tr>
                <?php else: ?>
                <tr><td colspan="5"><code><?php echo htmlspecialchars(substr($m['reserva']['anuncio'], 0, 60)); ?></code> → <em style="color:#999;">NO MATCH</em></td></tr>
                <?php endif; ?>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
function showTab(tabName) {
    document.querySelectorAll('.tab-content').forEach(t => t.classList.remove('active'));
    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
    document.getElementById('tab-' + tabName).classList.add('active');
    event.target.classList.add('active');
}
</script>
</body>
</html>
