<?php
/**
 * One-Time Bulk Linking: Propiedades <-> PMS Reservations (Cloudbeds + Hostify)
 *
 * Purpose: Link properties to PMS reservations using multi-tier fuzzy matching
 * Author: Claude Code (Thoth's Algorithm v2)
 * Date: 2026-01-03
 *
 * Usage:
 *   - Preview: ?action=preview (default)
 *   - Apply high confidence: ?action=apply_high
 *   - Apply all: ?action=apply_all
 *   - Export CSV: ?action=export_csv&target=[cloudbeds|hostify|both]
 */

require_once("../../inc/config.php");

// Configuration
$HIGH_CONFIDENCE_THRESHOLD = 80;
$DRY_RUN = true; // Safety flag

// Action handler
$action = $_GET['action'] ?? 'preview';
$target = $_GET['target'] ?? 'both'; // cloudbeds, hostify, both
$debug = isset($_GET['debug']); // Debug mode flag

?>
<!DOCTYPE html>
<html>
<head>
    <title>Propiedades ↔ PMS Fuzzy Matcher</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: #f5f5f5;
        }
        .container {
            max-width: 1600px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }

        /* Tabs */
        .tabs {
            display: flex;
            gap: 10px;
            margin: 20px 0;
            border-bottom: 2px solid #ecf0f1;
        }
        .tab {
            padding: 12px 24px;
            background: #ecf0f1;
            border: none;
            border-radius: 5px 5px 0 0;
            cursor: pointer;
            font-size: 1em;
            transition: all 0.3s;
        }
        .tab:hover {
            background: #d4dce0;
        }
        .tab.active {
            background: #3498db;
            color: white;
            font-weight: bold;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }

        .stats {
            background: #ecf0f1;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
        }
        .stat-box {
            background: white;
            padding: 15px;
            border-radius: 5px;
            text-align: center;
            border-left: 4px solid #3498db;
        }
        .stat-box h3 { margin: 0 0 5px 0; font-size: 2em; color: #3498db; }
        .stat-box p { margin: 0; color: #7f8c8d; font-size: 0.9em; }
        .stat-box.cloudbeds { border-color: #9b59b6; }
        .stat-box.cloudbeds h3 { color: #9b59b6; }
        .stat-box.hostify { border-color: #e74c3c; }
        .stat-box.hostify h3 { color: #e74c3c; }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 0.85em;
        }
        th {
            background: #34495e;
            color: white;
            padding: 12px;
            text-align: left;
            position: sticky;
            top: 0;
        }
        td {
            padding: 10px;
            border-bottom: 1px solid #ecf0f1;
        }
        tr:hover { background: #f8f9fa; }

        .confidence-100 { background-color: #d4edda; border-left: 4px solid #28a745; }
        .confidence-90 { background-color: #d1ecf1; border-left: 4px solid #17a2b8; }
        .confidence-70 { background-color: #fff3cd; border-left: 4px solid #ffc107; }
        .confidence-50 { background-color: #f8d7da; border-left: 4px solid #dc3545; }
        .confidence-0 { background-color: #f8f9fa; border-left: 4px solid #6c757d; }

        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.85em;
            font-weight: bold;
            color: white;
        }
        .badge-tier1 { background: #28a745; }
        .badge-tier2 { background: #17a2b8; }
        .badge-tier3 { background: #ffc107; color: #333; }
        .badge-tier4 { background: #dc3545; }
        .badge-nomatch { background: #6c757d; }

        .actions {
            margin: 30px 0;
            padding: 20px;
            background: #e8f4f8;
            border-radius: 5px;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            margin: 5px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 1em;
            text-decoration: none;
            transition: all 0.3s;
        }
        .btn-primary { background: #3498db; color: white; }
        .btn-primary:hover { background: #2980b9; }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-warning:hover { background: #e0a800; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; }

        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            border-left: 4px solid;
        }
        .alert-success { background: #d4edda; border-color: #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border-color: #17a2b8; color: #0c5460; }
        .alert-warning { background: #fff3cd; border-color: #ffc107; color: #856404; }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }

        .match-detail {
            font-size: 0.8em;
            color: #7f8c8d;
            font-style: italic;
        }
    </style>
</head>
<body>
<div class="container">
    <h1>🔱 Fuzzy Matcher: Propiedades ↔ PMS (Cloudbeds + Hostify)</h1>
    <p style="color: #7f8c8d; font-style: italic;">The Architect's Algorithm v2 - Dual-Target PMS Matching Engine</p>
    <?php if (!$debug): ?>
    <p style="color: #3498db; font-size: 0.9em;">
        💡 <strong>Tip:</strong> Add <code>?debug=1</code> to the URL to see detailed matching information
    </p>
    <?php endif; ?>

<?php

// ============================================================================
// CORE FUNCTIONS
// ============================================================================

/**
 * Safe htmlspecialchars that handles NULL values
 */
function esc($value) {
    return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Normalize text for comparison
 */
function normalize_text($text) {
    if (empty($text)) return '';

    // Remove accents FIRST (before lowercasing)
    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);

    // Then lowercase
    $text = mb_strtolower($text, 'UTF-8');

    // Normalize whitespace
    $text = trim(preg_replace('/\s+/', ' ', $text));

    return $text;
}

/**
 * Extract street name (text before first digit)
 */
function extract_street_name($text) {
    $normalized = normalize_text($text);

    // Find position of first digit
    if (preg_match('/\d/', $normalized, $matches, PREG_OFFSET_CAPTURE)) {
        $pos = $matches[0][1];
        return trim(substr($normalized, 0, $pos));
    }

    return $normalized;
}

/**
 * Extract unit number from text (extract digits/patterns like "103", "401", "PH1")
 */
function extract_unit_number($text) {
    $normalized = normalize_text($text);

    // Try to extract common unit patterns
    // Pattern 1: Pure numbers (103, 401, 302)
    if (preg_match('/\b(\d{2,4})\b/', $normalized, $matches)) {
        return $matches[1];
    }

    // Pattern 2: Penthouse (PH1, PH2, ph chico)
    if (preg_match('/ph\s*(\w+)/i', $normalized, $matches)) {
        return 'ph' . normalize_text($matches[1]);
    }

    // Pattern 3: Suite (SU4, Suite 4)
    if (preg_match('/su(?:ite)?\s*(\w+)/i', $normalized, $matches)) {
        return 'su' . normalize_text($matches[1]);
    }

    // Pattern 4: Single letter suffix (A, B, C)
    if (preg_match('/\b([A-Z])\b/', $text, $matches)) {
        return strtolower($matches[1]);
    }

    return '';
}

// ============================================================================
// CLOUDBEDS MATCHING FUNCTIONS
// ============================================================================

/**
 * Match Cloudbeds: Hybrid building + unit matching
 * Returns: ['match' => bool, 'confidence' => 0-100, 'building_score' => 0-100, 'unit_score' => 0-100]
 */
function match_cloudbeds($propiedad_nombre, $propiedad_direccion, $cb_property, $cb_room_number) {
    // Extract street from propiedad
    $prop_street = extract_street_name($propiedad_nombre);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($propiedad_direccion);
    }

    // Extract street from cloudbeds property
    $cb_street = extract_street_name($cb_property);

    // PART 1: Building match (60% weight)
    $building_score = 0;
    $norm_cb_property = normalize_text($cb_property);
    $norm_prop_nombre = normalize_text($propiedad_nombre);
    $norm_prop_dir = normalize_text($propiedad_direccion);

    // Check if property name contains the building name
    if (strlen($norm_cb_property) >= 5) {
        if (strpos($norm_prop_nombre, $norm_cb_property) !== false ||
            strpos($norm_prop_dir, $norm_cb_property) !== false) {
            $building_score = 100;
        } elseif (strlen($prop_street) >= 5 && strlen($cb_street) >= 5) {
            // Street name similarity
            similar_text($prop_street, $cb_street, $percent);
            $building_score = $percent;
        }
    }

    // PART 2: Unit match (40% weight)
    $unit_score = 0;
    $prop_unit = extract_unit_number($propiedad_nombre);
    $cb_unit = extract_unit_number($cb_room_number);

    if (!empty($prop_unit) && !empty($cb_unit)) {
        if ($prop_unit === $cb_unit) {
            $unit_score = 100;
        } else {
            // Partial match (e.g., "103" vs "03")
            similar_text($prop_unit, $cb_unit, $percent);
            $unit_score = $percent;
        }
    } elseif (empty($cb_unit) || $cb_room_number === 'N/A') {
        // If no unit specified in Cloudbeds, use only building score
        $unit_score = $building_score;
    }

    // Combined confidence: 60% building + 40% unit
    $confidence = round(($building_score * 0.6) + ($unit_score * 0.4));

    // Determine tier based on confidence
    $tier = 0;
    if ($confidence >= 95) $tier = 1;
    elseif ($confidence >= 80) $tier = 2;
    elseif ($confidence >= 65) $tier = 3;
    elseif ($confidence >= 40) $tier = 4;

    if ($tier > 0) {
        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => $tier,
            'building_score' => round($building_score),
            'unit_score' => round($unit_score),
            'prop_unit' => $prop_unit,
            'cb_unit' => $cb_unit
        ];
    }

    return ['match' => false];
}

// ============================================================================
// HOSTIFY MATCHING FUNCTIONS (4-Tier Fuzzy)
// ============================================================================

/**
 * TIER 1: Exact match (after normalization)
 */
function match_hostify_tier1($propiedad_name, $anuncio) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_anuncio = normalize_text($anuncio);

    if ($norm_prop === $norm_anuncio) {
        return ['match' => true, 'confidence' => 100, 'tier' => 1];
    }

    return ['match' => false];
}

/**
 * TIER 2: Contains match (handle pipe-separated lists and partial matches)
 */
function match_hostify_tier2($propiedad_name, $propiedad_direccion, $anuncio) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_dir = normalize_text($propiedad_direccion);
    $norm_anuncio = normalize_text($anuncio);

    // Split anuncio by common separators
    $anuncio_segments = preg_split('/[\|,]/', $norm_anuncio);
    $anuncio_segments = array_map('trim', $anuncio_segments);

    foreach ($anuncio_segments as $segment) {
        // Avoid very short matches
        if (strlen($segment) >= 5 && strlen($norm_prop) >= 5) {
            // Check if segment contains property name or vice versa
            if (strpos($segment, $norm_prop) !== false || strpos($norm_prop, $segment) !== false) {
                return ['match' => true, 'confidence' => 90, 'tier' => 2, 'segment' => $segment];
            }

            // Check against direccion too
            if (strlen($norm_dir) >= 5 && (strpos($segment, $norm_dir) !== false || strpos($norm_dir, $segment) !== false)) {
                return ['match' => true, 'confidence' => 90, 'tier' => 2, 'segment' => $segment];
            }
        }
    }

    return ['match' => false];
}

/**
 * TIER 3: Similarity match using similar_text()
 */
function match_hostify_tier3($propiedad_name, $propiedad_direccion, $anuncio, $threshold = 85) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_dir = normalize_text($propiedad_direccion);
    $norm_anuncio = normalize_text($anuncio);

    // Try against full anuncio
    similar_text($norm_prop, $norm_anuncio, $percent);
    if ($percent >= $threshold) {
        return ['match' => true, 'confidence' => 70, 'tier' => 3, 'similarity' => round($percent, 1)];
    }

    // Try direccion
    similar_text($norm_dir, $norm_anuncio, $percent);
    if ($percent >= $threshold) {
        return ['match' => true, 'confidence' => 70, 'tier' => 3, 'similarity' => round($percent, 1)];
    }

    // Try against segments
    $anuncio_segments = preg_split('/[\|,]/', $norm_anuncio);
    foreach ($anuncio_segments as $segment) {
        $segment = trim($segment);
        if (strlen($segment) >= 5) {
            similar_text($norm_prop, $segment, $percent);
            if ($percent >= $threshold) {
                return ['match' => true, 'confidence' => 70, 'tier' => 3, 'similarity' => round($percent, 1), 'segment' => $segment];
            }
        }
    }

    return ['match' => false];
}

/**
 * TIER 4: Partial street name + unit match
 */
function match_hostify_tier4($propiedad_name, $propiedad_direccion, $anuncio) {
    $prop_street = extract_street_name($propiedad_name);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($propiedad_direccion);
    }
    $anuncio_street = extract_street_name($anuncio);

    // Need at least 5 chars for valid street match
    if (strlen($prop_street) < 5 || strlen($anuncio_street) < 5) {
        return ['match' => false];
    }

    // Check if streets match
    if (strpos($anuncio_street, $prop_street) !== false || strpos($prop_street, $anuncio_street) !== false) {
        // Try to match units too
        $prop_unit = extract_unit_number($propiedad_name);
        $anuncio_unit = extract_unit_number($anuncio);

        $confidence = 50; // Base for street match

        if (!empty($prop_unit) && !empty($anuncio_unit) && $prop_unit === $anuncio_unit) {
            $confidence = 65; // Boost if unit also matches
        }

        return ['match' => true, 'confidence' => $confidence, 'tier' => 4, 'street' => $prop_street];
    }

    return ['match' => false];
}

/**
 * Find best Hostify match for a property
 */
function find_best_hostify_match($propiedad_nombre, $propiedad_direccion, $hostify_reservas) {
    $best_match = null;

    // Group by anuncio to avoid duplicate matching
    $unique_anuncios = [];
    foreach ($hostify_reservas as $reserva) {
        $anuncio = $reserva['anuncio'];
        if (!isset($unique_anuncios[$anuncio])) {
            $unique_anuncios[$anuncio] = $reserva;
        }
    }

    foreach ($unique_anuncios as $anuncio => $reserva) {
        // Try each tier in order
        $result = match_hostify_tier1($propiedad_nombre, $anuncio);
        if (!$result['match']) $result = match_hostify_tier2($propiedad_nombre, $propiedad_direccion, $anuncio);
        if (!$result['match']) $result = match_hostify_tier3($propiedad_nombre, $propiedad_direccion, $anuncio);
        if (!$result['match']) $result = match_hostify_tier4($propiedad_nombre, $propiedad_direccion, $anuncio);

        if ($result['match']) {
            $result['anuncio'] = $anuncio;
            $result['reserva'] = $reserva;

            // Keep best match (highest confidence)
            if (!$best_match || $result['confidence'] > $best_match['confidence']) {
                $best_match = $result;
            }

            // If we got a perfect match, stop searching
            if ($result['confidence'] == 100) {
                break;
            }
        }
    }

    return $best_match;
}

// ============================================================================
// DATA LOADING
// ============================================================================

// Load all propiedades
$sql_propiedades = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_propiedades);

// Load all cloudbeds reservations
$sql_cloudbeds = "SELECT * FROM cloudbeds_reserva ORDER BY property, room_number";
$cloudbeds_reservas = ia_sqlArrayIndx($sql_cloudbeds);

// Load all hostify reservations
$sql_hostify = "SELECT * FROM hostify_reserva ORDER BY anuncio";
$hostify_reservas = ia_sqlArrayIndx($sql_hostify);

// Data validation
if (empty($propiedades)) {
    echo "<div class='alert alert-warning'>";
    echo "<strong>ERROR:</strong> No propiedades loaded from database!<br>";
    echo "Please ensure you're logged in and the table has data.";
    echo "</div></div></body></html>";
    exit;
}

echo "<div class='alert alert-info'>";
echo "<strong>Data Loaded:</strong> ";
echo count($propiedades) . " properties, ";
echo count($cloudbeds_reservas) . " Cloudbeds reservations, ";
echo count($hostify_reservas) . " Hostify reservations";
echo "</div>";

// Debug mode: Show data structure
if ($debug) {
    echo "<div class='alert alert-warning'>";
    echo "<strong>🐛 DEBUG MODE ACTIVE</strong><br>";
    echo "<strong>Sample Propiedad:</strong><pre>" . print_r($propiedades[0], true) . "</pre>";
    if (!empty($cloudbeds_reservas)) {
        echo "<strong>Sample Cloudbeds:</strong><pre>" . print_r($cloudbeds_reservas[0], true) . "</pre>";
    }
    if (!empty($hostify_reservas)) {
        echo "<strong>Sample Hostify:</strong><pre>" . print_r($hostify_reservas[0], true) . "</pre>";
    }
    echo "</div>";
}

// ============================================================================
// MATCHING ALGORITHMS
// ============================================================================

$cloudbeds_matches = [];
$hostify_matches = [];
$debug_log = [];

$stats_cloudbeds = [
    'total_reservations' => count($cloudbeds_reservas),
    'unique_properties' => 0,
    'tier1' => 0,
    'tier2' => 0,
    'tier3' => 0,
    'tier4' => 0,
    'unmatched' => 0,
    'high_confidence' => 0,
    'low_confidence' => 0,
];

$stats_hostify = [
    'total_reservations' => count($hostify_reservas),
    'unique_anuncios' => 0,
    'tier1' => 0,
    'tier2' => 0,
    'tier3' => 0,
    'tier4' => 0,
    'unmatched' => 0,
    'high_confidence' => 0,
    'low_confidence' => 0,
];

// CLOUDBEDS MATCHING
// Group by unique property+room combinations
$cloudbeds_unique = [];
foreach ($cloudbeds_reservas as $reserva) {
    $key = $reserva['property'] . '||' . $reserva['room_number'];
    if (!isset($cloudbeds_unique[$key])) {
        $cloudbeds_unique[$key] = $reserva;
    }
}
$stats_cloudbeds['unique_properties'] = count($cloudbeds_unique);

foreach ($cloudbeds_unique as $key => $reserva) {
    $best_match = null;
    $best_confidence = 0;

    foreach ($propiedades as $propiedad) {
        $result = match_cloudbeds(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['property'],
            $reserva['room_number']
        );

        if ($result['match'] && $result['confidence'] > $best_confidence) {
            $best_match = $result;
            $best_match['propiedad'] = $propiedad;
            $best_confidence = $result['confidence'];
        }
    }

    if ($best_match) {
        $cloudbeds_matches[$key] = [
            'reserva' => $reserva,
            'match' => $best_match,
        ];

        $tier = $best_match['tier'];
        $stats_cloudbeds["tier{$tier}"]++;

        if ($best_match['confidence'] >= $HIGH_CONFIDENCE_THRESHOLD) {
            $stats_cloudbeds['high_confidence']++;
        } else {
            $stats_cloudbeds['low_confidence']++;
        }
    } else {
        $cloudbeds_matches[$key] = [
            'reserva' => $reserva,
            'match' => null,
        ];
        $stats_cloudbeds['unmatched']++;
    }
}

// HOSTIFY MATCHING
foreach ($propiedades as $propiedad) {
    $match_result = find_best_hostify_match(
        $propiedad['nombre_propiedad'],
        $propiedad['direccion'],
        $hostify_reservas
    );

    if ($match_result) {
        $hostify_matches[] = [
            'propiedad' => $propiedad,
            'match' => $match_result,
        ];

        $tier = $match_result['tier'];
        $stats_hostify["tier{$tier}"]++;

        if ($match_result['confidence'] >= $HIGH_CONFIDENCE_THRESHOLD) {
            $stats_hostify['high_confidence']++;
        } else {
            $stats_hostify['low_confidence']++;
        }
    } else {
        $hostify_matches[] = [
            'propiedad' => $propiedad,
            'match' => null,
        ];
        $stats_hostify['unmatched']++;
    }
}

// Count unique anuncios matched
$unique_anuncios_matched = [];
foreach ($hostify_matches as $match_data) {
    if ($match_data['match']) {
        $unique_anuncios_matched[$match_data['match']['anuncio']] = true;
    }
}
$stats_hostify['unique_anuncios'] = count($unique_anuncios_matched);

// ============================================================================
// ACTION HANDLERS
// ============================================================================

if ($action === 'apply_high' || $action === 'apply_all') {
    $threshold = ($action === 'apply_high') ? $HIGH_CONFIDENCE_THRESHOLD : 0;
    $updates_cloudbeds = 0;
    $updates_hostify = 0;

    echo "<div class='alert alert-warning'>";
    echo "<strong>Applying updates (confidence ≥ {$threshold}%)...</strong>";
    echo "</div>";

    // Apply Cloudbeds updates
    if ($target === 'both' || $target === 'cloudbeds') {
        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $match['propiedad']['propiedad_id'];
                $property = $reserva['property'];
                $room_number = $reserva['room_number'];

                $sql = "UPDATE cloudbeds_reserva
                        SET propiedad_id = '{$propiedad_id}'
                        WHERE property = '" . mysqli_real_escape_string($gIAsql['cnx'], $property) . "'
                        AND room_number = '" . mysqli_real_escape_string($gIAsql['cnx'], $room_number) . "'";
                ia_query($sql);
                $updates_cloudbeds++;
            }
        }
    }

    // Apply Hostify updates
    if ($target === 'both' || $target === 'hostify') {
        foreach ($hostify_matches as $match_data) {
            $propiedad = $match_data['propiedad'];
            $match = $match_data['match'];

            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $propiedad['propiedad_id'];
                $anuncio = $match['anuncio'];

                $sql = "UPDATE hostify_reserva
                        SET propiedad_id = '{$propiedad_id}'
                        WHERE anuncio = '" . mysqli_real_escape_string($gIAsql['cnx'], $anuncio) . "'";
                ia_query($sql);
                $updates_hostify++;
            }
        }
    }

    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Success!</strong><br>";
    echo "Updated {$updates_cloudbeds} Cloudbeds records<br>";
    echo "Updated {$updates_hostify} Hostify records";
    echo "</div>";

    echo "<a href='link_propiedades_pms.php' class='btn btn-primary'>← Back to Preview</a>";
    echo "</div></body></html>";
    exit;
}

if ($action === 'export_csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=pms_matching_results_' . date('Y-m-d_His') . '.csv');

    $output = fopen('php://output', 'w');

    if ($target === 'both' || $target === 'cloudbeds') {
        fputcsv($output, ['=== CLOUDBEDS MATCHES ===']);
        fputcsv($output, [
            'property', 'room_number', 'room_type',
            'propiedad_id', 'nombre_propiedad', 'direccion',
            'match_tier', 'confidence', 'building_score', 'unit_score'
        ]);

        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            if ($match) {
                fputcsv($output, [
                    $reserva['property'],
                    $reserva['room_number'],
                    $reserva['room_type'],
                    $match['propiedad']['propiedad_id'],
                    $match['propiedad']['nombre_propiedad'],
                    $match['propiedad']['direccion'],
                    $match['tier'],
                    $match['confidence'],
                    $match['building_score'],
                    $match['unit_score']
                ]);
            }
        }
        fputcsv($output, []);
    }

    if ($target === 'both' || $target === 'hostify') {
        fputcsv($output, ['=== HOSTIFY MATCHES ===']);
        fputcsv($output, [
            'anuncio',
            'propiedad_id', 'nombre_propiedad', 'direccion',
            'match_tier', 'confidence', 'notes'
        ]);

        foreach ($hostify_matches as $match_data) {
            $propiedad = $match_data['propiedad'];
            $match = $match_data['match'];

            if ($match) {
                fputcsv($output, [
                    $match['anuncio'],
                    $propiedad['propiedad_id'],
                    $propiedad['nombre_propiedad'],
                    $propiedad['direccion'],
                    $match['tier'],
                    $match['confidence'],
                    $match['segment'] ?? $match['street'] ?? ''
                ]);
            }
        }
    }

    fclose($output);
    exit;
}

// ============================================================================
// DISPLAY TABS & STATISTICS
// ============================================================================

echo "<div class='tabs'>";
echo "<button class='tab active' onclick='showTab(\"summary\")'>📊 Summary</button>";
echo "<button class='tab' onclick='showTab(\"cloudbeds\")'>☁️ Cloudbeds Matches</button>";
echo "<button class='tab' onclick='showTab(\"hostify\")'>🏠 Hostify Matches</button>";
echo "</div>";

// SUMMARY TAB
echo "<div id='tab-summary' class='tab-content active'>";
echo "<h2>Overall Statistics</h2>";

echo "<div class='stats'>";
echo "<div class='stat-box cloudbeds'><h3>" . $stats_cloudbeds['unique_properties'] . "</h3><p>Cloudbeds Unique</p></div>";
echo "<div class='stat-box cloudbeds'><h3>" . $stats_cloudbeds['high_confidence'] . "</h3><p>CB High Conf (≥80%)</p></div>";
echo "<div class='stat-box cloudbeds'><h3>" . $stats_cloudbeds['unmatched'] . "</h3><p>CB Unmatched</p></div>";
echo "<div class='stat-box hostify'><h3>" . $stats_hostify['unique_anuncios'] . "</h3><p>Hostify Matched</p></div>";
echo "<div class='stat-box hostify'><h3>" . $stats_hostify['high_confidence'] . "</h3><p>HF High Conf (≥80%)</p></div>";
echo "<div class='stat-box hostify'><h3>" . $stats_hostify['unmatched'] . "</h3><p>HF Unmatched</p></div>";
echo "</div>";

echo "<div class='actions'>";
echo "<h3>Actions</h3>";
$total_high = $stats_cloudbeds['high_confidence'] + $stats_hostify['high_confidence'];
echo "<a href='?action=apply_high&target=both' class='btn btn-success' onclick='return confirm(\"Apply {$total_high} high-confidence matches (≥80%)?\")'>✓ Apply All High Confidence</a>";
echo "<a href='?action=apply_high&target=cloudbeds' class='btn btn-success' onclick='return confirm(\"Apply {$stats_cloudbeds['high_confidence']} Cloudbeds matches?\")'>✓ Apply Cloudbeds Only</a>";
echo "<a href='?action=apply_high&target=hostify' class='btn btn-success' onclick='return confirm(\"Apply {$stats_hostify['high_confidence']} Hostify matches?\")'>✓ Apply Hostify Only</a>";
echo "<a href='?action=export_csv&target=both' class='btn btn-primary'>📥 Export Both CSV</a>";
echo "</div>";

echo "</div>"; // End summary tab

// CLOUDBEDS TAB
echo "<div id='tab-cloudbeds' class='tab-content'>";
echo "<h2>Cloudbeds Matches</h2>";

echo "<div class='stats'>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['unique_properties']}</h3><p>Unique Properties</p></div>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['tier1']}</h3><p>Tier 1 (95-100%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['tier2']}</h3><p>Tier 2 (80-94%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['tier3']}</h3><p>Tier 3 (65-79%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['tier4']}</h3><p>Tier 4 (40-64%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_cloudbeds['unmatched']}</h3><p>Unmatched</p></div>";
echo "</div>";

echo "<table>";
echo "<thead><tr>";
echo "<th>Property</th>";
echo "<th>Room #</th>";
echo "<th>Room Type</th>";
echo "<th>→ Matched To</th>";
echo "<th>Propiedad</th>";
echo "<th>Tier</th>";
echo "<th>Confidence</th>";
echo "<th>Details</th>";
echo "</tr></thead>";
echo "<tbody>";

foreach ($cloudbeds_matches as $match_data) {
    $reserva = $match_data['reserva'];
    $match = $match_data['match'];

    if ($match) {
        $confidence = $match['confidence'];
        $tier = $match['tier'];
        $propiedad = $match['propiedad'];

        $row_class = "confidence-" . (($confidence >= 95) ? '100' : (($confidence >= 80) ? '90' : (($confidence >= 65) ? '70' : '50')));
        $badge_class = "badge-tier{$tier}";

        echo "<tr class='{$row_class}'>";
        echo "<td><strong>" . esc($reserva['property']) . "</strong></td>";
        echo "<td>" . esc($reserva['room_number']) . "</td>";
        echo "<td><small>" . esc($reserva['room_type']) . "</small></td>";
        echo "<td>→</td>";
        echo "<td><strong>" . esc($propiedad['nombre_propiedad']) . "</strong><br><small>" . esc($propiedad['direccion']) . "</small></td>";
        echo "<td><span class='badge {$badge_class}'>Tier {$tier}</span></td>";
        echo "<td><strong>{$confidence}%</strong></td>";
        echo "<td class='match-detail'>Building: {$match['building_score']}% | Unit: {$match['unit_score']}%</td>";
        echo "</tr>";
    } else {
        echo "<tr class='confidence-0'>";
        echo "<td><strong>" . esc($reserva['property']) . "</strong></td>";
        echo "<td>" . esc($reserva['room_number']) . "</td>";
        echo "<td><small>" . esc($reserva['room_type']) . "</small></td>";
        echo "<td colspan='4' style='color: #999; font-style: italic;'>No match found</td>";
        echo "<td><span class='badge badge-nomatch'>0%</span></td>";
        echo "</tr>";
    }
}

echo "</tbody></table>";
echo "</div>"; // End cloudbeds tab

// HOSTIFY TAB
echo "<div id='tab-hostify' class='tab-content'>";
echo "<h2>Hostify Matches</h2>";

echo "<div class='stats'>";
echo "<div class='stat-box'><h3>{$stats_hostify['unique_anuncios']}</h3><p>Unique Anuncios</p></div>";
echo "<div class='stat-box'><h3>{$stats_hostify['tier1']}</h3><p>Tier 1 (100%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_hostify['tier2']}</h3><p>Tier 2 (90%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_hostify['tier3']}</h3><p>Tier 3 (70%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_hostify['tier4']}</h3><p>Tier 4 (50-65%)</p></div>";
echo "<div class='stat-box'><h3>{$stats_hostify['unmatched']}</h3><p>Unmatched</p></div>";
echo "</div>";

echo "<table>";
echo "<thead><tr>";
echo "<th>Propiedad</th>";
echo "<th>Dirección</th>";
echo "<th>→ Matched To</th>";
echo "<th>Anuncio</th>";
echo "<th>Tier</th>";
echo "<th>Confidence</th>";
echo "</tr></thead>";
echo "<tbody>";

foreach ($hostify_matches as $match_data) {
    $propiedad = $match_data['propiedad'];
    $match = $match_data['match'];

    if ($match) {
        $confidence = $match['confidence'];
        $tier = $match['tier'];

        $row_class = "confidence-" . (($confidence >= 95) ? '100' : (($confidence >= 80) ? '90' : (($confidence >= 65) ? '70' : '50')));
        $badge_class = "badge-tier{$tier}";

        echo "<tr class='{$row_class}'>";
        echo "<td><strong>" . esc($propiedad['nombre_propiedad']) . "</strong></td>";
        echo "<td><small>" . esc($propiedad['direccion']) . "</small></td>";
        echo "<td>→</td>";
        echo "<td><code>" . esc($match['anuncio']) . "</code></td>";
        echo "<td><span class='badge {$badge_class}'>Tier {$tier}</span></td>";
        echo "<td><strong>{$confidence}%</strong></td>";
        echo "</tr>";
    } else {
        echo "<tr class='confidence-0'>";
        echo "<td><strong>" . esc($propiedad['nombre_propiedad']) . "</strong></td>";
        echo "<td><small>" . esc($propiedad['direccion']) . "</small></td>";
        echo "<td colspan='3' style='color: #999; font-style: italic;'>No match found</td>";
        echo "<td><span class='badge badge-nomatch'>0%</span></td>";
        echo "</tr>";
    }
}

echo "</tbody></table>";
echo "</div>"; // End hostify tab

?>

<script>
function showTab(tabName) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(el => {
        el.classList.remove('active');
    });

    // Remove active from all tabs
    document.querySelectorAll('.tab').forEach(el => {
        el.classList.remove('active');
    });

    // Show selected tab
    document.getElementById('tab-' + tabName).classList.add('active');

    // Mark tab as active
    event.target.classList.add('active');
}
</script>

</div>
</body>
</html>
