<?php
/**
 * THOTH'S ALGORITHM: AI-Powered Propietarios → Propiedades Matcher
 *
 * Purpose: Reverse-match propietarios to propiedades using semantic intelligence
 * Author: Claude Code (THOTH's Algorithm v2.0)
 * Date: 2026-01-04
 *
 * Key Changes from v1.0:
 * - ✨ REVERSED DIRECTION: Propietarios (108) → Propiedades (350)
 * - ✨ SEMANTIC TOKEN EXTRACTION: 11-dimensional analysis
 * - ✨ AI EXPLANATIONS: Human-readable reasoning for every match
 * - ✨ MULTI-DIMENSIONAL SCORING: Street + Building + Unit
 * - ✨ COMBO DETECTION: Pipe-separated segments ("Vicente Suárez 146 | GH 2 | Amer")
 *
 * Usage:
 *   - Preview: ?action=preview (default)
 *   - Apply high confidence: ?action=apply_high (≥80%)
 *   - Apply all: ?action=apply_all
 *   - Export CSV: ?action=export_csv
 *   - Debug mode: Add ?debug=1 to URL
 *
 * Stores Results In:
 *   - propiedad.propietario_nombre_propiedad (TEXT - full departamento)
 *   - propiedad.propietario_match_tier (INT - 0-4)
 *   - propiedad.propietario_match_confidence (INT - 0-100)
 *   - propiedad.propietario_match_pattern (VARCHAR - algorithm used)
 *   - propiedad.propietario_match_explanation (TEXT - AI reasoning)
 *   - propiedad.propietario_match_scores (JSON - {street:95, building:100, unit:90})
 *   - propiedad.propietario_match_timestamp (TIMESTAMP)
 */

require_once("../../inc/config.php");

// ============================================================================
// CONFIGURATION
// ============================================================================

$HIGH_CONFIDENCE_THRESHOLD = 80;
$action = $_GET['action'] ?? 'preview';
$debug = isset($_GET['debug']);

// Safe HTML escaping helper
function esc($value) {
    return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>🔮 THOTH's Propietarios Matcher</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .container {
            max-width: 1600px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            border-bottom: 4px solid #667eea;
            padding-bottom: 15px;
            font-size: 2.2em;
            margin-top: 0;
        }
        h1 .emoji { font-size: 1.2em; }
        .subtitle {
            color: #764ba2;
            font-style: italic;
            margin-top: -10px;
            margin-bottom: 20px;
        }
        .stats {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }
        .stat-box {
            background: white;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            border-left: 4px solid #667eea;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stat-box.combo { border-color: #9b59b6; }
        .stat-box.tier1 { border-color: #28a745; }
        .stat-box.tier2 { border-color: #17a2b8; }
        .stat-box.tier3 { border-color: #ffc107; }
        .stat-box.tier4 { border-color: #dc3545; }
        .stat-box.unmatched { border-color: #6c757d; }
        .stat-box h3 { margin: 0 0 5px 0; font-size: 2.2em; color: #667eea; }
        .stat-box p { margin: 0; color: #7f8c8d; font-size: 0.85em; }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 0.85em;
        }
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 8px;
            text-align: left;
            position: sticky;
            top: 0;
            font-size: 0.9em;
        }
        td {
            padding: 10px 8px;
            border-bottom: 1px solid #ecf0f1;
        }
        tr:hover { background: #f8f9fa; }

        /* Confidence-based row colors */
        .confidence-100 { background-color: #d4edda !important; border-left: 4px solid #28a745; }
        .confidence-90 { background-color: #d1ecf1 !important; border-left: 4px solid #17a2b8; }
        .confidence-70 { background-color: #fff3cd !important; border-left: 4px solid #ffc107; }
        .confidence-50 { background-color: #f8d7da !important; border-left: 4px solid #dc3545; }
        .confidence-0 { background-color: #f8f9fa !important; border-left: 4px solid #6c757d; }

        /* Tier badges */
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: bold;
            color: white;
        }
        .badge-tier0 { background: #9b59b6; }
        .badge-tier1 { background: #28a745; }
        .badge-tier2 { background: #17a2b8; }
        .badge-tier3 { background: #ffc107; color: #333; }
        .badge-tier4 { background: #dc3545; }
        .badge-nomatch { background: #6c757d; }
        .badge-combo { background: #9b59b6; margin-left: 5px; font-size: 0.75em; }

        /* Action buttons */
        .actions {
            margin: 30px 0;
            padding: 20px;
            background: linear-gradient(135deg, #e8f4f8 0%, #f3e7f9 100%);
            border-radius: 8px;
            border: 2px solid #667eea;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            margin: 5px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1em;
            text-decoration: none;
            transition: all 0.3s;
            font-weight: bold;
        }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5568d3; transform: translateY(-2px); }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; transform: translateY(-2px); }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-warning:hover { background: #e0a800; transform: translateY(-2px); }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; transform: translateY(-2px); }

        /* Alert boxes */
        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success { background: #d4edda; border-color: #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border-color: #17a2b8; color: #0c5460; }
        .alert-warning { background: #fff3cd; border-color: #ffc107; color: #856404; }
        .alert-danger { background: #f8d7da; border-color: #dc3545; color: #721c24; }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }

        /* Debug mode styling */
        .debug-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            border-radius: 6px;
            font-size: 0.85em;
        }
        .debug-box pre {
            background: white;
            padding: 10px;
            border-radius: 4px;
            overflow-x: auto;
            max-height: 300px;
        }
    </style>
</head>
<body>
<div class="container">
    <h1><span class="emoji">🔮</span> THOTH'S ALGORITHM: Propietarios Matcher</h1>
    <p class="subtitle">AI-Powered Reverse Matching | Semantic Intelligence | Multi-Dimensional Scoring</p>

    <?php if (!$debug): ?>
    <div class="alert alert-info">
        💡 <strong>Tip:</strong> Add <code>?debug=1</code> to URL for detailed matching diagnostics
    </div>
    <?php endif; ?>

<?php

// ============================================================================
// THOTH'S CORE FUNCTIONS - SEMANTIC INTELLIGENCE
// ============================================================================

/**
 * Normalize text for comparison (accent removal, lowercase, trim)
 */
function normalize_text($text) {
    if (empty($text)) return '';

    // Remove accents
    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);

    // Lowercase
    $text = mb_strtolower($text, 'UTF-8');

    // Normalize whitespace
    $text = trim(preg_replace('/\s+/', ' ', $text));

    return $text;
}

/**
 * Extract street name (text before first digit or hyphen)
 */
function extract_street_name($text) {
    $normalized = normalize_text($text);

    // Find position of first digit or hyphen
    if (preg_match('/[\d\-]/', $normalized, $matches, PREG_OFFSET_CAPTURE)) {
        $pos = $matches[0][1];
        return trim(substr($normalized, 0, $pos));
    }

    return $normalized;
}

/**
 * Extract building number from address
 * Examples: "Amsterdam 210" → '210', "Alvaro Obregon 182" → '182'
 */
function extract_building_number($text) {
    $normalized = normalize_text($text);

    // Pattern: street name + space + 1-4 digit number
    if (preg_match('/\b(\d{1,4})\b/', $normalized, $matches)) {
        return $matches[1];
    }

    return null;
}

/**
 * Extract unit identifier (advanced multi-format parser)
 * Formats: "- 204", "- A", "201", "302", etc.
 */
function extract_unit_advanced($text) {
    $normalized = normalize_text($text);

    // Format 1: "- XXX" (hyphen + space + unit)
    if (preg_match('/\-\s+([a-z0-9]+)/', $normalized, $matches)) {
        return $matches[1];
    }

    // Format 2: Last 3-digit number (room number)
    if (preg_match_all('/\b(\d{3})\b/', $normalized, $matches)) {
        return end($matches[1]); // Return last occurrence
    }

    // Format 3: Single letter unit (- A, - B, etc.)
    if (preg_match('/\b([a-z])\b/', $normalized, $matches)) {
        return $matches[1];
    }

    return null;
}

/**
 * 🎯 CRITICAL FIX: Extract text AFTER slash (/) separator
 *
 * Purpose: In virgin data, propiedad.nombre_propiedad contains:
 *          "Property Name / Departamento Match Text"
 *          We need to extract ONLY the text after '/' to match against
 *          propietario.departamento
 *
 * Example:
 *   Input:  "Some Property Name / Vicente Suárez 146 | GH 2"
 *   Output: "Vicente Suárez 146 | GH 2"
 *
 * @param string $text Original property name
 * @return string Text after slash (or original if no slash found)
 */
function extract_after_slash($text) {
    if (empty($text)) return '';

    // Check if text contains '/' separator
    if (strpos($text, '/') !== false) {
        $parts = explode('/', $text, 2); // Split into max 2 parts
        // Return text AFTER slash (trimmed)
        return trim($parts[1]);
    }

    // No slash found, return original text (backward compatible)
    return $text;
}

/**
 * SEMANTIC TOKEN EXTRACTION - THE AI BRAIN
 * Extracts 11 semantic dimensions from chaotic address text
 */
function extract_semantic_tokens($text) {
    if (empty($text)) return ['raw' => ''];

    $norm = normalize_text($text);
    $tokens = [
        'street' => null,           // "amsterdam", "alvaro obregon", "vicente suarez"
        'building_number' => null,  // "210", "182", "146"
        'unit' => null,             // "204", "a", "302"
        'descriptors' => [],        // ["doble", "gh", "pmex"]
        'segments' => [],           // For pipe-separated combos
        'raw' => $text
    ];

    // 1. Check if combo (pipe-separated)
    if (strpos($text, '|') !== false) {
        $tokens['segments'] = array_map('trim', explode('|', $text));
    }

    // 2. Extract street name
    $tokens['street'] = extract_street_name($text);

    // 3. Extract building number
    $tokens['building_number'] = extract_building_number($text);

    // 4. Extract unit
    $tokens['unit'] = extract_unit_advanced($text);

    // 5. Extract descriptors (semantic keywords)
    $descriptor_patterns = [
        'doble' => '/\bdoble\b/i',
        'gh' => '/\bgh\s*\d/i',
        'pmex' => '/\bpmex\b/i',
        'el' => '/\bel\s+[a-z]+/i'
    ];
    foreach ($descriptor_patterns as $desc => $pattern) {
        if (preg_match($pattern, $norm)) {
            $tokens['descriptors'][] = $desc;
        }
    }

    return $tokens;
}

/**
 * COMBO DETECTION - Expand pipe-separated segments
 * Example: "Vicente Suárez 146 | GH 2 | Amer" → 3 segments
 */
function expand_combo_departamento($text) {
    if (strpos($text, '|') === false) {
        return null; // Not a combo
    }

    $segments = array_map('trim', explode('|', $text));

    if (count($segments) < 2) {
        return null;
    }

    return [
        'type' => 'pipe_separated',
        'segments' => $segments,
        'count' => count($segments),
        'full_text' => $text
    ];
}

/**
 * INTELLIGENT ADDRESS COMPARISON
 * Multi-dimensional scoring: street + building + unit
 */
function compare_addresses_intelligent($propiedad_name, $departamento_segment) {
    $prop_tokens = extract_semantic_tokens($propiedad_name);
    $dept_tokens = extract_semantic_tokens($departamento_segment);

    $scores = [
        'street' => 0,
        'building' => 0,
        'unit' => 0,
        'overall' => 0
    ];

    // 1. Street match (40% weight)
    if (!empty($prop_tokens['street']) && !empty($dept_tokens['street'])) {
        similar_text($prop_tokens['street'], $dept_tokens['street'], $percent);
        $scores['street'] = round($percent);
    }

    // 🎯 CRITICAL FIX: Check if streets match before awarding building/unit scores
    // This prevents nonsense matches like "El Alfonsa" → "Amatlán" (24% street but 100% building/unit)
    $streets_match = ($scores['street'] >= 80); // Require at least 80% street similarity

    // 2. Building number match (30% weight)
    if (!empty($prop_tokens['building_number']) && !empty($dept_tokens['building_number'])) {
        // Both have building numbers - compare them
        if ($prop_tokens['building_number'] === $dept_tokens['building_number']) {
            $scores['building'] = 100;
        } else {
            // Fuzzy match for close numbers (210 vs 212)
            $diff = abs(intval($prop_tokens['building_number']) - intval($dept_tokens['building_number']));
            if ($diff <= 5) {
                $scores['building'] = max(0, 100 - ($diff * 10));
            }
        }
    } elseif (empty($prop_tokens['building_number']) && empty($dept_tokens['building_number']) && $streets_match) {
        // ✅ FIXED: Both empty AND streets match = perfect match (street-only like "Lerma")
        $scores['building'] = 100;
    }
    // else: Both empty but streets DON'T match → building stays 0% (prevents nonsense)

    // 3. Unit match (30% weight)
    if (!empty($prop_tokens['unit']) && !empty($dept_tokens['unit'])) {
        // Both have units - compare them
        if ($prop_tokens['unit'] === $dept_tokens['unit']) {
            $scores['unit'] = 100;
        } else {
            // Fuzzy unit match
            similar_text($prop_tokens['unit'], $dept_tokens['unit'], $percent);
            $scores['unit'] = round($percent);
        }
    } elseif (empty($prop_tokens['unit']) && empty($dept_tokens['unit']) && $streets_match) {
        // ✅ FIXED: Both empty AND streets match = perfect match (no units)
        $scores['unit'] = 100;
    }
    // else: Both empty but streets DON'T match → unit stays 0% (prevents nonsense)

    // Calculate weighted overall score
    $scores['overall'] = round(
        ($scores['street'] * 0.40) +
        ($scores['building'] * 0.30) +
        ($scores['unit'] * 0.30)
    );

    return $scores;
}

/**
 * TIER 1: Perfect Match - EXACT MATCHING ONLY
 * 🎯 VIRGIN DATA FIX: No fuzzy/semantic matching - exact equality ONLY
 */
function match_tier1_perfect($propiedad_name, $departamento_segment) {
    $prop_norm = normalize_text($propiedad_name);
    $dept_norm = normalize_text($departamento_segment);

    // OPTION 1: Exact full match
    if ($prop_norm === $dept_norm) {
        return [
            'match' => true,
            'confidence' => 100,
            'tier' => 1,
            'pattern' => 'tier1_exact_normalized',
            'scores' => ['overall' => 100, 'street' => 100, 'building' => 100, 'unit' => 100]
        ];
    }

    // OPTION 2: Exact pipe-separated part match
    // ONLY if departamento is an EXACT pipe-delimited segment of propiedad
    // Example: "san luis potosi 37" is exact part of "san luis potosi 37 | 101"
    if (strlen($dept_norm) >= 5 && strpos($prop_norm, '|') !== false) {
        $prop_parts = array_map('trim', explode('|', $prop_norm));
        foreach ($prop_parts as $part) {
            if ($part === $dept_norm) {
                return [
                    'match' => true,
                    'confidence' => 100,
                    'tier' => 1,
                    'pattern' => 'tier1_pipe_exact_part',
                    'scores' => ['overall' => 100, 'street' => 100, 'building' => 100, 'unit' => 100]
                ];
            }
        }
    }

    // REJECT EVERYTHING ELSE - No fuzzy matching for virgin data!
    // This prevents nonsense like "Alfonso Reyes 120" matching "Alfonso Reyes 176"
    return ['match' => false];
}

/**
 * TIER 2: DISABLED for virgin data
 * 🎯 VIRGIN DATA: Only exact matching allowed - no fuzzy logic!
 */
function match_tier2_high($propiedad_name, $departamento_segment) {
    // DISABLED - Virgin data uses exact matching only
    // This prevents nonsense matches like "Alfonso Reyes 120" → "Alfonso Reyes 176"
    return ['match' => false];
}

/**
 * TIER 3: DISABLED for virgin data
 * 🎯 VIRGIN DATA: Only exact matching allowed - no fuzzy logic!
 */
function match_tier3_medium($propiedad_name, $departamento_segment) {
    // DISABLED - Virgin data uses exact matching only
    return ['match' => false];
}

/**
 * TIER 4: DISABLED for virgin data
 * 🎯 VIRGIN DATA: Only exact matching allowed - no fuzzy logic!
 */
function match_tier4_low($propiedad_name, $departamento_segment) {
    // DISABLED - Virgin data uses exact matching only
    // This is the tier that was causing street-only nonsense matches:
    //   "Av. México 195" matched "Av. México 175" (same street, different building)
    //   "Alfonso Reyes 120" matched "Alfonso Reyes 176" (same street, different building)
    return ['match' => false];
}

/**
 * AI EXPLANATION GENERATOR - Explain successful match
 */
function explain_match_propietario($match, $propietario, $propiedad) {
    $lines = [];
    $tier = $match['tier'] ?? 99;
    $confidence = $match['confidence'] ?? 0;

    // Tier name
    $tier_names = [
        0 => 'Combo Match',
        1 => 'Perfect Match',
        2 => 'High Confidence Match',
        3 => 'Medium Confidence Match',
        4 => 'Low Confidence Match'
    ];
    $tier_name = $tier_names[$tier] ?? 'Unknown Tier';
    $lines[] = "Tier {$tier}: {$tier_name}";

    // Pattern used
    if (!empty($match['pattern'])) {
        $lines[] = "Method: {$match['pattern']}";
    }

    // Score breakdown
    if (isset($match['scores'])) {
        $scores = $match['scores'];
        if (isset($scores['street'])) {
            $lines[] = "Street Match: {$scores['street']}%";
        }
        if (isset($scores['building']) && $scores['building'] > 0) {
            $lines[] = "Building Match: {$scores['building']}%";
        }
        if (isset($scores['unit']) && $scores['unit'] > 0) {
            $lines[] = "Unit Match: {$scores['unit']}%";
        }
    }

    // Combo details
    if ($tier === 0 && !empty($match['segment'])) {
        $lines[] = "🔗 Combo segment: {$match['segment']}";
    }

    // Warnings
    if ($confidence < 70) {
        $lines[] = "⚠️  LOW CONFIDENCE - Review recommended";
    }

    return implode("\n", $lines);
}

/**
 * NO-MATCH EXPLAINER - Explain why match failed
 */
function explain_no_match_propietario($propietario, $all_propiedades) {
    $lines = [];
    $lines[] = "❌ NO MATCH";

    $departamento = $propietario['departamento'] ?? '';
    $tokens = extract_semantic_tokens($departamento);

    // Reason 1: Extract what we're looking for
    if (!empty($tokens['street'])) {
        $lines[] = "Looking for street: '{$tokens['street']}'";
    }
    if (!empty($tokens['building_number'])) {
        $lines[] = "Looking for building: '{$tokens['building_number']}'";
    }

    // Find closest matches (top 3)
    $closest = [];
    foreach ($all_propiedades as $prop) {
        $scores = compare_addresses_intelligent($prop['nombre_propiedad'], $departamento);
        if ($scores['overall'] > 20) { // At least 20% similarity
            $closest[] = [
                'propiedad' => $prop['nombre_propiedad'],
                'score' => $scores['overall']
            ];
        }
    }

    // Sort by score desc
    usort($closest, function($a, $b) {
        return $b['score'] - $a['score'];
    });

    if (!empty($closest)) {
        $lines[] = "Closest matches:";
        foreach (array_slice($closest, 0, 3) as $match) {
            $lines[] = "  • {$match['propiedad']} ({$match['score']}%)";
        }
    } else {
        $lines[] = "No similar propiedades found";
    }

    return implode("\n", $lines);
}

/**
 * SIMPLIFIED MATCHING FUNCTION - Find propietario for a propiedad
 * 🎯 EXACT CHARACTER-BY-CHARACTER MATCHING ONLY
 *
 * Direction: propiedad → propietario (reversed from original)
 * Logic: Simple exact string comparison after normalization
 * No combo splitting, no fuzzy matching, no tier system
 */
function find_propietario_for_propiedad($propiedad, $propietarios) {
    $prop_dept_normalized = normalize_text($propiedad['departamento']);

    // Loop through all propietarios to find exact match
    foreach ($propietarios as $propietario) {
        $propietario_dept_normalized = normalize_text($propietario['departamento']);

        // EXACT character-by-character match
        if ($prop_dept_normalized === $propietario_dept_normalized) {
            return [
                'match' => true,
                'confidence' => 100,
                'tier' => 1,
                'pattern' => 'exact_match',
                'propietario' => $propietario,
                'scores' => ['overall' => 100, 'street' => 100, 'building' => 100, 'unit' => 100]
            ];
        }
    }

    // No match found
    return [
        'match' => false,
        'confidence' => 0,
        'tier' => 99,
        'pattern' => 'no_match',
        'propietario' => null
    ];
}

// ============================================================================
// DATA LOADING
// ============================================================================

$sql_propiedades = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_propiedades);

$sql_propietarios = "SELECT * FROM propietario ORDER BY propietario";
$propietarios = ia_sqlArrayIndx($sql_propietarios);

// Data validation
if (empty($propiedades) || empty($propietarios)) {
    echo "<div class='alert alert-danger'>";
    echo "<strong>ERROR:</strong> Failed to load data from database!<br><br>";
    if (empty($propiedades)) {
        echo "• Propiedades: 0 records (expected ~350)<br>";
    }
    if (empty($propietarios)) {
        echo "• Propietarios: 0 records (expected ~108)<br>";
    }
    echo "<br><strong>Possible causes:</strong><br>";
    echo "1. Not logged into Quantix (session required)<br>";
    echo "2. Database connection issue<br>";
    echo "3. Tables are empty<br>";
    echo "</div></div></body></html>";
    exit;
}

echo "<div class='alert alert-success'>";
echo "<strong>✓ Data Loaded:</strong> ";
echo count($propietarios) . " propietarios (active), ";
echo count($propiedades) . " propiedades";
echo "</div>";

// Debug mode: Show sample data
if ($debug) {
    echo "<div class='debug-box'>";
    echo "<strong>🐛 DEBUG MODE - Sample Data</strong><br>";
    echo "<strong>Sample Propietario:</strong><pre>";
    print_r($propietarios[0]);
    echo "</pre>";
    echo "<strong>Sample Propiedad:</strong><pre>";
    print_r($propiedades[0]);
    echo "</pre>";
    echo "<strong>🎯 Slash Extraction Test:</strong><br>";
    $test_prop_name = $propiedades[0]['nombre_propiedad'] ?? '';
    echo "Propiedad (raw): '{$test_prop_name}'<br>";
    echo "Propiedad (after slash): '<strong>" . extract_after_slash($test_prop_name) . "</strong>'<br><br>";

    echo "<strong>Token Extraction Test:</strong><br>";
    $test_dept = $propietarios[0]['departamento'] ?? '';
    echo "Departamento: '{$test_dept}'<br>";
    echo "<pre>";
    print_r(extract_semantic_tokens($test_dept));
    echo "</pre>";
    echo "</div>";
}

// ============================================================================
// MATCHING ALGORITHM - REVERSED DIRECTION (PROPIETARIOS → PROPIEDADES)
// ============================================================================

$matches = []; // Stores: propietario → [matched propiedades]
// 🎯 REVERSED DIRECTION: Iterate propiedades → find propietario for each
$stats = [
    'total_propiedades' => count($propiedades),
    'tier1' => 0, // Exact matches
    'unmatched' => 0
];

foreach ($propiedades as $propiedad) {
    $match = find_propietario_for_propiedad($propiedad, $propietarios);

    $matches[] = [
        'propiedad' => $propiedad,
        'match' => $match['match'] ? $match : null
    ];

    if ($match['match']) {
        $tier = $match['tier'];
        $stats["tier{$tier}"]++;
    } else {
        $stats['unmatched']++;
    }
}

// ============================================================================
// DISPLAY STATISTICS
// ============================================================================

echo "<div class='stats'>";
echo "<div class='stat-box'><h3>{$stats['total_propiedades']}</h3><p>Total Propiedades</p></div>";
echo "<div class='stat-box tier1'><h3>{$stats['tier1']}</h3><p>✅ Exact Matches</p></div>";
echo "<div class='stat-box unmatched'><h3>{$stats['unmatched']}</h3><p>❌ Unmatched</p></div>";
echo "</div>";

$coverage = $stats['total_propiedades'] > 0
    ? round((($stats['total_propiedades'] - $stats['unmatched']) / $stats['total_propiedades']) * 100, 1)
    : 0;

echo "<div class='alert alert-info'>";
echo "<strong>Coverage:</strong> {$coverage}% of propiedades matched to propietarios (exact character match)";
echo "</div>";

// ============================================================================
// ACTION HANDLERS - DATABASE UPDATES
// ============================================================================

if ($action === 'apply_high' || $action === 'apply_all') {
    $threshold = ($action === 'apply_high') ? $HIGH_CONFIDENCE_THRESHOLD : 0;
    $updates = 0;

    echo "<div class='alert alert-warning'>";
    echo "<strong>⚡ Applying updates (exact matches only)...</strong>";
    echo "</div>";

    foreach ($matches as $match_data) {
        $propiedad = $match_data['propiedad'];
        $match = $match_data['match'];

        if ($match && $match['confidence'] >= $threshold) {
            $propiedad_id = $propiedad['propiedad_id'];
            $propietario = $match['propietario'];

            // Generate explanation
            $explanation = "Exact match: propiedad.departamento = '{$propiedad['departamento']}' matches propietario.departamento = '{$propietario['departamento']}'";

            // Build scores JSON
            $scores_json = json_encode($match['scores'], JSON_UNESCAPED_UNICODE);

            // Build and execute UPDATE query
            $sql = "UPDATE propiedad
                    SET propietario_id = '{$propietario['propietario_id']}',
                        propietario_nombre_propiedad = " . strit($propietario['departamento']) . ",
                        propietario_match_tier = {$match['tier']},
                        propietario_match_confidence = {$match['confidence']},
                        propietario_match_pattern = " . strit($match['pattern']) . ",
                        propietario_match_explanation = " . strit($explanation) . ",
                        propietario_match_scores = " . strit($scores_json) . ",
                        propietario_match_timestamp = NOW()
                    WHERE propiedad_id = '{$propiedad_id}'";
            ia_query($sql);
            $updates++;
        }
    }

    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Success!</strong> Updated <strong>{$updates}</strong> propiedades.";
    echo "</div>";

    echo "<a href='link_propiedades_propietarios.php' class='btn btn-primary'>← Back to Preview</a>";
    echo "</div></body></html>";
    exit;
}

if ($action === 'export_csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=propietarios_matching_' . date('Y-m-d_His') . '.csv');

    $output = fopen('php://output', 'w');

    // CSV Header
    fputcsv($output, [
        'propietario_id',
        'propietario_name',
        'departamento',
        'propiedad_id',
        'propiedad_name',
        'match_tier',
        'confidence',
        'pattern',
        'explanation'
    ]);

    // CSV Data
    foreach ($matches as $match_data) {
        $propietario = $match_data['propietario'];
        $match = $match_data['match'];

        if ($match) {
            $explanation = explain_match_propietario($match, $propietario, $match['propiedad']);

            fputcsv($output, [
                $propietario['propietario_id'],
                $propietario['propietario'],
                $propietario['departamento'],
                $match['propiedad']['propiedad_id'],
                $match['propiedad']['nombre_propiedad'],
                $match['tier'],
                $match['confidence'],
                $match['pattern'],
                $explanation
            ]);
        } else {
            fputcsv($output, [
                $propietario['propietario_id'],
                $propietario['propietario'],
                $propietario['departamento'],
                '',
                '',
                'NO_MATCH',
                '0',
                '',
                'No match found'
            ]);
        }
    }

    fclose($output);
    exit;
}

// ============================================================================
// DISPLAY RESULTS TABLE
// ============================================================================

echo "<div class='actions'>";
echo "<h3>Actions</h3>";
echo "<a href='?action=apply_all' class='btn btn-success' onclick='return confirm(\"Apply {$stats['tier1']} exact matches?\")'>✓ Apply All Exact Matches</a>";
echo "<a href='?action=export_csv' class='btn btn-primary'>📥 Export CSV</a>";
echo "</div>";

echo "<h2>Matching Results (Propiedades → Propietarios)</h2>";
echo "<table>";
echo "<thead><tr>";
echo "<th>Propiedad</th>";
echo "<th>Departamento (Source)</th>";
echo "<th>→</th>";
echo "<th>Matched Propietario</th>";
echo "<th>Tier</th>";
echo "<th>Conf</th>";
echo "<th>Explanation</th>";
echo "</tr></thead>";
echo "<tbody>";

foreach ($matches as $match_data) {
    $propiedad = $match_data['propiedad'];
    $match = $match_data['match'];

    if ($match) {
        $confidence = $match['confidence'];
        $tier = $match['tier'];
        $propietario = $match['propietario'];

        $row_class = "confidence-100"; // All matches are 100%
        $badge_class = "badge-tier1";

        echo "<tr class='{$row_class}'>";
        echo "<td><strong>" . esc($propiedad['nombre_propiedad']) . "</strong></td>";
        echo "<td><code>" . esc($propiedad['departamento']) . "</code></td>";
        echo "<td>→</td>";
        echo "<td><strong>" . esc($propietario['propietario']) . "</strong></td>";
        echo "<td><span class='badge {$badge_class}'>T{$tier}</span></td>";
        echo "<td><strong>{$confidence}%</strong></td>";
        echo "<td><small style='color:#666;'>Exact match: '{$propiedad['departamento']}' = '{$propietario['departamento']}'</small></td>";
        echo "</tr>";
    } else {
        // NO MATCH - Show propiedad without matching propietario
        echo "<tr class='confidence-0'>";
        echo "<td><strong>" . esc($propiedad['nombre_propiedad']) . "</strong></td>";
        echo "<td><code>" . esc($propiedad['departamento']) . "</code></td>";
        echo "<td colspan='3' style='color: #999; font-style: italic;'>No exact match found</td>";
        echo "<td><span class='badge badge-nomatch'>0%</span></td>";
        echo "<td><small style='color:#999;'>❌ No propietario has matching departamento value</small></td>";
        echo "</tr>";
    }
}

echo "</tbody></table>";

?>

<div class="alert alert-info" style="margin-top: 30px;">
    <strong>💡 How to use matched data:</strong><br>
    After applying updates, the <code>propiedad</code> table will have:<br>
    • <code>propietario_nombre_propiedad</code> - Full departamento text from matched propietario<br>
    • <code>propietario_match_tier</code> - Match quality tier (0-4)<br>
    • <code>propietario_match_confidence</code> - Match confidence (0-100)<br>
    • <code>propietario_match_explanation</code> - AI reasoning<br>
    • <code>propietario_match_scores</code> - JSON with detailed scores<br>
    <br>
    Use these for validation, conflict detection, and quality analysis.
</div>

</div>
</body>
</html>
