        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);

    // Then lowercase
    $text = mb_strtolower($text, 'UTF-8');

    // Normalize whitespace
    $text = trim(preg_replace('/\s+/', ' ', $text));

    return $text;
}

/**
 * Extract street name (text before first digit)
 */
function extract_street_name($text) {
    $normalized = normalize_text($text);

    // Expand property code abbreviations FIRST
    $property_abbreviations = [
        'slp37' => 'san luis potosi 37',
        'slp 37' => 'san luis potosi 37',
        'vs146' => 'vicente suarez 146',
        'vs 146' => 'vicente suarez 146',
        'ver 2' => 'veracruz 26',
        'ver2' => 'veracruz 26',
    ];

    foreach ($property_abbreviations as $abbr => $full) {
        // Handle "CODE-UNIT" format (e.g., "slp37-203" → "san luis potosi 37 | 203")
        if (strpos($normalized, $abbr . '-') !== false) {
            $normalized = str_replace($abbr . '-', $full . ' | ', $normalized);
            break;
        }
        // Handle "CODE " or "CODE" format
        if (strpos($normalized, $abbr) !== false) {
            $normalized = str_replace($abbr, $full, $normalized);
            break;
        }
    }

    // Find position of first digit
    if (preg_match('/\d/', $normalized, $matches, PREG_OFFSET_CAPTURE)) {
        $pos = $matches[0][1];
        return trim(substr($normalized, 0, $pos));
    }

    return $normalized;
}

/**
 * Extract unit number from text (extract digits/patterns like "103", "401", "PH1")
 */
function extract_unit_number($text) {
    $normalized = normalize_text($text);

    // Try to extract common unit patterns
    // Pattern 1: Pure numbers (103, 401, 302)
    if (preg_match('/\b(\d{2,4})\b/', $normalized, $matches)) {
        return $matches[1];
    }

    // Pattern 2: Penthouse (PH1, PH2, ph chico)
    if (preg_match('/ph\s*(\w+)/i', $normalized, $matches)) {
        return 'ph' . normalize_text($matches[1]);
    }

    // Pattern 3: Suite (SU4, Suite 4)
    if (preg_match('/su(?:ite)?\s*(\w+)/i', $normalized, $matches)) {
        return 'su' . normalize_text($matches[1]);
    }

    // Pattern 4: Single letter suffix (A, B, C)
    if (preg_match('/\b([A-Z])\b/', $text, $matches)) {
        return strtolower($matches[1]);
    }

    return '';
}

// ============================================================================
// 🤖 THOTH'S ALGORITHM: SEMANTIC INTELLIGENCE LAYER
// ============================================================================
// AI-Powered token extraction, pattern recognition, and multi-dimensional matching
// Built by Filemón Prime — 2026-01-04
// ============================================================================

/**
 * SEMANTIC TOKEN EXTRACTOR - The AI Brain
 *
 * Extracts meaning from chaotic reservation text. Understands:
 * - Brands ("Mr W", "Casa", "El")
 * - Streets, building numbers, units
 * - Descriptors ("Arena", "Mar", "Grande", "Doble")
 * - Cryptic codes ("RoP2BQQ", "CoS1BK")
 * - Metro stations, person names, floors
 *
 * @param string $text - Raw text (anuncio, property name, room number, etc.)
 * @return array - Structured tokens with semantic meaning
 */
function extract_semantic_tokens($text) {
    if (empty($text)) return ['raw' => ''];

    $norm = normalize_text($text);
    $tokens = [
        'brand' => null,            // "mr w", "casa", "el"
        'street' => null,           // "tonala", "amsterdam", "medellin"
        'building_number' => null,  // "127", "210", "148"
        'unit' => null,             // Final normalized unit identifier
        'unit_type' => null,        // "suite", "ph", "su", "piso"
        'unit_number' => null,      // Numeric part of unit
        'descriptors' => [],        // ["doble", "grande", "arena"]
        'codes' => [],              // ["RoP2BQQ", "CoS1BK"]
        'metro_station' => null,    // "pantitlan", "mixcoac"
        'person_name' => null,      // "Karen Kling", "Chris"
        'floor' => null,            // "piso 1", "floor 3"
        'raw' => $text              // Original unmodified text
    ];

    // 0. Property code abbreviations - expand BEFORE other token extraction
    $property_abbreviations = [
        'slp37' => 'san luis potosi 37',
        'slp 37' => 'san luis potosi 37',
        'vs146' => 'vicente suarez 146',
        'vs 146' => 'vicente suarez 146',
        'ver 2' => 'veracruz 26',     // Maps to Tigre properties on Veracruz 26
        'ver2' => 'veracruz 26',
        'p.e.21' => '',               // Need to identify actual property
        'pe21' => '',
    ];

    // Expand abbreviations in normalized text
    // Handle both "CODE-UNIT" and "CODE UNIT" and "CODE" formats
    foreach ($property_abbreviations as $abbr => $full) {
        if (!empty($full)) {
            // Convert to regex-safe pattern
            $pattern = '/' . preg_quote($abbr, '/') . '/i';

            // Check if abbreviation exists in text
            if (preg_match($pattern, $norm)) {
                // Replace "CODE-" with "FULL |" (e.g., "slp37-203" → "san luis potosi 37 | 203")
                if (preg_match('/' . preg_quote($abbr, '/') . '-/i', $norm)) {
                    $norm = preg_replace('/' . preg_quote($abbr, '/') . '-/i', $full . ' | ', $norm);
                } else {
                    // Simple replace for other formats
                    $norm = preg_replace($pattern, $full, $norm);
                }
                break;
            }
        }
    }

    // 1. Extract and strip brand prefixes (Cloudbeds special handling)
    $brands = ['mr w ', 'casa ', 'el ', 'the ', 'casitas by the sea '];
    foreach ($brands as $brand) {
        if (strpos($norm, $brand) === 0) {
            $tokens['brand'] = trim($brand);
            $norm = substr($norm, strlen($brand));
            break;
        }
    }

    // 2. Extract cryptic codes (patterns: Capital + lowercase + numbers)
    // Examples: "RoP2BQQ", "CoS1BK", "CuS2BKK", "JuGPH2BrMM"
    if (preg_match_all('/\b[A-Z][a-z]{0,2}[A-Z0-9]{2,}\b/', $text, $matches)) {
        $tokens['codes'] = $matches[0];
        // Strip codes from normalized text
        foreach ($matches[0] as $code) {
            $norm = str_replace(strtolower($code), '', $norm);
        }
    }

    // 3. Extract unit type keywords
    $unit_types = [
        'suite' => '/\bsuite\b/i',
        'penthouse' => '/\bpenthouse\b/i',
        'ph' => '/\bph\b/i',
        'su' => '/\bsu\d/i',
        'piso' => '/\bpiso\b/i',
        'floor' => '/\bfloor\b/i'
    ];
    foreach ($unit_types as $type => $pattern) {
        if (preg_match($pattern, $norm)) {
            $tokens['unit_type'] = $type;
            break; // Take first match
        }
    }

    // 4. Extract descriptors (semantic qualifiers)
    $descriptors_patterns = [
        'doble' => '/\bdoble\b/i',
        'triple' => '/\btriple\b/i',
        'grande' => '/\bgrande\b/i',
        'chico' => '/\bchico\b/i',
        'arena' => '/\barena\b/i',
        'mar' => '/\bmar\b/i',
        'duplex' => '/\bduplex\b/i',
        'loft' => '/\bloft\b/i'
    ];
    foreach ($descriptors_patterns as $desc => $pattern) {
        if (preg_match($pattern, $norm)) {
            $tokens['descriptors'][] = $desc;
        }
    }

    // 5. Extract metro stations (Mexico City metro + common location names)
    $metro_stations = [
        'pantitlan', 'mixcoac', 'balderas', 'candelaria', 'tacuba',
        'copilco', 'nativitas', 'insurgentes', 'condesa', 'polanco',
        'roma', 'juarez', 'tabacalera', 'cuauhtemoc', 'escandon'
    ];
    foreach ($metro_stations as $station) {
        if (strpos($norm, $station) !== false) {
            $tokens['metro_station'] = $station;
            break;
        }
    }

    // 6. Extract person names (Capitalized words after properties)
    // Pattern: "VS146 - 102 - Karen Kling" → "Karen Kling"
    if (preg_match('/\b([A-Z][a-z]+ [A-Z][a-z]+)\b/', $text, $matches)) {
        // Exclude common words that might match pattern
        $exclude = ['Suite', 'Casa', 'Forma Reforma', 'Rio Elba'];
        if (!in_array($matches[1], $exclude)) {
            $tokens['person_name'] = $matches[1];
        }
    }

    // 7. Extract floor designation
    if (preg_match('/piso\s+(\d+)/i', $norm, $matches)) {
        $tokens['floor'] = 'piso' . $matches[1];
    } elseif (preg_match('/floor\s+(\d+)/i', $norm, $matches)) {
        $tokens['floor'] = 'floor' . $matches[1];
    }

    // 8. Extract street name (cleaned text before first digit or separator)
    $tokens['street'] = extract_street_name($norm);

    // 9. Extract building number (2-4 digit numbers in address context)
    // Priority: number AFTER street name but BEFORE unit
    $street_pos = strpos($norm, $tokens['street']);
    if ($street_pos !== false) {
        $after_street = substr($norm, $street_pos + strlen($tokens['street']));
        if (preg_match('/\b(\d{2,4})\b/', $after_street, $matches)) {
            $tokens['building_number'] = $matches[1];
        }
    }

    // 10. Extract unit (advanced logic - calls separate function)
    $tokens['unit'] = extract_unit_advanced($text, $norm, $tokens);

    // Extract unit number (numeric part of unit if exists)
    if ($tokens['unit']) {
        if (preg_match('/(\d+)/', $tokens['unit'], $matches)) {
            $tokens['unit_number'] = $matches[1];
        }
    }

    return $tokens;
}

/**
 * ADVANCED UNIT EXTRACTOR - Handles 15+ chaotic formats
 *
 * Supported formats:
 * 1. Simple numbers: "502", "401", "103"
 * 2. Suite formats: "Suite 5", "Suite 10", "SU4"
 * 3. Penthouse: "PH1", "PH2", "PH Chico", "PH Grande", "2PH(1)"
 * 4. With parentheses: "SU1(1)", "SU2(1), SU1(1)"
 * 5. Floor designation: "Piso 1", "Floor 3"
 * 6. Letter units: "- A", "- H", "OCHO - E"
 * 7. Two-digit: "- 01", "- 23"
 * 8. Descriptors as units: "Arena" → "a", "Mar" → "m"
 * 9. Complex: "302, 602, 402 y DOBLES - A"
 *
 * @param string $text - Original text (NOT normalized, to preserve case)
 * @param string $norm - Normalized text
 * @param array $tokens - Tokens extracted so far (for context)
 * @return string|null - Normalized unit identifier or null
 */
function extract_unit_advanced($text, $norm, $tokens) {
    // Format 1: "SU1(1)" → "su1" (common in Casa Tenue)
    if (preg_match('/\bSU(\d+)\(\d+\)/i', $text, $matches)) {
        return 'su' . $matches[1];
    }

    // Format 2: "PH Chico" / "PH Grande" (named penthouses)
    if (preg_match('/\bPH\s+(Chico|Grande)/i', $text, $matches)) {
        return 'ph' . strtolower($matches[1]);
    }

    // Format 3: "2PH(1)" → "ph2" (multiple penthouses)
    if (preg_match('/(\d)PH\(\d+\)/i', $text, $matches)) {
        return 'ph' . $matches[1];
    }

    // Format 4: "Suite 10", "Suite 5" (with space and number)
    if ($tokens['unit_type'] === 'suite') {
        if (preg_match('/\bsuite\s+(\d+)/i', $norm, $matches)) {
            return 'suite' . $matches[1];
        }
    }

    // Format 5: "PH1", "PH2" (no space)
    if (preg_match('/\bPH(\d+)\b/i', $text, $matches)) {
        return 'ph' . $matches[1];
    }

    // Format 6: "Piso 1", "Floor 3"
    if ($tokens['floor']) {
        return $tokens['floor']; // Already normalized
    }

    // Format 7: Letter units after dash or space: "Amsterdam - A", "OCHO - E"
    // Look for pattern: "- LETTER" or "WORD - LETTER"
    if (preg_match('/[-\s]([A-J])\b/', $text, $matches)) {
        return strtolower($matches[1]);
    }

    // Format 8: Two-digit units: "Tonalá - 01", "Casa Ofelia - 23"
    // Pattern: "- NN" where NN is 01-99
    if (preg_match('/[-\s](\d{2})\b/', $text, $matches)) {
        // Avoid matching building numbers (e.g., "Amsterdam 210")
        // Only match if after a dash or at end of string
        if (strpos($text, '- ' . $matches[1]) !== false ||
            preg_match('/' . $matches[1] . '\s*$/i', $text)) {
            return $matches[1];
        }
    }

    // Format 9: Descriptor as unit identifier
    // "Casitas by the Sea Arena" → "a"
    // "Casitas by the Sea Mar" → "m"
    if (in_array('arena', $tokens['descriptors'])) {
        return 'a'; // Arena = A
    }
    if (in_array('mar', $tokens['descriptors'])) {
        return 'm'; // Mar = M (or could be 'b' for second)
    }

    // Format 10: Three-digit room numbers: "502", "302", "401"
    // Common in Mr W properties
    if (preg_match('/\b([1-5]\d{2})\b/', $norm, $matches)) {
        return $matches[1];
    }

    // Format 11: Pure numbers (last resort): "103", "7", "10"
    if (preg_match('/\b(\d{1,4})\b/', $norm, $matches)) {
        $num = $matches[1];
        // Avoid matching building numbers (100-500 range is ambiguous)
        // Only match if it's clearly a unit (small numbers or after indicators)
        if ($num < 100 || $num > 500) {
            return $num;
        }
    }

    // Format 12: If unit_type exists but no number, try to extract nearby number
    if ($tokens['unit_type']) {
        $type = $tokens['unit_type'];
        if (preg_match("/{$type}\s*(\d+)/i", $norm, $matches)) {
            return $type . $matches[1];
        }
    }

    return null; // Cannot extract unit
}

// ============================================================================
// CLOUDBEDS MATCHING FUNCTIONS
// ============================================================================

/**
 * Match Cloudbeds: Hybrid building + unit matching
 * Returns: ['match' => bool, 'confidence' => 0-100, 'building_score' => 0-100, 'unit_score' => 0-100]
 */
function match_cloudbeds($propiedad_nombre, $propiedad_direccion, $cb_property, $cb_room_number) {
    // Extract street from propiedad
    $prop_street = extract_street_name($propiedad_nombre);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($propiedad_direccion);
    }

    // Extract street from cloudbeds property
    $cb_street = extract_street_name($cb_property);

    // PART 1: Building match (60% weight)
    $building_score = 0;
    $norm_cb_property = normalize_text($cb_property);
    $norm_prop_nombre = normalize_text($propiedad_nombre);
    $norm_prop_dir = normalize_text($propiedad_direccion);

    // Check if property name contains the building name
    if (strlen($norm_cb_property) >= 5) {
        if (strpos($norm_prop_nombre, $norm_cb_property) !== false ||
            strpos($norm_prop_dir, $norm_cb_property) !== false) {
            $building_score = 100;
        } elseif (strlen($prop_street) >= 5 && strlen($cb_street) >= 5) {
            // Street name similarity
            similar_text($prop_street, $cb_street, $percent);
            $building_score = $percent;
        }
    }

    // PART 2: Unit match (40% weight)
    $unit_score = 0;
    $prop_unit = extract_unit_number($propiedad_nombre);
    $cb_unit = extract_unit_number($cb_room_number);

    if (!empty($prop_unit) && !empty($cb_unit)) {
        if ($prop_unit === $cb_unit) {
            $unit_score = 100;
        } else {
            // Partial match (e.g., "103" vs "03")
            similar_text($prop_unit, $cb_unit, $percent);
            $unit_score = $percent;
        }
    } elseif (empty($cb_unit) || $cb_room_number === 'N/A') {
        // If no unit specified in Cloudbeds, use only building score
        $unit_score = $building_score;
    }

    // Combined confidence: 60% building + 40% unit
    $confidence = round(($building_score * 0.6) + ($unit_score * 0.4));

    // Determine tier based on confidence
    $tier = 0;
    if ($confidence >= 95) $tier = 1;
    elseif ($confidence >= 80) $tier = 2;
    elseif ($confidence >= 65) $tier = 3;
    elseif ($confidence >= 40) $tier = 4;

    if ($tier > 0) {
        // Generate pattern description
        $pattern = "hybrid_building+unit ({$cb_property}";
        if (!empty($cb_room_number)) {
            $pattern .= " + {$cb_room_number}";
        }
        $pattern .= ")";

        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => $tier,
            'building_score' => round($building_score),
            'unit_score' => round($unit_score),
            'prop_unit' => $prop_unit,
            'cb_unit' => $cb_unit,
            'pattern' => $pattern
        ];
    }

    return ['match' => false];
}

// ============================================================================
// COMBO UNIT DETECTION & EXPANSION (TIER 0 - HIGHEST PRIORITY)
// ============================================================================

/**
 * MEGA COMBO EXPANDER - Detects 8 multi-unit patterns
 *
 * Enhanced to handle INSANE real-world combo formats discovered in production data.
 *
 * Supported Patterns:
 * 1. "Doble X y Y" → [X, Y]
 * 2. "X y Y" → [X, Y]
 * 3. "X | Y" → [X, Y]
 * 4. "Triple - A, B y C" → [A, B, C]
 * 5. "Suite 1, Suite 4, Suite 10" → [1, 4, 10] (MEGA COMBO!)
 * 6. "204, 103, 401, 203, 303" → [204, 103, 401, 203, 303] (5-unit!)
 * 7. "SU2(1), SU1(1)" → [2, 1]
 * 8. "302, 602, 402 y DOBLES" → [302, 602, 402]
 *
 * @param string $text - Anuncio or room_number text
 * @return array - ['units' => [...], 'type' => 'combo_type', 'street' => '...'] or [] if no combo
 */
function expand_combo_anuncio($text) {
    $norm = normalize_text($text);
    $result = [];

    // PATTERN 1: "Doble X y Y" (Ometusco Doble 5 y 6)
    if (preg_match('/doble\s+(\d+)\s+y\s+(\d+)/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'doble_y',
            'street' => extract_street_name($text),
            'pattern_raw' => $matches[0]
        ];
    }

    // PATTERN 5: "Suite 1, Suite 4, Suite 10, Suite 3, Suite 5" (MEGA SUITE COMBO!)
    if (preg_match_all('/suite\s+(\d+)/i', $norm, $matches)) {
        if (count($matches[1]) > 1) {
            return [
                'units' => $matches[1],
                'type' => 'multi_suite',
                'street' => extract_street_name($text),
                'pattern_raw' => implode(', ', array_slice($matches[0], 0, 3)) . '...'
            ];
        }
    }

    // PATTERN 7: "SU2(1), SU1(1)" (Casa Tenue format)
    if (preg_match_all('/su(\d+)\(\d+\)/i', $norm, $matches)) {
        if (count($matches[1]) > 1) {
            return [
                'units' => $matches[1],
                'type' => 'multi_su',
                'street' => extract_street_name($text),
                'pattern_raw' => implode(', ', array_slice($matches[0], 0, 3))
            ];
        }
    }

    // PATTERN 6: "204, 103, 401, 203, 303" (comma-separated numbers)
    // Must have commas to avoid matching street addresses
    if (strpos($text, ',') !== false) {
        if (preg_match_all('/\b(\d{2,4})\b/', $norm, $matches)) {
            $units = $matches[1];
            // Filter out building numbers (typically first occurrence)
            if (count($units) >= 2) {
                // If first number is 100-500 range, might be building - keep all
                return [
                    'units' => $units,
                    'type' => 'comma_separated',
                    'street' => extract_street_name($text),
                    'pattern_raw' => implode(',', array_slice($units, 0, 5))
                ];
            }
        }
    }

    // PATTERN 8: "302, 602, 402 y DOBLES" (mixed comma + y + descriptor)
    if (preg_match('/(\d{2,4})\s*,\s*(\d{2,4})\s*,?\s*(\d{2,4})?\s+y\s+/i', $norm, $matches)) {
        $units = array_filter([$matches[1], $matches[2], $matches[3] ?? null]);
        if (count($units) >= 2) {
            return [
                'units' => $units,
                'type' => 'mixed_combo_y',
                'street' => extract_street_name($text),
                'pattern_raw' => $matches[0]
            ];
        }
    }

    // PATTERN 2: "X y Y" without "Doble" (Amsterdam 302 y 402)
    if (preg_match('/(\d{2,4})\s+y\s+(\d{2,4})/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'y_connector',
            'street' => extract_street_name($text),
            'pattern_raw' => $matches[0]
        ];
    }

    // PATTERN 3: "X | Y" pipe separator (Rio Elba 50 - Doble - 101 | 103)
    if (preg_match('/(\d{2,4})\s*\|\s*(\d{2,4})/i', $norm, $matches)) {
        return [
            'units' => [$matches[1], $matches[2]],
            'type' => 'pipe_separator',
            'street' => extract_street_name($text),
            'pattern_raw' => $matches[0]
        ];
    }

    // PATTERN 4: "Triple - A, B y C" (named units)
    if (preg_match('/triple/i', $norm)) {
        $parts = preg_split('/\s*,\s*|\s+y\s+/', $norm);
        $units = [];
        $exclude = ['triple', 'doble', 'laredo', 'balderas', 'candelaria', 'nativitas'];
        foreach ($parts as $part) {
            $part = trim($part);
            if (strlen($part) >= 5 && !in_array($part, $exclude)) {
                $units[] = $part;
            }
        }
        if (count($units) >= 2) {
            return [
                'units' => $units,
                'type' => 'triple',
                'street' => extract_street_name($text),
                'pattern_raw' => 'triple combo'
            ];
        }
    }

    return []; // Not a combo
}

/**
 * Match Hostify combo listings (TIER 0 - highest priority)
 * Handles multi-unit anuncios like "Ometusco Doble 5 y 6"
 *
 * @return array - Same structure as other match functions, with tier=0 for combos
 */
function match_hostify_combo($propiedad_name, $propiedad_direccion, $anuncio) {
    // Try to expand combo
    $combo = expand_combo_anuncio($anuncio);

    if (empty($combo) || empty($combo['units'])) {
        return ['match' => false];
    }

    // Extract street from propiedad
    $prop_street = extract_street_name($propiedad_name);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($propiedad_direccion);
    }

    // Check if streets match
    $norm_prop_street = normalize_text($prop_street);
    $norm_combo_street = normalize_text($combo['street']);

    if (strlen($norm_prop_street) < 5 || strlen($norm_combo_street) < 5) {
        return ['match' => false];
    }

    // Street must match (either contains or high similarity)
    $street_matches = false;
    if (strpos($norm_combo_street, $norm_prop_street) !== false ||
        strpos($norm_prop_street, $norm_combo_street) !== false) {
        $street_matches = true;
    } else {
        similar_text($norm_prop_street, $norm_combo_street, $percent);
        if ($percent >= 70) {
            $street_matches = true;
        }
    }

    if (!$street_matches) {
        return ['match' => false];
    }

    // SPECIAL CASE: Ometusco is ONE property with multiple room combos
    // "Ometusco Doble 5 y 6" → "Ometusco - 3" (regardless of numbers)
    if (strpos($norm_combo_street, 'ometusco') !== false &&
        strpos($norm_prop_street, 'ometusco') !== false) {
        return [
            'match' => true,
            'confidence' => 85,  // High confidence - street match confirmed
            'tier' => 0,
            'combo_type' => $combo['type'],
            'combo_units' => $combo['units'],
            'matched_unit' => 'property_level',  // Not unit-specific
            'pattern' => "combo_ometusco_special ({$anuncio} → {$propiedad_name})"
        ];
    }

    // Extract unit from propiedad
    $prop_unit = extract_unit_number($propiedad_name);

    if (empty($prop_unit)) {
        return ['match' => false];
    }

    // Check if propiedad unit matches ANY combo unit
    foreach ($combo['units'] as $combo_unit) {
        $norm_combo_unit = normalize_text($combo_unit);

        // Direct match
        if ($prop_unit === $norm_combo_unit) {
            return [
                'match' => true,
                'confidence' => 90,
                'tier' => 0, // Special tier for combos
                'combo_type' => $combo['type'],
                'combo_units' => $combo['units'],
                'matched_unit' => $combo_unit,
                'pattern' => "combo_{$combo['type']} ({$anuncio} → unit {$combo_unit})"
            ];
        }

        // NUMBER → LETTER conversion for properties like "Ometusco - E" (E=5)
        // If combo unit is a number (1-26), convert to letter (a-z)
        if (is_numeric($norm_combo_unit) && $norm_combo_unit >= 1 && $norm_combo_unit <= 26) {
            $letter_equivalent = chr(96 + intval($norm_combo_unit)); // 1=a, 2=b, ..., 5=e, 6=f
            if ($prop_unit === $letter_equivalent) {
                return [
                    'match' => true,
                    'confidence' => 88, // Slightly lower for number→letter conversion
                    'tier' => 0,
                    'combo_type' => $combo['type'],
                    'combo_units' => $combo['units'],
                    'matched_unit' => $combo_unit,
                    'pattern' => "combo_{$combo['type']}_num2letter ({$anuncio} → {$combo_unit}={$letter_equivalent})"
                ];
            }
        }

        // Partial match (e.g., "5" in combo vs "05" or "305" in propiedad)
        if (strlen($norm_combo_unit) >= 1 && strlen($prop_unit) >= 1) {
            if (strpos($prop_unit, $norm_combo_unit) !== false ||
                strpos($norm_combo_unit, $prop_unit) !== false) {
                return [
                    'match' => true,
                    'confidence' => 85, // Slightly lower for partial
                    'tier' => 0,
                    'combo_type' => $combo['type'],
                    'combo_units' => $combo['units'],
                    'matched_unit' => $combo_unit,
                    'pattern' => "combo_{$combo['type']}_partial ({$anuncio} → unit {$combo_unit})"
                ];
            }
        }
    }

    return ['match' => false];
}

// ============================================================================
// 📊 MULTI-DIMENSIONAL SCORING & EXPLANATION ENGINE
// ============================================================================

/**
 * INTELLIGENT UNIT COMPARISON - Handles number→letter, semantic equivalence
 *
 * @param string $unit1 - First unit identifier
 * @param string $unit2 - Second unit identifier
 * @return int - Similarity score 0-100
 */
function compare_units_intelligent($unit1, $unit2) {
    if (empty($unit1) || empty($unit2)) return 0;

    $u1 = normalize_text($unit1);
    $u2 = normalize_text($unit2);

    // Exact match
    if ($u1 === $u2) return 100;

    // Number → Letter conversion (5→e, 6→f, etc.)
    if (is_numeric($u1) && $u1 >= 1 && $u1 <= 26) {
        $letter = chr(96 + intval($u1));
        if ($letter === $u2) return 95; // High confidence
    }
    if (is_numeric($u2) && $u2 >= 1 && $u2 <= 26) {
        $letter = chr(96 + intval($u2));
        if ($letter === $u1) return 95;
    }

    // Extract numeric parts and compare
    preg_match('/(\d+)/', $u1, $m1);
    preg_match('/(\d+)/', $u2, $m2);
    if (!empty($m1[1]) && !empty($m2[1])) {
        if ($m1[1] === $m2[1]) return 90; // Numbers match
    }

    // String similarity fallback
    similar_text($u1, $u2, $percent);
    return round($percent);
}

/**
 * MATCH EXPLANATION GENERATOR
 *
 * Generates human-readable explanation of WHY a match succeeded.
 *
 * @param array $match - Match result array
 * @param array $reservation - Reservation data
 * @param array $propiedad - Propiedad data
 * @return string - Multi-line explanation text
 */
function explain_match($match, $reservation, $propiedad) {
    if (!$match || !$match['match']) {
        return '';
    }

    $lines = [];
    $tier = $match['tier'] ?? 99;
    $confidence = $match['confidence'] ?? 0;

    // Tier name
    $tier_names = [
        0 => 'Combo Match',
        1 => 'Perfect Match',
        2 => 'High Confidence Match',
        3 => 'Medium Confidence Match',
        4 => 'Low Confidence Match'
    ];
    $tier_name = $tier_names[$tier] ?? 'Unknown Tier';
    $lines[] = "Tier {$tier}: {$tier_name}";

    // Pattern used
    if (!empty($match['pattern'])) {
        $lines[] = "Method: {$match['pattern']}";
    }

    // Score breakdown (if available)
    if (isset($match['building_score']) && isset($match['unit_score'])) {
        $lines[] = "Building Match: {$match['building_score']}%";
        $lines[] = "Unit Match: {$match['unit_score']}%";
    }

    // Combo details
    if ($tier === 0 && !empty($match['combo_type'])) {
        $lines[] = "🔗 Multi-unit combo: {$match['combo_type']}";
        if (!empty($match['matched_unit'])) {
            $lines[] = "Matched unit: {$match['matched_unit']}";
        }
    }

    // Warnings
    if ($confidence < 70) {
        $lines[] = "⚠️  LOW CONFIDENCE - Recommend review";
    }

    return implode("\n", $lines);
}

/**
 * NO-MATCH EXPLANATION GENERATOR
 *
 * Explains WHY a reservation couldn't be matched and suggests fixes.
 *
 * @param array $reservation - Reservation data (anuncio or property+room_number)
 * @param array $all_propiedades - All available propiedades for finding closest matches
 * @param string $source - 'hostify' or 'cloudbeds'
 * @return string - Multi-line explanation with suggestions
 */
function explain_no_match($reservation, $all_propiedades, $source = 'hostify') {
    $lines = [];
    $lines[] = "❌ NO MATCH";

    // Extract text to analyze
    if ($source === 'hostify') {
        $text = $reservation['anuncio'] ?? '';
    } else {
        $text = ($reservation['property'] ?? '') . ' ' . ($reservation['room_number'] ?? '');
    }

    // Use semantic tokens to understand what we're looking for
    $tokens = extract_semantic_tokens($text);

    // Reason 1: Street not found
    $street_found = false;
    if ($tokens['street']) {
        foreach ($all_propiedades as $prop) {
            $prop_tokens = extract_semantic_tokens($prop['nombre_propiedad']);
            if ($prop_tokens['street']) {
                similar_text($tokens['street'], $prop_tokens['street'], $percent);
                if ($percent > 50) {
                    $street_found = true;
                    break;
                }
            }
        }
        if (!$street_found) {
            $lines[] = "Street '{$tokens['street']}' not in database";
        }
    }

    // Reason 2: Unit format issue
    if (!$tokens['unit'] && !empty($text)) {
        $lines[] = "Could not extract unit number";
    }

    // Find closest matches (top 3)
    $similarities = [];
    foreach ($all_propiedades as $prop) {
        similar_text(
            normalize_text($text),
            normalize_text($prop['nombre_propiedad']),
            $percent
        );
        if ($percent > 20) { // Only show if somewhat similar
            $similarities[$prop['nombre_propiedad']] = round($percent);
        }
    }
    arsort($similarities);
    $top3 = array_slice($similarities, 0, 3, true);

    if (!empty($top3)) {
        $lines[] = "Closest matches:";
        foreach ($top3 as $name => $sim) {
            $lines[] = "  • {$name} ({$sim}%)";
        }
    }

    return implode("\n", $lines);
}

// ============================================================================
// HOSTIFY MATCHING FUNCTIONS (4-Tier Fuzzy)
// ============================================================================

/**
 * TIER 1: Exact match (after normalization)
 */
function match_hostify_tier1($propiedad_name, $anuncio) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_anuncio = normalize_text($anuncio);

    if ($norm_prop === $norm_anuncio) {
        return ['match' => true, 'confidence' => 100, 'tier' => 1, 'pattern' => "exact_match ({$anuncio})"];
    }

    return ['match' => false];
}

/**
 * TIER 2: Contains match (handle pipe-separated lists and partial matches)
 */
function match_hostify_tier2($propiedad_name, $propiedad_direccion, $anuncio) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_dir = normalize_text($propiedad_direccion);
    $norm_anuncio = normalize_text($anuncio);

    // Split anuncio by common separators
    $anuncio_segments = preg_split('/[\|,]/', $norm_anuncio);
    $anuncio_segments = array_map('trim', $anuncio_segments);

    foreach ($anuncio_segments as $segment) {
        // Avoid very short matches
        if (strlen($segment) >= 5 && strlen($norm_prop) >= 5) {
            // Check if segment contains property name or vice versa
            if (strpos($segment, $norm_prop) !== false || strpos($norm_prop, $segment) !== false) {
                return [
                    'match' => true,
                    'confidence' => 90,
                    'tier' => 2,
                    'segment' => $segment,
                    'pattern' => "contains_segment ({$segment})"
                ];
            }

            // Check against direccion too
            if (strlen($norm_dir) >= 5 && (strpos($segment, $norm_dir) !== false || strpos($norm_dir, $segment) !== false)) {
                return [
                    'match' => true,
                    'confidence' => 90,
                    'tier' => 2,
                    'segment' => $segment,
                    'pattern' => "contains_segment_dir ({$segment})"
                ];
            }
        }
    }

    return ['match' => false];
}

/**
 * TIER 3: Similarity match using similar_text()
 */
function match_hostify_tier3($propiedad_name, $propiedad_direccion, $anuncio, $threshold = 85) {
    $norm_prop = normalize_text($propiedad_name);
    $norm_dir = normalize_text($propiedad_direccion);
    $norm_anuncio = normalize_text($anuncio);

    // Try against full anuncio
    similar_text($norm_prop, $norm_anuncio, $percent);
    if ($percent >= $threshold) {
        return [
            'match' => true,
            'confidence' => 70,
            'tier' => 3,
            'similarity' => round($percent, 1),
            'pattern' => "similarity_" . round($percent, 0) . "% ({$norm_anuncio})"
        ];
    }

    // Try direccion
    similar_text($norm_dir, $norm_anuncio, $percent);
    if ($percent >= $threshold) {
        return [
            'match' => true,
            'confidence' => 70,
            'tier' => 3,
            'similarity' => round($percent, 1),
            'pattern' => "similarity_dir_" . round($percent, 0) . "% ({$norm_anuncio})"
        ];
    }

    // Try against segments
    $anuncio_segments = preg_split('/[\|,]/', $norm_anuncio);
    foreach ($anuncio_segments as $segment) {
        $segment = trim($segment);
        if (strlen($segment) >= 5) {
            similar_text($norm_prop, $segment, $percent);
            if ($percent >= $threshold) {
                return [
                    'match' => true,
                    'confidence' => 70,
                    'tier' => 3,
                    'similarity' => round($percent, 1),
                    'segment' => $segment,
                    'pattern' => "similarity_segment_" . round($percent, 0) . "% ({$segment})"
                ];
            }
        }
    }

    return ['match' => false];
}

/**
 * TIER 4: Partial street name + unit match
 */
function match_hostify_tier4($propiedad_name, $propiedad_direccion, $anuncio) {
    $prop_street = extract_street_name($propiedad_name);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($propiedad_direccion);
    }
    $anuncio_street = extract_street_name($anuncio);

    // Need at least 5 chars for valid street match
    if (strlen($prop_street) < 5 || strlen($anuncio_street) < 5) {
        return ['match' => false];
    }

    // Check if streets match
    if (strpos($anuncio_street, $prop_street) !== false || strpos($prop_street, $anuncio_street) !== false) {
        // Try to match units too
        $prop_unit = extract_unit_number($propiedad_name);
        $anuncio_unit = extract_unit_number($anuncio);

        $confidence = 50; // Base for street match
        $pattern_type = "street_match";

        if (!empty($prop_unit) && !empty($anuncio_unit) && $prop_unit === $anuncio_unit) {
            $confidence = 65; // Boost if unit also matches
            $pattern_type = "street+unit_match";
        }

        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => 4,
            'street' => $prop_street,
            'pattern' => "{$pattern_type} ({$prop_street})"
        ];
    }

    return ['match' => false];
}

// ============================================================================
// DATA LOADING
// ============================================================================

// Load all propiedades
$sql_propiedades = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_propiedades);

// Load all cloudbeds reservations (2025 and onwards only)
$sql_cloudbeds = "SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' ORDER BY check_in_date DESC";
$cloudbeds_reservas = ia_sqlArrayIndx($sql_cloudbeds);

// Load all hostify reservations (2025 and onwards only)
$sql_hostify = "SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01' ORDER BY check_in DESC";
$hostify_reservas = ia_sqlArrayIndx($sql_hostify);

// Data validation
if (empty($propiedades)) {
    echo "<div class='alert alert-warning'>";
    echo "<strong>ERROR:</strong> No propiedades loaded from database!<br>";
    echo "Please ensure you're logged in and the table has data.";
    echo "</div></div></body></html>";
    exit;
}

echo "<div class='alert alert-info'>";
echo "<strong>Data Loaded:</strong> ";
echo count($propiedades) . " properties, ";
echo count($cloudbeds_reservas) . " Cloudbeds reservations, ";
echo count($hostify_reservas) . " Hostify reservations";
echo "</div>";

// Debug mode: Show data structure
if ($debug) {
    echo "<div class='alert alert-warning'>";
    echo "<strong>🐛 DEBUG MODE ACTIVE</strong><br>";
    echo "<strong>Sample Propiedad:</strong><pre>" . print_r($propiedades[0], true) . "</pre>";
    if (!empty($cloudbeds_reservas)) {
        echo "<strong>Sample Cloudbeds:</strong><pre>" . print_r($cloudbeds_reservas[0], true) . "</pre>";
    }
    if (!empty($hostify_reservas)) {
        echo "<strong>Sample Hostify:</strong><pre>" . print_r($hostify_reservas[0], true) . "</pre>";
    }
    echo "</div>";
}

// ============================================================================
// REVERSE MATCHING ALGORITHMS (RESERVATION → PROPIEDAD)
// ============================================================================

$cloudbeds_matches = [];
$hostify_matches = [];

$stats_cloudbeds = [
    'total_reservations' => count($cloudbeds_reservas),
    'tier1' => 0,
    'tier2' => 0,
    'tier3' => 0,
    'tier4' => 0,
    'unmatched' => 0,
    'high_confidence' => 0,
    'low_confidence' => 0,
];

$stats_hostify = [
    'total_reservations' => count($hostify_reservas),
    'tier0' => 0,  // Combo matches
    'tier1' => 0,
    'tier2' => 0,
    'tier3' => 0,
    'tier4' => 0,
    'unmatched' => 0,
    'high_confidence' => 0,
    'low_confidence' => 0,
];

// ============================================================================
// CLOUDBEDS MATCHING: Loop each reservation → find best propiedad
// ============================================================================

foreach ($cloudbeds_reservas as $reserva) {
    $best_match = null;
    $best_confidence = 0;

    // Try to match against ALL propiedades
    foreach ($propiedades as $propiedad) {
        $result = match_cloudbeds(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['property'],
            $reserva['room_number']
        );

        if ($result['match'] && $result['confidence'] > $best_confidence) {
            $best_match = $result;
            $best_match['propiedad'] = $propiedad;
            $best_confidence = $result['confidence'];

            // If we got a perfect match, stop searching
            if ($best_confidence == 100) {
                break;
            }
        }
    }

    if ($best_match) {
        $cloudbeds_matches[] = [
            'reserva' => $reserva,
            'match' => $best_match,
        ];

        $tier = $best_match['tier'];
        $stats_cloudbeds["tier{$tier}"]++;

        if ($best_match['confidence'] >= $HIGH_CONFIDENCE_THRESHOLD) {
            $stats_cloudbeds['high_confidence']++;
        } else {
            $stats_cloudbeds['low_confidence']++;
        }
    } else {
        $cloudbeds_matches[] = [
            'reserva' => $reserva,
            'match' => null,
        ];
        $stats_cloudbeds['unmatched']++;
    }
}

// ============================================================================
// HOSTIFY MATCHING: Loop each reservation → find best propiedad
// ============================================================================

foreach ($hostify_reservas as $reserva) {
    $best_match = null;
    $best_confidence = 0;

    $anuncio = $reserva['anuncio'];

    // Try to match against ALL propiedades
    foreach ($propiedades as $propiedad) {
        // TIER 0: Try combo matching FIRST (highest priority for multi-unit listings)
        $result = match_hostify_combo($propiedad['nombre_propiedad'], $propiedad['direccion'], $anuncio);

        // If no combo match, try regular tiers
        if (!$result['match']) $result = match_hostify_tier1($propiedad['nombre_propiedad'], $anuncio);
        if (!$result['match']) $result = match_hostify_tier2($propiedad['nombre_propiedad'], $propiedad['direccion'], $anuncio);
        if (!$result['match']) $result = match_hostify_tier3($propiedad['nombre_propiedad'], $propiedad['direccion'], $anuncio);
        if (!$result['match']) $result = match_hostify_tier4($propiedad['nombre_propiedad'], $propiedad['direccion'], $anuncio);

        if ($result['match'] && $result['confidence'] > $best_confidence) {
            $best_match = $result;
            $best_match['propiedad'] = $propiedad;
            $best_confidence = $result['confidence'];

            // If we got a perfect match, stop searching
            if ($best_confidence == 100) {
                break;
            }
        }
    }

    if ($best_match) {
        $hostify_matches[] = [
            'reserva' => $reserva,
            'match' => $best_match,
        ];

        $tier = $best_match['tier'];
        $stats_hostify["tier{$tier}"]++;

        if ($best_match['confidence'] >= $HIGH_CONFIDENCE_THRESHOLD) {
            $stats_hostify['high_confidence']++;
        } else {
            $stats_hostify['low_confidence']++;
        }
    } else {
        $hostify_matches[] = [
            'reserva' => $reserva,
            'match' => null,
        ];
        $stats_hostify['unmatched']++;
    }
}

// ============================================================================
// ACTION HANDLERS (ACTUALLY UPDATE DATABASE!)
// ============================================================================

if ($action === 'apply_high' || $action === 'apply_all') {
    $threshold = ($action === 'apply_high') ? $HIGH_CONFIDENCE_THRESHOLD : 0;
    $updates_cloudbeds = 0;
    $updates_hostify = 0;

    echo "<div class='alert alert-warning'>";
    echo "<strong>⚡ Applying updates (confidence ≥ {$threshold}%)...</strong>";
    echo "</div>";

    // Apply Cloudbeds updates
    if ($target === 'both' || $target === 'cloudbeds') {
        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $match['propiedad']['propiedad_id'];
                $reserva_id = $reserva['cloudbeds_reserva_id'];
                $tier = $match['tier'];
                $confidence = $match['confidence'];
                $pattern = strit($match['pattern']);

                // ✨ THOTH'S ALGORITHM: Generate AI explanation
                $explanation = explain_match($match, $reserva, $match['propiedad']);
                $explanation_escaped = strit($explanation);

                // ✨ Build multi-dimensional score JSON
                $scores = [
                    'street' => $match['street_score'] ?? 0,
                    'building' => $match['building_score'] ?? 0,
                    'unit' => $match['unit_score'] ?? 0,
                    'overall' => $match['confidence'] ?? 0,
                    'tier' => $tier,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                $scores_json = strit(json_encode($scores, JSON_UNESCAPED_UNICODE));

                $sql = "UPDATE cloudbeds_reserva
                        SET propiedad_id = '{$propiedad_id}',
                            match_tier = {$tier},
                            match_confidence = {$confidence},
                            match_pattern = {$pattern},
                            match_explanation = {$explanation_escaped},
                            match_scores = {$scores_json},
                            match_timestamp = NOW()
                        WHERE cloudbeds_reserva_id = '{$reserva_id}'";
                ia_query($sql);
                $updates_cloudbeds++;
            }
        }
    }

    // Apply Hostify updates
    if ($target === 'both' || $target === 'hostify') {
        foreach ($hostify_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            if ($match && $match['confidence'] >= $threshold) {
                $propiedad_id = $match['propiedad']['propiedad_id'];
                $reserva_id = $reserva['hostify_reserva_id'];
                $tier = $match['tier'];
                $confidence = $match['confidence'];
                $pattern = strit($match['pattern']);

                // ✨ THOTH'S ALGORITHM: Generate AI explanation
                $explanation = explain_match($match, $reserva, $match['propiedad']);
                $explanation_escaped = strit($explanation);

                // ✨ Build multi-dimensional score JSON
                $scores = [
                    'street' => $match['street_score'] ?? 0,
                    'building' => $match['building_score'] ?? 0,
                    'unit' => $match['unit_score'] ?? 0,
                    'overall' => $match['confidence'] ?? 0,
                    'tier' => $tier,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                $scores_json = strit(json_encode($scores, JSON_UNESCAPED_UNICODE));

                $sql = "UPDATE hostify_reserva
                        SET propiedad_id = '{$propiedad_id}',
                            match_tier = {$tier},
                            match_confidence = {$confidence},
                            match_pattern = {$pattern},
                            match_explanation = {$explanation_escaped},
                            match_scores = {$scores_json},
                            match_timestamp = NOW()
                        WHERE hostify_reserva_id = '{$reserva_id}'";

                // Debug: Show first SQL query
                if ($debug && $updates_hostify === 0) {
                    echo "<div class='alert alert-info'><strong>Debug SQL (first query):</strong><pre>" . htmlspecialchars($sql) . "</pre></div>";
                }

                ia_query($sql);
                $updates_hostify++;
            }
        }
    }

    // ============================================================================
    // ✨ THOTH'S ALGORITHM: Explain NO-MATCHES (Store explanations even for failures)
    // ============================================================================
    $explained_nomatch_cloudbeds = 0;
    $explained_nomatch_hostify = 0;

    // Explain Cloudbeds no-matches
    if ($target === 'both' || $target === 'cloudbeds') {
        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            // Only explain if NO match or confidence below threshold (won't be linked)
            if (!$match || $match['confidence'] < $threshold) {
                $reserva_id = $reserva['cloudbeds_reserva_id'];

                // Generate AI explanation for why it didn't match
                $explanation = explain_no_match($reserva, $propiedades, 'cloudbeds');
                $explanation_escaped = strit($explanation);

                // Build failure scores
                $scores = [
                    'overall' => 0,
                    'tier' => 99,
                    'reason' => 'no_match_found',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                $scores_json = strit(json_encode($scores, JSON_UNESCAPED_UNICODE));

                $sql = "UPDATE cloudbeds_reserva
                        SET match_explanation = {$explanation_escaped},
                            match_scores = {$scores_json},
                            match_timestamp = NOW()
                        WHERE cloudbeds_reserva_id = '{$reserva_id}'";
                ia_query($sql);
                $explained_nomatch_cloudbeds++;
            }
        }
    }

    // Explain Hostify no-matches
    if ($target === 'both' || $target === 'hostify') {
        foreach ($hostify_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            // Only explain if NO match or confidence below threshold (won't be linked)
            if (!$match || $match['confidence'] < $threshold) {
                $reserva_id = $reserva['hostify_reserva_id'];

                // Generate AI explanation for why it didn't match
                $explanation = explain_no_match($reserva, $propiedades, 'hostify');
                $explanation_escaped = strit($explanation);

                // Build failure scores
                $scores = [
                    'overall' => 0,
                    'tier' => 99,
                    'reason' => 'no_match_found',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                $scores_json = strit(json_encode($scores, JSON_UNESCAPED_UNICODE));

                $sql = "UPDATE hostify_reserva
                        SET match_explanation = {$explanation_escaped},
                            match_scores = {$scores_json},
                            match_timestamp = NOW()
                        WHERE hostify_reserva_id = '{$reserva_id}'";
                ia_query($sql);
                $explained_nomatch_hostify++;
            }
        }
    }

    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Success!</strong><br>";
    echo "Updated <strong>{$updates_cloudbeds}</strong> Cloudbeds reservations<br>";
    echo "Updated <strong>{$updates_hostify}</strong> Hostify reservations<br>";
    echo "<strong>Total: " . ($updates_cloudbeds + $updates_hostify) . " reservations linked!</strong><br><br>";

    if ($explained_nomatch_cloudbeds > 0 || $explained_nomatch_hostify > 0) {
        echo "✨ <strong>AI Explanations Generated:</strong><br>";
        if ($explained_nomatch_cloudbeds > 0) {
            echo "- {$explained_nomatch_cloudbeds} Cloudbeds no-matches explained<br>";
        }
        if ($explained_nomatch_hostify > 0) {
            echo "- {$explained_nomatch_hostify} Hostify no-matches explained<br>";
        }
    }
    echo "</div>";

    echo "<a href='link_pms_propiedades.php' class='btn btn-primary'>← Back to Preview</a>";
    echo "</div></body></html>";
    exit;
}

if ($action === 'export_csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=pms_reverse_matching_' . date('Y-m-d_His') . '.csv');

    $output = fopen('php://output', 'w');

    if ($target === 'both' || $target === 'cloudbeds') {
        fputcsv($output, ['=== CLOUDBEDS RESERVATION MATCHES ===']);
        fputcsv($output, [
            'reservation_number', 'guest_name', 'check_in', 'check_out', 'nights',
            'property', 'room_number', 'room_type',
            'propiedad_id', 'nombre_propiedad', 'direccion',
            'match_tier', 'confidence', 'building_score', 'unit_score'
        ]);

        foreach ($cloudbeds_matches as $match_data) {
            $reserva = $match_data['reserva'];
            $match = $match_data['match'];

            if ($match) {
                fputcsv($output, [
                    $reserva['reservation_number'],
                    $reserva['name'],
                    $reserva['check_in_date'],
                    $reserva['check_out_date'],
                    $reserva['nights'],
                    $reserva['property'],
                    $reserva['room_number'],
                    $reserva['room_type'],
