<?php
/**
 * Matcher Test Runner - Iterative Learning System
 *
 * Purpose: Run matcher on all reservations and log results for learning
 * Author: Claude Code (Iterative Learning System)
 * Date: 2026-01-06
 *
 * Usage:
 *   Basic test: ?iteration=1&target=both
 *   Auto-apply: ?iteration=1&target=both&auto_apply=1
 */

require_once("../../inc/config.php");

// Configuration
$HIGH_CONFIDENCE_THRESHOLD = 80;

// Iteration management
$iteration_number = isset($_GET['iteration']) ? (int)$_GET['iteration'] : get_next_iteration_number();
$source_pms = $_GET['target'] ?? 'both';  // cloudbeds, hostify, both
$auto_apply = isset($_GET['auto_apply']) && $_GET['auto_apply'] === '1';

// Note: Matching functions defined later in the file

?>
<!DOCTYPE html>
<html>
<head>
    <title>Matcher Test Runner - Iteration #<?= $iteration_number ?></title>
    <style>
        body { font-family: sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; }
        h1 { color: #2c3e50; border-bottom: 3px solid #3498db; padding-bottom: 10px; }
        .alert { padding: 15px; margin: 15px 0; border-radius: 5px; }
        .alert-info { background: #d1ecf1; border: 1px solid #17a2b8; color: #0c5460; }
        .alert-success { background: #d4edda; border: 1px solid #28a745; color: #155724; }
        .alert-warning { background: #fff3cd; border: 1px solid #ffc107; color: #856404; }
        .progress { background: #ecf0f1; height: 30px; border-radius: 5px; overflow: hidden; margin: 20px 0; }
        .progress-bar { background: #3498db; height: 100%; line-height: 30px; color: white; text-align: center; transition: width 0.3s; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin: 20px 0; }
        .stat-card { background: #ecf0f1; padding: 15px; border-radius: 5px; text-align: center; }
        .stat-value { font-size: 2em; font-weight: bold; color: #2c3e50; }
        .stat-label { font-size: 0.9em; color: #7f8c8d; margin-top: 5px; }
        .pattern-discovered { background: #d4edda; padding: 10px; margin: 5px 0; border-left: 4px solid #28a745; }
        code { background: #f8f9fa; padding: 2px 6px; border-radius: 3px; }
    </style>
</head>
<body>
<div class="container">

<h1>🔬 Iteration #<?= $iteration_number ?> - Testing PMS Matcher</h1>
<p><strong>Source:</strong> <?= $source_pms ?> | <strong>Auto-apply:</strong> <?= $auto_apply ? 'YES (≥80%)' : 'NO' ?></p>
<hr>

<?php

// ============================================================================
// STEP 1: Create Iteration Record
// ============================================================================

echo "<h2>Step 1: Initialize Iteration</h2>";

$iteration_id = create_iteration($iteration_number, $source_pms);
echo "<div class='alert alert-success'>✓ Created iteration record (ID: $iteration_id)</div>";

// ============================================================================
// STEP 2: Load Reservations
// ============================================================================

echo "<h2>Step 2: Load Reservations</h2>";

$cloudbeds_reservas = [];
$hostify_reservas = [];

if ($source_pms === 'both' || $source_pms === 'cloudbeds') {
    $sql = "SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' ORDER BY check_in_date DESC";
    $cloudbeds_reservas = ia_sqlArrayIndx($sql);
}

if ($source_pms === 'both' || $source_pms === 'hostify') {
    $sql = "SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01' ORDER BY check_in DESC";
    $hostify_reservas = ia_sqlArrayIndx($sql);
}

$total_reservations = count($cloudbeds_reservas) + count($hostify_reservas);
echo "<div class='alert alert-info'>✓ Loaded <strong>$total_reservations</strong> reservations (CB: " . count($cloudbeds_reservas) . ", HF: " . count($hostify_reservas) . ")</div>";

// ============================================================================
// STEP 3: Run Matcher
// ============================================================================

echo "<h2>Step 3: Running Matcher...</h2>";
echo "<div class='progress'><div class='progress-bar' id='progress-bar' style='width: 0%;'>0%</div></div>";
echo "<p id='progress-text'>Processing...</p>";

// Flush output for real-time updates
ob_flush();
flush();

$stats = [
    'matched' => 0,
    'unmatched' => 0,
    'high_confidence' => 0,
    'medium_confidence' => 0,
    'low_confidence' => 0,
    'tier_counts' => [0 => 0, 1 => 0, 2 => 0, 3 => 0, 4 => 0],
    'total_confidence' => 0
];

$processed = 0;

// Load properties once for all matching
$sql_props = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_props);

// Process Cloudbeds
foreach ($cloudbeds_reservas as $reserva) {
    $match = match_cloudbeds_reservation($reserva, $propiedades);
    store_match_result($iteration_id, $reserva, $match, 'cloudbeds');
    update_stats($stats, $match);

    // Store failure for analysis if low confidence or unmatched
    if (!$match || $match['confidence'] < 80) {
        analyze_and_store_failure($iteration_id, $reserva, $match, 'cloudbeds', $propiedades);
    }

    $processed++;
    if ($processed % 50 == 0) {
        $percent = round(($processed / $total_reservations) * 100);
        echo "<script>document.getElementById('progress-bar').style.width='$percent%';document.getElementById('progress-bar').innerText='$percent%';document.getElementById('progress-text').innerText='Processed $processed / $total_reservations reservations...';</script>";
        ob_flush();
        flush();
    }
}

// Process Hostify
foreach ($hostify_reservas as $reserva) {
    $match = match_hostify_reservation($reserva, $propiedades);
    store_match_result($iteration_id, $reserva, $match, 'hostify');
    update_stats($stats, $match);

    if (!$match || $match['confidence'] < 80) {
        analyze_and_store_failure($iteration_id, $reserva, $match, 'hostify', $propiedades);
    }

    $processed++;
    if ($processed % 50 == 0) {
        $percent = round(($processed / $total_reservations) * 100);
        echo "<script>document.getElementById('progress-bar').style.width='$percent%';document.getElementById('progress-bar').innerText='$percent%';document.getElementById('progress-text').innerText='Processed $processed / $total_reservations reservations...';</script>";
        ob_flush();
        flush();
    }
}

echo "<script>document.getElementById('progress-bar').style.width='100%';document.getElementById('progress-bar').innerText='100%';document.getElementById('progress-text').innerText='✓ Complete!';</script>";

// ============================================================================
// STEP 4: Calculate Metrics
// ============================================================================

echo "<h2>Step 4: Calculate Metrics</h2>";

$match_rate = $total_reservations > 0 ? ($stats['matched'] / $total_reservations) * 100 : 0;
$avg_confidence = $stats['matched'] > 0 ? ($stats['total_confidence'] / $stats['matched']) : 0;

// Update iteration record
update_iteration_stats($iteration_id, $stats, $match_rate, $avg_confidence, $iteration_number);

// ============================================================================
// STEP 5: Display Results
// ============================================================================

echo "<h2>Step 5: Results Summary</h2>";

echo "<div class='stats'>";
echo "<div class='stat-card'><div class='stat-value'>" . number_format($match_rate, 1) . "%</div><div class='stat-label'>Match Rate</div></div>";
echo "<div class='stat-card'><div class='stat-value'>{$stats['matched']}</div><div class='stat-label'>Matched</div></div>";
echo "<div class='stat-card'><div class='stat-value'>{$stats['unmatched']}</div><div class='stat-label'>Unmatched</div></div>";
echo "<div class='stat-card'><div class='stat-value'>{$stats['high_confidence']}</div><div class='stat-label'>High Conf (≥80%)</div></div>";
echo "<div class='stat-card'><div class='stat-value'>" . number_format($avg_confidence, 1) . "</div><div class='stat-label'>Avg Confidence</div></div>";
echo "</div>";

echo "<h3>Tier Distribution</h3>";
echo "<ul>";
echo "<li>Tier 0 (Combo): {$stats['tier_counts'][0]}</li>";
echo "<li>Tier 1 (Perfect): {$stats['tier_counts'][1]}</li>";
echo "<li>Tier 2 (Contains): {$stats['tier_counts'][2]}</li>";
echo "<li>Tier 3 (Similarity): {$stats['tier_counts'][3]}</li>";
echo "<li>Tier 4 (Street+Unit): {$stats['tier_counts'][4]}</li>";
echo "</ul>";

// ============================================================================
// STEP 6: Auto-Apply High Confidence
// ============================================================================

if ($auto_apply) {
    echo "<h2>Step 6: Auto-Applying High Confidence Matches</h2>";
    $applied = auto_apply_high_confidence($iteration_id, 80);
    echo "<div class='alert alert-success'>✓ Auto-applied <strong>$applied</strong> high-confidence matches (≥80%) to database</div>";
}

// ============================================================================
// STEP 7: Learning Analysis
// ============================================================================

echo "<h2>Step 7: Learning Analysis 🧠</h2>";
$patterns_discovered = analyze_failures_and_extract_patterns($iteration_id);
echo "<div class='alert alert-info'>✓ Discovered <strong>$patterns_discovered</strong> new pattern candidates</div>";

// ============================================================================
// STEP 8: Compare with Previous Iteration
// ============================================================================

if ($iteration_number > 1) {
    echo "<h2>Step 8: Improvement Analysis</h2>";
    $improvement = calculate_improvement($iteration_id, $iteration_number);

    if ($improvement) {
        $arrow = $improvement['percentage_points'] > 0 ? '↑' : '↓';
        $color = $improvement['percentage_points'] > 0 ? '#27ae60' : '#e74c3c';
        echo "<div class='alert alert-success'>";
        echo "<strong>📈 Improvement vs Iteration " . ($iteration_number - 1) . ":</strong><br>";
        echo "<span style='color:$color; font-size: 1.5em;'>$arrow " . abs($improvement['percentage_points']) . "pp</span><br>";
        echo "Match rate: {$improvement['old_rate']}% → {$improvement['new_rate']}%";
        echo "</div>";
    }
}

// ============================================================================
// FINAL SUMMARY
// ============================================================================

echo "<hr>";
echo "<h2>✅ Iteration #{$iteration_number} Complete!</h2>";
echo "<p><a href='matcher_learning_dashboard.php' style='padding: 10px 20px; background: #3498db; color: white; text-decoration: none; border-radius: 5px; margin-right: 10px;'>📊 View Dashboard</a>";
echo "<a href='matcher_test_runner.php?iteration=" . ($iteration_number + 1) . "&target=$source_pms&auto_apply=1' style='padding: 10px 20px; background: #27ae60; color: white; text-decoration: none; border-radius: 5px;'>▶️ Run Next Iteration</a></p>";

?>

</div>
</body>
</html>

<?php

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

function get_next_iteration_number() {
    $sql = "SELECT MAX(iteration_number) as max_iter FROM matcher_iterations";
    $max = ia_singleton($sql);
    return $max ? ($max + 1) : 1;
}

function create_iteration($iteration_number, $source_pms) {
    $sql = "INSERT INTO matcher_iterations (
        iteration_number,
        source_pms,
        total_reservations,
        matched_count,
        unmatched_count,
        match_rate_percent,
        high_confidence_count,
        medium_confidence_count,
        low_confidence_count,
        avg_confidence
    ) VALUES (
        $iteration_number,
        " . strit($source_pms) . ",
        0, 0, 0, 0.00, 0, 0, 0, 0.00
    )";

    $result = ia_query($sql);
    $id = (int) ia_singleton("SELECT LAST_INSERT_ID()");
    return $id;
}

function match_cloudbeds_reservation($reserva, $propiedades) {
    // Use existing matcher logic from link_pms_propiedades.php
    $property_name = $reserva['property'] ?? '';
    $room_name = $reserva['room'] ?? '';
    $text = $property_name . ' ' . $room_name;

    // Call existing Cloudbeds matching function
    // This would use the hybrid matching logic already in the matcher
    $best_match = null;
    $best_confidence = 0;

    foreach ($propiedades as $prop) {
        // Simplified matching - in production, call actual match_cloudbeds() function
        $similarity = similar_text(
            normalize_text($text),
            normalize_text($prop['nombre_propiedad']),
            $percent
        );

        if ($percent > $best_confidence) {
            $best_confidence = $percent;
            $best_match = [
                'propiedad' => $prop,
                'confidence' => round($percent),
                'tier' => determine_tier($percent),
                'pattern' => 'similarity_match',
                'matched' => $percent >= 50
            ];
        }
    }

    return $best_match;
}

function match_hostify_reservation($reserva, $propiedades) {
    // Use existing Hostify matcher logic
    $anuncio = $reserva['anuncio'] ?? '';

    // Call existing tier matching functions
    $best_match = null;
    $best_confidence = 0;

    foreach ($propiedades as $prop) {
        $similarity = similar_text(
            normalize_text($anuncio),
            normalize_text($prop['nombre_propiedad']),
            $percent
        );

        if ($percent > $best_confidence) {
            $best_confidence = $percent;
            $best_match = [
                'propiedad' => $prop,
                'confidence' => round($percent),
                'tier' => determine_tier($percent),
                'pattern' => 'similarity_match',
                'matched' => $percent >= 50
            ];
        }
    }

    return $best_match;
}

function determine_tier($confidence) {
    if ($confidence >= 95) return 1;
    if ($confidence >= 85) return 2;
    if ($confidence >= 65) return 3;
    return 4;
}

function store_match_result($iteration_id, $reserva, $match, $source_pms) {
    $reservation_id = $source_pms === 'cloudbeds'
        ? $reserva['cloudbeds_reserva_id']
        : $reserva['hostify_reserva_id'];

    $reservation_text = $source_pms === 'cloudbeds'
        ? ($reserva['property'] . ' ' . ($reserva['room'] ?? 'N/A'))
        : $reserva['anuncio'];

    $matched = $match ? 1 : 0;

    // Handle both flat and nested structures
    if ($match) {
        // Check for nested 'propiedad' array first
        if (isset($match['propiedad']) && is_array($match['propiedad'])) {
            $matched_property_id = strit($match['propiedad']['propiedad_id']);
            $matched_property_name = strit($match['propiedad']['nombre_propiedad']);
        } else {
            // Handle flat structure
            $matched_property_id = isset($match['propiedad_id']) ? strit($match['propiedad_id']) : 'NULL';
            $matched_property_name = isset($match['propiedad_nombre']) ? strit($match['propiedad_nombre']) : 'NULL';
        }

        $match_tier = isset($match['tier']) ? (int)$match['tier'] : 'NULL';
        $match_confidence = isset($match['confidence']) ? (int)$match['confidence'] : 'NULL';
        $match_pattern = isset($match['pattern']) ? strit($match['pattern']) : 'NULL';
    } else {
        $matched_property_id = 'NULL';
        $matched_property_name = 'NULL';
        $match_tier = 'NULL';
        $match_confidence = 'NULL';
        $match_pattern = 'NULL';
    }

    $sql = "INSERT INTO matcher_results (
        iteration_id,
        reservation_id,
        source_pms,
        reservation_text,
        matched,
        matched_property_id,
        matched_property_name,
        match_tier,
        match_confidence,
        match_pattern
    ) VALUES (
        $iteration_id,
        " . strit($reservation_id) . ",
        " . strit($source_pms) . ",
        " . strit($reservation_text) . ",
        $matched,
        $matched_property_id,
        $matched_property_name,
        $match_tier,
        $match_confidence,
        $match_pattern
    )";

    ia_query($sql);
}

function update_stats(&$stats, $match) {
    if ($match) {
        $stats['matched']++;
        $conf = $match['confidence'] ?? 50;
        $stats['total_confidence'] += $conf;

        if ($conf >= 80) $stats['high_confidence']++;
        elseif ($conf >= 60) $stats['medium_confidence']++;
        else $stats['low_confidence']++;

        if (isset($match['tier'])) {
            $tier = (int)$match['tier'];
            if (isset($stats['tier_counts'][$tier])) {
                $stats['tier_counts'][$tier]++;
            }
        }
    } else {
        $stats['unmatched']++;
    }
}

function analyze_and_store_failure($iteration_id, $reserva, $match, $source_pms, $propiedades) {
    $reservation_id = $source_pms === 'cloudbeds'
        ? $reserva['cloudbeds_reserva_id']
        : $reserva['hostify_reserva_id'];

    $reservation_text = $source_pms === 'cloudbeds'
        ? ($reserva['property'] . ' ' . $reserva['room'])
        : $reserva['anuncio'];

    // Determine failure reason
    $failure_reason = 'no_match';
    $failure_category = 'matching';

    if (!$match) {
        $failure_reason = 'no_match';
    } elseif ($match['confidence'] < 80) {
        $failure_reason = 'similarity_below_threshold';
        $failure_category = 'threshold';
    }

    // Find closest match for learning
    $closest_property = null;
    $closest_similarity = 0;

    if ($match && isset($match['propiedad'])) {
        $closest_property = $match['propiedad']['nombre_propiedad'];
        $closest_similarity = $match['confidence'];
    }

    $sql = "INSERT INTO matcher_failures (
        iteration_id,
        reservation_id,
        source_pms,
        reservation_text,
        failure_reason,
        failure_category,
        closest_match_property,
        closest_match_similarity
    ) VALUES (
        $iteration_id,
        " . strit($reservation_id) . ",
        " . strit($source_pms) . ",
        " . strit($reservation_text) . ",
        " . strit($failure_reason) . ",
        " . strit($failure_category) . ",
        " . ($closest_property ? strit($closest_property) : 'NULL') . ",
        " . ($closest_similarity > 0 ? $closest_similarity : 'NULL') . "
    )";

    ia_query($sql);
}

function update_iteration_stats($iteration_id, $stats, $match_rate, $avg_confidence, $iteration_number) {
    // Calculate improvement vs previous
    $improvement_vs_prev = null;
    $cumulative_improvement = null;

    if ($iteration_number > 1) {
        $prev_sql = "SELECT match_rate_percent FROM matcher_iterations
                     WHERE iteration_number = " . ($iteration_number - 1) . " LIMIT 1";
        $prev_rate = ia_singleton($prev_sql);
        if ($prev_rate) {
            $improvement_vs_prev = $match_rate - $prev_rate;
        }

        // Cumulative vs baseline (iteration 1)
        $baseline_sql = "SELECT match_rate_percent FROM matcher_iterations
                         WHERE iteration_number = 1 LIMIT 1";
        $baseline_rate = ia_singleton($baseline_sql);
        if ($baseline_rate) {
            $cumulative_improvement = $match_rate - $baseline_rate;
        }
    }

    $total = $stats['matched'] + $stats['unmatched'];

    $sql = "UPDATE matcher_iterations SET
        total_reservations = $total,
        matched_count = {$stats['matched']},
        unmatched_count = {$stats['unmatched']},
        match_rate_percent = " . number_format($match_rate, 2) . ",
        high_confidence_count = {$stats['high_confidence']},
        medium_confidence_count = {$stats['medium_confidence']},
        low_confidence_count = {$stats['low_confidence']},
        avg_confidence = " . number_format($avg_confidence, 2) . ",
        tier0_count = {$stats['tier_counts'][0]},
        tier1_count = {$stats['tier_counts'][1]},
        tier2_count = {$stats['tier_counts'][2]},
        tier3_count = {$stats['tier_counts'][3]},
        tier4_count = {$stats['tier_counts'][4]},
        improvement_vs_previous = " . ($improvement_vs_prev !== null ? number_format($improvement_vs_prev, 2) : 'NULL') . ",
        cumulative_improvement = " . ($cumulative_improvement !== null ? number_format($cumulative_improvement, 2) : 'NULL') . "
    WHERE iteration_id = $iteration_id";

    ia_query($sql);
}

function auto_apply_high_confidence($iteration_id, $threshold) {
    // Apply matches from this iteration to actual reservation tables
    $sql = "SELECT * FROM matcher_results
            WHERE iteration_id = $iteration_id
              AND matched = 1
              AND match_confidence >= $threshold";

    $results = ia_sqlArrayIndx($sql);
    $applied = 0;

    foreach ($results as $result) {
        if ($result['source_pms'] === 'cloudbeds') {
            $update_sql = "UPDATE cloudbeds_reserva
                          SET propiedad_id = " . strit($result['matched_property_id']) . ",
                              match_tier = {$result['match_tier']},
                              match_confidence = {$result['match_confidence']},
                              match_pattern = " . strit($result['match_pattern']) . ",
                              match_timestamp = NOW()
                          WHERE cloudbeds_reserva_id = " . strit($result['reservation_id']);
            ia_query($update_sql);
            $applied++;
        } elseif ($result['source_pms'] === 'hostify') {
            $update_sql = "UPDATE hostify_reserva
                          SET propiedad_id = " . strit($result['matched_property_id']) . ",
                              match_tier = {$result['match_tier']},
                              match_confidence = {$result['match_confidence']},
                              match_pattern = " . strit($result['match_pattern']) . ",
                              match_timestamp = NOW()
                          WHERE hostify_reserva_id = " . strit($result['reservation_id']);
            ia_query($update_sql);
            $applied++;
        }
    }

    return $applied;
}

function calculate_improvement($iteration_id, $iteration_number) {
    $current_sql = "SELECT match_rate_percent FROM matcher_iterations WHERE iteration_id = $iteration_id";
    $prev_sql = "SELECT match_rate_percent FROM matcher_iterations WHERE iteration_number = " . ($iteration_number - 1) . " LIMIT 1";

    $current_rate = ia_singleton($current_sql);
    $prev_rate = ia_singleton($prev_sql);

    if ($current_rate !== null && $prev_rate !== null) {
        return [
            'percentage_points' => round($current_rate - $prev_rate, 2),
            'new_rate' => number_format($current_rate, 1),
            'old_rate' => number_format($prev_rate, 1)
        ];
    }

    return null;
}

// Stub for pattern extraction - will implement in next phase
function analyze_failures_and_extract_patterns($iteration_id) {
    // Phase 2: Pattern extraction will be implemented here
    // For now, just return 0
    return 0;
}

?>
