<?php
/**
 * Merge duplicate scholarship student records in alumnos_becados table
 * Purpose: Automatically merge duplicate scholarship student records by estudiante_id + beca_tipo
 * Security: SQL injection prevention via parameterized queries
 * UI: Multistep preview workflow mirroring link_propiedades_propietarios.php styling
 */

require_once("../../inc/config.php");

// ============================================================================
// CONFIGURATION
// ============================================================================

define('HIGH_CONFIDENCE_THRESHOLD', 80);
$action = $_GET['action'] ?? 'preview';
$debug = isset($_GET['debug']);

// Safe HTML escaping helper
function esc($value) {
    return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>🔄 Merge Scholarship Student Records</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .container {
            max-width: 1600px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            border-bottom: 4px solid #667eea;
            padding-bottom: 15px;
            font-size: 2.2em;
            margin-top: 0;
        }
        h1 .emoji { font-size: 1.2em; }
        .subtitle {
            color: #764ba2;
            font-style: italic;
            margin-top: -10px;
            margin-bottom: 20px;
        }
        .stats {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }
        .stat-box {
            background: white;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            border-left: 4px solid #667eea;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .stat-box.duplicates { border-color: #ffc107; }
        .stat-box.merged { border-color: #28a745; }
        .stat-box.unmatched { border-color: #6c757d; }
        .stat-box h3 { margin: 0 0 5px 0; font-size: 2.2em; color: #667eea; }
        .stat-box p { margin: 0; color: #7f8c8d; font-size: 0.85em; }

        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            font-size: 0.85em;
        }
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 8px;
            text-align: left;
            position: sticky;
            top: 0;
            font-size: 0.9em;
        }
        td {
            padding: 10px 8px;
            border-bottom: 1px solid #ecf0f1;
        }
        tr:hover { background: #f8f9fa; }

        /* Confidence-based row colors */
        .confidence-100 { background-color: #d4edda !important; border-left: 4px solid #28a745; }

        /* Tier badges */
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: bold;
            color: white;
        }
        .badge-duplicates { background: #ffc107; }
        .badge-merged { background: #28a745; }

        /* Action buttons */
        .actions {
            margin: 30px 0;
            padding: 20px;
            background: linear-gradient(135deg, #e8f4f8 0%, #f3e7f9 100%);
            border-radius: 8px;
            border: 2px solid #667eea;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            margin: 5px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 1em;
            text-decoration: none;
            transition: all 0.3s;
            font-weight: bold;
        }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5568d3; transform: translateY(-2px); }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; transform: translateY(-2px); }
        .btn-warning { background: #ffc107; color: #333; }
        .btn-warning:hover { background: #e0a800; transform: translateY(-2px); }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; transform: translateY(-2px); }

        /* Alert boxes */
        .alert {
            padding: 15px;
            margin: 20px 0;
            border-radius: 6px;
            border-left: 4px solid;
        }
        .alert-success { background: #d4edda; border-color: #28a745; color: #155724; }
        .alert-info { background: #d1ecf1; border-color: #17a2b8; color: #0c5460; }
        .alert-warning { background: #fff3cd; border-color: #ffc107; color: #856404; }
        .alert-danger { background: #f8d7da; border-color: #dc3545; color: #721c24; }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
        }

        /* Debug mode styling */
        .debug-box {
            background: #fff3cd;
            border: 2px solid #ffc107;
            padding: 15px;
            margin: 15px 0;
            border-radius: 6px;
            font-size: 0.85em;
        }
        .debug-box pre {
            background: white;
            padding: 10px;
            border-radius: 4px;
            overflow-x: auto;
            max-height: 300px;
        }
    </style>
</head>
<body>
<div class="container">
    <h1><span class="emoji">🔄</span> Scholarship Student Merge Tool</h1>
    <p class="subtitle">Automatically merge duplicate scholarship student records</p>

    <?php if (!$debug): ?>
    <div class="alert alert-info">
        💡 <strong>Tip:</strong> Add <code>?debug=1</code> to URL for detailed merge diagnostics
    </div>
    <?php endif; ?>

<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', '/tmp/php_errors.log');
error_reporting(E_ALL);
// ============================================================================
// DATABASE QUERY FUNCTIONS WITH PARAMETERIZATION
// ============================================================================

/**
 * Fetch duplicate student groups with parameterized query
 */
function fetch_duplicate_groups() {
    
    $sql = "
        SELECT estudiante_id, beca_tipo, COUNT(*) AS duplicate_count
        FROM alumnos_becados
        GROUP BY estudiante_id, beca_tipo
        HAVING COUNT(*) > 1";
    
    return ia_sqlArrayIndx($sql);
}

/**
 * Merge duplicate records for a single estudiante_id + beca_tipo group
 */
function merge_duplicate_group($estudiante_id, $beca_tipo) {
    global $db;
    
    // Fetch all records for this duplicate group
    $sql_records = "
        SELECT alumno_becado_id, porcentaje_beca_colegiatura, porcentaje_beca_inscripcion
        FROM alumnos_becados
        WHERE estudiante_id = " . strit($estudiante_id) .
        " AND beca_tipo = " . strit($beca_tipo);

    //echo "<pre>".print_r($sql_records, true)."</pre>";
    
    $records = ia_sqlArrayIndx($sql_records);
    
    //echo "<pre>".print_r($records, true)."</pre>";
    if (count($records) <= 1) {
        return ['success' => false, 'message' => 'No duplicates found'];
    }
    
    // Identify survivor (first record)
    $survivor = reset($records);
    
    // Merge percentage values
    foreach ($records as $record) {
        if ($record['alumno_becado_id'] !== $survivor['alumno_becado_id']) {
            if (empty($survivor['porcentaje_beca_colegiatura']) 
                && !empty($record['porcentaje_beca_colegiatura'])) {
                $survivor['porcentaje_beca_colegiatura'] = $record['porcentaje_beca_colegiatura'];
            }
            if (empty($survivor['porcentaje_beca_inscripcion']) 
                && !empty($record['porcentaje_beca_inscripcion'])) {
                $survivor['porcentaje_beca_inscripcion'] = $record['porcentaje_beca_inscripcion'];
            }
        }
    }
    $sql_trans_survivors = array();
    $sql_trans_delete = array();

    try {
        // Update survivor record
        $sql_trans_survivors[] = ia_update('alumnos_becados', $survivor, "alumno_becado_id=" . strit($survivor['alumno_becado_id']));
        
        // Delete duplicate records
        foreach ($records as $record) {
            if ($record['alumno_becado_id'] !== $survivor['alumno_becado_id']) {
                $sql_trans_delete[] = "
                    DELETE FROM alumnos_becados 
                    WHERE alumno_becado_id = " . strit($record['alumno_becado_id']);
            }
        }

        //echo "<pre>".print_r($sql_trans_delete, true)."</pre>";
        //echo "<pre>".print_r($sql_trans_survivors, true)."</pre>";
        ia_transaction($sql_trans_delete );
        ia_transaction($sql_trans_survivors);

        
        // Audit trail logging
        audit_merge_operation($survivor['alumno_becado_id'], $records);
        
        return [
            'success' => true,
            'survivor' => $survivor,
            'duplicates' => count($records) - 1
        ];
    } catch (Exception $e) {
        trigger_error("Merge failed: " . $e->getMessage(), E_USER_WARNING);
        return [
            'success' => false,
            'message' => "Merge failed: " . $e->getMessage()
        ];
    }
}

/**
 * Audit trail logging for merged records
 */
function audit_merge_operation($survivor_id, $duplicate_records) {
    $audit_log = [
        'timestamp' => date('Y-m-d H:i:s'),
        'survivor_id' => $survivor_id,
        'merged_ids' => implode(',', array_column($duplicate_records, 'alumno_becado_id')),
        'action' => 'merge_duplicates'
    ];
    
    // Store audit trail (could also log to database)
    file_put_contents(
        '/lamp/www/quantix/backoffice/helper/audit_merge.log',
        json_encode($audit_log) . PHP_EOL,
        FILE_APPEND
    );
}

// ============================================================================
// MAIN EXECUTION PATHS
// ============================================================================

// Preview mode (default)
if ($action === 'preview') {
    $duplicate_groups = fetch_duplicate_groups();
    $duplicate_groups_count = count($duplicate_groups);
    $duplicate_groups_count = count($duplicate_groups);
    
    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Preview Mode:</strong> ";
    echo count($duplicate_groups) . " duplicate student groups found";
    echo "</div>";
    
    if (!empty($duplicate_groups)) {
        echo "<div class='stats'>";
        echo "<div class='stat-box duplicates'><h3>" . count($duplicate_groups) . "</h3><p>Duplicate Groups</p></div>";
        echo "</div>";
        
        echo "<h2>Duplicate Student Groups</h2>";
        echo "<table>";
        echo "<thead><tr>";
        echo "<th>Estudiante</th>";
        echo "<th>Beca Tipo</th>";
        echo "<th>Duplicate Count</th>";
        echo "<th>Merge</th>";
        echo "</tr></thead>";
        echo "<tbody>";
        
        foreach ($duplicate_groups as $group) {
            echo "<tr>";
            echo "<td>" . esc($group['estudiante_id']) . "</td>";
            echo "<td>" . esc($group['beca_tipo']) . "</td>";
            echo "<td>" . esc($group['duplicate_count']) . "</td>";
            echo "<td>";
            echo "<a href='?action=apply&estudiante_id=" . esc($group['estudiante_id']) 
                . "&beca_tipo=" . esc($group['beca_tipo']) 
                . "' class='btn btn-success' onclick='return confirm(\"Merge {$group['duplicate_count']} duplicates?\")'>Merge</a>";
            echo "</td>";
            echo "</tr>";
        }
        
        echo "</tbody></table>";
    } else {
        echo "<div class='alert alert-success'>";
        echo "<strong>✓ Success:</strong> No duplicate student groups found";
        echo "</div>";
    }
    
    echo "<div class='actions'>";
    echo "<h3>Actions</h3>";
    echo "<a href='?action=apply_all' class='btn btn-success' onclick='return confirm(\"Merge all {$duplicate_groups_count} duplicate student groups?\")'>✓ Merge All</a>";
    echo "</div>";
    
    if ($debug) {
        echo "<div class='debug-box'>";
        echo "<strong>🐛 DEBUG MODE</strong><br>";
        echo "<strong>Duplicate Groups:</strong><pre>";
        print_r($duplicate_groups);
        echo "</pre>";
        echo "</div>";
    }
    
    echo "<div class='alert alert-info'>";
    echo "<strong>💡 Merge Logic:</strong><br>";
    echo "• Keeps the first duplicate as survivor<br>";
    echo "• Preserves non-empty scholarship percentages<br>";
    echo "• Automatically deletes duplicate records";
    echo "</div>";
}

// Apply mode (single group)
elseif ($action === 'apply') {
    $estudiante_id = $_GET['estudiante_id'] ?? '';
    $beca_tipo = $_GET['beca_tipo'] ?? '';
    
    if (!$estudiante_id || !$beca_tipo) {
        die("<div class='alert alert-danger'>Missing parameters</div>");
    }
    
    $result = merge_duplicate_group($estudiante_id, $beca_tipo);
    
    if ($result['success']) {
        echo "<div class='alert alert-success'>";
        echo "<strong>✓ Success!</strong> Successfully merged {$result['duplicates']} duplicate records.";
        echo "</div>";
        
        echo "<div class='alert alert-info'>";
        echo "<strong>Survivor:</strong> {$result['survivor']['alumno_becado_id']}<br>";
        echo "<strong>Percentages:</strong> ";
        echo "{$result['survivor']['porcentaje_beca_colegiatura']}% college, ";
        echo "{$result['survivor']['porcentaje_beca_inscripcion']}% enrollment";
        echo "</div>";
    } else {
        echo "<div class='alert alert-danger'>";
        echo "<strong>Error:</strong> {$result['message']}";
        echo "</div>";
    }
    
    echo "<a href='merge_alumnos_becados.php' class='btn btn-primary'>← Back to Preview</a>";
}

// Apply mode (all groups)
elseif ($action === 'apply_all') {
    $duplicate_groups = fetch_duplicate_groups();
    
    $total_merged = 0;
    $failed_groups = [];
    
    foreach ($duplicate_groups as $group) {
        $result = merge_duplicate_group($group['estudiante_id'], $group['beca_tipo']);
        if ($result['success']) {
            $total_merged += $result['duplicates'];
        } else {
            $failed_groups[] = "{$group['estudiante_id']} {$group['beca_tipo']}";    
        }
    }
    
    echo "<div class='alert alert-success'>";
    echo "<strong>✓ Success!</strong> Successfully merged {$total_merged} duplicate records.";
    echo "</div>";
    
    if (!empty($failed_groups)) {
        echo "<div class='alert alert-warning'>";
        echo "<strong>⚠️ Failed:</strong> {$failed_groups}";
        echo "</div>";
    }
    
    echo "<a href='merge_alumnos_becados.php' class='btn btn-primary'>← Back to Preview</a>";
}

?>
</div>
</body>
</html>