#!/lamp/php/bin/php
<?php
/**
 * Create missing properties from PMS data
 */

declare(strict_types=1);

$pdo = new PDO("mysql:host=localhost;unix_socket=/lamp/mysql/mysql.sock;dbname=quantix", "root", "M@chiavell1", [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
]);

function generateId(): string {
    return bin2hex(random_bytes(16));
}

echo "Loading existing properties from DB...\n";
$existingProps = [];
$res = $pdo->query("
    SELECT propiedad_id, nombre_propiedad, departamento, direccion, numero_unidad, 
           codigo_postal, colonia, estado, municipio, propietario_id
    FROM propiedad
")->fetchAll();
foreach ($res as $p) {
    $key = strtolower(trim($p['nombre_propiedad']));
    $existingProps[$key] = $p;
}
echo "Loaded " . count($existingProps) . " properties.\n\n";

echo "Loading Cloudbeds data...\n";
$cloudbedsData = [];
$res = $pdo->query("
    SELECT property, room_number, COUNT(*) as cnt
    FROM cloudbeds_reserva 
    WHERE check_in_date >= '2025-01-01'
    GROUP BY property, room_number
    ORDER BY property, room_number
")->fetchAll();
foreach ($res as $r) {
    $building = trim($r['property']);
    $unit = trim($r['room_number']);
    if (empty($building)) continue;
    
    if (!isset($cloudbedsData[$building])) {
        $cloudbedsData[$building] = ['units' => [], 'total' => 0];
    }
    if (!empty($unit) && $unit !== 'N/A') {
        $cloudbedsData[$building]['units'][$unit] = $r['cnt'];
    }
    $cloudbedsData[$building]['total'] += $r['cnt'];
}
echo "Loaded " . count($cloudbedsData) . " buildings.\n\n";

echo "Loading Hostify data...\n";
$hostifyData = [];
$res = $pdo->query("
    SELECT anuncio, COUNT(*) as cnt
    FROM hostify_reserva 
    WHERE check_in >= '2025-01-01'
    GROUP BY anuncio
")->fetchAll();
foreach ($res as $r) {
    $anuncio = trim($r['anuncio']);
    if (empty($anuncio)) continue;
    
    // Extract building and unit
    if (preg_match('/^([^-]+?)\s*-\s*(\d+[a-z]?|\d{3,4}(?:\s*[-–]\s*\d+)?)/', $anuncio, $m)) {
        $building = trim($m[1]);
        $unit = trim($m[2]);
    } else {
        $building = $anuncio;
        $unit = '';
    }
    
    if (!isset($hostifyData[$building])) {
        $hostifyData[$building] = ['units' => [], 'anuncios' => [], 'total' => 0];
    }
    if (!empty($unit)) {
        $hostifyData[$building]['units'][$unit] = ($hostifyData[$building]['units'][$unit] ?? 0) + $r['cnt'];
    }
    $hostifyData[$building]['anuncios'][] = $anuncio;
    $hostifyData[$building]['total'] += $r['cnt'];
}
echo "Loaded " . count($hostifyData) . " buildings.\n\n";

$newProps = [];

// Find Cloudbeds buildings that need additional units
echo "Analyzing Cloudbeds buildings...\n";
$buildingMapping = [
    'Mr W Medellin' => ['base' => 'Medellin 148', 'addr' => 'Medellin 148, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    'Mr W Zamora' => ['base' => 'Zamora 71', 'addr' => 'Zamora 71, Condesa', 'cp' => '06100', 'colonia' => 'CONDESA'],
    'Mr W Tonalá' => ['base' => 'Tonalá 127', 'addr' => 'Tonalá 127, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    'El Alfonsa' => ['base' => 'El Alfonsa', 'addr' => 'Alfonso Reyes 120, Condesa', 'cp' => '06100', 'colonia' => 'CONDESA'],
    'Casa Tenue' => ['base' => 'Casa Tenue', 'addr' => 'Durazno 74, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
];

foreach ($cloudbedsData as $building => $data) {
    if (!isset($buildingMapping[$building])) continue;
    
    $map = $buildingMapping[$building];
    $baseName = $map['base'];
    
    // Find existing property
    $existingUnit = null;
    foreach ($existingProps as $key => $prop) {
        if (stripos($key, strtolower($baseName)) !== false) {
            $existingUnit = $prop;
            break;
        }
    }
    
    if (!$existingUnit) {
        echo "  WARNING: No existing property found for $building\n";
        continue;
    }
    
    // Check which units are missing
    $existingUnitNum = $existingUnit['numero_unidad'] ?? '';
    
    foreach ($data['units'] as $unit => $cnt) {
        // Normalize unit for comparison
        $unitStr = (string)$unit;
        $normUnit = strtolower(preg_replace('/[^a-z0-9]/', '', $unitStr));
        $normExisting = strtolower(preg_replace('/[^a-z0-9]/', '', $existingUnitNum));
        
        if ($normUnit !== $normExisting && $normUnit !== '') {
            // This unit needs to be created
            $propName = "$baseName - $unit";
            $normKey = strtolower($propName);
            
            if (!isset($existingProps[$normKey])) {
                $newProps[] = [
                    'nombre_propiedad' => $propName,
                    'departamento' => $baseName,
                    'direccion' => $map['addr'],
                    'numero_unidad' => $unit,
                    'codigo_postal' => $map['cp'],
                    'colonia' => $map['colonia'],
                    'propietario_id' => $existingUnit['propietario_id'],
                    'count' => $cnt,
                ];
                $existingProps[$normKey] = ['propiedad_id' => 'new']; // Mark as added
            }
        }
    }
}

// Find Hostify buildings that need additional units
echo "\nAnalyzing Hostify buildings...\n";
$hostifyMapping = [
    'Rio Elba 50' => ['base' => 'Rio Elba', 'addr' => 'Rio Elba 50, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    'Ometusco 104' => ['base' => 'Ometusco', 'addr' => 'Ometusco 104, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    'Huatusco 6' => ['base' => 'Huatusco', 'addr' => 'Huatusco 6, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
    'Guanajuato 197' => ['base' => 'Guanajuato 197', 'addr' => 'Guanajuato 197, Roma Norte', 'cp' => '06700', 'colonia' => 'ROMA NORTE'],
    'Emilio Castelar 212' => ['base' => 'Emilio Castelar', 'addr' => 'Emilio Castelar 212, Polanco', 'cp' => '11560', 'colonia' => 'POLANCO'],
    'Hamburgo 14' => ['base' => 'Hamburgo 14', 'addr' => 'Hamburgo 14, Juárez', 'cp' => '06600', 'colonia' => 'JUÁREZ'],
    'Tlacotalpan 6' => ['base' => 'Tlacotalpan', 'addr' => 'Tlacotalpan 6, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
    'Campeche Frida' => ['base' => 'Casa Frida', 'addr' => 'Campeche 44, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
    'Campeche Ana' => ['base' => 'Casa Ana', 'addr' => 'Campeche 28, Condesa', 'cp' => '06140', 'colonia' => 'CONDESA'],
];

foreach ($hostifyData as $building => $data) {
    if (!isset($hostifyMapping[$building])) continue;
    
    $map = $hostifyMapping[$building];
    $baseName = $map['base'];
    
    // Find existing property
    $existingUnit = null;
    foreach ($existingProps as $key => $prop) {
        if (stripos($key, strtolower($baseName)) !== false) {
            $existingUnit = $prop;
            break;
        }
    }
    
    if (!$existingUnit) {
        echo "  WARNING: No existing property found for $building\n";
        continue;
    }
    
    // Check which units are missing
    $existingUnitNum = $existingUnit['numero_unidad'] ?? '';
    
    foreach ($data['units'] as $unit => $cnt) {
        $unitStr = (string)$unit;
        $normUnit = strtolower(preg_replace('/[^a-z0-9]/', '', $unitStr));
        $normExisting = strtolower(preg_replace('/[^a-z0-9]/', '', (string)$existingUnitNum));
        
        if ($normUnit !== $normExisting && $normUnit !== '') {
            $propName = "$baseName - $unit";
            $normKey = strtolower($propName);
            
            if (!isset($existingProps[$normKey])) {
                $newProps[] = [
                    'nombre_propiedad' => $propName,
                    'departamento' => $baseName,
                    'direccion' => $map['addr'],
                    'numero_unidad' => $unit,
                    'codigo_postal' => $map['cp'],
                    'colonia' => $map['colonia'],
                    'propietario_id' => $existingUnit['propietario_id'],
                    'count' => $cnt,
                ];
                $existingProps[$normKey] = ['propiedad_id' => 'new'];
            }
        }
    }
}

echo "\n" . str_repeat("=", 60) . "\n";
echo "NEW PROPERTIES TO CREATE: " . count($newProps) . "\n";
echo str_repeat("=", 60) . "\n\n";

// Deduplicate by nombre_propiedad
$uniqueProps = [];
foreach ($newProps as $p) {
    $key = $p['nombre_propiedad'];
    if (!isset($uniqueProps[$key])) {
        $uniqueProps[$key] = $p;
    }
}

foreach ($uniqueProps as $p) {
    echo "  " . $p['nombre_propiedad'] . " (unit: " . $p['numero_unidad'] . ", reservations: " . $p['count'] . ")\n";
}

echo "\n" . count($uniqueProps) . " properties would be created.\n";
echo "This would cover approximately " . array_sum(array_column($uniqueProps, 'count')) . " reservations.\n";
