<?php
/**
 * PMS Matcher CLI - Iteration Baseline
 * 
 * Runs matching algorithm from command line for testing
 * Once verified, logic will be integrated into link_pms_propiedades.php
 *
 * Usage:
 *   php pms_matcher_cli.php [--target=both|cloudbeds|hostify] [--apply] [--threshold=80]
 * 
 * Examples:
 *   php pms_matcher_cli.php --target=both              # Preview all matches
 *   php pms_matcher_cli.php --target=hostify --apply   # Apply Hostify matches
 *   php pms_matcher_cli.php --target=cloudbeds --threshold=85 --apply
 */

require_once("../../inc/config.php");
require_once(__DIR__ . '/matcher_lib_extracted.php');

$opts = getopt('', ['target:', 'apply', 'threshold::', 'help']);

if (isset($opts['help'])) {
    echo "Usage: php pms_matcher_cli.php [options]\n\n";
    echo "Options:\n";
    echo "  --target=both|cloudbeds|hostify   Target system (default: both)\n";
    echo "  --apply                           Actually update database (default: preview only)\n";
    echo "  --threshold=N                     Confidence threshold (default: 80)\n";
    echo "  --help                            Show this help\n";
    exit(0);
}

$target = $opts['target'] ?? 'both';
$apply_changes = isset($opts['apply']);
$threshold = (int)($opts['threshold'] ?? 80);

echo "═══════════════════════════════════════════════════════════════\n";
echo "  PMS MATCHER CLI - ITERATION BASELINE\n";
echo "═══════════════════════════════════════════════════════════════\n";
echo "Target: $target\n";
echo "Mode: " . ($apply_changes ? "APPLY (will update DB)" : "PREVIEW (read-only)") . "\n";
echo "Threshold: $threshold%\n";
echo "───────────────────────────────────────────────────────────────\n\n";

$stats = [
    'cloudbeds' => ['total' => 0, 'matched' => 0, 'by_tier' => []],
    'hostify' => ['total' => 0, 'matched' => 0, 'by_tier' => []]
];

// Load properties
echo "[1/5] Loading properties...\n";
$sql_props = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_props);
echo "  ✓ Loaded " . count($propiedades) . " properties\n\n";

$unmatched_cloudbeds = [];
$unmatched_hostify = [];

// Process Cloudbeds
if ($target === 'both' || $target === 'cloudbeds') {
    echo "[2/5] Processing Cloudbeds reservations...\n";
    $sql_cb = "SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' AND propiedad_id IS NULL";
    $cb_reservas = ia_sqlArrayIndx($sql_cb);
    $stats['cloudbeds']['total'] = count($cb_reservas);
    
    echo "  Found " . count($cb_reservas) . " unmatched Cloudbeds reservations\n";
    
    $matches = [];
    foreach ($cb_reservas as $cb) {
        $best_match = null;
        $best_confidence = 0;
        
        foreach ($propiedades as $prop) {
            $result = match_cloudbeds(
                $prop['nombre_propiedad'],
                $prop['direccion'] ?? '',
                $cb['property'],
                $cb['room_number'] ?? ''
            );
            
            if ($result['match'] && $result['confidence'] > $best_confidence) {
                $best_confidence = $result['confidence'];
                $best_match = [
                    'propiedad' => $prop,
                    'reserva' => $cb,
                    'result' => $result
                ];
            }
        }
        
        if ($best_match && $best_confidence >= $threshold) {
            $matches[] = $best_match;
            $tier = $best_match['result']['tier'];
            $stats['cloudbeds']['by_tier'][$tier] = ($stats['cloudbeds']['by_tier'][$tier] ?? 0) + 1;
        } else {
            $unmatched_cloudbeds[] = $cb;
        }
    }
    
    $stats['cloudbeds']['matched'] = count($matches);
    echo "  ✓ Matched: " . count($matches) . " (≥{$threshold}%)\n";
    echo "  ✗ Unmatched: " . count($unmatched_cloudbeds) . "\n";
    
    if ($apply_changes && !empty($matches)) {
        echo "  Applying Cloudbeds matches to database...\n";
        foreach ($matches as $match) {
            $sql = "UPDATE cloudbeds_reserva 
                    SET propiedad_id = " . strit($match['propiedad']['propiedad_id']) . ",
                        match_confidence = " . $match['result']['confidence'] . ",
                        match_tier = " . $match['result']['tier'] . "
                    WHERE cloudbeds_reserva_id = " . strit($match['reserva']['cloudbeds_reserva_id']);
            ia_query($sql);
        }
        echo "  ✓ Applied " . count($matches) . " matches\n";
    }
    echo "\n";
}

// Process Hostify
if ($target === 'both' || $target === 'hostify') {
    echo "[3/5] Processing Hostify reservations...\n";
    $sql_hf = "SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01' AND propiedad_id IS NULL";
    $hf_reservas = ia_sqlArrayIndx($sql_hf);
    $stats['hostify']['total'] = count($hf_reservas);
    
    echo "  Found " . count($hf_reservas) . " unmatched Hostify reservations\n";
    
    $matches = [];
    foreach ($hf_reservas as $hf) {
        $best_match = null;
        $best_confidence = 0;
        
        // Try combo detection first
        $combo = expand_combo_anuncio($hf['anuncio']);
        if (!empty($combo['units'])) {
            foreach ($propiedades as $prop) {
                $prop_street = extract_street_name($prop['nombre_propiedad']);
                if (strpos($combo['street'], $prop_street) !== false) {
                    $result = ['match' => true, 'confidence' => 95, 'tier' => 0, 'pattern' => 'combo_match'];
                    if ($result['confidence'] > $best_confidence) {
                        $best_confidence = $result['confidence'];
                        $best_match = ['propiedad' => $prop, 'reserva' => $hf, 'result' => $result];
                    }
                }
            }
        }
        
        // If no combo match, try standard tiers
        if (!$best_match) {
            foreach ($propiedades as $prop) {
                $result = match_hostify(
                    $prop['nombre_propiedad'],
                    $prop['direccion'] ?? '',
                    $hf['anuncio']
                );
                
                if ($result['match'] && $result['confidence'] > $best_confidence) {
                    $best_confidence = $result['confidence'];
                    $best_match = ['propiedad' => $prop, 'reserva' => $hf, 'result' => $result];
                }
            }
        }
        
        if ($best_match && $best_confidence >= $threshold) {
            $matches[] = $best_match;
            $tier = $best_match['result']['tier'] ?? 0;
            $stats['hostify']['by_tier'][$tier] = ($stats['hostify']['by_tier'][$tier] ?? 0) + 1;
        } else {
            $unmatched_hostify[] = $hf;
        }
    }
    
    $stats['hostify']['matched'] = count($matches);
    echo "  ✓ Matched: " . count($matches) . " (≥{$threshold}%)\n";
    echo "  ✗ Unmatched: " . count($unmatched_hostify) . "\n";
    
    if ($apply_changes && !empty($matches)) {
        echo "  Applying Hostify matches to database...\n";
        foreach ($matches as $match) {
            $sql = "UPDATE hostify_reserva 
                    SET propiedad_id = " . strit($match['propiedad']['propiedad_id']) . ",
                        match_confidence = " . $match['result']['confidence'] . ",
                        match_tier = " . ($match['result']['tier'] ?? 0) . "
                    WHERE hostify_reserva_id = " . strit($match['reserva']['hostify_reserva_id']);
            ia_query($sql);
        }
        echo "  ✓ Applied " . count($matches) . " matches\n";
    }
    echo "\n";
}

// Final statistics
echo "[4/5] Final Statistics\n";
echo "───────────────────────────────────────────────────────────────\n";

if ($target === 'both' || $target === 'cloudbeds') {
    $cb_rate = $stats['cloudbeds']['total'] > 0 
        ? round(100 * $stats['cloudbeds']['matched'] / $stats['cloudbeds']['total'], 2) 
        : 0;
    echo "CLOUDBEDS:\n";
    echo "  Total: " . $stats['cloudbeds']['total'] . "\n";
    echo "  Matched: " . $stats['cloudbeds']['matched'] . " ($cb_rate%)\n";
    ksort($stats['cloudbeds']['by_tier']);
    foreach ($stats['cloudbeds']['by_tier'] as $tier => $count) {
        echo "  Tier $tier: $count\n";
    }
    echo "\n";
}

if ($target === 'both' || $target === 'hostify') {
    $hf_rate = $stats['hostify']['total'] > 0 
        ? round(100 * $stats['hostify']['matched'] / $stats['hostify']['total'], 2) 
        : 0;
    echo "HOSTIFY:\n";
    echo "  Total: " . $stats['hostify']['total'] . "\n";
    echo "  Matched: " . $stats['hostify']['matched'] . " ($hf_rate%)\n";
    ksort($stats['hostify']['by_tier']);
    foreach ($stats['hostify']['by_tier'] as $tier => $count) {
        echo "  Tier $tier: $count\n";
    }
    echo "\n";
}

$total_reservations = $stats['cloudbeds']['total'] + $stats['hostify']['total'];
$total_matched = $stats['cloudbeds']['matched'] + $stats['hostify']['matched'];
$total_rate = $total_reservations > 0 ? round(100 * $total_matched / $total_reservations, 2) : 0;

echo "COMBINED:\n";
echo "  Total: $total_reservations\n";
echo "  Matched: $total_matched ($total_rate%)\n";
echo "  Unmatched: " . ($total_reservations - $total_matched) . "\n";

// Show sample unmatched patterns
if (!empty($unmatched_hostify) && count($unmatched_hostify) <= 20) {
    echo "\n[5/5] Unmatched Hostify Sample:\n";
    echo "───────────────────────────────────────────────────────────────\n";
    foreach ($unmatched_hostify as $hf) {
        echo "  - " . substr($hf['anuncio'], 0, 60) . "\n";
    }
} elseif (!empty($unmatched_hostify)) {
    echo "\n[5/5] Top Unmatched Hostify Patterns:\n";
    echo "───────────────────────────────────────────────────────────────\n";
    $patterns = [];
    foreach ($unmatched_hostify as $hf) {
        $key = substr($hf['anuncio'], 0, 40);
        $patterns[$key] = ($patterns[$key] ?? 0) + 1;
    }
    arsort($patterns);
    foreach (array_slice($patterns, 0, 10) as $pattern => $count) {
        echo "  [$count] $pattern...\n";
    }
}

echo "\n" . ($apply_changes ? "✓ DATABASE UPDATED" : "Preview complete (use --apply to update)") . "\n";
echo "═══════════════════════════════════════════════════════════════\n";
