#!/lamp/php/bin/php
<?php
/**
 * PMS Matcher CLI - Iteration Runner
 *
 * Purpose: Run matching iterations from CLI to link PMS reservations to properties
 * Usage: php pms_matcher_iteration.php [--preview|--apply [--tier N]]
 *
 * Options:
 *   --preview    Show matches without applying (default)
 *   --apply      Apply matches to database
 *   --tier N     Apply only matches at confidence tier N (0=combo, 1=perfect, 2=high, 3=medium, 4=low)
 *   --limit N    Limit to N reservations for testing
 *   --source     Filter by source: cloudbeds, hostify, or both (default)
 *
 * Example:
 *   php pms_matcher_iteration.php --preview                    # Preview all matches
 *   php pms_matcher_iteration.php --apply --tier 1             # Apply tier 1 matches only
 *   php pms_matcher_iteration.php --apply --source cloudbeds   # Apply all cloudbeds matches
 *   php pms_matcher_iteration.php --preview --limit 50         # Preview first 50
 */

declare(strict_types=1);

// =============================================================================
// CLI ARGUMENT PARSING
// =============================================================================

$options = getopt('', ['preview', 'apply', 'tier:', 'limit:', 'source:', 'help']);

if (isset($options['help']) || isset($options['h'])) {
    echo <<<HELP
PMS Matcher CLI - Iteration Runner

Usage: php pms_matcher_iteration.php [options]

Options:
  --preview        Show matches without applying (default)
  --apply          Apply matches to database
  --tier N         Apply only matches at confidence tier N
                   0 = Combo matches (multi-unit listings)
                   1 = Perfect matches (exact normalization)
                   2 = High confidence (similar_text >= 85%)
                   3 = Medium confidence (contains match)
                   4 = Low confidence (street + unit partial)
  --limit N        Limit to N reservations for testing
  --source SOURCE  Filter by source: cloudbeds, hostify, or both
  --help, -h       Show this help message

Examples:
  php pms_matcher_iteration.php --preview
  php pms_matcher_iteration.php --apply --tier 1
  php pms_matcher_iteration.php --preview --limit 50 --source hostify

HELP;
    exit(0);
}

$mode = isset($options['apply']) ? 'apply' : 'preview';
$tierFilter = isset($options['tier']) ? (int)$options['tier'] : null;
$limit = isset($options['limit']) ? (int)$options['limit'] : null;
$source = isset($options['source']) ? strtolower($options['source']) : 'both';

if (!in_array($source, ['cloudbeds', 'hostify', 'both'])) {
    fwrite(STDERR, "Error: --source must be 'cloudbeds', 'hostify', or 'both'\n");
    exit(1);
}

// =============================================================================
// DATABASE CONNECTION
// =============================================================================

echo "Connecting to database...\n";

try {
    $dsn = 'mysql:host=localhost;unix_socket=/lamp/mysql/mysql.sock;dbname=quantix;charset=utf8mb4';
    $pdo = new PDO($dsn, 'root', 'M@chiavell1', [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
    echo "Connected successfully.\n";
} catch (PDOException $e) {
    fwrite(STDERR, "Database connection failed: " . $e->getMessage() . "\n");
    exit(1);
}

// =============================================================================
// LOAD DATA
// =============================================================================

echo "Loading properties...\n";
$propiedades = $pdo->query("
    SELECT propiedad_id, nombre_propiedad, direccion, departamento
    FROM propiedad
    ORDER BY nombre_propiedad
")->fetchAll();

echo "Loaded " . count($propiedades) . " properties.\n";

$cloudbedsReservas = [];
$hostifyReservas = [];

if ($source === 'cloudbeds' || $source === 'both') {
    echo "Loading Cloudbeds reservations...\n";
    $sql = "SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01'";
    if ($limit) $sql .= " LIMIT " . $limit;
    $cloudbedsReservas = $pdo->query($sql)->fetchAll();
    echo "Loaded " . count($cloudbedsReservas) . " Cloudbeds reservations.\n";
}

if ($source === 'hostify' || $source === 'both') {
    echo "Loading Hostify reservations...\n";
    $sql = "SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01'";
    if ($limit) $sql .= " LIMIT " . $limit;
    $hostifyReservas = $pdo->query($sql)->fetchAll();
    echo "Loaded " . count($hostifyReservas) . " Hostify reservations.\n";
}

// =============================================================================
// MATCHING FUNCTIONS (extracted from link_pms_propiedades.php)
// =============================================================================

/**
 * Normalize text for comparison
 * - Lowercase
 * - Remove accents
 * - Trim whitespace
 */
function normalize_text(string $text): string {
    if (empty($text)) return '';

    $text = strtolower(trim($text));

    // Remove accents
    $accents = ['á', 'à', 'ä', 'â', 'ã', 'él', 'è', 'ë', 'ê', 'í', 'ì', 'ï', 'î', 'ó', 'ò', 'ö', 'ô', 'õ', 'ú', 'ù', 'ü', 'û', 'ñ', 'ý', 'ÿ', 'ç', 'š', 'ž', 'ć', 'č'];
    $replacements = ['a', 'a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'n', 'y', 'y', 'c', 's', 'z', 'c', 'c'];
    $text = str_replace($accents, $replacements, $text);

    // Remove extra whitespace
    $text = preg_replace('/\s+/', ' ', $text);

    return $text;
}

/**
 * Extract street name from text
 * Returns everything before the first digit
 */
function extract_street_name(string $text): string {
    if (empty($text)) return '';

    $text = normalize_text($text);

    // Find first digit position
    preg_match('/(\d)/', $text, $matches, PREG_OFFSET_CAPTURE);

    if (!empty($matches[0][1])) {
        return trim(substr($text, 0, $matches[0][1]));
    }

    return $text;
}

/**
 * Extract unit number from text
 * Looks for patterns like: 103, PH1, A, B, etc.
 * For Hostify format: "Street BuildingNum - Unit" extracts the unit after "-"
 */
function extract_unit_number(string $text): string {
    if (empty($text)) return '';

    $text = normalize_text($text);

    // Pattern for Hostify: "Street BuildingNum - Unit" format
    // Extract number after the first "-" (for "Sinaloa 51 - 202", extract "202")
    if (preg_match('/-\s*(\d{1,4}[a-z]?)\b/i', $text, $matches)) {
        return trim($matches[1]);
    }

    // Pattern: word boundary followed by digits/letters commonly used for units
    // Matches: "103", "PH1", "PH A", "UNIDAD 5", etc.
    $patterns = [
        '/\b(ph\s*\d{0,2}[a-z]?)\b/i',        // PH patterns: PH1, PH A, PH 5B
        '/\b(unidad\s*\d+)\b/i',              // Unidad patterns
        '/\b(depto\.?\s*\d+)\b/i',            // Depto patterns
        '/\b(apt\.?\s*\d+)\b/i',              // Apt patterns
        '/\b(\d{1,3})\b/',                    // Just digits: 103, 5, 12 (fallback)
        '/\b([a-z])\b(?!\s*\d)/',             // Single letter: A, B, C (not followed by digit)
    ];

    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) {
            return trim($matches[1]);
        }
    }

    return '';
}

/**
 * Compare two strings and return similarity percentage
 */
function similar_text_pct(string $str1, string $str2): float {
    similar_text($str1, $str2, $percent);
    return round($percent, 2);
}

// =============================================================================
// CLOUDBEDS MATCHING FUNCTIONS
// =============================================================================

/**
 * Match Cloudbeds reservation to propiedad
 * Cloudbeds uses: property (building name) + room_number (unit)
 */
function match_cloudbeds(string $propiedadName, string $propiedadDireccion, string $property, string $roomNumber): array {
    if (empty($property)) {
        return ['match' => false];
    }

    $normPropiedad = normalize_text($propiedadName);
    $normDireccion = normalize_text($propiedadDireccion);
    $normProperty = normalize_text($property);
    $normRoom = normalize_text($roomNumber);

    // Extract components
    $propStreet = extract_street_name($normPropiedad);
    $propUnit = extract_unit_number($normPropiedad);
    $propStreetAddr = extract_street_name($normDireccion);

    $cloudStreet = extract_street_name($normProperty);
    $cloudUnit = extract_unit_number($normRoom);

    // Calculate building match (street name similarity)
    $buildingScore = 0;
    $bestBuildingMatch = '';

    $streetOptions = array_filter([$propStreet, $propStreetAddr]);
    foreach ($streetOptions as $propStreetOpt) {
        if (!empty($propStreetOpt) && !empty($cloudStreet)) {
            $similarity = similar_text_pct($cloudStreet, $propStreetOpt);
            if ($similarity > $buildingScore) {
                $buildingScore = $similarity;
                $bestBuildingMatch = $propStreetOpt;
            }
        }
    }

    // Calculate unit match
    $unitScore = 0;
    if (!empty($cloudUnit) && !empty($propUnit)) {
        // Direct match
        if ($cloudUnit === $propUnit) {
            $unitScore = 100;
        }
        // Number to letter conversion (5 -> e, 6 -> f, etc.)
        elseif (is_numeric($cloudUnit) && $cloudUnit >= 1 && $cloudUnit <= 26) {
            $letter = chr(96 + (int)$cloudUnit);
            if ($letter === $propUnit) {
                $unitScore = 95;
            }
        }
        // Partial match
        elseif (strpos($propUnit, $cloudUnit) !== false || strpos($cloudUnit, $propUnit) !== false) {
            $unitScore = 85;
        }
        // Fuzzy match
        else {
            $unitScore = similar_text_pct($cloudUnit, $propUnit);
        }
    }

    // Calculate overall confidence
    // Building score: 60%, Unit score: 40%
    $buildingWeight = 0.6;
    $unitWeight = 0.4;

    $confidence = round(($buildingScore * $buildingWeight) + ($unitScore * $unitWeight));

    // Determine tier based on confidence
    $tier = null;
    $pattern = '';

    if ($confidence >= 95) {
        $tier = 1;
        $pattern = "perfect ({$property} + {$roomNumber})";
    } elseif ($confidence >= 80) {
        $tier = 2;
        $pattern = "high_conf ({$buildingScore}% building, {$unitScore}% unit)";
    } elseif ($confidence >= 65) {
        $tier = 3;
        $pattern = "medium_conf ({$buildingScore}% building, {$unitScore}% unit)";
    } elseif ($confidence >= 40) {
        $tier = 4;
        $pattern = "low_conf ({$buildingScore}% building, {$unitScore}% unit)";
    }

    if ($tier !== null) {
        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => $tier,
            'pattern' => $pattern,
            'building_score' => $buildingScore,
            'unit_score' => $unitScore,
        ];
    }

    return ['match' => false];
}

// =============================================================================
// HOSTIFY MATCHING FUNCTIONS
// =============================================================================

/**
 * Detect multi-unit combo patterns in anuncio
 * Returns array with combo info or null if not a combo
 */
function expand_combo_anuncio(string $anuncio): ?array {
    $normAnuncio = normalize_text($anuncio);

    // Ometusco combo patterns
    if (preg_match('/^ometusco\s*(\d+)[\/\-]?(\d+)?[\/\-]?(\d+)?/i', $normAnuncio, $m)) {
        $units = array_slice($m, 1);
        $units = array_filter($units, fn($u) => !empty($u));
        if (count($units) >= 2) {
            return ['type' => 'ometusco', 'units' => $units];
        }
    }

    // Polanco combo patterns
    if (preg_match('/polanco\s*(\d+)[\/\-](\d+)/i', $normAnuncio, $m)) {
        return ['type' => 'polanco', 'units' => [$m[1], $m[2]]];
    }

    return null;
}

/**
 * Match Hostify reservation to propiedad - Tier 1 (Exact normalization match)
 */
function match_hostify_tier1(string $propiedadName, string $anuncio): array {
    $normPropiedad = normalize_text($propiedadName);
    $normAnuncio = normalize_text($anuncio);

    if ($normPropiedad === $normAnuncio) {
        return [
            'match' => true,
            'confidence' => 100,
            'tier' => 1,
            'pattern' => "exact ({$anuncio})"
        ];
    }

    return ['match' => false];
}

/**
 * Match Hostify reservation to propiedad - Tier 2 (Contains match)
 * Handles pipe-separated unit lists: "Anuncio|103|104|105"
 */
function match_hostify_tier2(string $propiedadName, string $direccion, string $anuncio): array {
    $normPropiedad = normalize_text($propiedadName);
    $normDireccion = normalize_text($direccion);
    $normAnuncio = normalize_text($anuncio);

    // Split by pipe to get individual segments
    $segments = explode('|', $normAnuncio);

    foreach ($segments as $segment) {
        $segment = trim($segment);
        if (empty($segment)) continue;

        // Check if propiedad is contained in the segmento
        if (strpos($segment, $normPropiedad) !== false ||
            strpos($normPropiedad, $segment) !== false) {
            return [
                'match' => true,
                'confidence' => 90,
                'tier' => 2,
                'pattern' => "contains ({$segment})"
            ];
        }

        // Check against address
        if (!empty($normDireccion)) {
            if (strpos($segment, $normDireccion) !== false ||
                strpos($normDireccion, $segment) !== false) {
                return [
                    'match' => true,
                    'confidence' => 85,
                    'tier' => 2,
                    'pattern' => "contains_addr ({$segment})"
                ];
            }
        }
    }

    return ['match' => false];
}

/**
 * Match Hostify reservation to propiedad - Tier 3 (Similarity match)
 */
function match_hostify_tier3(string $propiedadName, string $direccion, string $anuncio): array {
    $normPropiedad = normalize_text($propiedadName);
    $normDireccion = normalize_text($direccion);
    $normAnuncio = normalize_text($anuncio);

    // Try similarity against nombre_propiedad
    $similarity = similar_text_pct($normAnuncio, $normPropiedad);

    if ($similarity >= 85) {
        return [
            'match' => true,
            'confidence' => $similarity,
            'tier' => 3,
            'pattern' => "similarity ({$similarity}%)"
        ];
    }

    // Try similarity against direccion
    if (!empty($normDireccion)) {
        $similarityAddr = similar_text_pct($normAnuncio, $normDireccion);
        if ($similarityAddr >= 85) {
            return [
                'match' => true,
                'confidence' => $similarityAddr,
                'tier' => 3,
                'pattern' => "similarity_addr ({$similarityAddr}%)"
            ];
        }
    }

    return ['match' => false];
}

/**
 * Match Hostify reservation to propiedad - Tier 4 (Street + Unit partial match)
 */
function match_hostify_tier4(string $propiedadName, string $direccion, string $anuncio): array {
    $normPropiedad = normalize_text($propiedadName);
    $normDireccion = normalize_text($direccion);
    $normAnuncio = normalize_text($anuncio);

    // Extract components from anuncio
    $anuncioStreet = extract_street_name($normAnuncio);
    $anuncioUnit = extract_unit_number($normAnuncio);

    if (empty($anuncioStreet)) {
        return ['match' => false];
    }

    // Compare street against propiedad
    $propStreet = extract_street_name($normPropiedad);
    $propStreetAddr = extract_street_name($normDireccion);

    $streetScore = 0;
    $bestStreet = '';

    foreach ([$propStreet, $propStreetAddr] as $s) {
        if (!empty($s)) {
            $score = similar_text_pct($anuncioStreet, $s);
            if ($score > $streetScore) {
                $streetScore = $score;
                $bestStreet = $s;
            }
        }
    }

    // Unit comparison (if both have units)
    $unitScore = 0;
    $propUnit = extract_unit_number($normPropiedad);

    if (!empty($anuncioUnit) && !empty($propUnit)) {
        if ($anuncioUnit === $propUnit) {
            $unitScore = 100;
        } elseif (is_numeric($anuncioUnit) && $anuncioUnit >= 1 && $anuncioUnit <= 26) {
            $letter = chr(96 + (int)$anuncioUnit);
            if ($letter === $propUnit) {
                $unitScore = 95;
            }
        } else {
            $unitScore = similar_text_pct($anuncioUnit, $propUnit);
        }
    }

    // Calculate confidence
    $unitWeight = !empty($anuncioUnit) && !empty($propUnit) ? 0.3 : 0;
    $streetWeight = 1 - $unitWeight;

    $confidence = round(($streetScore * $streetWeight) + ($unitScore * $unitWeight));

    if ($confidence >= 40) {
        return [
            'match' => true,
            'confidence' => $confidence,
            'tier' => 4,
            'pattern' => "street_unit ({$streetScore}% street" . ($unitScore ? ", {$unitScore}% unit" : "") . ")"
        ];
    }

    return ['match' => false];
}

/**
 * Match Hostify combo (multi-unit listings)
 */
function match_hostify_combo(string $propiedadName, string $direccion, string $anuncio): array {
    $combo = expand_combo_anuncio($anuncio);

    if (!$combo) {
        return ['match' => false];
    }

    $normPropiedad = normalize_text($propiedadName);
    $propUnit = extract_unit_number($normPropiedad);

    if (empty($propUnit)) {
        return ['match' => false];
    }

    // Check if propiedad unit matches any combo unit
    foreach ($combo['units'] as $comboUnit) {
        $normComboUnit = normalize_text($comboUnit);

        // Direct match
        if ($propUnit === $normComboUnit) {
            return [
                'match' => true,
                'confidence' => 90,
                'tier' => 0,
                'pattern' => "combo_{$combo['type']} ({$anuncio} → unit {$comboUnit})"
            ];
        }

        // Number to letter conversion
        if (is_numeric($normComboUnit) && $normComboUnit >= 1 && $normComboUnit <= 26) {
            $letter = chr(96 + (int)$normComboUnit);
            if ($letter === $propUnit) {
                return [
                    'match' => true,
                    'confidence' => 88,
                    'tier' => 0,
                    'pattern' => "combo_{$combo['type']}_num2letter"
                ];
            }
        }

        // Partial match
        if (strlen($normComboUnit) >= 1 && strlen($propUnit) >= 1) {
            if (strpos($propUnit, $normComboUnit) !== false ||
                strpos($normComboUnit, $propUnit) !== false) {
                return [
                    'match' => true,
                    'confidence' => 85,
                    'tier' => 0,
                    'pattern' => "combo_{$combo['type']}_partial"
                ];
            }
        }
    }

    return ['match' => false];
}

/**
 * Match Hostify reservation to propiedad - Tier 1.5 (Departamento match)
 * Matches when anuncio contains the propiedad's departamento field
 * Example: "Laredo 20 - Balderas" matches propiedad with departamento="Laredo 20"
 */
function match_hostify_tier1_5(string $departamento, string $anuncio): array {
    if (empty($departamento)) return ['match' => false];

    $normDepto = normalize_text($departamento);
    $normAnuncio = normalize_text($anuncio);

    // Check if anuncio contains departamento (normalized)
    if (strpos($normAnuncio, $normDepto) !== false) {
        return [
            'match' => true,
            'confidence' => 95,
            'tier' => 1,
            'pattern' => "departamento_match ({$departamento})"
        ];
    }

    return ['match' => false];
}

/**
 * Main Hostify matching function - tries all tiers
 */
function match_hostify(string $propiedadName, string $direccion, string $anuncio, string $departamento = ''): array {
    // Try combo matching first
    $result = match_hostify_combo($propiedadName, $direccion, $anuncio);
    if ($result['match']) return $result;

    // Try departamento matching (tier 1.5)
    $result = match_hostify_tier1_5($departamento, $anuncio);
    if ($result['match']) return $result;

    // Try tiers 1-4
    $result = match_hostify_tier1($propiedadName, $anuncio);
    if ($result['match']) return $result;

    $result = match_hostify_tier2($propiedadName, $direccion, $anuncio);
    if ($result['match']) return $result;

    $result = match_hostify_tier3($propiedadName, $direccion, $anuncio);
    if ($result['match']) return $result;

    $result = match_hostify_tier4($propiedadName, $direccion, $anuncio);
    if ($result['match']) return $result;

    return ['match' => false];
}

// =============================================================================
// RUN MATCHING
// =============================================================================

echo "\nRunning matching...\n";

$cloudbedsMatches = [];
$cloudbedsStats = [
    'total' => count($cloudbedsReservas),
    'tier1' => 0, 'tier2' => 0, 'tier3' => 0, 'tier4' => 0,
    'unmatched' => 0, 'high_conf' => 0, 'low_conf' => 0
];

$hostifyMatches = [];
$hostifyStats = [
    'total' => count($hostifyReservas),
    'tier0' => 0, 'tier1' => 0, 'tier2' => 0, 'tier3' => 0, 'tier4' => 0,
    'unmatched' => 0, 'high_conf' => 0, 'low_conf' => 0
];

// Match Cloudbeds
foreach ($cloudbedsReservas as $reserva) {
    $bestMatch = null;
    $bestConfidence = 0;

    foreach ($propiedades as $propiedad) {
        $result = match_cloudbeds(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['property'],
            $reserva['room_number']
        );

        if ($result['match'] && $result['confidence'] > $bestConfidence) {
            $bestMatch = $result;
            $bestMatch['propiedad_id'] = $propiedad['propiedad_id'];
            $bestMatch['propiedad_name'] = $propiedad['nombre_propiedad'];
            $bestConfidence = $result['confidence'];

            if ($bestConfidence == 100) break;
        }
    }

    if ($bestMatch) {
        $cloudbedsMatches[] = [
            'reserva' => $reserva,
            'match' => $bestMatch
        ];

        $tier = $bestMatch['tier'];
        $cloudbedsStats["tier{$tier}"]++;
        $cloudbedsStats[$bestConfidence >= 80 ? 'high_conf' : 'low_conf']++;
    } else {
        $cloudbedsMatches[] = [
            'reserva' => $reserva,
            'match' => null
        ];
        $cloudbedsStats['unmatched']++;
    }
}

// Match Hostify
foreach ($hostifyReservas as $reserva) {
    $bestMatch = null;
    $bestConfidence = 0;

    foreach ($propiedades as $propiedad) {
        $result = match_hostify(
            $propiedad['nombre_propiedad'],
            $propiedad['direccion'],
            $reserva['anuncio'],
            $propiedad['departamento'] ?? ''
        );

        if ($result['match'] && $result['confidence'] > $bestConfidence) {
            $bestMatch = $result;
            $bestMatch['propiedad_id'] = $propiedad['propiedad_id'];
            $bestMatch['propiedad_name'] = $propiedad['nombre_propiedad'];
            $bestConfidence = $result['confidence'];

            if ($bestConfidence == 100) break;
        }
    }

    if ($bestMatch) {
        $hostifyMatches[] = [
            'reserva' => $reserva,
            'match' => $bestMatch
        ];

        $tier = $bestMatch['tier'];
        $hostifyStats["tier{$tier}"]++;
        $hostifyStats[$bestConfidence >= 80 ? 'high_conf' : 'low_conf']++;
    } else {
        $hostifyMatches[] = [
            'reserva' => $reserva,
            'match' => null
        ];
        $hostifyStats['unmatched']++;
    }
}

// =============================================================================
// OUTPUT RESULTS
// =============================================================================

echo "\n" . str_repeat("=", 60) . "\n";
echo "MATCHING RESULTS\n";
echo str_repeat("=", 60) . "\n\n";

// Stats
echo "CLOUDBEDS:\n";
echo "  Total: {$cloudbedsStats['total']}\n";
echo "  Tier 1 (Perfect 95%+): {$cloudbedsStats['tier1']}\n";
echo "  Tier 2 (High 80-94%): {$cloudbedsStats['tier2']}\n";
echo "  Tier 3 (Medium 65-79%): {$cloudbedsStats['tier3']}\n";
echo "  Tier 4 (Low 40-64%): {$cloudbedsStats['tier4']}\n";
echo "  High Confidence (80%+): {$cloudbedsStats['high_conf']}\n";
echo "  Low Confidence (<80%): {$cloudbedsStats['low_conf']}\n";
echo "  Unmatched: {$cloudbedsStats['unmatched']}\n";

echo "\nHOSTIFY:\n";
echo "  Total: {$hostifyStats['total']}\n";
echo "  Tier 0 (Combo): {$hostifyStats['tier0']}\n";
echo "  Tier 1 (Perfect): {$hostifyStats['tier1']}\n";
echo "  Tier 2 (Contains): {$hostifyStats['tier2']}\n";
echo "  Tier 3 (Similarity): {$hostifyStats['tier3']}\n";
echo "  Tier 4 (Street+Unit): {$hostifyStats['tier4']}\n";
echo "  High Confidence (80%+): {$hostifyStats['high_conf']}\n";
echo "  Low Confidence (<80%): {$hostifyStats['low_conf']}\n";
echo "  Unmatched: {$hostifyStats['unmatched']}\n";

$totalMatched = $cloudbedsStats['high_conf'] + $cloudbedsStats['low_conf'] +
                $hostifyStats['high_conf'] + $hostifyStats['low_conf'];
$totalUnmatched = $cloudbedsStats['unmatched'] + $hostifyStats['unmatched'];
$totalHighConf = $cloudbedsStats['high_conf'] + $hostifyStats['high_conf'];

echo "\n" . str_repeat("-", 40) . "\n";
echo "SUMMARY:\n";
echo "  Total Reservations: " . ($cloudbedsStats['total'] + $hostifyStats['total']) . "\n";
echo "  Matched: {$totalMatched}\n";
echo "  Unmatched: {$totalUnmatched}\n";
echo "  High Confidence: {$totalHighConf}\n";
echo str_repeat("-", 40) . "\n";

// Preview mode - show sample matches
if ($mode === 'preview') {
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "SAMPLE MATCHES (first 10 per source)\n";
    echo str_repeat("=", 60) . "\n";

    echo "\n--- CLOUDBEDS ---\n";
    $count = 0;
    foreach ($cloudbedsMatches as $m) {
        if ($count >= 10) break;
        $reserva = $m['reserva'];
        $match = $m['match'];

        if ($match) {
            echo sprintf(
                "  [%2d%%] %s + %s → %s (%s)\n",
                $match['confidence'],
                $reserva['property'],
                $reserva['room_number'],
                $match['propiedad_name'],
                $match['pattern']
            );
        } else {
            echo "  [??] {$reserva['property']} + {$reserva['room_number']} → NO MATCH\n";
        }
        $count++;
    }

    echo "\n--- HOSTIFY ---\n";
    $count = 0;
    foreach ($hostifyMatches as $m) {
        if ($count >= 10) break;
        $reserva = $m['reserva'];
        $match = $m['match'];

        if ($match) {
            echo sprintf(
                "  [%2d%%] %s → %s (%s)\n",
                $match['confidence'],
                substr($reserva['anuncio'], 0, 40),
                $match['propiedad_name'],
                $match['pattern']
            );
        } else {
            echo "  [??] " . substr($reserva['anuncio'], 0, 40) . " → NO MATCH\n";
        }
        $count++;
    }
}

// =============================================================================
// APPLY MODE
// =============================================================================

if ($mode === 'apply') {
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "APPLYING MATCHES\n";
    echo str_repeat("=", 60) . "\n";

    $updatesCloudbeds = 0;
    $updatesHostify = 0;

    // Apply Cloudbeds matches
    foreach ($cloudbedsMatches as $m) {
        $reserva = $m['reserva'];
        $match = $m['match'];

        if (!$match) continue;

        // Apply tier filter (apply all tiers at or below the threshold)
        // Lower tier = higher confidence, so we apply tiers 0-N where N is the threshold
        if ($tierFilter !== null && $match['tier'] > $tierFilter) continue;

        $propiedadId = $match['propiedad_id'];
        $reservaId = $reserva['cloudbeds_reserva_id'];

        $sql = "UPDATE cloudbeds_reserva SET propiedad_id = ? WHERE cloudbeds_reserva_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$propiedadId, $reservaId]);

        $updatesCloudbeds++;
        echo "  [CLOUD] ID {$reservaId} → propiedad_id {$propiedadId} ({$match['confidence']}%)\n";
    }

    // Apply Hostify matches
    foreach ($hostifyMatches as $m) {
        $reserva = $m['reserva'];
        $match = $m['match'];

        if (!$match) continue;

        // Apply tier filter (apply all tiers at or below the threshold)
        // Lower tier = higher confidence, so we apply tiers 0-N where N is the threshold
        if ($tierFilter !== null && $match['tier'] > $tierFilter) continue;

        $propiedadId = $match['propiedad_id'];
        $reservaId = $reserva['hostify_reserva_id'];

        $sql = "UPDATE hostify_reserva SET propiedad_id = ? WHERE hostify_reserva_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$propiedadId, $reservaId]);

        $updatesHostify++;
        echo "  [HOST] ID {$reservaId} → propiedad_id {$propiedadId} ({$match['confidence']}%)\n";
    }

    echo "\n";
    echo str_repeat("-", 40) . "\n";
    echo "APPLIED:\n";
    echo "  Cloudbeds: {$updatesCloudbeds} updates\n";
    echo "  Hostify: {$updatesHostify} updates\n";
    echo str_repeat("-", 40) . "\n";
}

echo "\nDone.\n";
