<?php
/**
 * PMS Matcher - Standalone CLI Version
 * 
 * Uses direct MySQL to avoid framework compatibility issues
 * Can be run from command line
 * 
 * Usage:
 *   php pms_matcher_standalone.php [--apply]
 */

$opts = getopt('', ['apply', 'help']);

if (isset($opts['help'])) {
    echo "Usage: php pms_matcher_standalone.php [--apply]\n";
    echo "  --apply  Actually update database (default: preview only)\n";
    exit(0);
}

$apply = isset($opts['apply']);

// MySQL connection
$host = 'localhost';
$socket = '/lamp/mysql/mysql.sock';
$user = 'root';
$pass = 'M@chiavell1';
$db = 'quantix';

$mysqli = new mysqli($host, $user, $pass, $db, 3306, $socket);
if ($mysqli->connect_error) {
    die("Connection failed: " . $mysqli->connect_error);
}

echo "═══════════════════════════════════════════════════════════════\n";
echo "  PMS MATCHER CLI - BASELINE (Standalone)\n";
echo "═══════════════════════════════════════════════════════════════\n";
echo "Mode: " . ($apply ? "APPLY" : "PREVIEW") . "\n";
echo "───────────────────────────────────────────────────────────────\n\n";

// Load properties
$result = $mysqli->query("SELECT * FROM propiedad ORDER BY nombre_propiedad");
$propiedades = [];
while ($row = $result->fetch_assoc()) {
    $propiedades[$row['propiedad_id']] = $row;
}
echo "[1/5] Loaded " . count($propiedades) . " properties\n\n";

// Helper functions (copied here to avoid framework)
function normalize_text($text) {
    if (empty($text)) return '';
    $accents = ['Á'=>'A','É'=>'E','Í'=>'I','Ó'=>'O','Ú'=>'U','á'=>'a','é'=>'e','í'=>'i','ó'=>'o','ú'=>'u','Ñ'=>'N','ñ'=>'n','Ü'=>'U','ü'=>'u'];
    $text = strtr($text, $accents);
    $text = strtolower($text);
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $text;
}

function extract_street_name($text) {
    $normalized = normalize_text($text);
    $abbr_map = ['slp37'=>'san luis potosi 37','slp 37'=>'san luis potosi 37','vs146'=>'vicente suarez 146','vs 146'=>'vicente suarez 146','ver 2'=>'veracruz 26','ver2'=>'veracruz 26'];
    foreach ($abbr_map as $abbr => $full) {
        if (strpos($normalized, $abbr . '-') !== false) { $normalized = str_replace($abbr . '-', $full . ' | ', $normalized); break; }
        if (strpos($normalized, $abbr) !== false) { $normalized = str_replace($abbr, $full, $normalized); break; }
    }
    if (preg_match('/\d/', $normalized, $matches, PREG_OFFSET_CAPTURE)) {
        $pos = $matches[0][1];
        return trim(substr($normalized, 0, $pos));
    }
    return $normalized;
}

function extract_unit_number($text) {
    $normalized = normalize_text($text);
    if (preg_match('/\b(\d{2,4})\b/', $normalized, $matches)) return $matches[1];
    if (preg_match('/ph\s*(\w+)/i', $normalized, $matches)) return 'ph' . normalize_text($matches[1]);
    if (preg_match('/\b([A-Z])\b/', $text, $matches)) return strtolower($matches[1]);
    return '';
}

function match_cloudbeds($prop, $cb_property, $cb_room_number) {
    $prop_street = extract_street_name($prop['nombre_propiedad']);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($prop['direccion'] ?? '');
    }
    $cb_street = extract_street_name($cb_property);
    
    $norm_cb_property = normalize_text($cb_property);
    $norm_prop_nombre = normalize_text($prop['nombre_propiedad']);
    $norm_prop_dir = normalize_text($prop['direccion'] ?? '');
    
    $building_score = 0;
    if (strlen($norm_cb_property) >= 5) {
        if (strpos($norm_prop_nombre, $norm_cb_property) !== false || strpos($norm_prop_dir, $norm_cb_property) !== false) {
            $building_score = 100;
        } elseif (strlen($prop_street) >= 5 && strlen($cb_street) >= 5) {
            similar_text($prop_street, $cb_street, $percent);
            $building_score = $percent;
        }
    }
    
    $unit_score = 0;
    $prop_unit = extract_unit_number($prop['nombre_propiedad']);
    $cb_unit = extract_unit_number($cb_room_number);
    
    if (!empty($prop_unit) && !empty($cb_unit)) {
        if ($prop_unit === $cb_unit) $unit_score = 100;
        else { similar_text($prop_unit, $cb_unit, $percent); $unit_score = $percent; }
    } elseif (empty($cb_unit) || $cb_room_number === 'N/A') {
        $unit_score = $building_score;
    }
    
    $confidence = round(($building_score * 0.6) + ($unit_score * 0.4));
    
    $tier = 0;
    if ($confidence >= 95) $tier = 1;
    elseif ($confidence >= 80) $tier = 2;
    elseif ($confidence >= 65) $tier = 3;
    elseif ($confidence >= 40) $tier = 4;
    
    return ($tier > 0) ? ['match' => true, 'confidence' => $confidence, 'tier' => $tier] : ['match' => false];
}

function match_hostify($prop, $anuncio) {
    $norm_prop = normalize_text($prop['nombre_propiedad']);
    $norm_dir = normalize_text($prop['direccion'] ?? '');
    $norm_anuncio = normalize_text($anuncio);
    
    // Tier 1: Exact
    if ($norm_prop === $norm_anuncio) {
        return ['match' => true, 'confidence' => 100, 'tier' => 1];
    }
    
    // Tier 2: Contains
    $segments = preg_split('/[\|,]/', $norm_anuncio);
    foreach ($segments as $segment) {
        $segment = trim($segment);
        if (strlen($segment) >= 5 && strlen($norm_prop) >= 5) {
            if (strpos($segment, $norm_prop) !== false || strpos($norm_prop, $segment) !== false) {
                return ['match' => true, 'confidence' => 90, 'tier' => 2];
            }
            if (strlen($norm_dir) >= 5 && (strpos($segment, $norm_dir) !== false || strpos($norm_dir, $segment) !== false)) {
                return ['match' => true, 'confidence' => 90, 'tier' => 2];
            }
        }
    }
    
    // Tier 3: Similarity
    similar_text($norm_prop, $norm_anuncio, $percent);
    if ($percent >= 85) return ['match' => true, 'confidence' => 70, 'tier' => 3];
    similar_text($norm_dir, $norm_anuncio, $percent);
    if ($percent >= 85) return ['match' => true, 'confidence' => 70, 'tier' => 3];
    
    // Tier 4: Street
    $prop_street = extract_street_name($prop['nombre_propiedad']);
    if (empty($prop_street) || strlen($prop_street) < 3) {
        $prop_street = extract_street_name($prop['direccion'] ?? '');
    }
    $anuncio_street = extract_street_name($anuncio);
    
    if (strlen($prop_street) >= 5 && strlen($anuncio_street) >= 5) {
        if (strpos($anuncio_street, $prop_street) !== false || strpos($prop_street, $anuncio_street) !== false) {
            $prop_unit = extract_unit_number($prop['nombre_propiedad']);
            $anuncio_unit = extract_unit_number($anuncio);
            $confidence = (!empty($prop_unit) && !empty($anuncio_unit) && $prop_unit === $anuncio_unit) ? 65 : 50;
            return ['match' => true, 'confidence' => $confidence, 'tier' => 4];
        }
    }
    
    return ['match' => false];
}

// Stats
$stats = ['cloudbeds' => ['total' => 0, 'matched' => 0, 'by_tier' => []], 'hostify' => ['total' => 0, 'matched' => 0, 'by_tier' => []]];

// Process Cloudbeds
echo "[2/5] Processing Cloudbeds...\n";
$result = $mysqli->query("SELECT * FROM cloudbeds_reserva WHERE check_in_date >= '2025-01-01' AND (propiedad_id IS NULL OR propiedad_id = '')");
$cb_reservas = [];
while ($row = $result->fetch_assoc()) { $cb_reservas[] = $row; }
$stats['cloudbeds']['total'] = count($cb_reservas);
echo "  Found " . count($cb_reservas) . " unmatched\n";

$cb_matches = [];
foreach ($cb_reservas as $cb) {
    $best = null; $best_conf = 0;
    foreach ($propiedades as $prop) {
        $r = match_cloudbeds($prop, $cb['property'], $cb['room_number'] ?? '');
        if ($r['match'] && $r['confidence'] > $best_conf) { $best_conf = $r['confidence']; $best = ['prop' => $prop, 'res' => $cb, 'r' => $r]; }
    }
    if ($best && $best_conf >= 80) { $cb_matches[] = $best; $stats['cloudbeds']['by_tier'][$best['r']['tier']]++; }
}
$stats['cloudbeds']['matched'] = count($cb_matches);
echo "  Matched: " . count($cb_matches) . "\n\n";

// Process Hostify
echo "[3/5] Processing Hostify...\n";
$result = $mysqli->query("SELECT * FROM hostify_reserva WHERE check_in >= '2025-01-01' AND (propiedad_id IS NULL OR propiedad_id = '')");
$hf_reservas = [];
while ($row = $result->fetch_assoc()) { $hf_reservas[] = $row; }
$stats['hostify']['total'] = count($hf_reservas);
echo "  Found " . count($hf_reservas) . " unmatched\n";

$hf_matches = [];
foreach ($hf_reservas as $hf) {
    $best = null; $best_conf = 0;
    foreach ($propiedades as $prop) {
        $r = match_hostify($prop, $hf['anuncio']);
        if ($r['match'] && $r['confidence'] > $best_conf) { $best_conf = $r['confidence']; $best = ['prop' => $prop, 'res' => $hf, 'r' => $r]; }
    }
    if ($best && $best_conf >= 80) { $hf_matches[] = $best; $stats['hostify']['by_tier'][$best['r']['tier']]++; }
}
$stats['hostify']['matched'] = count($hf_matches);
echo "  Matched: " . count($hf_matches) . "\n\n";

// Apply matches
if ($apply) {
    echo "[4/5] Applying matches to database...\n";
    foreach ($cb_matches as $m) {
        $mysqli->query("UPDATE cloudbeds_reserva SET propiedad_id = '{$m['prop']['propiedad_id']}', match_confidence = {$m['r']['confidence']}, match_tier = {$m['r']['tier']} WHERE cloudbeds_reserva_id = '{$m['res']['cloudbeds_reserva_id']}'");
    }
    foreach ($hf_matches as $m) {
        $mysqli->query("UPDATE hostify_reserva SET propiedad_id = '{$m['prop']['propiedad_id']}', match_confidence = {$m['r']['confidence']}, match_tier = {$m['r']['tier']} WHERE hostify_reserva_id = '{$m['res']['hostify_reserva_id']}'");
    }
    echo "  Applied " . count($cb_matches) . " Cloudbeds + " . count($hf_matches) . " Hostify matches\n\n";
}

// Statistics
echo "[5/5] Statistics\n";
echo "───────────────────────────────────────────────────────────────\n";

$cb_rate = $stats['cloudbeds']['total'] > 0 ? round(100 * $stats['cloudbeds']['matched'] / $stats['cloudbeds']['total'], 2) : 0;
$hf_rate = $stats['hostify']['total'] > 0 ? round(100 * $stats['hostify']['matched'] / $stats['hostify']['total'], 2) : 0;
$total = $stats['cloudbeds']['total'] + $stats['hostify']['total'];
$matched = $stats['cloudbeds']['matched'] + $stats['hostify']['matched'];
$total_rate = $total > 0 ? round(100 * $matched / $total, 2) : 0;

echo "CLOUDBEDS: {$stats['cloudbeds']['matched']}/{$stats['cloudbeds']['total']} ({$cb_rate}%)\n";
foreach ($stats['cloudbeds']['by_tier'] as $t => $c) echo "  Tier $t: $c\n";

echo "\nHOSTIFY: {$stats['hostify']['matched']}/{$stats['hostify']['total']} ({$hf_rate}%)\n";
foreach ($stats['hostify']['by_tier'] as $t => $c) echo "  Tier $t: $c\n";

echo "\nCOMBINED: $matched/$total ({$total_rate}%)\n";
echo ($apply ? "✓ Database updated" : "Preview complete (use --apply to update)") . "\n";
echo "═══════════════════════════════════════════════════════════════\n";

$mysqli->close();
