<?php
require_once("../../inc/config.php");
require_once("PropertyRevenueLib.php");

$lib = new PropertyRevenueLib();

$fecha_inicio = param('fecha_inicio', date('Y-m-01'));
$fecha_fin = param('fecha_fin', date('Y-m-t'));
$propiedad_id = param('propiedad_id', '');
$propietario_id = param('propietario_id', '');
$sistema = param('sistema', '');
$canal = param('canal', '');
$view_mode = param('view', 'property'); // property, owner, channel
$export = param('export', '');

$filters = [
    'fecha_inicio' => $fecha_inicio,
    'fecha_fin' => $fecha_fin,
    'propiedad_id' => $propiedad_id,
    'propietario_id' => $propietario_id,
    'sistema' => $sistema,
    'canal' => $canal
];

$reservations = $lib->getUnifiedReservations($filters);

switch ($view_mode) {
    case 'owner':
        $grouped_data = $lib->aggregateByOwner($reservations);
        break;
    case 'channel':
        $grouped_data = $lib->aggregateByChannel($reservations);
        break;
    default:
        $grouped_data = $lib->aggregateByProperty($reservations);
}

$grand_totals = $lib->calculateFinancialSummary($lib->aggregateByProperty($reservations));

if ($export === 'csv') {
    $lib->exportToCSV($lib->aggregateByProperty($reservations));
}

if ($export === 'summary') {
    $lib->exportToCSV($lib->aggregateByProperty($reservations), null, false);
}

$property_list = $lib->getPropertyList(true);
$owner_list = $lib->getOwnerList(true);
$channel_list = $lib->getChannelList();
$data_stats = $lib->getDataSourceStats();

function formatMoney($amount) {
    return '$' . number_format((float)$amount, 2);
}

function formatPercent($value, $decimals = 1) {
    return number_format((float)$value, $decimals) . '%';
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reporte de Ingresos por Propiedad</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 20px; }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 12px;
            margin-bottom: 25px;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        .header h1 { font-size: 2em; margin-bottom: 5px; }
        .header p { opacity: 0.9; }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            text-align: center;
        }
        .stat-card .value { font-size: 1.8em; font-weight: 700; color: #667eea; }
        .stat-card .label { font-size: 0.85em; color: #666; margin-top: 5px; }
        .stat-card.highlight { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
        .stat-card.highlight .value { color: white; }
        .stat-card.highlight .label { color: rgba(255,255,255,0.9); }
        
        .filters-panel {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 25px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        .filters-panel h3 { margin-bottom: 15px; color: #444; }
        .filters-row {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: flex-end;
        }
        .filter-group { display: flex; flex-direction: column; }
        .filter-group label { font-size: 0.85em; color: #666; margin-bottom: 5px; }
        .filter-group input, .filter-group select {
            padding: 10px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 0.95em;
            min-width: 150px;
        }
        .filter-group select { min-width: 180px; }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.95em;
            font-weight: 500;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.2s;
        }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5a6fd6; }
        .btn-success { background: #28a745; color: white; }
        .btn-success:hover { background: #218838; }
        .btn-outline { background: white; border: 1px solid #ddd; color: #666; }
        .btn-outline:hover { background: #f8f9fa; }
        
        .view-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        .view-tab {
            padding: 10px 20px;
            background: white;
            border: 1px solid #ddd;
            border-radius: 6px;
            text-decoration: none;
            color: #666;
            font-weight: 500;
        }
        .view-tab.active { background: #667eea; color: white; border-color: #667eea; }
        
        .data-source-info {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            padding: 15px;
            background: #e8f4f8;
            border-radius: 8px;
            font-size: 0.9em;
        }
        .data-source-info .source { display: flex; align-items: center; gap: 8px; }
        .source-dot { width: 10px; height: 10px; border-radius: 50%; }
        .source-dot.cloudbeds { background: #3498db; }
        .source-dot.hostify { background: #e74c3c; }
        
        .property-card {
            background: white;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            overflow: hidden;
        }
        .property-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .property-header h3 { font-size: 1.1em; }
        .property-header .owner { opacity: 0.9; font-size: 0.9em; }
        .property-header .subtotal { text-align: right; }
        .property-header .subtotal .amount { font-size: 1.3em; font-weight: 700; }
        
        .reservations-table {
            width: 100%;
            border-collapse: collapse;
        }
        .reservations-table th {
            background: #f8f9fa;
            padding: 12px 15px;
            text-align: left;
            font-weight: 600;
            color: #555;
            font-size: 0.85em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border-bottom: 2px solid #eee;
        }
        .reservations-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #eee;
            font-size: 0.95em;
        }
        .reservations-table tr:hover { background: #f8f9fa; }
        .reservations-table .money { text-align: right; font-family: 'SF Mono', Monaco, monospace; }
        .reservations-table .center { text-align: center; }
        
        .system-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.75em;
            font-weight: 600;
        }
        .system-badge.cloudbeds { background: #e3f2fd; color: #1565c0; }
        .system-badge.hostify { background: #ffebee; color: #c62828; }
        
        .channel-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 0.75em;
            background: #f3e5f5;
            color: #7b1fa2;
        }
        
        .subtotal-row {
            background: #fff3cd !important;
            font-weight: 600;
        }
        .subtotal-row td { border-top: 2px solid #ffc107; }
        
        .grand-total-card {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            padding: 25px;
            border-radius: 10px;
            margin-top: 30px;
        }
        .grand-total-card h3 { margin-bottom: 20px; font-size: 1.3em; }
        .grand-total-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 20px;
        }
        .grand-total-item .value { font-size: 1.5em; font-weight: 700; }
        .grand-total-item .label { opacity: 0.9; font-size: 0.9em; }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        .empty-state h3 { color: #333; margin-bottom: 10px; }
        
        .collapsible { cursor: pointer; }
        .collapsible-content { display: none; }
        .collapsible-content.show { display: table-row-group; }
        
        @media (max-width: 768px) {
            .filters-row { flex-direction: column; }
            .filter-group input, .filter-group select { width: 100%; }
            .property-header { flex-direction: column; gap: 10px; }
            .reservations-table { font-size: 0.85em; }
            .reservations-table th, .reservations-table td { padding: 8px 10px; }
        }
        
        @media print {
            .filters-panel, .view-tabs, .btn { display: none; }
            .property-card { break-inside: avoid; }
        }
    </style>
</head>
<body>

<div class="container">
    <div class="header">
        <h1>Reporte de Ingresos por Propiedad</h1>
        <p>Datos unificados de Cloudbeds y Hostify | <?= date('d/m/Y H:i') ?></p>
    </div>
    
    <div class="stats-grid">
        <div class="stat-card highlight">
            <div class="value"><?= formatMoney($grand_totals['ingreso_bruto']) ?></div>
            <div class="label">Ingresos Totales</div>
        </div>
        <div class="stat-card">
            <div class="value"><?= number_format($grand_totals['count']) ?></div>
            <div class="label">Reservaciones</div>
        </div>
        <div class="stat-card">
            <div class="value"><?= number_format($grand_totals['noches']) ?></div>
            <div class="label">Noches Ocupadas</div>
        </div>
        <div class="stat-card">
            <div class="value"><?= number_format($grand_totals['propiedades']) ?></div>
            <div class="label">Propiedades</div>
        </div>
        <div class="stat-card">
            <div class="value"><?= formatMoney($grand_totals['avg_nightly_rate']) ?></div>
            <div class="label">Tarifa Prom/Noche</div>
        </div>
        <div class="stat-card">
            <div class="value"><?= $grand_totals['avg_stay_length'] ?></div>
            <div class="label">Estancia Promedio</div>
        </div>
    </div>
    
    <div class="filters-panel">
        <h3>Filtros</h3>
        <form method="GET">
            <div class="filters-row">
                <div class="filter-group">
                    <label>Fecha Inicio</label>
                    <input type="date" name="fecha_inicio" value="<?= htmlspecialchars($fecha_inicio) ?>">
                </div>
                <div class="filter-group">
                    <label>Fecha Fin</label>
                    <input type="date" name="fecha_fin" value="<?= htmlspecialchars($fecha_fin) ?>">
                </div>
                <div class="filter-group">
                    <label>Propiedad</label>
                    <select name="propiedad_id">
                        <option value="">Todas las propiedades</option>
                        <?php foreach ($property_list as $id => $name): ?>
                            <option value="<?= htmlspecialchars($id) ?>" <?= $propiedad_id === $id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($name) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="filter-group">
                    <label>Propietario</label>
                    <select name="propietario_id">
                        <option value="">Todos los propietarios</option>
                        <?php foreach ($owner_list as $id => $name): ?>
                            <option value="<?= htmlspecialchars($id) ?>" <?= $propietario_id === $id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($name) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="filter-group">
                    <label>Sistema</label>
                    <select name="sistema">
                        <option value="">Todos</option>
                        <option value="Cloudbeds" <?= $sistema === 'Cloudbeds' ? 'selected' : '' ?>>Cloudbeds</option>
                        <option value="Hostify" <?= $sistema === 'Hostify' ? 'selected' : '' ?>>Hostify</option>
                    </select>
                </div>
                <div class="filter-group">
                    <label>Canal</label>
                    <select name="canal">
                        <option value="">Todos los canales</option>
                        <?php foreach ($channel_list as $ch): ?>
                            <option value="<?= htmlspecialchars($ch) ?>" <?= $canal === $ch ? 'selected' : '' ?>>
                                <?= htmlspecialchars($ch) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <input type="hidden" name="view" value="<?= htmlspecialchars($view_mode) ?>">
                <button type="submit" class="btn btn-primary">Filtrar</button>
                <a href="?<?= http_build_query(array_merge($_GET, ['export' => 'csv'])) ?>" class="btn btn-success">CSV Detallado</a>
                <a href="?<?= http_build_query(array_merge($_GET, ['export' => 'summary'])) ?>" class="btn btn-outline">CSV Resumen</a>
            </div>
        </form>
    </div>
    
    <div class="data-source-info">
        <div class="source">
            <span class="source-dot cloudbeds"></span>
            <strong>Cloudbeds:</strong> <?= number_format($data_stats['cloudbeds']['linked_to_property']) ?> reservas vinculadas
        </div>
        <div class="source">
            <span class="source-dot hostify"></span>
            <strong>Hostify:</strong> <?= number_format($data_stats['hostify']['linked_to_property']) ?> reservas vinculadas
        </div>
    </div>
    
    <div class="view-tabs">
        <a href="?<?= http_build_query(array_merge($_GET, ['view' => 'property'])) ?>" 
           class="view-tab <?= $view_mode === 'property' ? 'active' : '' ?>">Por Propiedad</a>
        <a href="?<?= http_build_query(array_merge($_GET, ['view' => 'owner'])) ?>" 
           class="view-tab <?= $view_mode === 'owner' ? 'active' : '' ?>">Por Propietario</a>
        <a href="?<?= http_build_query(array_merge($_GET, ['view' => 'channel'])) ?>" 
           class="view-tab <?= $view_mode === 'channel' ? 'active' : '' ?>">Por Canal</a>
    </div>
    
    <?php if (empty($grouped_data)): ?>
        <div class="empty-state">
            <h3>No se encontraron reservaciones</h3>
            <p>Intenta ajustar los filtros o el rango de fechas</p>
        </div>
    <?php else: ?>
    
        <?php if ($view_mode === 'property'): ?>
            <?php foreach ($grouped_data as $prop_id => $prop): ?>
                <div class="property-card">
                    <div class="property-header">
                        <div>
                            <h3><?= htmlspecialchars($prop['nombre_propiedad']) ?></h3>
                            <?php if ($prop['propietario']): ?>
                                <div class="owner">Propietario: <?= htmlspecialchars($prop['propietario']) ?></div>
                            <?php endif; ?>
                            <?php if ($prop['colonia']): ?>
                                <div class="owner"><?= htmlspecialchars($prop['colonia']) ?></div>
                            <?php endif; ?>
                        </div>
                        <div class="subtotal">
                            <div class="amount"><?= formatMoney($prop['totals']['ingreso_bruto']) ?></div>
                            <div><?= $prop['totals']['count'] ?> reservas | <?= $prop['totals']['noches'] ?> noches</div>
                        </div>
                    </div>
                    <table class="reservations-table">
                        <thead>
                            <tr>
                                <th>Fecha Entrada</th>
                                <th>Fecha Salida</th>
                                <th class="center">Noches</th>
                                <th>Sistema</th>
                                <th>Canal</th>
                                <th>Huesped</th>
                                <th class="money">Ingreso Bruto</th>
                                <th class="money">Recibido</th>
                                <th class="money">Pendiente</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($prop['reservations'] as $res): ?>
                                <tr>
                                    <td><?= $res['fecha_inicio'] ?></td>
                                    <td><?= $res['fecha_fin'] ?></td>
                                    <td class="center"><?= $res['noches'] ?></td>
                                    <td>
                                        <span class="system-badge <?= strtolower($res['sistema']) ?>">
                                            <?= $res['sistema'] ?>
                                        </span>
                                    </td>
                                    <td><span class="channel-badge"><?= htmlspecialchars($res['canal'] ?: 'Direct') ?></span></td>
                                    <td><?= htmlspecialchars($res['huesped']) ?></td>
                                    <td class="money"><?= formatMoney($res['ingreso_bruto']) ?></td>
                                    <td class="money"><?= formatMoney($res['ingreso_recibido']) ?></td>
                                    <td class="money"><?= formatMoney($res['saldo_pendiente']) ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endforeach; ?>
            
        <?php elseif ($view_mode === 'owner'): ?>
            <?php foreach ($grouped_data as $owner_id => $owner): ?>
                <div class="property-card">
                    <div class="property-header">
                        <div>
                            <h3><?= htmlspecialchars($owner['propietario']) ?></h3>
                            <div class="owner"><?= $owner['property_count'] ?> propiedades</div>
                        </div>
                        <div class="subtotal">
                            <div class="amount"><?= formatMoney($owner['totals']['ingreso_bruto']) ?></div>
                            <div><?= $owner['totals']['count'] ?> reservas | <?= $owner['totals']['noches'] ?> noches</div>
                        </div>
                    </div>
                    <table class="reservations-table">
                        <thead>
                            <tr>
                                <th>Propiedad</th>
                                <th>Fecha Entrada</th>
                                <th class="center">Noches</th>
                                <th>Canal</th>
                                <th>Huesped</th>
                                <th class="money">Ingreso</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($owner['reservations'] as $res): ?>
                                <tr>
                                    <td><?= htmlspecialchars($res['nombre_propiedad']) ?></td>
                                    <td><?= $res['fecha_inicio'] ?></td>
                                    <td class="center"><?= $res['noches'] ?></td>
                                    <td><span class="channel-badge"><?= htmlspecialchars($res['canal'] ?: 'Direct') ?></span></td>
                                    <td><?= htmlspecialchars($res['huesped']) ?></td>
                                    <td class="money"><?= formatMoney($res['ingreso_bruto']) ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endforeach; ?>
            
        <?php elseif ($view_mode === 'channel'): ?>
            <div class="property-card">
                <table class="reservations-table">
                    <thead>
                        <tr>
                            <th>Canal</th>
                            <th class="center">Reservaciones</th>
                            <th class="center">Noches</th>
                            <th class="center">Propiedades</th>
                            <th class="money">Ingresos</th>
                            <th class="money">Tarifa Prom/Noche</th>
                            <th class="center">% del Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($grouped_data as $channel => $data): 
                            $avg_rate = $data['totals']['noches'] > 0 
                                ? $data['totals']['ingreso_bruto'] / $data['totals']['noches'] 
                                : 0;
                            $pct = $grand_totals['ingreso_bruto'] > 0 
                                ? ($data['totals']['ingreso_bruto'] / $grand_totals['ingreso_bruto']) * 100 
                                : 0;
                        ?>
                            <tr>
                                <td><span class="channel-badge"><?= htmlspecialchars($channel) ?></span></td>
                                <td class="center"><?= number_format($data['totals']['count']) ?></td>
                                <td class="center"><?= number_format($data['totals']['noches']) ?></td>
                                <td class="center"><?= $data['property_count'] ?></td>
                                <td class="money"><?= formatMoney($data['totals']['ingreso_bruto']) ?></td>
                                <td class="money"><?= formatMoney($avg_rate) ?></td>
                                <td class="center"><?= formatPercent($pct) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    
    <?php endif; ?>
    
    <div class="grand-total-card">
        <h3>Totales del Periodo: <?= $fecha_inicio ?> al <?= $fecha_fin ?></h3>
        <div class="grand-total-grid">
            <div class="grand-total-item">
                <div class="value"><?= formatMoney($grand_totals['ingreso_bruto']) ?></div>
                <div class="label">Ingresos Brutos</div>
            </div>
            <div class="grand-total-item">
                <div class="value"><?= formatMoney($grand_totals['ingreso_recibido']) ?></div>
                <div class="label">Ingresos Recibidos</div>
            </div>
            <div class="grand-total-item">
                <div class="value"><?= formatMoney($grand_totals['saldo_pendiente']) ?></div>
                <div class="label">Saldo Pendiente</div>
            </div>
            <div class="grand-total-item">
                <div class="value"><?= number_format($grand_totals['count']) ?></div>
                <div class="label">Total Reservaciones</div>
            </div>
            <div class="grand-total-item">
                <div class="value"><?= number_format($grand_totals['noches']) ?></div>
                <div class="label">Total Noches</div>
            </div>
            <div class="grand-total-item">
                <div class="value"><?= number_format($grand_totals['propiedades']) ?></div>
                <div class="label">Propiedades Activas</div>
            </div>
        </div>
    </div>
    
    <div style="margin-top: 30px; text-align: center; color: #666; font-size: 0.85em;">
        <p>Reporte generado: <?= date('d/m/Y H:i:s') ?> | PropertyRevenueLib v1.0</p>
        <p>Datos: Cloudbeds (<?= number_format($data_stats['cloudbeds']['linked_to_property']) ?> reservas) + Hostify (<?= number_format($data_stats['hostify']['linked_to_property']) ?> reservas)</p>
    </div>
</div>

<script>
document.querySelectorAll('.collapsible').forEach(function(el) {
    el.addEventListener('click', function() {
        var content = this.nextElementSibling;
        if (content && content.classList.contains('collapsible-content')) {
            content.classList.toggle('show');
        }
    });
});
</script>

</body>
</html>
