<?php
/**
 * Iteration 4: Actually RUN the matcher on unmatched reservations
 * This script UPDATES the database with new matches
 */

require_once("../../inc/config.php");

echo "<!DOCTYPE html><html><head><title>Iteration 4 - Running Matcher</title></head><body>";
echo "<h1>Iteration 4 - Running Matcher with New Patterns</h1>";
echo "<p><strong>Action:</strong> Re-run matcher on unmatched/low-confidence reservations</p>";
echo "<hr>";

// Get all unmatched or low-confidence Hostify reservations
$sql = "SELECT *
        FROM hostify_reserva
        WHERE check_in >= '2025-01-01'
          AND (propiedad_id IS NULL OR match_confidence < 80)
        LIMIT 100";  // Start with 100 to test

$unmatched = ia_sqlArrayIndx($sql);

echo "<p>Found " . count($unmatched) . " unmatched/low-confidence reservations to process</p>";

// Load all propiedades
$sql_props = "SELECT * FROM propiedad ORDER BY nombre_propiedad";
$propiedades = ia_sqlArrayIndx($sql_props);

echo "<p>Loaded " . count($propiedades) . " properties for matching</p>";
echo "<hr>";

// We'll manually code the matching logic here to avoid include conflicts
function normalize_text_local($text) {
    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);
    $text = mb_strtolower($text, 'UTF-8');
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $text;
}

function expand_combo_local($text) {
    $norm = normalize_text_local($text);

    // PATTERN 0: "ABBR##-###" (SLP37-301, VS146-102)
    if (preg_match('/^([a-z]{2,4})(\d{2,3})\s*[\-]\s*(\d{2,4})/i', $norm, $matches)) {
        $abbr = $matches[1];
        $building_num = $matches[2];
        $unit_num = $matches[3];

        $abbr_map = [
            'slp' => 'san luis potosi',
            'vs' => 'versalles',
            'ver' => 'veracruz',
            'pe' => 'paseo de la reforma',
            'p' => 'paseo de la reforma'
        ];

        $street = $abbr_map[strtolower($abbr)] ?? $abbr;
        $full_building = $street . ' ' . $building_num;

        return [
            'units' => [$unit_num],
            'type' => 'abbr_building_unit',
            'street' => $full_building,
            'building_number' => $building_num
        ];
    }

    // PATTERN 0.3: "ABBR## - #unit"
    if (preg_match('/^([a-z]{2,4})(\d{2,3})\s+\-\s+\#?([a-z]*\d+)/i', $norm, $matches)) {
        $abbr = $matches[1];
        $building_num = $matches[2];
        $unit_num = $matches[3];

        $abbr_map = [
            'slp' => 'san luis potosi',
            'vs' => 'versalles',
            'ver' => 'veracruz',
            'pe' => 'paseo de la reforma',
            'p' => 'paseo de la reforma'
        ];

        $street = $abbr_map[strtolower($abbr)] ?? $abbr;
        $full_building = $street . ' ' . $building_num;

        return [
            'units' => [$unit_num],
            'type' => 'abbr_space_dash_unit',
            'street' => $full_building,
            'building_number' => $building_num
        ];
    }

    return []; // Not a combo
}

// Track stats
$newly_matched = 0;
$improved_confidence = 0;
$still_unmatched = 0;

echo "<h2>Processing Reservations:</h2>";
echo "<table border='1' cellpadding='5' style='border-collapse: collapse;'>";
echo "<tr><th>Anuncio</th><th>Pattern</th><th>Matched Property</th><th>Confidence</th></tr>";

foreach ($unmatched as $reserva) {
    $anuncio = $reserva['anuncio'];
    $reserva_id = $reserva['hostify_reserva_id'];
    $old_confidence = $reserva['match_confidence'] ?? 0;

    // Try to expand combo
    $combo = expand_combo_local($anuncio);

    $best_match = null;
    $best_confidence = $old_confidence;

    if (!empty($combo) && !empty($combo['units'])) {
        // Try to match against properties
        $combo_street = normalize_text_local($combo['street']);
        $combo_unit = $combo['units'][0];

        foreach ($propiedades as $prop) {
            $prop_name = normalize_text_local($prop['nombre_propiedad']);

            // Check if street and unit match
            // Property format: "Street Building | Unit"
            $expected_pattern = $combo_street . ' ' . $combo_unit;
            $alt_pattern = $combo_street . ' ' . $combo['building_number'] . ' | ' . $combo_unit;

            if (strpos($prop_name, $combo_street) !== false) {
                // Street matches! Check unit
                if (strpos($prop_name, $combo_unit) !== false) {
                    // Exact match!
                    if ($best_confidence < 90) {
                        $best_match = $prop;
                        $best_confidence = 90;
                    }
                }
            }
        }
    }

    if ($best_match && $best_confidence > $old_confidence) {
        // Update the database
        $propiedad_id = $best_match['propiedad_id'];
        $pattern = $combo['type'] ?? 'combo';

        $sql_update = "UPDATE hostify_reserva
                      SET propiedad_id = " . strit($propiedad_id) . ",
                          match_tier = 0,
                          match_confidence = $best_confidence,
                          match_pattern = " . strit($pattern) . ",
                          match_timestamp = NOW()
                      WHERE hostify_reserva_id = " . strit($reserva_id);
        ia_query($sql_update);

        if ($old_confidence == 0 || $old_confidence === null) {
            $newly_matched++;
        } else {
            $improved_confidence++;
        }

        echo "<tr>";
        echo "<td>$anuncio</td>";
        echo "<td>{$combo['type']}</td>";
        echo "<td>{$best_match['nombre_propiedad']}</td>";
        echo "<td style='color: green;'><strong>$best_confidence%</strong> (was: $old_confidence%)</td>";
        echo "</tr>";
    } else {
        $still_unmatched++;
    }
}

echo "</table>";
echo "<hr>";

echo "<h2>Results:</h2>";
echo "<ul>";
echo "<li><strong style='color: green;'>Newly Matched:</strong> $newly_matched</li>";
echo "<li><strong style='color: blue;'>Improved Confidence:</strong> $improved_confidence</li>";
echo "<li><strong style='color: orange;'>Still Unmatched:</strong> $still_unmatched</li>";
echo "<li><strong>Total Gain:</strong> " . ($newly_matched + $improved_confidence) . " improvements</li>";
echo "</ul>";

echo "<p><a href='iteration4_test.php'>→ Run Iteration 4 Full Test</a></p>";
echo "</body></html>";
?>
