<?php
/**
 * Quick test to verify slash extraction achieves 100% match
 * Run via CLI: /lamp/php/bin/php test_slash_extraction.php
 */

// Change directory to match web execution context
chdir(__DIR__);
require_once("../../inc/config.php");

// Import the slash extraction function
function extract_after_slash($text) {
    if (empty($text)) return '';
    if (strpos($text, '/') !== false) {
        $parts = explode('/', $text, 2);
        return trim($parts[1]);
    }
    return $text;
}

function normalize_text($text) {
    if (empty($text)) return '';
    $accents = [
        'Á'=>'A', 'É'=>'E', 'Í'=>'I', 'Ó'=>'O', 'Ú'=>'U',
        'á'=>'a', 'é'=>'e', 'í'=>'i', 'ó'=>'o', 'ú'=>'u',
        'Ñ'=>'N', 'ñ'=>'n', 'Ü'=>'U', 'ü'=>'u'
    ];
    $text = strtr($text, $accents);
    $text = mb_strtolower($text, 'UTF-8');
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $text;
}

echo "=============================================================\n";
echo "🎯 SLASH EXTRACTION TEST - VERIFY 100% MATCH\n";
echo "=============================================================\n\n";

// Load data
$sql_propiedades = "SELECT propiedad_id, nombre_propiedad FROM propiedad ORDER BY nombre_propiedad LIMIT 10";
$propiedades = ia_sqlArrayIndx($sql_propiedades);

$sql_propietarios = "SELECT propietario_id, propietario, departamento FROM propietario WHERE vale='Active' ORDER BY propietario LIMIT 10";
$propietarios = ia_sqlArrayIndx($sql_propietarios);

echo "✓ Loaded " . count($propiedades) . " propiedades (sample)\n";
echo "✓ Loaded " . count($propietarios) . " propietarios (sample)\n\n";

// Test slash extraction
echo "--- SLASH EXTRACTION TEST ---\n\n";
$perfect_matches = 0;
$total_tests = 0;

foreach ($propiedades as $idx => $propiedad) {
    if ($idx >= 5) break; // Test first 5

    $raw = $propiedad['nombre_propiedad'];
    $extracted = extract_after_slash($raw);
    $normalized = normalize_text($extracted);

    echo "Propiedad #{$idx}:\n";
    echo "  Raw:        {$raw}\n";
    echo "  Extracted:  {$extracted}\n";
    echo "  Normalized: {$normalized}\n\n";
}

echo "--- MATCHING TEST (Propiedad vs Propietario) ---\n\n";

// Test a few specific matches
$test_cases = [
    [
        'propiedad' => 'Alvaro Obregon 182 - 101 / Álvaro Obregón 182 - 101',
        'departamento' => 'Álvaro Obregón 182 - 101',
        'propietario' => 'Test Owner 1'
    ],
    [
        'propiedad' => 'Alfonso Reyes 176 - 201 / Alfonso Reyes 176 - 201',
        'departamento' => 'Alfonso Reyes 176 - 201',
        'propietario' => 'Test Owner 2'
    ],
    [
        'propiedad' => 'Amsterdam 210 - 302, 602, 402 y DOBLES / Amsterdam 210',
        'departamento' => 'Amsterdam 210',
        'propietario' => 'Test Owner 3'
    ]
];

foreach ($test_cases as $idx => $test) {
    $prop_extracted = extract_after_slash($test['propiedad']);
    $prop_norm = normalize_text($prop_extracted);
    $dept_norm = normalize_text($test['departamento']);

    $match = ($prop_norm === $dept_norm);
    $match_symbol = $match ? '✅' : '❌';

    echo "Test Case #{$idx}:\n";
    echo "  Propiedad (raw):       {$test['propiedad']}\n";
    echo "  Propiedad (extracted): {$prop_extracted}\n";
    echo "  Propiedad (normalized): {$prop_norm}\n";
    echo "  Departamento (normalized): {$dept_norm}\n";
    echo "  Match: {$match_symbol} " . ($match ? "PERFECT MATCH (100%)" : "NO MATCH") . "\n\n";

    $total_tests++;
    if ($match) $perfect_matches++;
}

// Count total matches from real data
echo "--- REAL DATA MATCHING TEST ---\n\n";
$real_matches = 0;
$real_total = 0;

foreach ($propietarios as $propietario) {
    $dept_norm = normalize_text($propietario['departamento']);

    foreach ($propiedades as $propiedad) {
        $prop_extracted = extract_after_slash($propiedad['nombre_propiedad']);
        $prop_norm = normalize_text($prop_extracted);

        if ($prop_norm === $dept_norm) {
            $real_matches++;
            echo "✅ MATCH: '{$propiedad['nombre_propiedad']}' → '{$propietario['departamento']}' ({$propietario['propietario']})\n";
        }

        $real_total++;
    }
}

echo "\n=============================================================\n";
echo "📊 RESULTS\n";
echo "=============================================================\n";
echo "Test Cases: {$perfect_matches}/{$total_tests} perfect matches (" . round(($perfect_matches/$total_tests)*100, 1) . "%)\n";
echo "Real Data Sample: {$real_matches} matches found in " . count($propietarios) . " propietarios × " . count($propiedades) . " propiedades\n";
echo "\n";

if ($perfect_matches === $total_tests) {
    echo "🎉 SUCCESS! Slash extraction achieves 100% match on test cases!\n";
} else {
    echo "⚠️  WARNING: Not all test cases matched. Check normalization.\n";
}

echo "=============================================================\n";
