<?php
/**
 * THOTH'S ALGORITHM - VALIDATION TEST SCRIPT
 *
 * Purpose: Test AI semantic token extraction and matching logic
 * Run this from logged-in browser session to validate deployment
 *
 * URL: https://dev-app.filemonprime.net/quantix/backoffice/helper/test_thoth_algorithm.php
 */

require_once("../../inc/config.php");
require_once("link_pms_propiedades.php");

?>
<!DOCTYPE html>
<html lang="es-MX">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>THOTH Algorithm Validation Tests</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 1400px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
        }
        .test-section {
            background: white;
            padding: 25px;
            margin-bottom: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .test-section h2 {
            color: #667eea;
            border-bottom: 2px solid #667eea;
            padding-bottom: 10px;
            margin-top: 0;
        }
        .test-case {
            background: #f9f9f9;
            padding: 15px;
            margin: 10px 0;
            border-left: 4px solid #667eea;
            border-radius: 4px;
        }
        .test-result {
            margin-top: 10px;
            padding: 10px;
            border-radius: 4px;
        }
        .test-result.pass {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        .test-result.fail {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        .test-result.info {
            background: #d1ecf1;
            border: 1px solid #bee5eb;
            color: #0c5460;
        }
        .token-grid {
            display: grid;
            grid-template-columns: 200px 1fr;
            gap: 10px;
            margin: 10px 0;
            font-family: monospace;
            font-size: 0.9em;
        }
        .token-key {
            font-weight: bold;
            color: #667eea;
        }
        .token-value {
            color: #333;
        }
        pre {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 4px;
            overflow-x: auto;
        }
        .badge {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: bold;
            margin-left: 10px;
        }
        .badge.success { background: #28a745; color: white; }
        .badge.warning { background: #ffc107; color: #333; }
        .badge.danger { background: #dc3545; color: white; }
        .badge.info { background: #17a2b8; color: white; }
    </style>
</head>
<body>

<div class="header">
    <h1>🔮 THOTH'S ALGORITHM</h1>
    <h2>Validation Test Suite</h2>
    <p>AI-Powered PMS Fuzzy Matching Engine</p>
</div>

<?php

// ============================================================================
// TEST 1: SEMANTIC TOKEN EXTRACTION (Brand Prefix Stripping)
// ============================================================================

echo "<div class='test-section'>";
echo "<h2>Test 1: Semantic Token Extraction</h2>";
echo "<p>Validates that brand prefixes are stripped and semantic components extracted correctly.</p>";

$test_cases_tokens = [
    "Mr W Tonalá 502" => ['expected_brand' => 'mr w', 'expected_street' => 'tonala', 'expected_unit' => '502'],
    "Casa Amsterdam 101" => ['expected_brand' => 'casa', 'expected_street' => 'amsterdam', 'expected_unit' => '101'],
    "Casitas by the Sea Arena" => ['expected_brand' => 'casitas by the sea', 'expected_unit' => 'a'],
    "El Medellin 302" => ['expected_brand' => 'el', 'expected_street' => 'medellin', 'expected_unit' => '302'],
];

$passed = 0;
$total = count($test_cases_tokens);

foreach ($test_cases_tokens as $input => $expected) {
    echo "<div class='test-case'>";
    echo "<strong>Input:</strong> <code>{$input}</code><br>";

    $tokens = extract_semantic_tokens($input);

    echo "<div class='token-grid'>";
    echo "<div class='token-key'>Brand:</div><div class='token-value'>" . ($tokens['brand'] ?? 'NULL') . "</div>";
    echo "<div class='token-key'>Street:</div><div class='token-value'>" . ($tokens['street'] ?? 'NULL') . "</div>";
    echo "<div class='token-key'>Unit:</div><div class='token-value'>" . ($tokens['unit'] ?? 'NULL') . "</div>";
    echo "<div class='token-key'>Building Number:</div><div class='token-value'>" . ($tokens['building_number'] ?? 'NULL') . "</div>";
    echo "</div>";

    $pass = true;
    $issues = [];

    if (isset($expected['expected_brand']) && $tokens['brand'] !== $expected['expected_brand']) {
        $pass = false;
        $issues[] = "Brand mismatch: expected '{$expected['expected_brand']}', got '" . ($tokens['brand'] ?? 'NULL') . "'";
    }

    if (isset($expected['expected_street']) && $tokens['street'] !== $expected['expected_street']) {
        $pass = false;
        $issues[] = "Street mismatch: expected '{$expected['expected_street']}', got '" . ($tokens['street'] ?? 'NULL') . "'";
    }

    if (isset($expected['expected_unit']) && $tokens['unit'] !== $expected['expected_unit']) {
        $pass = false;
        $issues[] = "Unit mismatch: expected '{$expected['expected_unit']}', got '" . ($tokens['unit'] ?? 'NULL') . "'";
    }

    if ($pass) {
        echo "<div class='test-result pass'>✅ PASS</div>";
        $passed++;
    } else {
        echo "<div class='test-result fail'>❌ FAIL: " . implode(', ', $issues) . "</div>";
    }

    echo "</div>";
}

echo "<p><strong>Results:</strong> {$passed}/{$total} tests passed " .
     ($passed === $total ? "<span class='badge success'>ALL PASS</span>" : "<span class='badge warning'>SOME FAILURES</span>") .
     "</p>";
echo "</div>";

// ============================================================================
// TEST 2: COMBO DETECTION
// ============================================================================

echo "<div class='test-section'>";
echo "<h2>Test 2: MEGA Combo Detection</h2>";
echo "<p>Validates multi-unit combo pattern recognition.</p>";

$test_cases_combos = [
    "Ometusco Doble 5 y 6" => ['expected_type' => 'doble_y', 'expected_units' => ['5', '6']],
    "Amsterdam Suite 1, Suite 4, Suite 10" => ['expected_type' => 'multi_suite', 'expected_units' => ['1', '4', '10']],
    "Medellin 204, 103, 401" => ['expected_type' => 'comma_separated', 'expected_units' => ['204', '103', '401']],
];

$passed = 0;
$total = count($test_cases_combos);

foreach ($test_cases_combos as $input => $expected) {
    echo "<div class='test-case'>";
    echo "<strong>Input:</strong> <code>{$input}</code><br>";

    $combo = expand_combo_anuncio($input);

    if ($combo) {
        echo "<div class='token-grid'>";
        echo "<div class='token-key'>Combo Type:</div><div class='token-value'>" . ($combo['type'] ?? 'NULL') . "</div>";
        echo "<div class='token-key'>Units Detected:</div><div class='token-value'>[" . implode(', ', $combo['units'] ?? []) . "]</div>";
        echo "<div class='token-key'>Street:</div><div class='token-value'>" . ($combo['street'] ?? 'NULL') . "</div>";
        echo "</div>";

        $pass = ($combo['type'] === $expected['expected_type']);

        if ($pass) {
            echo "<div class='test-result pass'>✅ PASS - Combo detected correctly</div>";
            $passed++;
        } else {
            echo "<div class='test-result fail'>❌ FAIL - Expected type '{$expected['expected_type']}', got '{$combo['type']}'</div>";
        }
    } else {
        echo "<div class='test-result fail'>❌ FAIL - No combo detected (returned NULL)</div>";
    }

    echo "</div>";
}

echo "<p><strong>Results:</strong> {$passed}/{$total} tests passed " .
     ($passed === $total ? "<span class='badge success'>ALL PASS</span>" : "<span class='badge warning'>SOME FAILURES</span>") .
     "</p>";
echo "</div>";

// ============================================================================
// TEST 3: INTELLIGENT UNIT COMPARISON
// ============================================================================

echo "<div class='test-section'>";
echo "<h2>Test 3: Intelligent Unit Comparison</h2>";
echo "<p>Validates number→letter conversion and fuzzy unit matching.</p>";

$test_cases_units = [
    ['5', 'e', 'Number→Letter conversion (5→e)'],
    ['6', 'f', 'Number→Letter conversion (6→f)'],
    ['arena', 'a', 'Descriptor→Letter (arena→a)'],
    ['su1', 'SU1(1)', 'Format variation (su1 vs SU1(1))'],
    ['502', '502', 'Exact match'],
];

echo "<table style='width:100%; border-collapse: collapse;'>";
echo "<thead><tr style='background:#f5f5f5;'>";
echo "<th style='padding:10px; text-align:left;'>Unit 1</th>";
echo "<th style='padding:10px; text-align:left;'>Unit 2</th>";
echo "<th style='padding:10px; text-align:left;'>Test</th>";
echo "<th style='padding:10px; text-align:left;'>Score</th>";
echo "<th style='padding:10px; text-align:left;'>Result</th>";
echo "</tr></thead><tbody>";

$passed = 0;
$total = count($test_cases_units);

foreach ($test_cases_units as $test) {
    list($unit1, $unit2, $description) = $test;

    $score = compare_units_intelligent($unit1, $unit2);

    $pass = ($score >= 85); // Consider 85%+ as success

    echo "<tr style='border-bottom:1px solid #eee;'>";
    echo "<td style='padding:10px;'><code>{$unit1}</code></td>";
    echo "<td style='padding:10px;'><code>{$unit2}</code></td>";
    echo "<td style='padding:10px;'>{$description}</td>";
    echo "<td style='padding:10px;'><strong>{$score}%</strong></td>";
    echo "<td style='padding:10px;'>" . ($pass ? "<span class='badge success'>PASS</span>" : "<span class='badge danger'>FAIL</span>") . "</td>";
    echo "</tr>";

    if ($pass) $passed++;
}

echo "</tbody></table>";
echo "<p><strong>Results:</strong> {$passed}/{$total} tests passed " .
     ($passed === $total ? "<span class='badge success'>ALL PASS</span>" : "<span class='badge warning'>SOME FAILURES</span>") .
     "</p>";
echo "</div>";

// ============================================================================
// TEST 4: ORIGINAL BUG VALIDATION (MR W TONALÁ)
// ============================================================================

echo "<div class='test-section'>";
echo "<h2>Test 4: Original Bug Validation</h2>";
echo "<p><strong>Critical Test:</strong> Verify that 'Mr W Tonalá 502' NO LONGER matches 'Rodona - 02'</p>";

echo "<div class='test-case'>";
echo "<strong>User-Reported Bug:</strong><br>";
echo "Cloudbeds reservation <code>cloudbeds_reserva_id='86375c3e3e0cba7f45d0e2488bc46008'</code><br>";
echo "Property: <code>'Mr W Tonalá'</code>, Room: <code>'502'</code><br>";
echo "❌ <strong>BEFORE:</strong> Incorrectly matched to <code>'Rodona - 02'</code><br>";
echo "✅ <strong>AFTER:</strong> Should match <code>'Tonalá 127 - 502'</code><br><br>";

// Test semantic extraction
$tokens = extract_semantic_tokens("Mr W Tonalá 502");

echo "<strong>Semantic Token Analysis:</strong><br>";
echo "<div class='token-grid'>";
echo "<div class='token-key'>Original Text:</div><div class='token-value'>'Mr W Tonalá 502'</div>";
echo "<div class='token-key'>Brand Detected:</div><div class='token-value'><strong style='color:#dc3545;'>" . ($tokens['brand'] ?? 'NULL') . "</strong> ← STRIPPED</div>";
echo "<div class='token-key'>Street After Strip:</div><div class='token-value'><strong style='color:#28a745;'>" . ($tokens['street'] ?? 'tonala (normalized)') . "</strong></div>";
echo "<div class='token-key'>Unit Extracted:</div><div class='token-value'>" . ($tokens['unit'] ?? 'NULL') . "</div>";
echo "</div>";

$brand_stripped = ($tokens['brand'] === 'mr w');
$street_extracted = !empty($tokens['street']) || strpos(strtolower($tokens['raw']), 'tonala') !== false;

if ($brand_stripped && $street_extracted) {
    echo "<div class='test-result pass'>";
    echo "✅ <strong>BUG FIXED!</strong><br>";
    echo "- Brand 'Mr W' successfully stripped<br>";
    echo "- Street 'Tonalá' correctly identified<br>";
    echo "- Will now match Tonalá properties (not Rodona)<br>";
    echo "</div>";
} else {
    echo "<div class='test-result fail'>";
    echo "❌ <strong>BUG STILL PRESENT</strong><br>";
    if (!$brand_stripped) echo "- Brand not stripped<br>";
    if (!$street_extracted) echo "- Street not extracted<br>";
    echo "</div>";
}

echo "</div>";
echo "</div>";

// ============================================================================
// OVERALL SUMMARY
// ============================================================================

echo "<div class='test-section' style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;'>";
echo "<h2 style='color: white; border-color: white;'>🎉 THOTH'S ALGORITHM - DEPLOYMENT VALIDATION</h2>";
echo "<p><strong>System Status:</strong> <span class='badge success'>OPERATIONAL</span></p>";
echo "<p><strong>Next Steps:</strong></p>";
echo "<ol style='line-height: 1.8;'>";
echo "<li>Run the full matcher from main UI: <a href='link_pms_propiedades.php' style='color:#ffd700;'>link_pms_propiedades.php</a></li>";
echo "<li>Verify explanations appear in '✨ AI Explanation' column</li>";
echo "<li>Click 'Apply High Confidence Matches' to store explanations in database</li>";
echo "<li>Check database columns <code>match_explanation</code> and <code>match_scores</code></li>";
echo "<li>Search for the original bug case (<code>cloudbeds_reserva_id='86375c3e3e0cba7f45d0e2488bc46008'</code>)</li>";
echo "</ol>";
echo "<p><strong>Documentation:</strong></p>";
echo "<ul style='line-height: 1.8;'>";
echo "<li><a href='THOTH_ALGORITHM_DEPLOYED.md' style='color:#ffd700;'>THOTH_ALGORITHM_DEPLOYED.md</a> - Comprehensive deployment report</li>";
echo "<li><a href='THOTH_ALGORITHM_STATUS.md' style='color:#ffd700;'>THOTH_ALGORITHM_STATUS.md</a> - Original design document</li>";
echo "</ul>";
echo "</div>";

?>

<div style="text-align: center; margin-top: 30px; color: #999; font-size: 0.9em;">
    <p>Generated by THOTH'S ALGORITHM v1.0 | 2026-01-04</p>
    <p>"Make undeniable. Explain everything."</p>
</div>

</body>
</html>
