<?php
/**
 * Test script for Tier -1 Estado sequence matching
 * Run via: /lamp/php/bin/php test_tier_minus1.php
 */

// Set up paths for CLI execution
$quantix_root = '/lamp/www/quantix';
require_once("$quantix_root/inc/config.php");
require_once(__DIR__ . "/cfdi_matcher_lib.php");

echo "=== Testing Tier -1 Estado Sequence Matching ===\n\n";

// Simulate an invoice with Estado_de_Cuenta
$test_invoice = [
    'eleyeme_cfdi_emitido_id' => 'test-invoice-001',
    'Total' => 5000.00,
    'Fecha' => '2024-10-15',
    'Estado_de_Cuenta' => 'ING 03 OCT 24 BBVA',  // 3rd deposit in October 2024, BBVA bank
    'Receptor_Nombre' => 'Test Client'
];

// Simulate deposits for October 2024, BBVA bank (banco_cuenta_id = 1)
$test_deposits = [
    // September deposits (should be ignored)
    [
        'banco_cuenta_mov_id' => 'dep-sep-001',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-09-25',
        'deposit' => 3000.00,
        'reference' => 'September deposit'
    ],

    // October deposits for BBVA (chronological order)
    [
        'banco_cuenta_mov_id' => 'dep-oct-001',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-03',
        'deposit' => 2000.00,
        'reference' => 'First October deposit'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-002',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-08',
        'deposit' => 4500.00,
        'reference' => 'Second October deposit'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-003',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-15',
        'deposit' => 5000.00,
        'reference' => 'Third October deposit - SHOULD MATCH'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-004',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-22',
        'deposit' => 7000.00,
        'reference' => 'Fourth October deposit'
    ],

    // October deposits for SANTANDER (should be ignored - wrong bank)
    [
        'banco_cuenta_mov_id' => 'dep-oct-sant-001',
        'banco_cuenta_id' => 2,  // SANTANDER
        'fecha' => '2024-10-05',
        'deposit' => 5000.00,
        'reference' => 'SANTANDER deposit - wrong bank'
    ],

    // November deposits (should be ignored)
    [
        'banco_cuenta_mov_id' => 'dep-nov-001',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-11-05',
        'deposit' => 6000.00,
        'reference' => 'November deposit'
    ],
];

echo "Test Invoice:\n";
echo "  ID: {$test_invoice['eleyeme_cfdi_emitido_id']}\n";
echo "  Amount: \$" . number_format($test_invoice['Total'], 2) . "\n";
echo "  Date: {$test_invoice['Fecha']}\n";
echo "  Estado_de_Cuenta: {$test_invoice['Estado_de_Cuenta']}\n";
echo "  Expected: Should match 3rd deposit in October 2024 for BBVA (dep-oct-003)\n\n";

echo "Available Deposits (total: " . count($test_deposits) . "):\n";
foreach ($test_deposits as $idx => $dep) {
    $bank_name = ($dep['banco_cuenta_id'] == 1) ? 'BBVA' : 'SANTANDER';
    echo sprintf("  [%d] %s | %s | %s | \$%s | %s\n",
        $idx + 1,
        $dep['banco_cuenta_mov_id'],
        $dep['fecha'],
        $bank_name,
        number_format($dep['deposit'], 2),
        $dep['reference']
    );
}
echo "\n";

// Test the matching function
echo "Running match_invoice_to_all_deposits()...\n\n";
$result = match_invoice_to_all_deposits($test_invoice, $test_deposits);

echo "=== RESULT ===\n";
if ($result['match']) {
    echo "✓ MATCH FOUND!\n";
    echo "  Tier: {$result['tier']}\n";
    echo "  Confidence: {$result['confidence']}%\n";
    echo "  Pattern: {$result['pattern']}\n";
    echo "  Explanation: {$result['explanation']}\n";
    echo "  Matched Deposit ID: {$result['matched_deposit']['banco_cuenta_mov_id']}\n";
    echo "  Deposit Date: {$result['matched_deposit']['fecha']}\n";
    echo "  Deposit Amount: \$" . number_format($result['matched_deposit']['deposit'], 2) . "\n";
    echo "  Deposit Reference: {$result['matched_deposit']['reference']}\n";

    // Verify it's the correct deposit
    if ($result['matched_deposit']['banco_cuenta_mov_id'] === 'dep-oct-003') {
        echo "\n✓✓✓ SUCCESS! Matched the correct deposit (3rd in October 2024 for BBVA)\n";
    } else {
        echo "\n✗✗✗ ERROR! Matched wrong deposit!\n";
        echo "  Expected: dep-oct-003\n";
        echo "  Got: {$result['matched_deposit']['banco_cuenta_mov_id']}\n";
    }
} else {
    echo "✗ NO MATCH\n";
    if (isset($result['reason'])) {
        echo "  Reason: {$result['reason']}\n";
    }
}

echo "\n=== Test Complete ===\n";
