<?php
/**
 * Standalone test for Tier -1 Estado sequence matching
 * Tests the core matching logic without full framework dependencies
 */

// Load only the database utilities we need
require_once('/lamp/www/quantix/inc/ia_utilerias.php');
require_once(__DIR__ . '/cfdi_matcher_lib.php');

echo "=== Testing Tier -1 Estado Sequence Matching ===\n\n";

// Test 1: Parse Estado_de_Cuenta
echo "TEST 1: Parsing Estado_de_Cuenta\n";
echo "----------------------------------------\n";

$test_estados = [
    'ING 03 OCT 24 BBVA',
    'ING 13 MAR 24 SANTANDER',
    'ING 01 ENE 25 BBVA',
    'ING 25 DIC 24 SANTANDER',
    'Invalid formato',
];

foreach ($test_estados as $estado) {
    $parsed = parse_estado_cuenta($estado);
    echo "Input: '$estado'\n";
    if ($parsed) {
        echo "  ✓ Parsed: Sequence={$parsed['sequence']}, Month={$parsed['month']}, Year={$parsed['year']}, Bank={$parsed['bank_id']}\n";
        echo "  Batch ID: {$parsed['batch_id']}\n";
    } else {
        echo "  ✗ Failed to parse\n";
    }
    echo "\n";
}

// Test 2: Monthly deposit indexing
echo "\nTEST 2: Monthly Deposit Indexing\n";
echo "----------------------------------------\n";

$test_deposits = [
    // October 2024 BBVA deposits (out of chronological order to test sorting)
    [
        'banco_cuenta_mov_id' => 'dep-oct-004',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-22',
        'deposit' => 7000.00,
        'reference' => 'Fourth'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-001',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-03',
        'deposit' => 2000.00,
        'reference' => 'First'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-003',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-15',
        'deposit' => 5000.00,
        'reference' => 'Third'
    ],
    [
        'banco_cuenta_mov_id' => 'dep-oct-002',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-10-08',
        'deposit' => 4500.00,
        'reference' => 'Second'
    ],
    // Wrong month (should be filtered out)
    [
        'banco_cuenta_mov_id' => 'dep-sep-001',
        'banco_cuenta_id' => 1,  // BBVA
        'fecha' => '2024-09-25',
        'deposit' => 3000.00,
        'reference' => 'September'
    ],
    // Wrong bank (should be filtered out)
    [
        'banco_cuenta_mov_id' => 'dep-oct-sant',
        'banco_cuenta_id' => 2,  // SANTANDER
        'fecha' => '2024-10-10',
        'deposit' => 8000.00,
        'reference' => 'Wrong bank'
    ],
];

echo "Building monthly index for BBVA (ID=1), October 2024...\n";
$monthly_index = build_monthly_deposit_index($test_deposits, 1, 2024, 10);

echo "Indexed deposits: " . count($monthly_index) . " (expected: 4)\n\n";

foreach ($monthly_index as $idx => $deposit) {
    echo sprintf("  Sequence %d: %s | %s | \$%s | %s\n",
        $deposit['_sequence'],
        $deposit['banco_cuenta_mov_id'],
        $deposit['fecha'],
        number_format($deposit['deposit'], 2),
        $deposit['reference']
    );
}

// Test 3: Get deposit by sequence
echo "\n\nTEST 3: Get Deposit by Sequence\n";
echo "----------------------------------------\n";

for ($seq = 1; $seq <= 5; $seq++) {
    $deposit = get_deposit_by_sequence($monthly_index, $seq);
    if ($deposit) {
        echo "Sequence $seq: {$deposit['banco_cuenta_mov_id']} ({$deposit['reference']})\n";
    } else {
        echo "Sequence $seq: Not found\n";
    }
}

// Test 4: Full Tier -1 matching
echo "\n\nTEST 4: Full Tier -1 Matching\n";
echo "----------------------------------------\n";

$test_invoice = [
    'eleyeme_cfdi_emitido_id' => 'test-invoice-001',
    'Total' => 5000.00,
    'Fecha' => '2024-10-15',
    'Estado_de_Cuenta' => 'ING 03 OCT 24 BBVA',  // 3rd deposit in October 2024
    'Receptor_Nombre' => 'Test Client'
];

echo "Invoice: {$test_invoice['eleyeme_cfdi_emitido_id']}\n";
echo "Amount: \$" . number_format($test_invoice['Total'], 2) . "\n";
echo "Estado: {$test_invoice['Estado_de_Cuenta']}\n";
echo "Expected: Should match deposit #3 (dep-oct-003, the Third deposit)\n\n";

$result = match_cfdi_tier_minus1_estado_sequence($test_invoice, $test_deposits);

if ($result['match']) {
    echo "✓✓✓ MATCH FOUND!\n";
    echo "  Tier: {$result['tier']}\n";
    echo "  Confidence: {$result['confidence']}%\n";
    echo "  Pattern: {$result['pattern']}\n";
    echo "  Explanation: {$result['explanation']}\n";
    echo "  Matched Deposit: {$result['deposit']['banco_cuenta_mov_id']}\n";
    echo "  Deposit Date: {$result['deposit']['fecha']}\n";
    echo "  Deposit Amount: \$" . number_format($result['deposit']['deposit'], 2) . "\n";
    echo "  Deposit Reference: {$result['deposit']['reference']}\n";

    if (isset($result['monthly_index_stats'])) {
        echo "\n  Monthly Index Stats:\n";
        foreach ($result['monthly_index_stats'] as $key => $value) {
            echo "    $key: $value\n";
        }
    }

    // Verify correctness
    if ($result['deposit']['banco_cuenta_mov_id'] === 'dep-oct-003') {
        echo "\n✓✓✓ PERFECT! Correctly matched the 3rd deposit in October 2024.\n";
    } else {
        echo "\n✗✗✗ ERROR! Wrong deposit matched.\n";
        echo "  Expected: dep-oct-003\n";
        echo "  Got: {$result['deposit']['banco_cuenta_mov_id']}\n";
    }
} else {
    echo "✗✗✗ NO MATCH\n";
    if (isset($result['reason'])) {
        echo "Reason: {$result['reason']}\n";
    }
}

echo "\n=== All Tests Complete ===\n";
