<?php
/**
 * Unit test for Tier -1 Estado sequence matching functions
 * Tests ONLY the new functions without any database dependencies
 */

// Copy the core functions we need to test (isolated from dependencies)

/**
 * Parse Estado_de_Cuenta field to extract deposit sequence information
 */
function parse_estado_cuenta($estado_texto) {
    if (empty($estado_texto) || !is_string($estado_texto)) {
        return null;
    }

    // Pattern: ING [SEQUENCE] [MONTH] [YEAR] [BANK]
    // Example: "ING 13 MAR 24 SANTANDER"
    // Sequence: 13 (13th deposit in that month)
    // Month: MAR (March)
    // Year: 24 (2024)
    // Bank: SANTANDER
    $pattern = '/ING\s+(\d{1,2})\s+([A-Z]{3})\s+(\d{2})\s+(SANTANDER|BBVA)/i';

    if (preg_match($pattern, $estado_texto, $matches)) {
        $sequence = (int)$matches[1];  // Deposit sequence number (1-indexed)
        $month_abbr = strtoupper($matches[2]);
        $year_2digit = $matches[3];
        $bank_name = strtoupper($matches[4]);

        // Month abbreviation to number mapping
        $meses = [
            'ENE' => 1, 'FEB' => 2, 'MAR' => 3, 'ABR' => 4,
            'MAY' => 5, 'JUN' => 6, 'JUL' => 7, 'AGO' => 8,
            'SEP' => 9, 'OCT' => 10, 'NOV' => 11, 'DIC' => 12
        ];

        if (!array_key_exists($month_abbr, $meses)) {
            return null;  // Invalid month
        }

        $month = $meses[$month_abbr];
        $year = (int)('20' . $year_2digit);  // Convert 24 -> 2024

        // Bank name to banco_cuenta_id mapping
        $bank_id = ($bank_name === 'SANTANDER') ? 2 : 1;  // 1=BBVA, 2=SANTANDER

        return [
            'sequence' => $sequence,      // Deposit sequence number (1, 2, 3, ...)
            'month' => $month,            // Month number (1-12)
            'year' => $year,              // Full year (2024)
            'bank_id' => $bank_id,        // banco_cuenta_id
            'bank_name' => $bank_name,    // Bank name for display
            'batch_id' => sprintf('%d_%d_%d_%d', $bank_id, $year, $month, $sequence)
        ];
    }

    return null;  // Pattern didn't match
}

/**
 * Build monthly deposit index: filter, sort chronologically, add sequence numbers
 */
function build_monthly_deposit_index($deposits, $banco_id, $year, $month) {
    // Filter deposits for this specific bank/month/year
    $filtered = array_filter($deposits, function($deposit) use ($banco_id, $year, $month) {
        // Must match bank
        if ($deposit['banco_cuenta_id'] != $banco_id) return false;

        // Must be a deposit (positive amount)
        if ($deposit['deposit'] <= 0) return false;

        // Extract year and month from fecha
        $deposit_year = (int)date('Y', strtotime($deposit['fecha']));
        $deposit_month = (int)date('n', strtotime($deposit['fecha']));

        // Must match year and month
        return ($deposit_year === $year && $deposit_month === $month);
    });

    // Sort by fecha chronologically (ASC = oldest first)
    usort($filtered, function($a, $b) {
        return strtotime($a['fecha']) - strtotime($b['fecha']);
    });

    // Add 1-based sequence numbers
    $indexed = [];
    $sequence = 1;
    foreach ($filtered as $deposit) {
        $deposit['_sequence'] = $sequence;
        $indexed[] = $deposit;
        $sequence++;
    }

    return $indexed;
}

/**
 * Get deposit at specific sequence position in monthly index
 */
function get_deposit_by_sequence($monthly_index, $sequence) {
    $index = $sequence - 1;  // Convert 1-based to 0-based
    return isset($monthly_index[$index]) ? $monthly_index[$index] : null;
}

// ============================================
// START TESTS
// ============================================

echo "=== TIER -1 SEQUENCE MATCHING - UNIT TESTS ===\n\n";

// TEST 1: Parse Estado_de_Cuenta
echo "TEST 1: parse_estado_cuenta()\n";
echo str_repeat("-", 60) . "\n";

$test_cases = [
    ['ING 03 OCT 24 BBVA', true, 3, 10, 2024, 1, 'BBVA'],
    ['ING 13 MAR 24 SANTANDER', true, 13, 3, 2024, 2, 'SANTANDER'],
    ['ING 01 ENE 25 BBVA', true, 1, 1, 2025, 1, 'BBVA'],
    ['ING 25 DIC 24 SANTANDER', true, 25, 12, 2024, 2, 'SANTANDER'],
    ['Invalid formato', false, null, null, null, null, null],
    ['', false, null, null, null, null, null],
    ['ING 05 XYZ 24 BBVA', false, null, null, null, null, null],  // Invalid month
];

$passed = 0;
$failed = 0;

foreach ($test_cases as $idx => $test) {
    list($input, $should_parse, $exp_seq, $exp_month, $exp_year, $exp_bank_id, $exp_bank_name) = $test;

    $result = parse_estado_cuenta($input);
    $test_num = $idx + 1;

    echo "Test $test_num: '$input'\n";

    if ($should_parse) {
        if ($result &&
            $result['sequence'] === $exp_seq &&
            $result['month'] === $exp_month &&
            $result['year'] === $exp_year &&
            $result['bank_id'] === $exp_bank_id &&
            $result['bank_name'] === $exp_bank_name) {
            echo "  ✓ PASS - Seq={$result['sequence']}, Month={$result['month']}, Year={$result['year']}, Bank={$result['bank_name']}\n";
            $passed++;
        } else {
            echo "  ✗ FAIL - Got: " . json_encode($result) . "\n";
            echo "         Expected: Seq=$exp_seq, Month=$exp_month, Year=$exp_year, Bank=$exp_bank_name\n";
            $failed++;
        }
    } else {
        if ($result === null) {
            echo "  ✓ PASS - Correctly returned null\n";
            $passed++;
        } else {
            echo "  ✗ FAIL - Should have returned null, got: " . json_encode($result) . "\n";
            $failed++;
        }
    }
}

echo "\nTest 1 Results: $passed passed, $failed failed\n\n";

// TEST 2: Monthly deposit indexing
echo "TEST 2: build_monthly_deposit_index()\n";
echo str_repeat("-", 60) . "\n";

$deposits = [
    // BBVA October 2024 - intentionally out of order
    ['banco_cuenta_mov_id' => 'D', 'banco_cuenta_id' => 1, 'fecha' => '2024-10-22', 'deposit' => 7000],
    ['banco_cuenta_mov_id' => 'A', 'banco_cuenta_id' => 1, 'fecha' => '2024-10-03', 'deposit' => 2000],
    ['banco_cuenta_mov_id' => 'C', 'banco_cuenta_id' => 1, 'fecha' => '2024-10-15', 'deposit' => 5000],
    ['banco_cuenta_mov_id' => 'B', 'banco_cuenta_id' => 1, 'fecha' => '2024-10-08', 'deposit' => 4500],

    // Should be filtered out - wrong month
    ['banco_cuenta_mov_id' => 'Z1', 'banco_cuenta_id' => 1, 'fecha' => '2024-09-25', 'deposit' => 3000],
    ['banco_cuenta_mov_id' => 'Z2', 'banco_cuenta_id' => 1, 'fecha' => '2024-11-05', 'deposit' => 6000],

    // Should be filtered out - wrong bank
    ['banco_cuenta_mov_id' => 'Z3', 'banco_cuenta_id' => 2, 'fecha' => '2024-10-10', 'deposit' => 8000],

    // Should be filtered out - negative amount
    ['banco_cuenta_mov_id' => 'Z4', 'banco_cuenta_id' => 1, 'fecha' => '2024-10-12', 'deposit' => -100],
];

echo "Input: 8 deposits (4 valid BBVA Oct 2024, 4 should be filtered out)\n";
echo "Expected order: A (Oct 3), B (Oct 8), C (Oct 15), D (Oct 22)\n\n";

$index = build_monthly_deposit_index($deposits, 1, 2024, 10);

echo "Result: " . count($index) . " deposits indexed\n";

$test2_passed = true;
if (count($index) !== 4) {
    echo "  ✗ FAIL - Expected 4 deposits, got " . count($index) . "\n";
    $test2_passed = false;
}

$expected_order = ['A', 'B', 'C', 'D'];
foreach ($index as $i => $dep) {
    $exp_id = $expected_order[$i];
    $exp_seq = $i + 1;
    echo "  [$exp_seq] {$dep['banco_cuenta_mov_id']} - {$dep['fecha']} - Seq={$dep['_sequence']}";

    if ($dep['banco_cuenta_mov_id'] === $exp_id && $dep['_sequence'] === $exp_seq) {
        echo " ✓\n";
    } else {
        echo " ✗ (Expected ID=$exp_id, Seq=$exp_seq)\n";
        $test2_passed = false;
    }
}

echo $test2_passed ? "\n✓ TEST 2 PASSED\n\n" : "\n✗ TEST 2 FAILED\n\n";

// TEST 3: Get deposit by sequence
echo "TEST 3: get_deposit_by_sequence()\n";
echo str_repeat("-", 60) . "\n";

$test3_passed = true;

for ($seq = 1; $seq <= 6; $seq++) {
    $deposit = get_deposit_by_sequence($index, $seq);
    $expected_exists = ($seq <= 4);

    echo "Sequence $seq: ";

    if ($expected_exists) {
        $exp_id = $expected_order[$seq - 1];
        if ($deposit && $deposit['banco_cuenta_mov_id'] === $exp_id) {
            echo "{$deposit['banco_cuenta_mov_id']} ✓\n";
        } else {
            echo "✗ (Expected $exp_id, got " . ($deposit ? $deposit['banco_cuenta_mov_id'] : 'null') . ")\n";
            $test3_passed = false;
        }
    } else {
        if ($deposit === null) {
            echo "null ✓ (correctly not found)\n";
        } else {
            echo "✗ (Expected null, got {$deposit['banco_cuenta_mov_id']})\n";
            $test3_passed = false;
        }
    }
}

echo $test3_passed ? "\n✓ TEST 3 PASSED\n\n" : "\n✗ TEST 3 FAILED\n\n";

// SUMMARY
echo str_repeat("=", 60) . "\n";
echo "SUMMARY\n";
echo str_repeat("=", 60) . "\n";
echo "Test 1 (parse_estado_cuenta):        $passed/$test_num passed\n";
echo "Test 2 (build_monthly_deposit_index): " . ($test2_passed ? "PASS" : "FAIL") . "\n";
echo "Test 3 (get_deposit_by_sequence):     " . ($test3_passed ? "PASS" : "FAIL") . "\n";

$all_passed = ($failed === 0 && $test2_passed && $test3_passed);
echo "\n" . ($all_passed ? "✓✓✓ ALL TESTS PASSED ✓✓✓" : "✗✗✗ SOME TESTS FAILED ✗✗✗") . "\n";
